  -------------------------------------------------------------------------------
  -- axi_datamover_reset.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_reset.vhd
  --
  -- Description:     
  --   This file implements the DataMover Reset module.               
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library lib_cdc_v1_0_2;  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_reset is
    generic (
      
      C_STSCMD_IS_ASYNC    : Integer range  0 to  1 :=  0
        -- 0 = Use Synchronous Command/Statys User Interface
        -- 1 = Use Asynchronous Command/Statys User Interface
      );
    port (
      
      -- Primary Clock and Reset Inputs -----------------
                                                       --
      primary_aclk         : in  std_logic;            --
      primary_aresetn      : in  std_logic;            --
      ---------------------------------------------------
      
                                                       
      -- Async operation clock and reset from User ------
      -- Used for Command/Status User interface        --
      -- synchronization when C_STSCMD_IS_ASYNC = 1    --
                                                       --
      secondary_awclk      : in  std_logic;            --
      secondary_aresetn    : in  std_logic;            --
      ---------------------------------------------------
  
                                           
     
      -- Halt request input control -------------------------------
      halt_req             : in  std_logic;                      --
         -- Active high soft shutdown request (can be a pulse)   --
                                                                 --
      -- Halt Complete status flag                               --
      halt_cmplt           : Out std_logic;                      --
         -- Active high soft shutdown complete status            --
      -------------------------------------------------------------
      
       
                                                 
      -- Soft Shutdown internal interface ------------------------------------------------
                                                                                        --
      flush_stop_request   : Out std_logic;                                             --
         -- Active high soft stop request to modules                                    --
                                                                                        --
      data_cntlr_stopped   : in  std_logic;                                             --
         -- Active high flag indicating the data controller is flushed and stopped      --
                                                                                        --
      addr_cntlr_stopped   : in  std_logic;                                             --
         -- Active high flag indicating the address controller is flushed and stopped   --
                                                                                        --
      aux1_stopped         : in  std_logic;                                             --
         -- Active high flag flush complete for auxillary 1 module                      --
         -- Tie high if unused                                                          --
                                                                                        --
      aux2_stopped         : in  std_logic;                                             --
         -- Active high flag flush complete for auxillary 2 module                      --
         -- Tie high if unused                                                          --
      ------------------------------------------------------------------------------------
      
       
          
      -- HW Reset outputs to reset groups  -------------------------------------     
                                                                              --
      cmd_stat_rst_user    : Out std_logic;                                   --
         -- The reset to the Command/Status Module User interface side        --
                                                                              --
      cmd_stat_rst_int     : Out std_logic;                                   --
         -- The reset to the Command/Status Module internal interface side    --
                                                                              --
      mmap_rst             : Out std_logic;                                   --
         -- The reset to the Memory Map interface side                        --
                                                                              --
      stream_rst           : Out std_logic                                    --
         -- The reset to the Stream interface side                            --
      --------------------------------------------------------------------------
      
      );
  
  end entity axi_datamover_reset;
  
  
  architecture implementation of axi_datamover_reset is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

    constant MTBF_STAGES : integer := 4;
  
      --  ATTRIBUTE async_reg                      : STRING;
    -- Signals
    
    signal sig_cmd_stat_rst_user_n     : std_logic := '0';
    signal sig_cmd_stat_rst_user_reg_n_cdc_from : std_logic := '0';
    signal sig_cmd_stat_rst_int_reg_n  : std_logic := '0';
    signal sig_mmap_rst_reg_n          : std_logic := '0';
    signal sig_stream_rst_reg_n        : std_logic := '0';
    signal sig_syncd_sec_rst           : std_logic := '0';
    
    -- soft shutdown support
    signal sig_internal_reset          : std_logic := '0';
    signal sig_s_h_halt_reg            : std_logic := '0';
    signal sig_halt_cmplt              : std_logic := '0';
                               
    -- additional CDC synchronization signals
    signal sig_sec_neg_edge_plus_delay : std_logic := '0';
    signal sig_secondary_aresetn_reg   : std_logic := '0';
    signal sig_prim2sec_rst_reg1_n_cdc_to     : std_logic := '0';
    signal sig_prim2sec_rst_reg2_n     : std_logic := '0';
    
--  ATTRIBUTE async_reg OF sig_prim2sec_rst_reg1_n_cdc_to  : SIGNAL IS "true";
--  ATTRIBUTE async_reg OF sig_prim2sec_rst_reg2_n  : SIGNAL IS "true";
                               
                               
  begin --(architecture implementation)
  
  
    -- Assign outputs
   
    cmd_stat_rst_user <=   not(sig_cmd_stat_rst_user_n);
    
    cmd_stat_rst_int  <=   not(sig_cmd_stat_rst_int_reg_n) or
                           sig_syncd_sec_rst;
    
    mmap_rst          <=   not(sig_mmap_rst_reg_n) or
                           sig_syncd_sec_rst;
    
    stream_rst        <=   not(sig_stream_rst_reg_n) or
                           sig_syncd_sec_rst;
    
    
    
    
    
    
    
    -- Internal logic Implmentation
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_SYNC_CMDSTAT_RESET
    --
    -- If Generate Description:
    --  This IfGen assigns the reset for the 
    -- Synchronous Command/Status User interface case
    --
    ------------------------------------------------------------
    GEN_SYNC_CMDSTAT_RESET : if (C_STSCMD_IS_ASYNC = 0) generate
       
       begin
    
          sig_syncd_sec_rst       <= '0';
          
          sig_cmd_stat_rst_user_n <=  not(sig_cmd_stat_rst_user_reg_n_cdc_from);
                    
       end generate GEN_SYNC_CMDSTAT_RESET;
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ASYNC_CMDSTAT_RESET
    --
    -- If Generate Description:
    --  This IfGen assigns the reset for the 
    -- Asynchronous Command/Status User interface case
    --
    ------------------------------------------------------------
    GEN_ASYNC_CMDSTAT_RESET : if (C_STSCMD_IS_ASYNC = 1) generate
     --   ATTRIBUTE async_reg                      : STRING;
 
      signal sig_sec_reset_in_reg_n      : std_logic := '0';
      signal sig_secondary_aresetn_reg_tmp : std_logic := '0';
      
      -- Secondary reset pulse stretcher
      signal sig_secondary_dly1          : std_logic := '0';
      signal sig_secondary_dly2          : std_logic := '0';
      signal sig_neg_edge_detect         : std_logic := '0';
      signal sig_sec2prim_reset          : std_logic := '0';
      signal sig_sec2prim_reset_reg_cdc_tig      : std_logic := '0';
      signal sig_sec2prim_reset_reg2     : std_logic := '0';
      signal sig_sec2prim_rst_syncro1_cdc_tig    : std_logic := '0';
      signal sig_sec2prim_rst_syncro2    : std_logic := '0';
--  ATTRIBUTE async_reg OF sig_sec2prim_reset_reg_cdc_tig  : SIGNAL IS "true";
--  ATTRIBUTE async_reg OF sig_sec2prim_reset_reg2  : SIGNAL IS "true";
      
--  ATTRIBUTE async_reg OF sig_sec2prim_rst_syncro1_cdc_tig  : SIGNAL IS "true";
--  ATTRIBUTE async_reg OF sig_sec2prim_rst_syncro2  : SIGNAL IS "true";
                                 
      begin
         
        -- Generate the reset in the primary clock domain. Use the longer
        -- of the pulse stretched reset or the actual reset.
        sig_syncd_sec_rst <= sig_sec2prim_reset_reg2 or
                             sig_sec2prim_rst_syncro2;
  
        
        -- Check for falling edge of secondary_aresetn input
        sig_neg_edge_detect <=  '1'
          when (sig_sec_reset_in_reg_n = '1' and 
                secondary_aresetn      = '0')
          else '0';
          

        
         
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_PUSE_STRETCH_FLOPS
        --
        -- Process Description:
        -- This process implements a 3 clock wide pulse whenever the 
        -- secondary reset is asserted
        --
        -------------------------------------------------------------
        IMP_PUSE_STRETCH_FLOPS : process (secondary_awclk)
          begin
            if (secondary_awclk'event and secondary_awclk = '1') then
            
              If (sig_secondary_dly2 = '1') Then
                
                sig_secondary_dly1 <= '0' ;
                sig_secondary_dly2 <= '0' ;
            
              Elsif (sig_neg_edge_detect = '1') Then
                
                sig_secondary_dly1 <= '1';
            
              else

                sig_secondary_dly2 <= sig_secondary_dly1 ;
              
              End if;
            
            
            
            
            end if;       
          end process IMP_PUSE_STRETCH_FLOPS; 
        
        
        
   --  CDC add     
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: SYNC_NEG_EDGE
        --
        -- Process Description:
        --  First (source clock) stage synchronizer for CDC of 
        -- negative edge detection,
        --
        -------------------------------------------------------------
        SYNC_NEG_EDGE : process (secondary_awclk)
          begin
            if (secondary_awclk'event and secondary_awclk = '1') then
            
              sig_sec_neg_edge_plus_delay <= sig_neg_edge_detect or
                                             sig_secondary_dly1  or
                                             sig_secondary_dly2;
            
            end if;       
          end process SYNC_NEG_EDGE; 
        
  --      
         
         
         
          
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: SEC2PRIM_RST_SYNCRO
        --
        -- Process Description:
        --    This process registers the secondary reset input to 
        -- the primary clock domain.
        --
        -------------------------------------------------------------
SEC2PRIM_RST_SYNCRO : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => sig_sec_neg_edge_plus_delay,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => primary_aclk,
        scndry_resetn              => '0',
        scndry_out                 => sig_sec2prim_reset_reg2,
        scndry_vect_out            => open
    );


--        SEC2PRIM_RST_SYNCRO : process (primary_aclk)
--           begin
--             if (primary_aclk'event and primary_aclk = '1') then
--             
--               
--               sig_sec2prim_reset_reg_cdc_tig   <=  sig_sec_neg_edge_plus_delay ;
--               
--               sig_sec2prim_reset_reg2  <=  sig_sec2prim_reset_reg_cdc_tig;
--                            
--             end if;       
--           end process SEC2PRIM_RST_SYNCRO; 
        
       
       
 
 
 
 
 
 
   --  CDC add     
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: REG_SEC_RST
        --
        -- Process Description:
        --  First (source clock) stage synchronizer for CDC of 
        -- secondary reset input,
        --
        -------------------------------------------------------------
        REG_SEC_RST : process (secondary_awclk)
          begin
            if (secondary_awclk'event and secondary_awclk = '1') then
            
              sig_secondary_aresetn_reg <= secondary_aresetn;
            
            end if;       
          end process REG_SEC_RST; 
        
  --      
         
         
         
         
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: SEC2PRIM_RST_SYNCRO_2
        --
        -- Process Description:
        --    Second stage (destination) synchronizers for the secondary
        -- reset CDC to the primary clock.
        --
        -------------------------------------------------------------

         sig_secondary_aresetn_reg_tmp <= not(sig_secondary_aresetn_reg);

SEC2PRIM_RST_SYNCRO_2 : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => sig_secondary_aresetn_reg_tmp,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => primary_aclk,
        scndry_resetn              => '0',
        scndry_out                 => sig_sec2prim_rst_syncro2,
        scndry_vect_out            => open
    );


--        SEC2PRIM_RST_SYNCRO_2 : process (primary_aclk)
--           begin
--             if (primary_aclk'event and primary_aclk = '1') then
--             
--               
--      -- CDC   sig_sec2prim_rst_syncro1_cdc_tig  <= not(secondary_aresetn);
--               sig_sec2prim_rst_syncro1_cdc_tig  <= not(sig_secondary_aresetn_reg);
--               sig_sec2prim_rst_syncro2  <= sig_sec2prim_rst_syncro1_cdc_tig;
-- 
-- 
--             end if;       
--           end process SEC2PRIM_RST_SYNCRO_2; 
        
       
         
         
         -- Generate the Command and Status side reset
         sig_cmd_stat_rst_user_n <= sig_sec_reset_in_reg_n and
                                    sig_prim2sec_rst_reg2_n;
    -- CDC                          sig_cmd_stat_rst_user_reg_n_cdc_from;
   
   
    
    
         
         -------------------------------------------------------------
         -- Synchronous Process with Sync Reset
         --
         -- Label: REG_RESET_ASYNC
         --
         -- Process Description:
         --    This process registers the secondary reset input to 
         -- generate the Command/Status User interface reset.
         --
         -------------------------------------------------------------
         REG_RESET_ASYNC : process (secondary_awclk)
            begin
              if (secondary_awclk'event and secondary_awclk = '1') then
              
                 sig_sec_reset_in_reg_n <= secondary_aresetn;
              
              end if;       
            end process REG_RESET_ASYNC; 
         
   
   
   
   
   --  CDC add     
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: SYNC_PRIM2SEC_RST
        --
        -- Process Description:
        --  Second (destination clock) stage synchronizers for CDC of 
        -- primary reset input,
        --
        -------------------------------------------------------------

SYNC_PRIM2SEC_RST : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => sig_cmd_stat_rst_user_reg_n_cdc_from,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => secondary_awclk,
        scndry_resetn              => '0',
        scndry_out                 => sig_prim2sec_rst_reg2_n,
        scndry_vect_out            => open
    );


--        SYNC_PRIM2SEC_RST : process (secondary_awclk)
--          begin
--            if (secondary_awclk'event and secondary_awclk = '1') then
--            
--              sig_prim2sec_rst_reg1_n_cdc_to <= sig_cmd_stat_rst_user_reg_n_cdc_from;
--              sig_prim2sec_rst_reg2_n <= sig_prim2sec_rst_reg1_n_cdc_to;
--            
--            end if;       
--          end process SYNC_PRIM2SEC_RST; 
        
  --      
         
         
        
        
      end generate GEN_ASYNC_CMDSTAT_RESET;
 
 
 
  
  
  
  
  
  
  
  
 
 
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_CMDSTAT_PRIM_RESET
    --
    -- Process Description:
    --    This process registers the primary reset input to 
    -- generate the Command/Status User interface reset.
    --
    -------------------------------------------------------------
    REG_CMDSTAT_PRIM_RESET : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
         
            sig_cmd_stat_rst_user_reg_n_cdc_from <= primary_aresetn;
         
         end if;       
       end process REG_CMDSTAT_PRIM_RESET; 
    
   
   
 
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_CMDSTAT_INT_RESET
    --
    -- Process Description:
    --    This process registers the primary reset input to 
    -- generate the Command/Status internal interface reset.
    --
    -------------------------------------------------------------
    REG_CMDSTAT_INT_RESET : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
         
            sig_cmd_stat_rst_int_reg_n <= primary_aresetn;
         
         end if;       
       end process REG_CMDSTAT_INT_RESET; 
    
   
   
 
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_MMAP_RESET
    --
    -- Process Description:
    --    This process registers the primary reset input to 
    -- generate the Memory Map interface reset.
    --
    -------------------------------------------------------------
    REG_MMAP_RESET : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
         
            sig_mmap_rst_reg_n <= primary_aresetn;
         
         end if;       
       end process REG_MMAP_RESET; 
    
   
   
 
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_STREAM_RESET
    --
    -- Process Description:
    --    This process registers the primary reset input to 
    -- generate the Stream interface reset.
    --
    -------------------------------------------------------------
    REG_STREAM_RESET : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
         
            sig_stream_rst_reg_n <= primary_aresetn;
         
         end if;       
       end process REG_STREAM_RESET; 
    
  
  
  
  
  
  
  
  
  -- Soft Shutdown logic ------------------------------------------------------
  
    
    
    sig_internal_reset  <= not(sig_cmd_stat_rst_int_reg_n) or
                           sig_syncd_sec_rst;
    
    
    flush_stop_request  <= sig_s_h_halt_reg;
    
    
    halt_cmplt          <= sig_halt_cmplt;
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_HALT_REQ
    --
    -- Process Description:
    --  Implements a sample and hold flop for the halt request 
    -- input. Can only be cleared on a HW reset.
    --
    -------------------------------------------------------------
    REG_HALT_REQ : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_internal_reset = '1') then
              
              sig_s_h_halt_reg <= '0';
            
            elsif (halt_req = '1') then
              
              sig_s_h_halt_reg <= '1';
            
            else
              null;  -- hold current state
            end if; 
         end if;       
       end process REG_HALT_REQ; 
     
     
  
  
  
  
  
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_CMPLT
    --
    -- Process Description:
    --  Implements a the flop for the halt complete status 
    -- output. Can only be cleared on a HW reset.
    --
    -------------------------------------------------------------
    IMP_HALT_CMPLT : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_internal_reset = '1') then
              
              sig_halt_cmplt <= '0';
            
            elsif (data_cntlr_stopped = '1' and
                   addr_cntlr_stopped = '1' and
                   aux1_stopped       = '1' and
                   aux2_stopped       = '1') then
              
              sig_halt_cmplt <= '1';
            
            else
              null;  -- hold current state
            end if; 
         end if;       
       end process IMP_HALT_CMPLT; 
     
     
  
          
          
  
    
  end implementation;


-------------------------------------------------------------------------------
-- axi_datamover_afifo_autord.vhd - entity/architecture pair

-------------------------------------------------------------------------------
--
-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_datamover_afifo_autord.vhd
-- Version:         initial
-- Description:     
--    This file contains the logic to generate a CoreGen call to create a
-- asynchronous FIFO as part of the synthesis process of XST. This eliminates
-- the need for multiple fixed netlists for various sizes and widths of FIFOs. 
-- 
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

library lib_fifo_v1_0_8;
use lib_fifo_v1_0_8.async_fifo_fg;

-----------------------------------------------------------------------------
-- Entity section
-----------------------------------------------------------------------------

entity axi_datamover_afifo_autord is
  generic (
     C_DWIDTH        : integer := 32;
       -- Sets the width of the FIFO Data
       
     C_DEPTH         : integer := 16;
       -- Sets the depth of the FIFO
       
     C_CNT_WIDTH     : Integer := 5;
       -- Sets the width of the FIFO Data Count output
       
     C_USE_BLKMEM    : Integer := 1 ;
       -- Sets the type of memory to use for the FIFO
       -- 0 = Distributed Logic
       -- 1 = Block Ram
       
     C_FAMILY        : String  := "virtex7"
       -- Specifies the target FPGA Family
       
    );
  port (
    -- FIFO Inputs --------------------------------------------------------------
     AFIFO_Ainit                : In  std_logic;                               --
     AFIFO_Ainit_Rd_clk         : In  std_logic;                               --
     AFIFO_Wr_clk               : In  std_logic;                               --
     AFIFO_Wr_en                : In  std_logic;                               --
     AFIFO_Din                  : In  std_logic_vector(C_DWIDTH-1 downto 0);   --
     AFIFO_Rd_clk               : In  std_logic;                               --
     AFIFO_Rd_en                : In  std_logic;                               --
     AFIFO_Clr_Rd_Data_Valid    : In  std_logic;                               --
     ----------------------------------------------------------------------------
     
    -- FIFO Outputs --------------------------------------------------------------
     AFIFO_DValid               : Out std_logic;                                --
     AFIFO_Dout                 : Out std_logic_vector(C_DWIDTH-1 downto 0);    --
     AFIFO_Full                 : Out std_logic;                                --
     AFIFO_Empty                : Out std_logic;                                --
     AFIFO_Almost_full          : Out std_logic;                                --
     AFIFO_Almost_empty         : Out std_logic;                                --
     AFIFO_Wr_count             : Out std_logic_vector(C_CNT_WIDTH-1 downto 0); --
     AFIFO_Rd_count             : Out std_logic_vector(C_CNT_WIDTH-1 downto 0); --
     AFIFO_Corr_Rd_count        : Out std_logic_vector(C_CNT_WIDTH downto 0);   --
     AFIFO_Corr_Rd_count_minus1 : Out std_logic_vector(C_CNT_WIDTH downto 0);   --
     AFIFO_Rd_ack               : Out std_logic                                 --
     -----------------------------------------------------------------------------
     
    );
end entity axi_datamover_afifo_autord;

 
-----------------------------------------------------------------------------
-- Architecture section
-----------------------------------------------------------------------------

architecture imp of axi_datamover_afifo_autord is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of imp : architecture is "yes";

   constant MTBF_STAGES : integer := 4;
   constant C_FIFO_MTBF : integer := 4;
-- Constant declarations
  -- none 
  ATTRIBUTE async_reg                      : STRING;
-- Signal declarations
   signal write_data_lil_end       : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
   signal read_data_lil_end        : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');

   signal wr_count_lil_end         : std_logic_vector(C_CNT_WIDTH-1 downto 0) := (others => '0');
   signal rd_count_lil_end         : std_logic_vector(C_CNT_WIDTH-1 downto 0) := (others => '0');
   signal rd_count_int             : natural   :=  0;
   signal rd_count_int_corr        : natural   :=  0;
   signal rd_count_int_corr_minus1 : natural   :=  0;
   Signal corrected_empty          : std_logic := '0';
   Signal corrected_almost_empty   : std_logic := '0';
   Signal sig_afifo_empty          : std_logic := '0';
   Signal sig_afifo_almost_empty   : std_logic := '0';
  
 -- backend fifo read ack sample and hold
   Signal sig_rddata_valid         : std_logic := '0';
   Signal hold_ff_q                : std_logic := '0';
   Signal ored_ack_ff_reset        : std_logic := '0';
   Signal autoread                 : std_logic := '0';
   Signal sig_wrfifo_rdack         : std_logic := '0';
   Signal fifo_read_enable         : std_logic := '0';

   signal AFIFO_Ainit_d2_cdc_tig : std_logic;
   signal AFIFO_Ainit_d2 : std_logic;
--     ATTRIBUTE async_reg OF AFIFO_Ainit_d2_cdc_tig  : SIGNAL IS "true";
--  ATTRIBUTE async_reg OF AFIFO_Ainit_d2  : SIGNAL IS "true";
   
 
-----------------------------------------------------------------------------
-- Begin architecture
-----------------------------------------------------------------------------
begin  

 -- Bit ordering translations
       
    write_data_lil_end   <=  AFIFO_Din;  -- translate from Big Endian to little
                                         -- endian.
    AFIFO_Rd_ack         <= sig_wrfifo_rdack; 
       
    AFIFO_Dout           <= read_data_lil_end;  -- translate from Little Endian to 
                                                -- Big endian.                            
 
    AFIFO_Almost_empty   <= corrected_almost_empty;
 
    AFIFO_Empty          <= corrected_empty;
 
    AFIFO_Wr_count       <= wr_count_lil_end;   
   
    AFIFO_Rd_count       <= rd_count_lil_end;                       
   
    AFIFO_Corr_Rd_count  <= CONV_STD_LOGIC_VECTOR(rd_count_int_corr, 
                                                  C_CNT_WIDTH+1);                       
   
    AFIFO_Corr_Rd_count_minus1 <= CONV_STD_LOGIC_VECTOR(rd_count_int_corr_minus1, 
                                                        C_CNT_WIDTH+1);                                          
                                                                        
    AFIFO_DValid         <= sig_rddata_valid; -- Output data valid indicator
   
   
    fifo_read_enable     <= AFIFO_Rd_en or autoread;
       

                     
   -------------------------------------------------------------------------------
   -- Instantiate the CoreGen FIFO
   --
   -- NOTE:
   -- This instance refers to a wrapper file that interm will use the 
   -- CoreGen FIFO Generator Async FIFO utility.
   --
   -------------------------------------------------------------------------------
    I_ASYNC_FIFOGEN_FIFO : entity lib_fifo_v1_0_8.async_fifo_fg 
       generic map (
          C_ALLOW_2N_DEPTH      =>  1 , 
          C_FAMILY              =>  C_FAMILY,
          C_DATA_WIDTH          =>  C_DWIDTH, 
          C_ENABLE_RLOCS        =>  0, 
          C_FIFO_DEPTH          =>  C_DEPTH, 
          C_HAS_ALMOST_EMPTY    =>  1, 
          C_HAS_ALMOST_FULL     =>  1, 
          C_HAS_RD_ACK          =>  1, 
          C_HAS_RD_COUNT        =>  1, 
          C_HAS_RD_ERR          =>  0, 
          C_HAS_WR_ACK          =>  0, 
          C_HAS_WR_COUNT        =>  1, 
          C_EN_SAFETY_CKT       =>  1, 
          C_HAS_WR_ERR          =>  0, 
          C_RD_ACK_LOW          =>  0, 
          C_RD_COUNT_WIDTH      =>  C_CNT_WIDTH, 
          C_RD_ERR_LOW          =>  0, 
          C_USE_BLOCKMEM        =>  C_USE_BLKMEM, 
          C_WR_ACK_LOW          =>  0, 
          C_WR_COUNT_WIDTH      =>  C_CNT_WIDTH, 
          C_WR_ERR_LOW          =>  0,
          C_SYNCHRONIZER_STAGE  =>  C_FIFO_MTBF,
          C_USE_EMBEDDED_REG    =>  0 -- 0 ;
   --       C_PRELOAD_REGS        =>  0, -- 0 ;
   --       C_PRELOAD_LATENCY     =>  1  -- 1 ;
         )
      port Map (
         Din                 =>  write_data_lil_end,     
         Wr_en               =>  AFIFO_Wr_en,            
         Wr_clk              =>  AFIFO_Wr_clk,           
         Rd_en               =>  fifo_read_enable,       
         Rd_clk              =>  AFIFO_Rd_clk,           
         Ainit               =>  AFIFO_Ainit,            
         Dout                =>  read_data_lil_end,      
         Full                =>  AFIFO_Full,             
         Empty               =>  sig_afifo_empty,        
         Almost_full         =>  AFIFO_Almost_full,      
         Almost_empty        =>  sig_afifo_almost_empty, 
         Wr_count            =>  wr_count_lil_end,       
         Rd_count            =>  rd_count_lil_end,       
         Rd_ack              =>  sig_wrfifo_rdack,       
         Rd_err              =>  open,                   
         Wr_ack              =>  open,                   
         Wr_err              =>  open                    
        );                


   ----------------------------------------------------------------------------                              
   -- Read Ack assert & hold logic (needed because:
   --     1) The Async FIFO has to be read once to get valid
   --        data to the read data port (data is discarded). 
   --     2) The Read ack from the fifo is only asserted for 1 clock.
   --     3) A signal is needed that indicates valid data is at the read
   --        port of the FIFO and has not yet been read. This signal needs
   --        to be held until the next read operation occurs or a clear
   --        signal is received.
      
    
--    ored_ack_ff_reset  <=  fifo_read_enable or 
--                           AFIFO_Ainit_Rd_clk or
--                           AFIFO_Clr_Rd_Data_Valid;
--    
--    sig_rddata_valid   <=  hold_ff_q or 
--                           sig_wrfifo_rdack;
-- 
 
   
     ored_ack_ff_reset <= '1'                     
      when (fifo_read_enable = '1' or  
            AFIFO_Ainit_Rd_clk = '1' or  
            AFIFO_Clr_Rd_Data_Valid = '1')      
      Else '0';
      
    sig_rddata_valid <= '1'                     
      when (hold_ff_q = '1' or  
            sig_wrfifo_rdack = '1')      
      Else '0';
      
           
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ACK_HOLD_FLOP
    --
    -- Process Description:
    --  Flop for registering the hold flag
    --
    -------------------------------------------------------------
--IMP_SYNC_FLOP : entity  proc_common_v4_0_2.cdc_sync
--    generic map (
--        C_CDC_TYPE                 => 1,
--        C_RESET_STATE              => 0,
--        C_SINGLE_BIT               => 1,
--        C_VECTOR_WIDTH             => 32,
--        C_MTBF_STAGES              => MTBF_STAGES
--    )
--    port map (
--        prmry_aclk                 => '0',
--        prmry_resetn               => '0',
--        prmry_in                   => AFIFO_Ainit,
--        prmry_vect_in              => (others => '0'),

--        scndry_aclk                => AFIFO_Rd_clk,
--        scndry_resetn              => '0',
--        scndry_out                 => AFIFO_Ainit_d2,
--        scndry_vect_out            => open
--    );



--    IMP_SYNC_FLOP : process (AFIFO_Rd_clk)
--       begin
--         if (AFIFO_Rd_clk'event and AFIFO_Rd_clk = '1') then
--            AFIFO_Ainit_d2_cdc_tig <= AFIFO_Ainit; 
--            AFIFO_Ainit_d2 <= AFIFO_Ainit_d2_cdc_tig; 
--         end if;       
--       end process IMP_SYNC_FLOP; 

    IMP_ACK_HOLD_FLOP : process (AFIFO_Rd_clk)
       begin
         if (AFIFO_Rd_clk'event and AFIFO_Rd_clk = '1') then
           if (ored_ack_ff_reset = '1') then
             hold_ff_q  <= '0';
           else
             hold_ff_q  <= sig_rddata_valid;
           end if; 
         end if;       
       end process IMP_ACK_HOLD_FLOP; 
    
    
    
  -- generate auto-read enable. This keeps fresh data at the output
  -- of the FIFO whenever it is available.
    autoread <= '1'                     -- create a read strobe when the 
      when (sig_rddata_valid = '0' and  -- output data is NOT valid
            sig_afifo_empty = '0')      -- and the FIFO is not empty
      Else '0';
      
    
    rd_count_int <=  CONV_INTEGER(rd_count_lil_end);
    
    
    -------------------------------------------------------------
    -- Combinational Process
    --
    -- Label: CORRECT_RD_CNT
    --
    -- Process Description:
    --  This process corrects the FIFO Read Count output for the
    -- auto read function.
    --
    -------------------------------------------------------------
    CORRECT_RD_CNT : process (sig_rddata_valid,
                              sig_afifo_empty ,
                              sig_afifo_almost_empty,
                              rd_count_int)
       begin
    
          if (sig_rddata_valid = '0') then

             rd_count_int_corr        <= 0;
             rd_count_int_corr_minus1 <= 0;
             corrected_empty          <= '1';
             corrected_almost_empty   <= '0';
             
          elsif (sig_afifo_empty = '1') then         -- rddata valid and fifo empty
             
             rd_count_int_corr        <= 1;
             rd_count_int_corr_minus1 <= 0;
             corrected_empty          <= '0';
             corrected_almost_empty   <= '1';
          
          Elsif (sig_afifo_almost_empty = '1') Then  -- rddata valid and fifo almost empty
             
             rd_count_int_corr        <= 2;
             rd_count_int_corr_minus1 <= 1;
             corrected_empty          <= '0';
             corrected_almost_empty   <= '0';
          
          else                                       -- rddata valid and modify rd count from FIFO 
             
             rd_count_int_corr        <= rd_count_int+1;
             rd_count_int_corr_minus1 <= rd_count_int;
             corrected_empty          <= '0';
             corrected_almost_empty   <= '0';
          
          end if;
    
       end process CORRECT_RD_CNT; 
    


end imp;


-------------------------------------------------------------------------------
-- axi_datamover_sfifo_autord.vhd - entity/architecture pair

-------------------------------------------------------------------------------
--
-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_datamover_sfifo_autord.vhd
-- Version:         initial
-- Description:     
--    This file contains the logic to generate a CoreGen call to create a
-- synchronous FIFO as part of the synthesis process of XST. This eliminates
-- the need for multiple fixed netlists for various sizes and widths of FIFOs. 
-- 
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

library lib_fifo_v1_0_8;
use lib_fifo_v1_0_8.sync_fifo_fg;


-------------------------------------------------------------------------------

entity axi_datamover_sfifo_autord is
  generic (
     C_DWIDTH                : integer := 32;
       -- Sets the width of the FIFO Data
       
     C_DEPTH                 : integer := 128;
       -- Sets the depth of the FIFO
       
     C_DATA_CNT_WIDTH        : integer := 8;
       -- Sets the width of the FIFO Data Count output
       
     C_NEED_ALMOST_EMPTY     : Integer range 0 to 1 := 0;
       -- Indicates the need for an almost empty flag from the internal FIFO
     
     C_NEED_ALMOST_FULL      : Integer range 0 to 1 := 0;
       -- Indicates the need for an almost full flag from the internal FIFO
     
     C_USE_BLKMEM            : Integer range 0 to 1 := 1;
       -- Sets the type of memory to use for the FIFO
       -- 0 = Distributed Logic
       -- 1 = Block Ram
       
     C_FAMILY                : String  := "virtex7"
       -- Specifies the target FPGA Family
       
    );
  port (
    
    -- FIFO Inputs ------------------------------------------------------------------
     SFIFO_Sinit             : In  std_logic;                                      --
     SFIFO_Clk               : In  std_logic;                                      --
     SFIFO_Wr_en             : In  std_logic;                                      --
     SFIFO_Din               : In  std_logic_vector(C_DWIDTH-1 downto 0);          --
     SFIFO_Rd_en             : In  std_logic;                                      --
     SFIFO_Clr_Rd_Data_Valid : In  std_logic;                                      --
     --------------------------------------------------------------------------------
     
    -- FIFO Outputs -----------------------------------------------------------------
     SFIFO_DValid            : Out std_logic;                                      --
     SFIFO_Dout              : Out std_logic_vector(C_DWIDTH-1 downto 0);          --
     SFIFO_Full              : Out std_logic;                                      --
     SFIFO_Empty             : Out std_logic;                                      --
     SFIFO_Almost_full       : Out std_logic;                                      --
     SFIFO_Almost_empty      : Out std_logic;                                      --
     SFIFO_Rd_count          : Out std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0);  --
     SFIFO_Rd_count_minus1   : Out std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0);  --
     SFIFO_Wr_count          : Out std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0);  --
     SFIFO_Rd_ack            : Out std_logic                                       --
     --------------------------------------------------------------------------------
     
    );
end entity axi_datamover_sfifo_autord;

-----------------------------------------------------------------------------
-- Architecture section
-----------------------------------------------------------------------------

architecture imp of axi_datamover_sfifo_autord is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of imp : architecture is "yes";


-- Constant declarations

   -- none
 
-- Signal declarations

   signal write_data_lil_end         : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
   signal read_data_lil_end          : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
   signal raw_data_cnt_lil_end       : std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0) := (others => '0');
   signal raw_data_count_int         : natural := 0;
   signal raw_data_count_corr        : std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0) := (others => '0');
   signal raw_data_count_corr_minus1 : std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0) := (others => '0');
   Signal corrected_empty            : std_logic := '0';
   Signal corrected_almost_empty     : std_logic := '0';
   Signal sig_SFIFO_empty            : std_logic := '0';
  
   -- backend fifo read ack sample and hold
   Signal sig_rddata_valid           : std_logic := '0';
   Signal hold_ff_q                  : std_logic := '0';
   Signal ored_ack_ff_reset          : std_logic := '0';
   Signal autoread                   : std_logic := '0';
   Signal sig_sfifo_rdack            : std_logic := '0';
   Signal fifo_read_enable           : std_logic := '0';
   

 
begin  

 -- Bit ordering translations
       
    write_data_lil_end    <=  SFIFO_Din;  -- translate from Big Endian to little
                                          -- endian.
       
    SFIFO_Dout            <= read_data_lil_end;  -- translate from Little Endian to 
                                                 -- Big endian.                            
 
 
 -- Other port usages and assignments
    SFIFO_Rd_ack          <= sig_sfifo_rdack; 
 
    SFIFO_Almost_empty    <= corrected_almost_empty;
 
    SFIFO_Empty           <= corrected_empty;
 
    SFIFO_Wr_count        <= raw_data_cnt_lil_end;   
                                                    
    
    SFIFO_Rd_count        <= raw_data_count_corr;    
                                                                    
   
    SFIFO_Rd_count_minus1 <= raw_data_count_corr_minus1;    
        
        
        
    SFIFO_DValid          <= sig_rddata_valid; -- Output data valid indicator
   
NON_BLK_MEM : if (C_USE_BLKMEM = 0)
  generate   
    fifo_read_enable      <= SFIFO_Rd_en or autoread;
       
    ------------------------------------------------------------
    -- Instance: I_SYNC_FIFOGEN_FIFO 
    --
    -- Description:
    --  Instance for the synchronous fifo from proc common.   
    --
    ------------------------------------------------------------
    I_SYNC_FIFOGEN_FIFO : entity lib_fifo_v1_0_8.sync_fifo_fg 
      generic map(
         C_FAMILY             =>  C_FAMILY,        -- requred for FIFO Gen       
         C_DCOUNT_WIDTH       =>  C_DATA_CNT_WIDTH,     
         C_ENABLE_RLOCS       =>  0,                    
         C_HAS_DCOUNT         =>  1,                    
         C_HAS_RD_ACK         =>  1,                    
         C_HAS_RD_ERR         =>  0,                    
         C_HAS_WR_ACK         =>  1,                    
         C_HAS_WR_ERR         =>  0,                    
         C_MEMORY_TYPE        =>  C_USE_BLKMEM,         
         C_PORTS_DIFFER       =>  0,                    
         C_RD_ACK_LOW         =>  0,                    
         C_READ_DATA_WIDTH    =>  C_DWIDTH,             
         C_READ_DEPTH         =>  C_DEPTH,              
         C_RD_ERR_LOW         =>  0,                    
         C_WR_ACK_LOW         =>  0,                    
         C_WR_ERR_LOW         =>  0,                    
         C_WRITE_DATA_WIDTH   =>  C_DWIDTH,             
         C_WRITE_DEPTH        =>  C_DEPTH
     --    C_PRELOAD_REGS       =>  0, -- 1 = first word fall through
     --    C_PRELOAD_LATENCY    =>  1 -- 0 = first word fall through
     --    C_USE_EMBEDDED_REG   =>  1  -- 0 ;
         )
      port map(  
         Clk                  =>  SFIFO_Clk,            
         Sinit                =>  SFIFO_Sinit,          
         Din                  =>  write_data_lil_end,   
         Wr_en                =>  SFIFO_Wr_en,          
         Rd_en                =>  fifo_read_enable,     
         Dout                 =>  read_data_lil_end,    
         Almost_full          =>  open,
         Full                 =>  SFIFO_Full,           
         Empty                =>  sig_SFIFO_empty,      
         Rd_ack               =>  sig_sfifo_rdack,      
         Wr_ack               =>  open,                 
         Rd_err               =>  open,                 
         Wr_err               =>  open,                 
         Data_count           =>  raw_data_cnt_lil_end  
        );  
end generate NON_BLK_MEM;                     

      
BLK_MEM : if (C_USE_BLKMEM = 1)
  generate   
    fifo_read_enable      <= SFIFO_Rd_en; -- or autoread;
       
    ------------------------------------------------------------
    -- Instance: I_SYNC_FIFOGEN_FIFO 
    --
    -- Description:
    --  Instance for the synchronous fifo from proc common.   
    --
    ------------------------------------------------------------
    I_SYNC_FIFOGEN_FIFO : entity lib_fifo_v1_0_8.sync_fifo_fg 
      generic map(
         C_FAMILY             =>  C_FAMILY,        -- requred for FIFO Gen       
         C_DCOUNT_WIDTH       =>  C_DATA_CNT_WIDTH,     
         C_ENABLE_RLOCS       =>  0,                    
         C_HAS_DCOUNT         =>  1,                    
         C_HAS_RD_ACK         =>  1,                    
         C_HAS_RD_ERR         =>  0,                    
         C_HAS_WR_ACK         =>  1,                    
         C_HAS_WR_ERR         =>  0,                    
         C_MEMORY_TYPE        =>  C_USE_BLKMEM,         
         C_PORTS_DIFFER       =>  0,                    
         C_RD_ACK_LOW         =>  0,                    
         C_READ_DATA_WIDTH    =>  C_DWIDTH,             
         C_READ_DEPTH         =>  C_DEPTH,              
         C_RD_ERR_LOW         =>  0,                    
         C_WR_ACK_LOW         =>  0,                    
         C_WR_ERR_LOW         =>  0,                    
         C_WRITE_DATA_WIDTH   =>  C_DWIDTH,             
         C_WRITE_DEPTH        =>  C_DEPTH,
         C_PRELOAD_REGS       =>  1, -- 1 = first word fall through
         C_PRELOAD_LATENCY    =>  0, -- 0 = first word fall through
         C_USE_EMBEDDED_REG   =>  1  -- 0 ;
         )
      port map(  
         Clk                  =>  SFIFO_Clk,            
         Sinit                =>  SFIFO_Sinit,          
         Din                  =>  write_data_lil_end,   
         Wr_en                =>  SFIFO_Wr_en,          
         Rd_en                =>  fifo_read_enable,     
         Dout                 =>  read_data_lil_end,    
         Almost_full          =>  open,
         Full                 =>  SFIFO_Full,           
         Empty                =>  sig_SFIFO_empty,      
         Rd_ack               =>  sig_sfifo_rdack,      
         Wr_ack               =>  open,                 
         Rd_err               =>  open,                 
         Wr_err               =>  open,                 
         Data_count           =>  raw_data_cnt_lil_end  
        );  
end generate BLK_MEM;                     
    
    
    
         
    
   -------------------------------------------------------------------------------




    
                                 
   -------------------------------------------------------------------------------
   -- Read Ack assert & hold logic Needed because....
   -------------------------------------------------------------------------------
   --     1) The CoreGen Sync FIFO has to be read once to get valid
   --        data to the read data port. 
   --     2) The Read ack from the fifo is only asserted for 1 clock.
   --     3) A signal is needed that indicates valid data is at the read
   --        port of the FIFO and has not yet been used. This signal needs
   --        to be held until the next read operation occurs or a clear
   --        signal is received.
      
    
    ored_ack_ff_reset  <=  fifo_read_enable or 
                           SFIFO_Sinit or
                           SFIFO_Clr_Rd_Data_Valid;
    
    sig_rddata_valid   <=  hold_ff_q or 
                           sig_sfifo_rdack;
 
 
   
            
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ACK_HOLD_FLOP
    --
    -- Process Description:
    --  Flop for registering the hold flag
    --
    -------------------------------------------------------------
    IMP_ACK_HOLD_FLOP : process (SFIFO_Clk)
       begin
         if (SFIFO_Clk'event and SFIFO_Clk = '1') then
           if (ored_ack_ff_reset = '1') then
             hold_ff_q  <= '0';
           else
             hold_ff_q  <= sig_rddata_valid;
           end if; 
         end if;       
       end process IMP_ACK_HOLD_FLOP; 
    
    
    
    -- generate auto-read enable. This keeps fresh data at the output
    -- of the FIFO whenever it is available.
    autoread <= '1'                     -- create a read strobe when the 
      when (sig_rddata_valid = '0' and  -- output data is NOT valid
            sig_SFIFO_empty = '0')      -- and the FIFO is not empty
      Else '0';
      
    
    raw_data_count_int <=  CONV_INTEGER(raw_data_cnt_lil_end);
    
    
 
 
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: INCLUDE_ALMOST_EMPTY
    --
    -- If Generate Description:
    --  This IFGen corrects the FIFO Read Count output for the
    --  auto read function and includes the generation of the
    --  Almost_Empty flag.
    --
    ------------------------------------------------------------
    INCLUDE_ALMOST_EMPTY : if (C_NEED_ALMOST_EMPTY = 1) generate
    
       -- local signals
       
          Signal raw_data_count_int_corr        : integer := 0;
          Signal raw_data_count_int_corr_minus1 : integer := 0;
       
       begin
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: CORRECT_RD_CNT_IAE
         --
         -- Process Description:
         --  This process corrects the FIFO Read Count output for the
         --  auto read function and includes the generation of the
         --  Almost_Empty flag.
         --
         -------------------------------------------------------------
         CORRECT_RD_CNT_IAE : process (sig_rddata_valid,
                                       sig_SFIFO_empty,
                                       raw_data_count_int)
            begin
         
               
               if (sig_rddata_valid = '0') then

                  raw_data_count_int_corr        <= 0;
                  raw_data_count_int_corr_minus1 <= 0;
                  corrected_empty                <= '1';
                  corrected_almost_empty         <= '0';
                  
               elsif (sig_SFIFO_empty = '1') then   -- rddata valid and fifo empty
                  
                  raw_data_count_int_corr        <= 1;
                  raw_data_count_int_corr_minus1 <= 0;
                  corrected_empty                <= '0';
                  corrected_almost_empty         <= '1';
               
               Elsif (raw_data_count_int = 1) Then  -- rddata valid and fifo almost empty
                  
                  raw_data_count_int_corr        <= 2;
                  raw_data_count_int_corr_minus1 <= 1;
                  corrected_empty                <= '0';
                  corrected_almost_empty         <= '0';
               
               else                                 -- rddata valid and modify rd count from FIFO 
                  
                  raw_data_count_int_corr        <= raw_data_count_int+1;
                  raw_data_count_int_corr_minus1 <= raw_data_count_int;
                  corrected_empty                <= '0';
                  corrected_almost_empty         <= '0';
               
               end if;
         
            end process CORRECT_RD_CNT_IAE; 
      
    
            raw_data_count_corr <= CONV_STD_LOGIC_VECTOR(raw_data_count_int_corr,
                                                         C_DATA_CNT_WIDTH);
        
            raw_data_count_corr_minus1 <= CONV_STD_LOGIC_VECTOR(raw_data_count_int_corr_minus1,
                                                                 C_DATA_CNT_WIDTH);      
             
       end generate INCLUDE_ALMOST_EMPTY;
 
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: OMIT_ALMOST_EMPTY
    --
    -- If Generate Description:
    --    This process corrects the FIFO Read Count output for the
    -- auto read function and omits the generation of the
    -- Almost_Empty flag.
    --
    ------------------------------------------------------------
    OMIT_ALMOST_EMPTY : if (C_NEED_ALMOST_EMPTY = 0) generate
    
       -- local signals
       
          Signal raw_data_count_int_corr : integer := 0;
       
       begin
    
          corrected_almost_empty  <= '0'; -- always low
         
         
          -------------------------------------------------------------
          -- Combinational Process
          --
          -- Label: CORRECT_RD_CNT
          --
          -- Process Description:
          --    This process corrects the FIFO Read Count output for the
          -- auto read function and omits the generation of the
          -- Almost_Empty flag.
          --
          -------------------------------------------------------------
          CORRECT_RD_CNT : process (sig_rddata_valid,
                                    sig_SFIFO_empty,
                                    raw_data_count_int)
             begin
          
              
                if (sig_rddata_valid = '0') then

                   raw_data_count_int_corr <= 0;
                   corrected_empty         <= '1';
                   
                elsif (sig_SFIFO_empty = '1') then   -- rddata valid and fifo empty
                   
                   raw_data_count_int_corr <= 1;
                   corrected_empty         <= '0';
                
                Elsif (raw_data_count_int = 1) Then  -- rddata valid and fifo almost empty
                   
                   raw_data_count_int_corr <= 2;
                   corrected_empty         <= '0';
                
                else                                 -- rddata valid and modify rd count from FIFO 
                   
                   raw_data_count_int_corr <= raw_data_count_int+1;
                   corrected_empty         <= '0';
                
                end if;
          
             end process CORRECT_RD_CNT; 
       
            
             raw_data_count_corr <= CONV_STD_LOGIC_VECTOR(raw_data_count_int_corr,
                                                          C_DATA_CNT_WIDTH);
        
 
         
       end generate OMIT_ALMOST_EMPTY;
  
  
  
       
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: INCLUDE_ALMOST_FULL
    --
    -- If Generate Description:
    --  This IfGen Includes the generation of the Amost_Full flag.
    --
    --
    ------------------------------------------------------------
    INCLUDE_ALMOST_FULL : if (C_NEED_ALMOST_FULL = 1) generate
    
       -- Local Constants
          
         Constant ALMOST_FULL_VALUE : integer := 2**(C_DATA_CNT_WIDTH-1)-1;
       
       begin
    
          SFIFO_Almost_full <= '1'
             When raw_data_count_int = ALMOST_FULL_VALUE
             Else '0';
                
                
       end generate INCLUDE_ALMOST_FULL;

   
   
   
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: OMIT_ALMOST_FULL
    --
    -- If Generate Description:
    --  This IfGen Omits the generation of the Amost_Full flag.
    --
    --
    ------------------------------------------------------------
    OMIT_ALMOST_FULL : if (C_NEED_ALMOST_FULL = 0) generate
    
       begin
    
           SFIFO_Almost_full <= '0';  -- always low   
                
       end generate OMIT_ALMOST_FULL;



end imp;


-------------------------------------------------------------------------------
-- axi_datamover_fifo.vhd - entity/architecture pair

-------------------------------------------------------------------------------
--
-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_datamover_fifo.vhd
-- Version:         initial
-- Description:     
--    This file is a wrapper file for the Synchronous FIFO used by the DataMover. 
-- 
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
---------------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;



library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.all;
use lib_pkg_v1_0_2.lib_pkg.clog2;

library lib_srl_fifo_v1_0_2;
use lib_srl_fifo_v1_0_2.srl_fifo_f;


library axi_datamover_v5_1_14;
use axi_datamover_v5_1_14.axi_datamover_sfifo_autord;
use axi_datamover_v5_1_14.axi_datamover_afifo_autord;


-------------------------------------------------------------------------------

entity axi_datamover_fifo is
  generic (
     C_DWIDTH            : integer := 32  ;
       -- Bit width of the FIFO
       
     C_DEPTH             : integer := 4   ;
       -- Depth of the fifo in fifo width words
     
     C_IS_ASYNC          : Integer range 0 to 1 := 0 ;
       -- 0 = Syncronous FIFO
       -- 1 = Asynchronous (2 clock) FIFO
     
     C_PRIM_TYPE         : Integer range 0 to 2 := 2 ;
       -- 0 = Register
       -- 1 = Block Memory
       -- 2 = SRL
     
     C_FAMILY            : String  := "virtex7"
       -- Specifies the Target FPGA device family
     
    );
  port (
     
     
     
     -- Write Clock and reset -----------------
     fifo_wr_reset        : In  std_logic;   --
     fifo_wr_clk          : In  std_logic;   --
     ------------------------------------------
     
     -- Write Side ------------------------------------------------------
     fifo_wr_tvalid       : In  std_logic;                             --
     fifo_wr_tready       : Out std_logic;                             --
     fifo_wr_tdata        : In  std_logic_vector(C_DWIDTH-1 downto 0); --
     fifo_wr_full         : Out std_logic;                             --
     --------------------------------------------------------------------
    
    
     -- Read Clock and reset -----------------------------------------------
     fifo_async_rd_reset  : In  std_logic; -- only used if C_IS_ASYNC = 1 --  
     fifo_async_rd_clk    : In  std_logic; -- only used if C_IS_ASYNC = 1 --
     -----------------------------------------------------------------------
     
     -- Read Side --------------------------------------------------------
     fifo_rd_tvalid       : Out std_logic;                              --
     fifo_rd_tready       : In  std_logic;                              --
     fifo_rd_tdata        : Out std_logic_vector(C_DWIDTH-1 downto 0);  --
     fifo_rd_empty        : Out std_logic                               --
     ---------------------------------------------------------------------
    
    );
end entity axi_datamover_fifo;

-----------------------------------------------------------------------------
-- Architecture section
-----------------------------------------------------------------------------

architecture imp of axi_datamover_fifo is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of imp : architecture is "yes";

 -- function Declarations
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: funct_get_prim_type
  --
  -- Function Description:
  --  Sorts out the FIFO Primitive type selection based on fifo
  -- depth and original primitive choice.
  --
  -------------------------------------------------------------------
  function funct_get_prim_type (depth            : integer;
                                input_prim_type  : integer) return integer is
  
    Variable temp_prim_type : Integer := 0;
  
  begin
  
    If (depth > 64) Then
    
      temp_prim_type := 1;  -- use BRAM
   
    Elsif (depth <= 64 and 
           input_prim_type = 0) Then
    
      temp_prim_type := 0;  -- use regiaters
    
    else

      temp_prim_type := 1;  -- use BRAM
    
    End if;
    
    
    Return (temp_prim_type);
    
  end function funct_get_prim_type;
  
 
  
  
-- Signal declarations
  
  Signal sig_init_reg          : std_logic := '0';
  Signal sig_init_reg2         : std_logic := '0';
  Signal sig_init_done         : std_logic := '0';
  signal sig_inhibit_rdy_n     : std_logic := '0';
 
   
   
 
-----------------------------------------------------------------------------
-- Begin architecture
-----------------------------------------------------------------------------
begin  


  -------------------------------------------------------------
  -- Synchronous Process with Sync Reset
  --
  -- Label: IMP_INIT_REG
  --
  -- Process Description:
  --  Registers the reset signal input.
  --
  -------------------------------------------------------------
  IMP_INIT_REG : process (fifo_wr_clk)
     begin
       if (fifo_wr_clk'event and fifo_wr_clk = '1') then
          if (fifo_wr_reset = '1') then
            sig_init_reg  <= '1';
            sig_init_reg2 <= '1';
          else
            sig_init_reg <= '0';
            sig_init_reg2 <= sig_init_reg;
          end if; 
       end if;       
     end process IMP_INIT_REG; 
  
  
  -------------------------------------------------------------
  -- Synchronous Process with Sync Reset
  --
  -- Label: IMP_INIT_DONE_REG
  --
  -- Process Description:
  -- Create a 1 clock wide init done pulse. 
  --
  -------------------------------------------------------------
  IMP_INIT_DONE_REG : process (fifo_wr_clk)
     begin
       if (fifo_wr_clk'event and fifo_wr_clk = '1') then
          if (fifo_wr_reset = '1' or
              sig_init_done = '1') then
            
            sig_init_done <= '0';
          
          Elsif (sig_init_reg  = '1' and
                 sig_init_reg2 = '1') Then
          
            sig_init_done <= '1';
          
          else
            null;  -- hold current state
          end if; 
       end if;       
     end process IMP_INIT_DONE_REG; 
  
  
  -------------------------------------------------------------
  -- Synchronous Process with Sync Reset
  --
  -- Label: IMP_RDY_INHIBIT_REG
  --
  -- Process Description:
  --  Implements a ready inhibit flop.
  --
  -------------------------------------------------------------
  IMP_RDY_INHIBIT_REG : process (fifo_wr_clk)
     begin
       if (fifo_wr_clk'event and fifo_wr_clk = '1') then
          if (fifo_wr_reset = '1') then
            
            sig_inhibit_rdy_n  <= '0';
          
          Elsif (sig_init_done = '1') Then
          
            sig_inhibit_rdy_n <= '1';
          
          else
            null;  -- hold current state
          end if; 
       end if;       
     end process IMP_RDY_INHIBIT_REG; 
  
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: USE_SINGLE_REG
  --
  -- If Generate Description:
  --  Implements a 1 deep register FIFO (synchronous mode only)
  --
  --
  ------------------------------------------------------------
  USE_SINGLE_REG : if (C_IS_ASYNC  = 0 and 
                       C_DEPTH    <= 1) generate
  
     -- Local Constants
     
     -- local signals
     signal sig_data_in           : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
     signal sig_regfifo_dout_reg  : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
     signal sig_regfifo_full_reg  : std_logic := '0';
     signal sig_regfifo_empty_reg : std_logic := '0';
     signal sig_push_regfifo      : std_logic := '0';
     signal sig_pop_regfifo       : std_logic := '0';
     
     
  
     begin

       -- Internal signals
       
       -- Write signals
       fifo_wr_tready    <=  sig_regfifo_empty_reg;
       
       fifo_wr_full      <=  sig_regfifo_full_reg ;
       
       sig_push_regfifo  <=  fifo_wr_tvalid and
                             sig_regfifo_empty_reg;
       
       sig_data_in       <=  fifo_wr_tdata ; 

       
       -- Read signals
       fifo_rd_tdata     <=  sig_regfifo_dout_reg ;
       
       fifo_rd_tvalid    <=  sig_regfifo_full_reg ;
       
       fifo_rd_empty     <=  sig_regfifo_empty_reg;
       
       sig_pop_regfifo   <=  sig_regfifo_full_reg and
                             fifo_rd_tready;
       
       
       
       
       -------------------------------------------------------------
       -- Synchronous Process with Sync Reset
       --
       -- Label: IMP_REG_FIFO
       --
       -- Process Description:
       --    This process implements the data and full flag for the 
       -- register fifo.
       --
       -------------------------------------------------------------
       IMP_REG_FIFO : process (fifo_wr_clk)
          begin
            if (fifo_wr_clk'event and fifo_wr_clk = '1') then
               if (fifo_wr_reset    = '1' or
                   sig_pop_regfifo  = '1') then
                 
                 sig_regfifo_full_reg  <= '0';
                 
               elsif (sig_push_regfifo = '1') then
                 
                 sig_regfifo_full_reg  <= '1';
                 
               else
                 null;  -- don't change state
               end if; 
            end if;       
          end process IMP_REG_FIFO; 
       
      
       IMP_REG_FIFO1 : process (fifo_wr_clk)
          begin
            if (fifo_wr_clk'event and fifo_wr_clk = '1') then
               if (fifo_wr_reset    = '1') then
                
                 
                 sig_regfifo_dout_reg  <= (others => '0');
                 
               elsif (sig_push_regfifo = '1') then
                 
                 sig_regfifo_dout_reg  <= sig_data_in;
                 
               else
                 null;  -- don't change state
               end if; 
            end if;       
          end process IMP_REG_FIFO1; 
       
       
       -------------------------------------------------------------
       -- Synchronous Process with Sync Reset
       --
       -- Label: IMP_REG_EMPTY_FLOP
       --
       -- Process Description:
       --    This process implements the empty flag for the 
       -- register fifo.
       --
       -------------------------------------------------------------
       IMP_REG_EMPTY_FLOP : process (fifo_wr_clk)
          begin
            if (fifo_wr_clk'event and fifo_wr_clk = '1') then
               if (fifo_wr_reset    = '1') then
                 
                 sig_regfifo_empty_reg <= '0'; -- since this is used for the ready (invertd)
                                               -- it can't be asserted during reset
                 
               elsif (sig_pop_regfifo  = '1' or
                      sig_init_done    = '1') then
                 
                 sig_regfifo_empty_reg <= '1';
                 
               elsif (sig_push_regfifo = '1') then
                 
                 sig_regfifo_empty_reg <= '0';
                 
               else
                 null;  -- don't change state
               end if; 
            end if;       
          end process IMP_REG_EMPTY_FLOP; 
       

  
     end generate USE_SINGLE_REG;
 
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: USE_SRL_FIFO
  --
  -- If Generate Description:
  --  Generates a fifo implementation usinf SRL based FIFOa
  --
  --
  ------------------------------------------------------------
  USE_SRL_FIFO : if (C_IS_ASYNC  =  0 and
                     C_DEPTH    <= 64 and
                     C_DEPTH     >  1 and
                     C_PRIM_TYPE =  2 ) generate
                     
  
  
    -- Local Constants
    Constant  LOGIC_LOW         : std_logic := '0';
    Constant  NEED_ALMOST_EMPTY : Integer := 0;
    Constant  NEED_ALMOST_FULL  : Integer := 0;
    
    
    -- local signals

    signal sig_wr_full          : std_logic := '0';
    signal sig_wr_fifo          : std_logic := '0';
    signal sig_wr_ready         : std_logic := '0';
    signal sig_rd_fifo          : std_logic := '0';
    signal sig_rd_empty         : std_logic := '0';
    signal sig_rd_valid         : std_logic := '0';
    signal sig_fifo_rd_data     : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_wr_data     : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
      
    begin
  
       

      -- Write side signals
      fifo_wr_tready    <=  sig_wr_ready;
      
      fifo_wr_full      <=  sig_wr_full;
      
      sig_wr_ready      <=  not(sig_wr_full) and 
                            sig_inhibit_rdy_n;
 
      sig_wr_fifo       <=  fifo_wr_tvalid and 
                            sig_wr_ready;
      
      sig_fifo_wr_data  <=  fifo_wr_tdata;
      
      
      
      
      -- Read Side Signals
      fifo_rd_tvalid    <=  sig_rd_valid;
      
      sig_rd_valid      <=  not(sig_rd_empty);
      
      fifo_rd_tdata     <=  sig_fifo_rd_data ;
      
      fifo_rd_empty     <=  not(sig_rd_valid);
      
      sig_rd_fifo       <=  sig_rd_valid and
                            fifo_rd_tready;
       
      
       
      ------------------------------------------------------------
      -- Instance: I_SYNC_FIFO 
      --
      -- Description:
      -- Implement the synchronous FIFO using SRL FIFO elements    
      --
      ------------------------------------------------------------
       I_SYNC_FIFO : entity lib_srl_fifo_v1_0_2.srl_fifo_f
       generic map (

         C_DWIDTH            =>  C_DWIDTH   ,  
         C_DEPTH             =>  C_DEPTH    ,  
         C_FAMILY            =>  C_FAMILY      

         )
       port map (

         Clk           =>  fifo_wr_clk      ,  
         Reset         =>  fifo_wr_reset    ,  
         FIFO_Write    =>  sig_wr_fifo      ,  
         Data_In       =>  sig_fifo_wr_data ,  
         FIFO_Read     =>  sig_rd_fifo      ,  
         Data_Out      =>  sig_fifo_rd_data ,  
         FIFO_Empty    =>  sig_rd_empty     ,  
         FIFO_Full     =>  sig_wr_full      ,  
         Addr          =>  open                
     
         );

       
       
     end generate USE_SRL_FIFO;




 
 
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: USE_SYNC_FIFO
  --
  -- If Generate Description:
  --  Instantiates a synchronous FIFO design for use in the 
  -- synchronous operating mode.
  --
  ------------------------------------------------------------
  USE_SYNC_FIFO : if (C_IS_ASYNC  =  0 and
                     (C_DEPTH     > 64  or
                     (C_DEPTH     >  1 and C_PRIM_TYPE < 2 ))) 
                     or 
                     (C_IS_ASYNC  =  0 and
                     C_DEPTH    <= 64 and
                     C_DEPTH     >  1 and
                     C_PRIM_TYPE =  0 ) 
generate
  
    -- Local Constants
    Constant  LOGIC_LOW         : std_logic := '0';
    Constant  NEED_ALMOST_EMPTY : Integer   := 0;
    Constant  NEED_ALMOST_FULL  : Integer   := 0;
    Constant  DATA_CNT_WIDTH    : Integer   := clog2(C_DEPTH)+1;
    Constant  PRIM_TYPE         : Integer   := funct_get_prim_type(C_DEPTH, C_PRIM_TYPE);
    
    
    -- local signals
    signal sig_wr_full          : std_logic := '0';
    signal sig_wr_fifo          : std_logic := '0';
    signal sig_wr_ready         : std_logic := '0';
    signal sig_rd_fifo          : std_logic := '0';
    signal sig_rd_valid         : std_logic := '0';
    signal sig_fifo_rd_data     : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_wr_data     : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
    
    begin

      -- Write side signals
      fifo_wr_tready    <=  sig_wr_ready;
      
      fifo_wr_full      <=  sig_wr_full;
      
      sig_wr_ready      <=  not(sig_wr_full) and 
                            sig_inhibit_rdy_n;
 
      sig_wr_fifo       <=  fifo_wr_tvalid and 
                            sig_wr_ready;
      
      sig_fifo_wr_data  <=  fifo_wr_tdata;
      
      
      
      
      -- Read Side Signals
      fifo_rd_tvalid    <=  sig_rd_valid;
      
      fifo_rd_tdata     <=  sig_fifo_rd_data ;
      
      fifo_rd_empty     <=  not(sig_rd_valid);
      
      sig_rd_fifo       <=  sig_rd_valid and
                            fifo_rd_tready;
       
      
      
      
      ------------------------------------------------------------
      -- Instance: I_SYNC_FIFO 
      --
      -- Description:
      -- Implement the synchronous FIFO    
      --
      ------------------------------------------------------------
       I_SYNC_FIFO : entity axi_datamover_v5_1_14.axi_datamover_sfifo_autord
       generic map (

         C_DWIDTH                =>  C_DWIDTH          ,  
         C_DEPTH                 =>  C_DEPTH           ,  
         C_DATA_CNT_WIDTH        =>  DATA_CNT_WIDTH    ,  
         C_NEED_ALMOST_EMPTY     =>  NEED_ALMOST_EMPTY ,  
         C_NEED_ALMOST_FULL      =>  NEED_ALMOST_FULL  ,  
         C_USE_BLKMEM            =>  PRIM_TYPE         ,  
         C_FAMILY                =>  C_FAMILY             

         )
       port map (

        -- Inputs 
         SFIFO_Sinit             =>  fifo_wr_reset     ,  
         SFIFO_Clk               =>  fifo_wr_clk       ,  
         SFIFO_Wr_en             =>  sig_wr_fifo       ,  
         SFIFO_Din               =>  fifo_wr_tdata     ,  
         SFIFO_Rd_en             =>  sig_rd_fifo       ,  
         SFIFO_Clr_Rd_Data_Valid =>  LOGIC_LOW         ,  
         
        -- Outputs
         SFIFO_DValid            =>  sig_rd_valid      ,  
         SFIFO_Dout              =>  sig_fifo_rd_data  ,  
         SFIFO_Full              =>  sig_wr_full       ,  
         SFIFO_Empty             =>  open              ,  
         SFIFO_Almost_full       =>  open              ,  
         SFIFO_Almost_empty      =>  open              ,  
         SFIFO_Rd_count          =>  open              ,  
         SFIFO_Rd_count_minus1   =>  open              ,  
         SFIFO_Wr_count          =>  open              ,  
         SFIFO_Rd_ack            =>  open                 

         );


      
 
     end generate USE_SYNC_FIFO;
 
 
 
 
 
 
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: USE_ASYNC_FIFO
  --
  -- If Generate Description:
  --  Instantiates an asynchronous FIFO design for use in the 
  -- asynchronous operating mode.
  --
  ------------------------------------------------------------
  USE_ASYNC_FIFO : if (C_IS_ASYNC = 1) generate
  
    -- Local Constants
    Constant  LOGIC_LOW         : std_logic := '0';
    Constant  CNT_WIDTH         : Integer := clog2(C_DEPTH);
    
    
    -- local signals

    signal sig_async_wr_full       : std_logic := '0';
    signal sig_async_wr_fifo       : std_logic := '0';
    signal sig_async_wr_ready      : std_logic := '0';
    signal sig_async_rd_fifo       : std_logic := '0';
    signal sig_async_rd_valid      : std_logic := '0';
    signal sig_afifo_rd_data       : std_logic_vector(C_DWIDTH-1 downto 0);
    signal sig_afifo_wr_data       : std_logic_vector(C_DWIDTH-1 downto 0);
    signal sig_fifo_ainit          : std_logic := '0';
    Signal sig_init_reg            : std_logic := '0';
    
    
    begin

      sig_fifo_ainit  <= fifo_async_rd_reset or fifo_wr_reset;
      
      

      -- Write side signals
      fifo_wr_tready      <=  sig_async_wr_ready;
      
      fifo_wr_full        <=  sig_async_wr_full;
      
      sig_async_wr_ready  <=  not(sig_async_wr_full) and
                              sig_inhibit_rdy_n;
 
      sig_async_wr_fifo   <=  fifo_wr_tvalid and 
                              sig_async_wr_ready;
      
      sig_afifo_wr_data   <=  fifo_wr_tdata;
      
      
      
      
      -- Read Side Signals
      fifo_rd_tvalid    <=  sig_async_rd_valid;
      
      fifo_rd_tdata     <=  sig_afifo_rd_data ;
      
      fifo_rd_empty     <=  not(sig_async_rd_valid);
      
      sig_async_rd_fifo <=  sig_async_rd_valid and
                            fifo_rd_tready;
       
      
 
       
       
       
      ------------------------------------------------------------
      -- Instance: I_ASYNC_FIFO 
      --
      -- Description:
      -- Implement the asynchronous FIFO    
      --
      ------------------------------------------------------------
       I_ASYNC_FIFO : entity axi_datamover_v5_1_14.axi_datamover_afifo_autord
       generic map (

         C_DWIDTH                   =>  C_DWIDTH          ,  
         C_DEPTH                    =>  C_DEPTH           ,  
         C_CNT_WIDTH                =>  CNT_WIDTH         ,  
         C_USE_BLKMEM               =>  C_PRIM_TYPE       ,  
         C_FAMILY                   =>  C_FAMILY             

         )
       port map (

        -- Inputs 
         AFIFO_Ainit                =>  sig_fifo_ainit    ,  
         AFIFO_Ainit_Rd_clk         =>  fifo_async_rd_reset    ,  
         AFIFO_Wr_clk               =>  fifo_wr_clk       ,  
         AFIFO_Wr_en                =>  sig_async_wr_fifo ,  
         AFIFO_Din                  =>  sig_afifo_wr_data ,  
         AFIFO_Rd_clk               =>  fifo_async_rd_clk ,  
         AFIFO_Rd_en                =>  sig_async_rd_fifo ,  
         AFIFO_Clr_Rd_Data_Valid    =>  LOGIC_LOW         ,  
         
        -- Outputs
         AFIFO_DValid               =>  sig_async_rd_valid,  
         AFIFO_Dout                 =>  sig_afifo_rd_data ,  
         AFIFO_Full                 =>  sig_async_wr_full ,  
         AFIFO_Empty                =>  open              ,  
         AFIFO_Almost_full          =>  open              ,  
         AFIFO_Almost_empty         =>  open              ,  
         AFIFO_Wr_count             =>  open              ,   
         AFIFO_Rd_count             =>  open              ,  
         AFIFO_Corr_Rd_count        =>  open              ,  
         AFIFO_Corr_Rd_count_minus1 =>  open              ,  
         AFIFO_Rd_ack               =>  open                 

         );


      
 
     end generate USE_ASYNC_FIFO;
 
 

end imp;


  -------------------------------------------------------------------------------
  -- axi_datamover_cmd_status.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_cmd_status.vhd
  --
  -- Description:     
  --    This file implements the DataMover Command and Status interfaces.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library axi_datamover_v5_1_14;
  Use axi_datamover_v5_1_14.axi_datamover_fifo;
  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_cmd_status is
    generic (
      
      C_ADDR_WIDTH         : Integer range 32 to 64 := 32;
        -- Indictes the width of the DataMover Address bus
       
      C_INCLUDE_STSFIFO    : Integer range  0 to  1 :=  1;
        -- Indicates if a Stus FIFO is to be included or omitted
        -- 0 = Omit
        -- 1 = Include
       
      C_STSCMD_FIFO_DEPTH  : Integer range  1 to 16 :=  4;
        -- Sets the depth of the Command and Status FIFOs
       
      C_STSCMD_IS_ASYNC    : Integer range  0 to  1 :=  0;
        -- Indicates if the Command and Status Stream Channels are clocked with
        -- a different clock than the Main dataMover Clock
        -- 0 = Same Clock
        -- 1 = Different clocks
       
      C_CMD_WIDTH          : Integer                := 68;
        -- Sets the width of the input command
       
      C_STS_WIDTH          : Integer                :=  8;
        -- Sets the width of the output status

      C_ENABLE_CACHE_USER  : Integer range 0 to 1   :=  0;
       
      C_FAMILY             : string                 := "virtex7"
        -- Sets the target FPGA family
      
      );
    port (
      
      -- Clock inputs ----------------------------------------------------
      primary_aclk           : in  std_logic;                           --
         -- Primary synchronization clock for the Master side           --
         -- interface and internal logic. It is also used               --
         -- for the User interface synchronization when                 --
         -- C_STSCMD_IS_ASYNC = 0.                                      --
                                                                        --
      secondary_awclk        : in  std_logic;                           --
         -- Clock used for the Command and Status User Interface        --
         --  when the User Command and Status interface is Async        --
         -- to the MMap interface. Async mode is set by the assigned    --
         -- value to C_STSCMD_IS_ASYNC = 1.                             --
      --------------------------------------------------------------------
     
     
      -- Reset inputs ----------------------------------------------------
      user_reset             : in  std_logic;                           --
        -- Reset used for the User Stream interface logic               --
                                                                        --
      internal_reset         : in  std_logic;                           --
        -- Reset used for the internal master interface logic           --
      --------------------------------------------------------------------
      
      
      -- User Command Stream Ports (AXI Stream) -------------------------------
      cmd_wvalid             : in  std_logic;                                --
      cmd_wready             : out std_logic;                                --
      cmd_wdata              : in  std_logic_vector(C_CMD_WIDTH-1 downto 0); --
      cache_data             : in  std_logic_vector(7 downto 0); --
      -------------------------------------------------------------------------
      
      -- User Status Stream Ports (AXI Stream) ------------------------------------
      sts_wvalid             : out std_logic;                                    --
      sts_wready             : in  std_logic;                                    --
      sts_wdata              : out std_logic_vector(C_STS_WIDTH-1 downto 0);     --
      sts_wstrb              : out std_logic_vector((C_STS_WIDTH/8)-1 downto 0); --
      sts_wlast              : out std_logic;                                    --
      -----------------------------------------------------------------------------
      
      
      -- Internal Command Out Interface -----------------------------------------------
      cmd2mstr_command       : Out std_logic_vector(C_CMD_WIDTH-1 downto 0);         --
         -- The next command value available from the Command FIFO/Register          --

      cache2mstr_command       : Out std_logic_vector(7 downto 0);         --
         -- The cache value available from the FIFO/Register          --

                                                                                     --
      mst2cmd_cmd_valid      : Out std_logic;                                        --
         -- Handshake bit indicating the Command FIFO/Register has at least 1 valid  --
         -- command entry                                                            --
                                                                                     --
      cmd2mstr_cmd_ready     : in  std_logic;                                        --
         -- Handshake bit indicating the Command Calculator is ready to accept       --
         -- another command                                                          --
      ---------------------------------------------------------------------------------
      
      
      -- Internal Status In Interface  -----------------------------------------------------
      mstr2stat_status       : in  std_logic_vector(C_STS_WIDTH-1 downto 0);              --
         -- The input for writing the status value to the Status FIFO/Register            --
                                                                                          --
      stat2mstr_status_ready : Out std_logic;                                             --
         -- Handshake bit indicating that the Status FIFO/Register is ready for transfer  --
                                                                                          --
      mst2stst_status_valid  : In  std_logic                                              --
         -- Handshake bit for writing the Status value into the Status FIFO/Register      --
      --------------------------------------------------------------------------------------
     
      );
  
  end entity axi_datamover_cmd_status;
  
  
  architecture implementation of axi_datamover_cmd_status is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    -- Function
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: get_fifo_prim_type
    --
    -- Function Description:
    --  Returns the fifo primitiver type to use for the given input
    -- conditions.
    --
    --  0 = Not used or allowed here
    --  1 = BRAM Primitives (Block Memory)
    --  2 = Distributed memory
    --
    -------------------------------------------------------------------
    function get_fifo_prim_type (is_async : integer;
                                 depth    : integer) return integer is
    
      Variable var_temp_prim_type : Integer := 1;
    
    begin
    
      if (is_async = 1) then   -- Async FIFOs always use Blk Mem (BRAM)
      
        var_temp_prim_type := 1;
      
      elsif (depth <= 64) then -- (use srls or distrubuted)
      
        var_temp_prim_type := 2; 
      
      else  -- depth is too big for SRLs so use Blk Memory (BRAM)
      
        var_temp_prim_type := 1;
      
      end if;
      
     Return (var_temp_prim_type);
      
    end function get_fifo_prim_type;
    
   
   
    
    
    -- Constants 
    
    Constant REGISTER_TYPE  : integer := 0; 
    Constant BRAM_TYPE      : integer := 1; 
    --Constant SRL_TYPE       : integer := 2; 
    --Constant FIFO_PRIM_TYPE : integer := SRL_TYPE;
    Constant FIFO_PRIM_TYPE : integer := get_fifo_prim_type(C_STSCMD_IS_ASYNC, 
                                                            C_STSCMD_FIFO_DEPTH);
    
    
    -- Signals
    
    signal sig_cmd_fifo_wr_clk  : std_logic := '0';
    signal sig_cmd_fifo_wr_rst  : std_logic := '0';
    signal sig_cmd_fifo_rd_clk  : std_logic := '0';
    signal sig_cmd_fifo_rd_rst  : std_logic := '0';
    signal sig_sts_fifo_wr_clk  : std_logic := '0';
    signal sig_sts_fifo_wr_rst  : std_logic := '0';
    signal sig_sts_fifo_rd_clk  : std_logic := '0';
    signal sig_sts_fifo_rd_rst  : std_logic := '0';
    signal sig_reset_mstr       : std_logic := '0';
    signal sig_reset_user       : std_logic := '0';
 
 
  
  
  begin --(architecture implementation)
  
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_SYNC_RESET
    --
    -- If Generate Description:
    --  This IfGen assigns the clock and reset signals for the 
    -- synchronous User interface case
    --
    ------------------------------------------------------------
    GEN_SYNC_RESET : if (C_STSCMD_IS_ASYNC = 0) generate
    
       begin
    
          sig_reset_mstr       <= internal_reset  ;
          sig_reset_user       <= internal_reset  ;
        
          sig_cmd_fifo_wr_clk   <=  primary_aclk  ;  
          sig_cmd_fifo_wr_rst   <=  sig_reset_user; 
          sig_cmd_fifo_rd_clk   <=  primary_aclk  ; 
          sig_cmd_fifo_rd_rst   <=  sig_reset_mstr; 
         
          sig_sts_fifo_wr_clk   <=  primary_aclk  ; 
          sig_sts_fifo_wr_rst   <=  sig_reset_mstr; 
          sig_sts_fifo_rd_clk   <=  primary_aclk  ; 
          sig_sts_fifo_rd_rst   <=  sig_reset_user; 
           
         
        
             
         
       end generate GEN_SYNC_RESET;
  
  
    
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ASYNC_RESET
    --
    -- If Generate Description:
    --  This IfGen assigns the clock and reset signals for the 
    -- Asynchronous User interface case
    --
    ------------------------------------------------------------
    GEN_ASYNC_RESET : if (C_STSCMD_IS_ASYNC = 1) generate
    
       begin
    
         sig_reset_mstr        <= internal_reset  ;
         sig_reset_user        <= user_reset      ;
       
         sig_cmd_fifo_wr_clk   <=  secondary_awclk;  
         sig_cmd_fifo_wr_rst   <=  sig_reset_user ; 
         sig_cmd_fifo_rd_clk   <=  primary_aclk   ; 
         sig_cmd_fifo_rd_rst   <=  sig_reset_mstr ; 
        
         sig_sts_fifo_wr_clk   <=  primary_aclk   ; 
         sig_sts_fifo_wr_rst   <=  sig_reset_mstr ; 
         sig_sts_fifo_rd_clk   <=  secondary_awclk; 
         sig_sts_fifo_rd_rst   <=  sig_reset_user ; 
          
        
            
         
       end generate GEN_ASYNC_RESET;
  
  
  
  
       
    ------------------------------------------------------------
    -- Instance: I_CMD_FIFO 
    --
    -- Description:
    -- Instance for the Command FIFO
    -- The User Interface is the Write Side
    -- The Internal Interface is the Read side    
    --
    ------------------------------------------------------------
     I_CMD_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
     generic map (
   
       C_DWIDTH            =>  C_CMD_WIDTH          ,  
       C_DEPTH             =>  C_STSCMD_FIFO_DEPTH  ,  
       C_IS_ASYNC          =>  C_STSCMD_IS_ASYNC    ,  
       C_PRIM_TYPE         =>  FIFO_PRIM_TYPE       ,  
       C_FAMILY            =>  C_FAMILY                
      
       )
     port map (
       
       -- Write Clock and reset
       fifo_wr_reset        =>  sig_cmd_fifo_wr_rst ,  
       fifo_wr_clk          =>  sig_cmd_fifo_wr_clk ,  
       
       -- Write Side
       fifo_wr_tvalid       =>  cmd_wvalid          ,  
       fifo_wr_tready       =>  cmd_wready          ,  
       fifo_wr_tdata        =>  cmd_wdata           ,  
       fifo_wr_full         =>  open                ,  
      
      
       -- Read Clock and reset
       fifo_async_rd_reset  =>  sig_cmd_fifo_rd_rst ,     
       fifo_async_rd_clk    =>  sig_cmd_fifo_rd_clk ,   
       
       -- Read Side
       fifo_rd_tvalid       =>  mst2cmd_cmd_valid   ,  
       fifo_rd_tready       =>  cmd2mstr_cmd_ready  ,  
       fifo_rd_tdata        =>  cmd2mstr_command    ,  
       fifo_rd_empty        =>  open                   
      
       );

CACHE_ENABLE : if C_ENABLE_CACHE_USER = 1 generate
begin
   
     I_CACHE_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
     generic map (
   
       C_DWIDTH            =>  8          ,  
       C_DEPTH             =>  C_STSCMD_FIFO_DEPTH  ,  
       C_IS_ASYNC          =>  C_STSCMD_IS_ASYNC    ,  
       C_PRIM_TYPE         =>  FIFO_PRIM_TYPE       ,  
       C_FAMILY            =>  C_FAMILY                
      
       )
     port map (
       
       -- Write Clock and reset
       fifo_wr_reset        =>  sig_cmd_fifo_wr_rst ,  
       fifo_wr_clk          =>  sig_cmd_fifo_wr_clk ,  
       
       -- Write Side
       fifo_wr_tvalid       =>  cmd_wvalid          ,  
       fifo_wr_tready       =>  open ,--cmd_wready          ,  
       fifo_wr_tdata        =>  cache_data           ,  
       fifo_wr_full         =>  open                ,  
      
      
       -- Read Clock and reset
       fifo_async_rd_reset  =>  sig_cmd_fifo_rd_rst ,     
       fifo_async_rd_clk    =>  sig_cmd_fifo_rd_clk ,   
       
       -- Read Side
       fifo_rd_tvalid       =>  open ,--mst2cmd_cmd_valid   ,  
       fifo_rd_tready       =>  cmd2mstr_cmd_ready  ,  
       fifo_rd_tdata        =>  cache2mstr_command  ,  
       fifo_rd_empty        =>  open                   
      
       );
   
end generate;    
    
    
CACHE_DISABLE : if C_ENABLE_CACHE_USER = 0 generate
begin
   
 cache2mstr_command <= (others => '0'); 
    
end generate CACHE_DISABLE;    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_INCLUDE_STATUS_FIFO
    --
    -- If Generate Description:
    --  Instantiates a Status FIFO
    --
    --
    ------------------------------------------------------------
    GEN_INCLUDE_STATUS_FIFO : if (C_INCLUDE_STSFIFO = 1) generate
    
       begin
 
  
         -- Set constant outputs for Status Interface
         sts_wstrb             <=  (others => '1');    
         sts_wlast             <=  '1';                
         
         
       
         ------------------------------------------------------------
         -- Instance: I_STS_FIFO 
         --
         -- Description:
         -- Instance for the Status FIFO
         -- The Internal Interface is the Write Side
         -- The User Interface is the Read side    
         --
         ------------------------------------------------------------
         I_STS_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
         generic map (
       
           C_DWIDTH            =>  C_STS_WIDTH            ,  
           C_DEPTH             =>  C_STSCMD_FIFO_DEPTH    ,  
           C_IS_ASYNC          =>  C_STSCMD_IS_ASYNC      ,  
           C_PRIM_TYPE         =>  FIFO_PRIM_TYPE         ,  
           C_FAMILY            =>  C_FAMILY                  
          
           )
         port map (
           
           -- Write Clock and reset
           fifo_wr_reset        =>  sig_sts_fifo_wr_rst   ,  
           fifo_wr_clk          =>  sig_sts_fifo_wr_clk   ,  
           
           -- Write Side
           fifo_wr_tvalid       =>  mst2stst_status_valid ,  
           fifo_wr_tready       =>  stat2mstr_status_ready,  
           fifo_wr_tdata        =>  mstr2stat_status      ,  
           fifo_wr_full         =>  open                  ,  
          
          
           -- Read Clock and reset
           fifo_async_rd_reset  =>  sig_sts_fifo_rd_rst   ,     
           fifo_async_rd_clk    =>  sig_sts_fifo_rd_clk   ,   
           
           -- Read Side
           fifo_rd_tvalid       =>  sts_wvalid            ,  
           fifo_rd_tready       =>  sts_wready            ,  
           fifo_rd_tdata        =>  sts_wdata             ,  
           fifo_rd_empty        =>  open                     
          
           );
        
    
       end generate GEN_INCLUDE_STATUS_FIFO;
    
    
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_OMIT_STATUS_FIFO
    --
    -- If Generate Description:
    --  Omits the Status FIFO
    --
    --
    ------------------------------------------------------------
    GEN_OMIT_STATUS_FIFO : if (C_INCLUDE_STSFIFO = 0) generate
    
       begin
  
         -- Status FIFO User interface housekeeping
         sts_wvalid            <=  '0';
         -- sts_wready         -- ignored
         sts_wdata             <=  (others => '0');
         sts_wstrb             <=  (others => '0');    
         sts_wlast             <=  '0';                
 
         
         
         -- Status FIFO Internal interface housekeeping
         stat2mstr_status_ready <= '1';
         -- mstr2stat_status       -- ignored
         -- mst2stst_status_valid  -- ignored
 
    
       end generate GEN_OMIT_STATUS_FIFO;
    
    
 
  
  end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_scc.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_scc.vhd
  --
  -- Description:     
  --    This file implements the DataMover Lite Master Simple Command Calculator (SCC).                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_scc is
    generic (
      
      C_SEL_ADDR_WIDTH     : Integer range  1 to  8 :=  5;
        -- Sets the width of the LS address bus used for 
        -- Muxing/Demuxing data to/from a wider AXI4 data bus
      
      C_ADDR_WIDTH         : Integer range 32 to 64 := 32;
        -- Sets the width of the AXi Address Channel 
      
      C_STREAM_DWIDTH      : Integer range  8 to 64 := 32;
        -- Sets the width of the Native Data width that
        -- is being supported by the PCC
      
      C_MAX_BURST_LEN      : Integer range 2 to 64 := 16;
        -- Indicates the max allowed burst length to use for
        -- AXI4 transfer calculations
      
      C_CMD_WIDTH          : Integer                := 68;
        -- Sets the width of the input command port

      C_MICRO_DMA          : integer range 0 to 1   := 0;
      
      C_TAG_WIDTH          : Integer range  1 to  8 := 4
        -- Sets the width of the Tag field in the input command
      
      );
    port (
      
      -- Clock and Reset inputs -------------------------------------
      primary_aclk         : in  std_logic;                        --
         -- Primary synchronization clock for the Master side      --
         -- interface and internal logic. It is also used          --
         -- for the User interface synchronization when            --
         -- C_STSCMD_IS_ASYNC = 0.                                 --
                                                                   --
      -- Reset input                                               --
      mmap_reset           : in  std_logic;                        --
        -- Reset used for the internal master logic                --
      ---------------------------------------------------------------
      
      
     
      -- Command Input Interface ---------------------------------------------------------
                                                                                        --
      cmd2mstr_command      : in std_logic_vector(C_CMD_WIDTH-1 downto 0);              --
         -- The next command value available from the Command FIFO/Register             --
                                                                                        --
      cache2mstr_command      : in std_logic_vector(7 downto 0);              --
         -- The next command value available from the Command FIFO/Register             --
                                                                                        --
      cmd2mstr_cmd_valid    : in std_logic;                                             --
         -- Handshake bit indicating if the Command FIFO/Register has at leasdt 1 entry --
                                                                                        --
      mst2cmd_cmd_ready     : out  std_logic;                                           --
         -- Handshake bit indicating the Command Calculator is ready to accept          --
         -- another command                                                             --
      ------------------------------------------------------------------------------------
      
      
      
      -- Address Channel Controller Interface --------------------------------------------
                                                                                        --
      mstr2addr_tag       : out std_logic_vector(C_TAG_WIDTH-1 downto 0);               --
         -- The next command tag                                                        --
                                                                                        --
      mstr2addr_addr      : out std_logic_vector(C_ADDR_WIDTH-1 downto 0);              --
         -- The next command address to put on the AXI MMap ADDR                        --
                                                                                        --
      mstr2addr_len       : out std_logic_vector(7 downto 0);                           --
         -- The next command length to put on the AXI MMap LEN                          --
                                                                                        --
      mstr2addr_size      : out std_logic_vector(2 downto 0);                           --
         -- The next command size to put on the AXI MMap SIZE                           --
                                                                                        --
      mstr2addr_burst     : out std_logic_vector(1 downto 0);                           --
         -- The next command burst type to put on the AXI MMap BURST                    --
                                                                                        --
      mstr2addr_cache     : out std_logic_vector(3 downto 0);                           --
         -- The next command burst type to put on the AXI MMap BURST                    --
                                                                                        --
      mstr2addr_user      : out std_logic_vector(3 downto 0);                           --
         -- The next command burst type to put on the AXI MMap BURST                    --
                                                                                        --
      mstr2addr_cmd_cmplt : out std_logic;                                              --
         -- The indication to the Address Channel that the current                      --
         -- sub-command output is the last one compiled from the                        --
         -- parent command pulled from the Command FIFO                                 --
                                                                                        --
      mstr2addr_calc_error : out std_logic;                                             --
         -- Indication if the next command in the calculation pipe                      --
         -- has a calcualtion error                                                     --
                                                                                        --
      mstr2addr_cmd_valid : out std_logic;                                              --
         -- The next command valid indication to the Address Channel                    --
         -- Controller for the AXI MMap                                                 --
                                                                                        --
      addr2mstr_cmd_ready : In std_logic;                                               --
         -- Indication from the Address Channel Controller that the                     --
         -- command is being accepted                                                   --
      ------------------------------------------------------------------------------------
      
      
      
      -- Data Channel Controller Interface  ----------------------------------------------
                                                                                        --
      mstr2data_tag        : out std_logic_vector(C_TAG_WIDTH-1 downto 0);              --
         -- The next command tag                                                        --
                                                                                        --
      mstr2data_saddr_lsb  : out std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0);         --
         -- The next command start address LSbs to use for the read data                --
         -- mux (only used if Stream data width is 8 or 16 bits).                       --
                                                                                        --
      mstr2data_len        : out std_logic_vector(7 downto 0);                          --
         -- The LEN value output to the Address Channel                                 --
                                                                                        --
      mstr2data_strt_strb  : out std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);      --
         -- The starting strobe value to use for the data transfer                      --
                                                                                        --
      mstr2data_last_strb  : out std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);      --
         -- The endiing (LAST) strobe value to use for the data transfer                --
                                                                                        --
      mstr2data_sof        : out std_logic;                                             --
         -- The starting tranfer of a sequence of transfers                             --
                                                                                        --
      mstr2data_eof        : out std_logic;                                             --
         -- The endiing tranfer of a sequence of parent transfer commands               --
                                                                                        --
      mstr2data_calc_error : out std_logic;                                             --
         -- Indication if the next command in the calculation pipe                      --
         -- has a calculation error                                                     --
                                                                                        --
      mstr2data_cmd_cmplt  : out std_logic;                                             --
         -- The indication to the Data Channel that the current                         --
         -- sub-command output is the last one compiled from the                        --
         -- parent command pulled from the Command FIFO                                 --
                                                                                        --
      mstr2data_cmd_valid  : out std_logic;                                             --
         -- The next command valid indication to the Data Channel                       --
         -- Controller for the AXI MMap                                                 --
                                                                                        --
      data2mstr_cmd_ready  : In std_logic ;                                             --
         -- Indication from the Data Channel Controller that the                        --
         -- command is being accepted on the AXI Address                                --
         -- Channel                                                                     --
                                                                                        --
      calc_error           : Out std_logic                                              --
         -- Indication from the Command Calculator that a calculation                   --
         -- error has occured.                                                          --
      ------------------------------------------------------------------------------------
     
      );
  
  end entity axi_datamover_scc;
  
  
  architecture implementation of axi_datamover_scc is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
  
  
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_slice_width
    --
    -- Function Description:
    -- Calculates the bits to rip from the Command BTT field to calculate
    -- the LEN value output to the AXI Address Channel.
    --
    -------------------------------------------------------------------
    function funct_get_slice_width (max_burst_len : integer) return integer is
                                    
    
      Variable temp_slice_width : Integer := 0;
    
    begin
  
      case max_burst_len is
        
        when 64 =>
          temp_slice_width := 7;
        when 32 =>
          temp_slice_width := 6;
        when 16 =>
          temp_slice_width := 5;
        when 8 =>
          temp_slice_width := 4;
        when 4 =>
          temp_slice_width := 3;
        when others =>   -- assume 16 dbeats is max LEN
          temp_slice_width := 2;
      end case;
      
      Return (temp_slice_width);
     
    end function funct_get_slice_width;
    
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_residue_width
    --
    -- Function Description:
    --  Calculates the number of Least significant bits of the BTT field
    -- that are unused for the LEN calculation
    --
    -------------------------------------------------------------------
    function funct_get_btt_ls_unused (transfer_width : integer) return integer is
    
      Variable temp_btt_ls_unused : Integer := 0; -- 8-bit stream
      
    begin
    
      case transfer_width is
        when 64 =>
            temp_btt_ls_unused := 3;
        when 32 =>
            temp_btt_ls_unused := 2;
        when 16 =>
            temp_btt_ls_unused := 1;
        when others =>  -- assume 8-bit transfers
            temp_btt_ls_unused := 0;
      end case;
      
      Return (temp_btt_ls_unused);
     
    end function funct_get_btt_ls_unused;
    
    
    
    
    
    
    
    
    
    -- Constant Declarations  ----------------------------------------
    
    Constant BASE_CMD_WIDTH      : integer := 32; -- Bit Width of Command LS (no address)
    Constant CMD_TYPE_INDEX      : integer := 23;
    Constant CMD_ADDR_LS_INDEX   : integer := BASE_CMD_WIDTH;
    Constant CMD_EOF_INDEX     : integer := BASE_CMD_WIDTH-2;
    Constant CMD_ADDR_MS_INDEX   : integer := (C_ADDR_WIDTH+BASE_CMD_WIDTH)-1;
    Constant CMD_TAG_WIDTH       : integer := C_TAG_WIDTH;
    Constant CMD_TAG_LS_INDEX    : integer := C_ADDR_WIDTH+BASE_CMD_WIDTH;
    Constant CMD_TAG_MS_INDEX    : integer := (CMD_TAG_LS_INDEX+CMD_TAG_WIDTH)-1;
    Constant AXI_BURST_FIXED     : std_logic_vector(1 downto 0) := "00";
    Constant AXI_BURST_INCR      : std_logic_vector(1 downto 0) := "01";
    Constant AXI_BURST_WRAP      : std_logic_vector(1 downto 0) := "10";
    Constant AXI_BURST_RESVD     : std_logic_vector(1 downto 0) := "11";
    Constant AXI_SIZE_1BYTE      : std_logic_vector(2 downto 0) := "000"; 
    Constant AXI_SIZE_2BYTE      : std_logic_vector(2 downto 0) := "001"; 
    Constant AXI_SIZE_4BYTE      : std_logic_vector(2 downto 0) := "010"; 
    Constant AXI_SIZE_8BYTE      : std_logic_vector(2 downto 0) := "011"; 
    Constant AXI_SIZE_16BYTE     : std_logic_vector(2 downto 0) := "100"; 
    Constant AXI_SIZE_32BYTE     : std_logic_vector(2 downto 0) := "101"; 
    Constant AXI_SIZE_64BYTE     : std_logic_vector(2 downto 0) := "110"; 
    Constant AXI_SIZE_128BYTE    : std_logic_vector(2 downto 0) := "111"; 
    Constant BTT_SLICE_SIZE      : integer := funct_get_slice_width(C_MAX_BURST_LEN);
    Constant MAX_BURST_LEN_US    : unsigned(BTT_SLICE_SIZE-1 downto 0) := 
                                   TO_UNSIGNED(C_MAX_BURST_LEN-1, BTT_SLICE_SIZE);
    Constant BTT_LS_UNUSED_WIDTH : integer := funct_get_btt_ls_unused(C_STREAM_DWIDTH);
    Constant CMD_BTT_WIDTH       : integer :=  BTT_SLICE_SIZE+BTT_LS_UNUSED_WIDTH;
    Constant CMD_BTT_LS_INDEX    : integer :=  0;
    Constant CMD_BTT_MS_INDEX    : integer :=  CMD_BTT_WIDTH-1;
    Constant BTT_ZEROS           : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0'); 
    Constant BTT_RESIDUE_ZEROS   : unsigned(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0'); 
    Constant BTT_SLICE_ONE       : unsigned(BTT_SLICE_SIZE-1 downto 0) := TO_UNSIGNED(1, BTT_SLICE_SIZE); 
    Constant STRB_WIDTH          : integer := C_STREAM_DWIDTH/8; -- Number of bytes in the Stream
    Constant LEN_WIDTH           : integer := 8; 
    
               
               
    -- Type Declarations  --------------------------------------------
    
    type SCC_SM_STATE_TYPE is (
                INIT,
                POP_RECOVER,
                GET_NXT_CMD,
                CHK_AND_CALC,
                PUSH_TO_AXI,
                ERROR_TRAP
                );
  
  
    
    
    
    -- Signal Declarations  --------------------------------------------
  
    signal sm_scc_state              : SCC_SM_STATE_TYPE := INIT;
    signal sm_scc_state_ns           : SCC_SM_STATE_TYPE := INIT;
    signal sm_pop_input_cmd          : std_logic := '0';
    signal sm_pop_input_cmd_ns       : std_logic := '0';
    signal sm_set_push2axi           : std_logic := '0';
    signal sm_set_push2axi_ns        : std_logic := '0';
    signal sm_set_error              : std_logic := '0';
    signal sm_set_error_ns           : std_logic := '0';
    Signal sm_scc_sm_ready           : std_logic := '0';
    Signal sm_scc_sm_ready_ns        : std_logic := '0';
    signal sig_cmd2data_valid        : std_logic := '0';
    signal sig_clr_cmd2data_valid    : std_logic := '0';
    signal sig_cmd2addr_valid        : std_logic := '0';
    signal sig_clr_cmd2addr_valid    : std_logic := '0';
    signal sig_addr_data_rdy_pending : std_logic := '0';
    signal sig_cmd_btt_slice         : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    signal sig_load_input_cmd        : std_logic := '0';
    signal sig_cmd_reg_empty         : std_logic := '0';
    signal sig_cmd_reg_full          : std_logic := '0';
    signal sig_cmd_addr_reg          : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd_btt_reg           : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd_type_reg          : std_logic := '0';
    signal sig_cmd_burst_reg         : std_logic_vector (1 downto 0) := "00";
    signal sig_cmd_tag_reg           : std_logic_vector(CMD_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_data_rdy4cmd     : std_logic := '0';
    signal sig_btt_raw               : std_logic := '0';
    signal sig_btt_is_zero           : std_logic := '0';
    signal sig_btt_is_zero_reg       : std_logic := '0';
    signal sig_next_tag              : std_logic_vector(CMD_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_addr             : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_len              : std_logic_vector(LEN_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_size             : std_logic_vector(2 downto 0) := (others => '0');
    signal sig_next_burst            : std_logic_vector(1 downto 0) := (others => '0');
    signal sig_next_cache            : std_logic_vector(3 downto 0) := (others => '0');
    signal sig_next_user             : std_logic_vector(3 downto 0) := (others => '0');
    signal sig_next_strt_strb        : std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0) := (others => '0');
    signal sig_next_end_strb         : std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0) := (others => '0');
    signal sig_input_eof_reg         : std_logic;
    
  
    
    
  begin --(architecture implementation)
     
    -- Assign calculation error output 
    calc_error            <= sm_set_error;
   
    -- Assign the ready output to the Command FIFO 
    mst2cmd_cmd_ready     <= sig_cmd_reg_empty and sm_scc_sm_ready;
    
    -- Assign the Address Channel Controller Qualifiers
    mstr2addr_tag         <= sig_next_tag  ; 
    mstr2addr_addr        <= sig_next_addr ; 
    mstr2addr_len         <= sig_next_len  ; 
    mstr2addr_size        <= sig_next_size ;
    mstr2addr_burst       <= sig_cmd_burst_reg; 
    mstr2addr_cache       <= sig_next_cache; 
    mstr2addr_user        <= sig_next_user; 
    mstr2addr_cmd_valid   <= sig_cmd2addr_valid;
    mstr2addr_calc_error  <= sm_set_error  ; 
    mstr2addr_cmd_cmplt   <= '1'           ;   -- Lite mode is always 1 
    
    -- Assign the Data Channel Controller Qualifiers
    mstr2data_tag         <= sig_next_tag ; 
    mstr2data_saddr_lsb   <= sig_cmd_addr_reg(C_SEL_ADDR_WIDTH-1 downto 0);
    
    mstr2data_len         <= sig_next_len ;
    
    mstr2data_strt_strb   <= sig_next_strt_strb;
    mstr2data_last_strb   <= sig_next_end_strb;
    mstr2data_sof         <= '1';  -- Lite mode is always 1 cmd
    mstr2data_eof         <= sig_input_eof_reg;  -- Lite mode is always 1 cmd
    mstr2data_cmd_cmplt   <= '1';  -- Lite mode is always 1 cmd
    mstr2data_cmd_valid   <= sig_cmd2data_valid;
    mstr2data_calc_error  <= sm_set_error;   
    
    
    -- Internal logic ------------------------------
    sig_addr_data_rdy_pending  <=  sig_cmd2addr_valid or 
                                   sig_cmd2data_valid;
   
    sig_clr_cmd2data_valid     <=  sig_cmd2data_valid and data2mstr_cmd_ready;
    
    sig_clr_cmd2addr_valid     <=  sig_cmd2addr_valid and addr2mstr_cmd_ready;
    
    
    sig_load_input_cmd         <=  cmd2mstr_cmd_valid and 
                                   sig_cmd_reg_empty  and
                                   sm_scc_sm_ready;
    
    sig_next_tag               <=  sig_cmd_tag_reg;
    
    sig_next_addr              <=  sig_cmd_addr_reg;
    
    sig_addr_data_rdy4cmd      <=  addr2mstr_cmd_ready and data2mstr_cmd_ready;
    
    sig_cmd_btt_slice          <=  cmd2mstr_command(CMD_BTT_MS_INDEX downto CMD_BTT_LS_INDEX);
    
    sig_btt_is_zero  <= '1'
      when  (sig_cmd_btt_slice = BTT_ZEROS)
      Else '0';
    
  
  
    
 
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_NO_RESIDUE_BITS
    --
    -- If Generate Description:
    --
    --
    --
    ------------------------------------------------------------
    GEN_NO_RESIDUE_BITS : if (BTT_LS_UNUSED_WIDTH = 0) generate
    
    
       -- signals 
       signal sig_len_btt_slice         : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len_btt_slice_minus_1 : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len2use               : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       
     
       begin
    
       -- LEN Calculation logic ------------------------------------------ 
        
         sig_next_len         <= STD_LOGIC_VECTOR(RESIZE(sig_len2use, LEN_WIDTH));
        
         sig_len_btt_slice    <= UNSIGNED(sig_cmd_btt_reg(CMD_BTT_MS_INDEX downto 0));
         
         sig_len_btt_slice_minus_1 <= sig_len_btt_slice-BTT_SLICE_ONE
          when sig_btt_is_zero_reg = '0'
          else (others => '0');    -- clip at zero
         
         
         -- If most significant bit of BTT set then limit to 
         -- Max Burst Len, else rip it from the BTT value,
         -- otheriwse subtract 1 from the BTT ripped value
         -- 1 from the BTT ripped value
         sig_len2use <= MAX_BURST_LEN_US 
           When (sig_cmd_btt_reg(CMD_BTT_MS_INDEX) = '1')
           Else sig_len_btt_slice_minus_1;
           
        
       end generate GEN_NO_RESIDUE_BITS;
        
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_HAS_RESIDUE_BITS
    --
    -- If Generate Description:
    --
    --
    --
    ------------------------------------------------------------
    GEN_HAS_RESIDUE_BITS : if (BTT_LS_UNUSED_WIDTH > 0) generate
    
       -- signals 
       signal sig_btt_len_residue       : unsigned(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       signal sig_len_btt_slice         : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len_btt_slice_minus_1 : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len2use               : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       
     
       begin
    
       -- LEN Calculation logic ------------------------------------------ 
        
         sig_next_len         <= STD_LOGIC_VECTOR(RESIZE(sig_len2use, LEN_WIDTH));
        
         sig_len_btt_slice    <= UNSIGNED(sig_cmd_btt_reg(CMD_BTT_MS_INDEX downto BTT_LS_UNUSED_WIDTH));
         
         sig_len_btt_slice_minus_1 <= sig_len_btt_slice-BTT_SLICE_ONE
          when sig_btt_is_zero_reg = '0'
          else (others => '0');    -- clip at zero
         
         sig_btt_len_residue  <= UNSIGNED(sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0));
            
         
         -- If most significant bit of BTT set then limit to 
         -- Max Burst Len, else rip it from the BTT value
         -- However if residue bits are zeroes then subtract
         -- 1 from the BTT ripped value
         sig_len2use <= MAX_BURST_LEN_US 
           When (sig_cmd_btt_reg(CMD_BTT_MS_INDEX) = '1')
           Else sig_len_btt_slice_minus_1
           when (sig_btt_len_residue = BTT_RESIDUE_ZEROS)
           Else sig_len_btt_slice;

        
       end generate GEN_HAS_RESIDUE_BITS;
        
        
        
        
        
        
        
     
     
     
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_INPUT_CMD
    --
    -- Process Description:
    --  Implements the input command holding registers
    --
    -------------------------------------------------------------
    REG_INPUT_CMD : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset       = '1' or
                sm_pop_input_cmd = '1') then
            
              sig_cmd_btt_reg      <=  (others => '0');
              sig_cmd_type_reg     <=  '0';
              sig_cmd_addr_reg     <=  (others => '0');
              sig_cmd_tag_reg      <=  (others => '0');
              sig_btt_is_zero_reg  <=  '0';
              
              sig_cmd_reg_empty    <=  '1';
              sig_cmd_reg_full     <=  '0';
              sig_input_eof_reg    <= '0'; 
              sig_cmd_burst_reg    <=  "00";
            
            elsif (sig_load_input_cmd = '1') then
              
              sig_cmd_btt_reg      <= sig_cmd_btt_slice;
              sig_cmd_type_reg     <= cmd2mstr_command(CMD_TYPE_INDEX);
              sig_cmd_addr_reg     <= cmd2mstr_command(CMD_ADDR_MS_INDEX downto CMD_ADDR_LS_INDEX);  
              sig_cmd_tag_reg      <=  cmd2mstr_command(CMD_TAG_MS_INDEX downto CMD_TAG_LS_INDEX);
              sig_btt_is_zero_reg  <= sig_btt_is_zero;
              
              sig_cmd_reg_empty    <=  '0';
              sig_cmd_reg_full     <=  '1';

              sig_cmd_burst_reg    <= sig_next_burst;
              if (C_MICRO_DMA = 1) then  
                sig_input_eof_reg    <= cmd2mstr_command(CMD_EOF_INDEX);
              else
                sig_input_eof_reg    <= '1';
              end if;
            
            else
              null; -- Hold current State
            end if; 
         end if;       
       end process REG_INPUT_CMD; 
   
   
    
    
    -- Only Incrementing Burst type supported (per Interface_X guidelines)
    sig_next_burst <= AXI_BURST_INCR when (cmd2mstr_command(CMD_TYPE_INDEX) = '1') else
                      AXI_BURST_FIXED;
    sig_next_user <= cache2mstr_command (7 downto 4);   
    sig_next_cache  <= cache2mstr_command (3 downto 0);   
 
 
  
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_64
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 64-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_64 : if (C_STREAM_DWIDTH = 64) generate
    
       -- Local Constants
       Constant AXI_SIZE2USE      : std_logic_vector(2 downto 0) := AXI_SIZE_8BYTE;
       Constant RESIDUE_BIT_WIDTH : integer := 3;
       
       
       
       -- local signals
       signal sig_last_strb2use              : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       signal sig_last_strb                  : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       Signal sig_btt_ms_bit_value           : std_logic := '0';
       signal lsig_btt_len_residue           : std_logic_vector(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       signal sig_btt_len_residue_composite  : std_logic_vector(RESIDUE_BIT_WIDTH downto 0) := (others => '0');
                                                                -- note 1 extra bit implied
                                                                
       
       begin
           
         -- Assign the Address Channel Controller Size Qualifier Value
         sig_next_size        <= AXI_SIZE2USE;
         
         -- Assign the Strobe Values
         sig_next_strt_strb   <= (others => '1'); -- always aligned on first databeat for LITE DataMover
         sig_next_end_strb    <= sig_last_strb;
        
        
         -- Local calculations ------------------------------
         
         lsig_btt_len_residue  <= sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0);
         
         sig_btt_ms_bit_value  <= sig_cmd_btt_reg(CMD_BTT_MS_INDEX);
         
         sig_btt_len_residue_composite <= sig_btt_ms_bit_value &
                                          lsig_btt_len_residue;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: IMP_LAST_STRB_8bit
         --
         -- Process Description:
         -- Generates the Strobe values for the LAST databeat of the
         -- Burst to MMap when the Stream is 64 bits wide and 8 strobe
         -- bits are required.
         --
         -------------------------------------------------------------
         IMP_LAST_STRB_8bit : process (sig_btt_len_residue_composite)
            begin
         
             case sig_btt_len_residue_composite is
               when "0001" =>
                 sig_last_strb <= "00000001";
               when "0010" =>
                 sig_last_strb <= "00000011";
               when "0011" =>
                 sig_last_strb <= "00000111";
               when "0100" =>
                 sig_last_strb <= "00001111";
               when "0101" =>
                 sig_last_strb <= "00011111";
               when "0110" =>
                 sig_last_strb <= "00111111";
               when "0111" =>
                 sig_last_strb <= "01111111";
               when others =>
                 sig_last_strb <= "11111111";
             end case;
             
             
            end process IMP_LAST_STRB_8bit; 
         
         
       end generate GEN_LEN_SDWIDTH_64;
  
  
  
  
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_32
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 32-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_32 : if (C_STREAM_DWIDTH = 32) generate
    
       -- Local Constants
       Constant AXI_SIZE2USE                 : std_logic_vector(2 downto 0) := AXI_SIZE_4BYTE;
       Constant RESIDUE_BIT_WIDTH            : integer := 2;
       
       -- local signals
       signal sig_last_strb2use              : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       signal sig_last_strb                  : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       Signal sig_btt_ms_bit_value           : std_logic := '0';
       signal sig_btt_len_residue_composite  : std_logic_vector(RESIDUE_BIT_WIDTH downto 0) := (others => '0'); -- 1 extra bit
       signal lsig_btt_len_residue           : std_logic_vector(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       
       
       begin
           
         -- Assign the Address Channel Controller Size Qualifier Value
         sig_next_size        <= AXI_SIZE2USE;
         
         -- Assign the Strobe Values
         sig_next_strt_strb   <= (others => '1'); -- always aligned on first databeat for LITE DataMover
         sig_next_end_strb    <= sig_last_strb;
        
        
         -- Local calculations ------------------------------
         
         lsig_btt_len_residue  <= sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0);
         
         sig_btt_ms_bit_value <= sig_cmd_btt_reg(CMD_BTT_MS_INDEX);
         
         sig_btt_len_residue_composite <= sig_btt_ms_bit_value &
                                          lsig_btt_len_residue;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: IMP_LAST_STRB_4bit
         --
         -- Process Description:
         -- Generates the Strobe values for the LAST databeat of the
         -- Burst to MMap when the Stream is 32 bits wide and 4 strobe
         -- bits are required.
         --
         -------------------------------------------------------------
         IMP_LAST_STRB_4bit : process (sig_btt_len_residue_composite)
            begin
         
             case sig_btt_len_residue_composite is
               when "001" =>
                 sig_last_strb <= "0001";
               when "010" =>
                 sig_last_strb <= "0011";
               when "011" =>
                 sig_last_strb <= "0111";
               when others =>
                 sig_last_strb <= "1111";
             end case;
             
             
            end process IMP_LAST_STRB_4bit; 
          
       end generate GEN_LEN_SDWIDTH_32;
  
  
  
  
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_16
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 16-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_16 : if (C_STREAM_DWIDTH = 16) generate
    
       -- Local Constants
       Constant AXI_SIZE2USE      : std_logic_vector(2 downto 0) := AXI_SIZE_2BYTE;
       Constant RESIDUE_BIT_WIDTH : integer := 1;
       
       
       -- local signals
       signal sig_last_strb2use              : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       signal sig_last_strb                  : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       Signal sig_btt_ms_bit_value           : std_logic := '0';
       signal sig_btt_len_residue_composite  : std_logic_vector(RESIDUE_BIT_WIDTH downto 0) := (others => '0'); -- 1 extra bit
       signal lsig_btt_len_residue           : std_logic_vector(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       
       
       begin
           
         -- Assign the Address Channel Controller Size Qualifier Value
         sig_next_size        <= AXI_SIZE2USE;
         
         -- Assign the Strobe Values
         sig_next_strt_strb   <= (others => '1'); -- always aligned on first databeat for LITE DataMover
         sig_next_end_strb    <= sig_last_strb;
        
        
         -- Local calculations ------------------------------
         
         lsig_btt_len_residue  <= sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0);
         
         sig_btt_ms_bit_value  <= sig_cmd_btt_reg(CMD_BTT_MS_INDEX);
         
         sig_btt_len_residue_composite <= sig_btt_ms_bit_value &
                                          lsig_btt_len_residue;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: IMP_LAST_STRB_2bit
         --
         -- Process Description:
         -- Generates the Strobe values for the LAST databeat of the
         -- Burst to MMap when the Stream is 16 bits wide and 2 strobe
         -- bits are required.
         --
         -------------------------------------------------------------
         IMP_LAST_STRB_2bit : process (sig_btt_len_residue_composite)
            begin
         
             case sig_btt_len_residue_composite is
               when "01" =>
                 sig_last_strb <= "01";
               when others =>
                 sig_last_strb <= "11";
             end case;
             
             
            end process IMP_LAST_STRB_2bit; 
         
         
       end generate GEN_LEN_SDWIDTH_16;
  
  
  
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_8
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 8-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_8 : if (C_STREAM_DWIDTH = 8) generate
       
       -- Local Constants
       Constant AXI_SIZE2USE : std_logic_vector(2 downto 0) := AXI_SIZE_1BYTE;
       
       begin
    
           -- Assign the Address Channel Controller Qualifiers
         sig_next_size        <= AXI_SIZE2USE;
           
         -- Assign the Data Channel Controller Qualifiers
         sig_next_strt_strb   <= (others => '1');
         sig_next_end_strb    <= (others => '1');
          
          
       end generate GEN_LEN_SDWIDTH_8;
  
  
  
  
  
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: CMD2DATA_VALID_FLOP
    --
    -- Process Description:
    --  Implements the set/reset flop for the Command Ready control
    -- to the Data Controller Module.
    --
    -------------------------------------------------------------
    CMD2DATA_VALID_FLOP : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset             = '1' or
                sig_clr_cmd2data_valid = '1') then
    
              sig_cmd2data_valid <= '0';
              
            elsif (sm_set_push2axi_ns = '1') then
    
              sig_cmd2data_valid <= '1';
              
            else
              null; -- hold current state
            end if; 
         end if;       
       end process CMD2DATA_VALID_FLOP; 
      
      
      
      
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: CMD2ADDR_VALID_FLOP
    --
    -- Process Description:
    --  Implements the set/reset flop for the Command Ready control
    -- to the Address Controller Module.
    --
    -------------------------------------------------------------
    CMD2ADDR_VALID_FLOP : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset             = '1' or
                sig_clr_cmd2addr_valid = '1') then
    
              sig_cmd2addr_valid <= '0';
              
            elsif (sm_set_push2axi_ns = '1') then
    
              sig_cmd2addr_valid <= '1';
              
            else
              null; -- hold current state
            end if; 
         end if;       
       end process CMD2ADDR_VALID_FLOP; 
      
      
      
      
    
    
    -------------------------------------------------------------
    -- Combinational Process
    --
    -- Label: SCC_SM_COMB
    --
    -- Process Description:
    -- Implements combinational portion of state machine
    --
    -------------------------------------------------------------
    SCC_SM_COMB : process (sm_scc_state,
                           cmd2mstr_cmd_valid,
                           sig_addr_data_rdy_pending, 
                           sig_cmd_reg_full,
                           sig_btt_is_zero_reg
                          )
       begin
    
         -- Set default State machine outputs
         sm_pop_input_cmd_ns  <= '0';
         sm_set_push2axi_ns   <= '0';
         sm_scc_state_ns      <= sm_scc_state;
         sm_set_error_ns      <= '0';
         sm_scc_sm_ready_ns   <= '1';
          
          
         case sm_scc_state is
           
           ----------------------------------------------------
           when INIT =>
             
             -- if (sig_addr_data_rdy4cmd = '1') then
             if (cmd2mstr_cmd_valid = '1') then  -- wait for first cmd valid after reset
             
               sm_scc_state_ns   <= GET_NXT_CMD;  -- jump to get command
       
              else
               
               sm_scc_sm_ready_ns <= '0';
               sm_scc_state_ns    <= INIT;  -- Stay in Init
              
              End if;
             
               
           ----------------------------------------------------
           when POP_RECOVER =>
           
               sm_scc_state_ns    <= GET_NXT_CMD;  -- jump to next state
           
             
           ----------------------------------------------------
           when GET_NXT_CMD =>
             
             if (sig_cmd_reg_full = '1') then
             
               sm_scc_state_ns    <= CHK_AND_CALC;  -- jump to next state
               
             else
             
               sm_scc_state_ns    <= GET_NXT_CMD;  -- stay in this state
             
             end if;
             
           
           ----------------------------------------------------
           when CHK_AND_CALC =>
             
             sm_set_push2axi_ns <= '1';  -- Push the command to ADDR and DATA
             
             if (sig_btt_is_zero_reg = '1') then
             
               sm_scc_state_ns    <= ERROR_TRAP;  -- jump to error trap
               sm_set_error_ns    <= '1';         -- Set internal error flag
               
             else
             
               sm_scc_state_ns    <= PUSH_TO_AXI;  
             
             end if;
             
           
           ----------------------------------------------------
           when PUSH_TO_AXI =>
           
             if (sig_addr_data_rdy_pending = '1') then
             
               sm_scc_state_ns    <= PUSH_TO_AXI;  -- stay in this state
                                                   -- until both Addr and Data have taken commands
             else
             
               sm_pop_input_cmd_ns  <= '1';
               sm_scc_state_ns      <= POP_RECOVER; -- jump back to fetch new cmd input 
             
             end if;
             
           
           
           ----------------------------------------------------
           when ERROR_TRAP =>
           
             sm_scc_state_ns    <= ERROR_TRAP;  -- stay in this state
             sm_set_error_ns    <= '1';
           
             
           ----------------------------------------------------
           when others =>
               
             sm_scc_state_ns    <= INIT; -- error so always jump to init state
               
         end case;
         
    
       end process SCC_SM_COMB; 
    
  
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: SCC_SM_REG
    --
    -- Process Description:
    -- Implements registered portion of state machine
    --
    -------------------------------------------------------------
    SCC_SM_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sm_scc_state      <= INIT;
              sm_pop_input_cmd  <= '0' ;           
              sm_set_push2axi   <= '0' ;           
              sm_set_error      <= '0' ;
              sm_scc_sm_ready   <= '0' ;
              
            else
              
              sm_scc_state      <= sm_scc_state_ns     ;
              sm_pop_input_cmd  <= sm_pop_input_cmd_ns ;           
              sm_set_push2axi   <= sm_set_push2axi_ns  ;           
              sm_set_error      <= sm_set_error_ns     ;
              sm_scc_sm_ready   <= sm_scc_sm_ready_ns  ;
              
            end if; 
         end if;       
       end process SCC_SM_REG; 
    
    
  
  
  
  end implementation;


-------------------------------------------------------------------------------
-- axi_datamover_strb_gen2.vhd
-------------------------------------------------------------------------------
--
-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_datamover_strb_gen2.vhd
--
-- Description:     
--   Second generation AXI Strobe Generator module. This design leverages
-- look up table approach vs real-time calculation. This design method is 
-- used to reduce logic levels and improve final Fmax timing.               
--                  
--                  
--                  
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;




-------------------------------------------------------------------------------

entity axi_datamover_strb_gen2 is
  generic (
    C_OP_MODE            : Integer range 0 to 1 := 0; 
      -- 0 = offset/length mode 
      -- 1 = offset/offset mode,
      
    C_STRB_WIDTH         : Integer := 8; 
    -- number of addr bits needed
    
    C_OFFSET_WIDTH       : Integer := 3; 
    -- log2(C_STRB_WIDTH)
    
    C_NUM_BYTES_WIDTH    : Integer := 4 
      -- log2(C_STRB_WIDTH)+1 in offset/length mode (C_OP_MODE = 0)
      -- log2(C_STRB_WIDTH) in offset/offset mode   (C_OP_MODE = 1)
    );
  port (
    
    -- Starting offset input -----------------------------------------------------
                                                                                --
    start_addr_offset    : In  std_logic_vector(C_OFFSET_WIDTH-1 downto 0);     --
      -- Specifies the starting address offset of the strobe value              --
    ------------------------------------------------------------------------------
      -- used in both offset/offset and offset/length modes
      
    
    
    -- Endig Offset Input --------------------------------------------------------
                                                                                --
    end_addr_offset      : In  std_logic_vector(C_OFFSET_WIDTH-1 downto 0);     --
      -- Specifies the ending address offset of the strobe value                --
      -- used in only offset/offset mode (C_OP_MODE = 1)                        --
    ------------------------------------------------------------------------------
    
      
    -- Number of valid Bytes input (from starting offset) ------------------------
                                                                                --
    num_valid_bytes      : In  std_logic_vector(C_NUM_BYTES_WIDTH-1 downto 0);  --
      -- Specifies the number of valid bytes from starting offset               --
      -- used in only offset/length mode (C_OP_MODE = 0)                        --
    ------------------------------------------------------------------------------  
      
    
    -- Generated Strobe output ---------------------------------------------------  
                                                                                --
    strb_out             : out std_logic_vector(C_STRB_WIDTH-1 downto 0)        --
    ------------------------------------------------------------------------------
    
    
    );

end entity axi_datamover_strb_gen2;


architecture implementation of axi_datamover_strb_gen2 is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_start_2
  --
  -- Function Description:
  --   returns the 2-bit vector filled with '1's from the start
  -- offset to the end of of the vector
  --
  -------------------------------------------------------------------
  function get_start_2 (start_offset : natural) return std_logic_vector is
  
    Variable var_start_vector : std_logic_vector(1 downto 0) := (others => '0');
  
  begin
  
    case start_offset is
      when 0 =>
        var_start_vector := "11";
      when others =>
        var_start_vector := "10";
    end case;
   
    Return (var_start_vector);
   
  end function get_start_2;
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_end_2
  --
  -- Function Description:
  --   Returns the 2-bit vector filled with '1's from the lsbit
  -- of the vector to the end offset.
  --
  -------------------------------------------------------------------
  function get_end_2 (end_offset : natural) return std_logic_vector is
  
    Variable var_end_vector : std_logic_vector(1 downto 0) := (others => '0');
  
  begin
  
    case end_offset is
      when 0 =>
        var_end_vector := "01";
      when others =>
        var_end_vector := "11";
    end case;
   
    Return (var_end_vector);
   
  end function get_end_2; 
  
  
  
  
  
  
  
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_start_4
  --
  -- Function Description:
  --   returns the 4-bit vector filled with '1's from the start
  -- offset to the end of of the vector
  --
  -------------------------------------------------------------------
  function get_start_4 (start_offset : natural) return std_logic_vector is
  
    Variable var_start_vector : std_logic_vector(3 downto 0) := (others => '0');
  
  begin
  
    case start_offset is
      when 0 =>
        var_start_vector := "1111";
      when 1 =>
        var_start_vector := "1110";
      when 2 =>
        var_start_vector := "1100";
      when others =>
        var_start_vector := "1000";
    end case;
   
    Return (var_start_vector);
   
  end function get_start_4;
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_end_4
  --
  -- Function Description:
  --   Returns the 4-bit vector filled with '1's from the lsbit
  -- of the vector to the end offset.
  --
  -------------------------------------------------------------------
  function get_end_4 (end_offset : natural) return std_logic_vector is
  
    Variable var_end_vector : std_logic_vector(3 downto 0) := (others => '0');
  
  begin
  
    case end_offset is
      when 0 =>
        var_end_vector := "0001";
      when 1 =>
        var_end_vector := "0011";
      when 2 =>
        var_end_vector := "0111";
      when others =>
        var_end_vector := "1111";
    end case;
   
    Return (var_end_vector);
   
  end function get_end_4; 
  
  
  
  
  
  
  
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_start_8
  --
  -- Function Description:
  --   returns the 8-bit vector filled with '1's from the start
  -- offset to the end of of the vector
  --
  -------------------------------------------------------------------
  function get_start_8 (start_offset : natural) return std_logic_vector is
  
    Variable var_start_vector : std_logic_vector(7 downto 0) := (others => '0');
  
  begin
  
    case start_offset is
      when 0 =>
        var_start_vector := "11111111";
      when 1 =>
        var_start_vector := "11111110";
      when 2 =>
        var_start_vector := "11111100";
      when 3 =>
        var_start_vector := "11111000";
      when 4 =>
        var_start_vector := "11110000";
      when 5 =>
        var_start_vector := "11100000";
      when 6 =>
        var_start_vector := "11000000";
      when others =>
        var_start_vector := "10000000";
    end case;
   
    Return (var_start_vector);
   
  end function get_start_8;
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_end_8
  --
  -- Function Description:
  --   Returns the 8-bit vector filled with '1's from the lsbit
  -- of the vector to the end offset.
  --
  -------------------------------------------------------------------
  function get_end_8 (end_offset : natural) return std_logic_vector is
  
    Variable var_end_vector : std_logic_vector(7 downto 0) := (others => '0');
  
  begin
  
    case end_offset is
      when 0 =>
        var_end_vector := "00000001";
      when 1 =>
        var_end_vector := "00000011";
      when 2 =>
        var_end_vector := "00000111";
      when 3 =>
        var_end_vector := "00001111";
      when 4 =>
        var_end_vector := "00011111";
      when 5 =>
        var_end_vector := "00111111";
      when 6 =>
        var_end_vector := "01111111";
      when others =>
        var_end_vector := "11111111";
    end case;
   
    Return (var_end_vector);
   
  end function get_end_8; 
  
  
  
  
  
  
  
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_start_16
  --
  -- Function Description:
  --   returns the 16-bit vector filled with '1's from the start
  -- offset to the end of of the vector
  --
  -------------------------------------------------------------------
  function get_start_16 (start_offset : natural) return std_logic_vector is
  
    Variable var_start_vector : std_logic_vector(15 downto 0) := (others => '0');
  
  begin
  
    case start_offset is
      when 0 =>
        var_start_vector := "1111111111111111";
      when 1 =>
        var_start_vector := "1111111111111110";
      when 2 =>
        var_start_vector := "1111111111111100";
      when 3 =>
        var_start_vector := "1111111111111000";
      when 4 =>
        var_start_vector := "1111111111110000";
      when 5 =>
        var_start_vector := "1111111111100000";
      when 6 =>
        var_start_vector := "1111111111000000";
      when 7 =>
        var_start_vector := "1111111110000000";
      when 8 =>
        var_start_vector := "1111111100000000";
      when 9 =>
        var_start_vector := "1111111000000000";
      when 10 =>
        var_start_vector := "1111110000000000";
      when 11 =>
        var_start_vector := "1111100000000000";
      when 12 =>
        var_start_vector := "1111000000000000";
      when 13 =>
        var_start_vector := "1110000000000000";
      when 14 =>
        var_start_vector := "1100000000000000";
      when others =>
        var_start_vector := "1000000000000000";
    end case;
   
    Return (var_start_vector);
   
  end function get_start_16;
  
  
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_end_16
  --
  -- Function Description:
  --   Returns the 16-bit vector filled with '1's from the lsbit
  -- of the vector to the end offset.
  --
  -------------------------------------------------------------------
  function get_end_16 (end_offset : natural) return std_logic_vector is
  
    Variable var_end_vector : std_logic_vector(15 downto 0) := (others => '0');
  
  begin
  
    case end_offset is
      when 0 =>
        var_end_vector := "0000000000000001";
      when 1 =>
        var_end_vector := "0000000000000011";
      when 2 =>
        var_end_vector := "0000000000000111";
      when 3 =>
        var_end_vector := "0000000000001111";
      when 4 =>
        var_end_vector := "0000000000011111";
      when 5 =>
        var_end_vector := "0000000000111111";
      when 6 =>
        var_end_vector := "0000000001111111";
      when 7 =>
        var_end_vector := "0000000011111111";
      when 8 =>
        var_end_vector := "0000000111111111";
      when 9 =>
        var_end_vector := "0000001111111111";
      when 10 =>
        var_end_vector := "0000011111111111";
      when 11 =>
        var_end_vector := "0000111111111111";
      when 12 =>
        var_end_vector := "0001111111111111";
      when 13 =>
        var_end_vector := "0011111111111111";
      when 14 =>
        var_end_vector := "0111111111111111";
      when others =>
        var_end_vector := "1111111111111111";
    end case;
   
    Return (var_end_vector);
   
  end function get_end_16;
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_start_32
  --
  -- Function Description:
  --   returns the 32-bit vector filled with '1's from the start
  -- offset to the end of of the vector
  --
  -------------------------------------------------------------------
  function get_start_32 (start_offset : natural) return std_logic_vector is
  
    Variable var_start_vector : std_logic_vector(31 downto 0) := (others => '0');
  
  begin
  
    case start_offset is
      when 0 =>
        var_start_vector := "11111111111111111111111111111111";
      when 1 =>
        var_start_vector := "11111111111111111111111111111110";
      when 2 =>
        var_start_vector := "11111111111111111111111111111100";
      when 3 =>
        var_start_vector := "11111111111111111111111111111000";
      when 4 =>
        var_start_vector := "11111111111111111111111111110000";
      when 5 =>
        var_start_vector := "11111111111111111111111111100000";
      when 6 =>
        var_start_vector := "11111111111111111111111111000000";
      when 7 =>
        var_start_vector := "11111111111111111111111110000000";
      when 8 =>
        var_start_vector := "11111111111111111111111100000000";
      when 9 =>
        var_start_vector := "11111111111111111111111000000000";
      when 10 =>
        var_start_vector := "11111111111111111111110000000000";
      when 11 =>
        var_start_vector := "11111111111111111111100000000000";
      when 12 =>
        var_start_vector := "11111111111111111111000000000000";
      when 13 =>
        var_start_vector := "11111111111111111110000000000000";
      when 14 =>
        var_start_vector := "11111111111111111100000000000000";
      when 15 =>
        var_start_vector := "11111111111111111000000000000000";
      when 16 =>
        var_start_vector := "11111111111111110000000000000000";
      when 17 =>
        var_start_vector := "11111111111111100000000000000000";
      when 18 =>
        var_start_vector := "11111111111111000000000000000000";
      when 19 =>
        var_start_vector := "11111111111110000000000000000000";
      when 20 =>
        var_start_vector := "11111111111100000000000000000000";
      when 21 =>
        var_start_vector := "11111111111000000000000000000000";
      when 22 =>
        var_start_vector := "11111111110000000000000000000000";
      when 23 =>
        var_start_vector := "11111111100000000000000000000000";
      when 24 =>
        var_start_vector := "11111111000000000000000000000000";
      when 25 =>
        var_start_vector := "11111110000000000000000000000000";
      when 26 =>
        var_start_vector := "11111100000000000000000000000000";
      when 27 =>
        var_start_vector := "11111000000000000000000000000000";
      when 28 =>
        var_start_vector := "11110000000000000000000000000000";
      when 29 =>
        var_start_vector := "11100000000000000000000000000000";
      when 30 =>
        var_start_vector := "11000000000000000000000000000000";
      when others =>
        var_start_vector := "10000000000000000000000000000000";
    end case;
   
    Return (var_start_vector);
   
  end function get_start_32;
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_end_32
  --
  -- Function Description:
  --   Returns the 32-bit vector filled with '1's from the lsbit
  -- of the vector to the end offset.
  --
  -------------------------------------------------------------------
  function get_end_32 (end_offset : natural) return std_logic_vector is
  
    Variable var_end_vector : std_logic_vector(31 downto 0) := (others => '0');
  
  begin
  
    case end_offset is
      when 0 =>
        var_end_vector := "00000000000000000000000000000001";
      when 1 =>
        var_end_vector := "00000000000000000000000000000011";
      when 2 =>
        var_end_vector := "00000000000000000000000000000111";
      when 3 =>
        var_end_vector := "00000000000000000000000000001111";
      when 4 =>
        var_end_vector := "00000000000000000000000000011111";
      when 5 =>
        var_end_vector := "00000000000000000000000000111111";
      when 6 =>
        var_end_vector := "00000000000000000000000001111111";
      when 7 =>
        var_end_vector := "00000000000000000000000011111111";
      when 8 =>
        var_end_vector := "00000000000000000000000111111111";
      when 9 =>
        var_end_vector := "00000000000000000000001111111111";
      when 10 =>
        var_end_vector := "00000000000000000000011111111111";
      when 11 =>
        var_end_vector := "00000000000000000000111111111111";
      when 12 =>
        var_end_vector := "00000000000000000001111111111111";
      when 13 =>
        var_end_vector := "00000000000000000011111111111111";
      when 14 =>
        var_end_vector := "00000000000000000111111111111111";
      when 15 =>
        var_end_vector := "00000000000000001111111111111111";
      when 16 =>
        var_end_vector := "00000000000000011111111111111111";
      when 17 =>
        var_end_vector := "00000000000000111111111111111111";
      when 18 =>
        var_end_vector := "00000000000001111111111111111111";
      when 19 =>
        var_end_vector := "00000000000011111111111111111111";
      when 20 =>
        var_end_vector := "00000000000111111111111111111111";
      when 21 =>
        var_end_vector := "00000000001111111111111111111111";
      when 22 =>
        var_end_vector := "00000000011111111111111111111111";
      when 23 =>
        var_end_vector := "00000000111111111111111111111111";
      when 24 =>
        var_end_vector := "00000001111111111111111111111111";
      when 25 =>
        var_end_vector := "00000011111111111111111111111111";
      when 26 =>
        var_end_vector := "00000111111111111111111111111111";
      when 27 =>
        var_end_vector := "00001111111111111111111111111111";
      when 28 =>
        var_end_vector := "00011111111111111111111111111111";
      when 29 =>
        var_end_vector := "00111111111111111111111111111111";
      when 30 =>
        var_end_vector := "01111111111111111111111111111111";
      when others =>
        var_end_vector := "11111111111111111111111111111111";
    end case;
   
    Return (var_end_vector);
   
  end function get_end_32;
  
  
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_start_64
  --
  -- Function Description:
  --   returns the 64-bit vector filled with '1's from the start
  -- offset to the end of of the vector
  --
  -------------------------------------------------------------------
  function get_start_64 (start_offset : natural) return std_logic_vector is
  
    Variable var_start_vector : std_logic_vector(63 downto 0) := (others => '0');
  
  begin
  
    case start_offset is
      when 0 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111111111111111";
      when 1 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111111111111110";
      when 2 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111111111111100";
      when 3 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111111111111000";
      when 4 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111111111110000";
      when 5 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111111111100000";
      when 6 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111111111000000";
      when 7 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111111110000000";
      when 8 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111111100000000";
      when 9 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111111000000000";
      when 10 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111110000000000";
      when 11 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111100000000000";
      when 12 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111111000000000000";
      when 13 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111110000000000000";
      when 14 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111100000000000000";
      when 15 =>
        var_start_vector := "1111111111111111111111111111111111111111111111111000000000000000";
      when 16 =>
        var_start_vector := "1111111111111111111111111111111111111111111111110000000000000000";
      when 17 =>
        var_start_vector := "1111111111111111111111111111111111111111111111100000000000000000";
      when 18 =>
        var_start_vector := "1111111111111111111111111111111111111111111111000000000000000000";
      when 19 =>
        var_start_vector := "1111111111111111111111111111111111111111111110000000000000000000";
      when 20 =>
        var_start_vector := "1111111111111111111111111111111111111111111100000000000000000000";
      when 21 =>
        var_start_vector := "1111111111111111111111111111111111111111111000000000000000000000";
      when 22 =>
        var_start_vector := "1111111111111111111111111111111111111111110000000000000000000000";
      when 23 =>
        var_start_vector := "1111111111111111111111111111111111111111100000000000000000000000";
      when 24 =>
        var_start_vector := "1111111111111111111111111111111111111111000000000000000000000000";
      when 25 =>
        var_start_vector := "1111111111111111111111111111111111111110000000000000000000000000";
      when 26 =>
        var_start_vector := "1111111111111111111111111111111111111100000000000000000000000000";
      when 27 =>
        var_start_vector := "1111111111111111111111111111111111111000000000000000000000000000";
      when 28 =>
        var_start_vector := "1111111111111111111111111111111111110000000000000000000000000000";
      when 29 =>
        var_start_vector := "1111111111111111111111111111111111100000000000000000000000000000";
      when 30 =>
        var_start_vector := "1111111111111111111111111111111111000000000000000000000000000000";
      when 31 =>
        var_start_vector := "1111111111111111111111111111111110000000000000000000000000000000";
      when 32 =>
        var_start_vector := "1111111111111111111111111111111100000000000000000000000000000000";
      when 33 =>
        var_start_vector := "1111111111111111111111111111111000000000000000000000000000000000";
      when 34 =>
        var_start_vector := "1111111111111111111111111111110000000000000000000000000000000000";
      when 35 =>                                                                             
        var_start_vector := "1111111111111111111111111111100000000000000000000000000000000000";
      when 36 =>                                                                             
        var_start_vector := "1111111111111111111111111111000000000000000000000000000000000000";
      when 37 =>                                                                             
        var_start_vector := "1111111111111111111111111110000000000000000000000000000000000000";
      when 38 =>                                                                             
        var_start_vector := "1111111111111111111111111100000000000000000000000000000000000000";
      when 39 =>                                                                             
        var_start_vector := "1111111111111111111111111000000000000000000000000000000000000000";
      when 40 =>                                                                             
        var_start_vector := "1111111111111111111111110000000000000000000000000000000000000000";
      when 41 =>                                                                             
        var_start_vector := "1111111111111111111111100000000000000000000000000000000000000000";
      when 42 =>                                                                             
        var_start_vector := "1111111111111111111111000000000000000000000000000000000000000000";
      when 43 =>                                                                             
        var_start_vector := "1111111111111111111110000000000000000000000000000000000000000000";
      when 44 =>                                                                             
        var_start_vector := "1111111111111111111100000000000000000000000000000000000000000000";
      when 45 =>                                                                             
        var_start_vector := "1111111111111111111000000000000000000000000000000000000000000000";
      when 46 =>                                                                             
        var_start_vector := "1111111111111111110000000000000000000000000000000000000000000000";
      when 47 =>                                                                             
        var_start_vector := "1111111111111111100000000000000000000000000000000000000000000000";
      when 48 =>                                                                             
        var_start_vector := "1111111111111111000000000000000000000000000000000000000000000000";
      when 49 =>                                                                             
        var_start_vector := "1111111111111110000000000000000000000000000000000000000000000000";
      when 50 =>                                                                             
        var_start_vector := "1111111111111100000000000000000000000000000000000000000000000000";
      when 51 =>                                                                             
        var_start_vector := "1111111111111000000000000000000000000000000000000000000000000000";
      when 52 =>                                                                             
        var_start_vector := "1111111111110000000000000000000000000000000000000000000000000000";
      when 53 =>                                                                             
        var_start_vector := "1111111111100000000000000000000000000000000000000000000000000000";
      when 54 =>                                                                             
        var_start_vector := "1111111111000000000000000000000000000000000000000000000000000000";
      when 55 =>                                                                             
        var_start_vector := "1111111110000000000000000000000000000000000000000000000000000000";
      when 56 =>                                                                             
        var_start_vector := "1111111100000000000000000000000000000000000000000000000000000000";
      when 57 =>                                                                             
        var_start_vector := "1111111000000000000000000000000000000000000000000000000000000000";
      when 58 =>                                                                             
        var_start_vector := "1111110000000000000000000000000000000000000000000000000000000000";
      when 59 =>                                                                             
        var_start_vector := "1111100000000000000000000000000000000000000000000000000000000000";
      when 60 =>                                                                             
        var_start_vector := "1111000000000000000000000000000000000000000000000000000000000000";
      when 61 =>                                                                             
        var_start_vector := "1110000000000000000000000000000000000000000000000000000000000000";
      when 62 =>                                                                             
        var_start_vector := "1100000000000000000000000000000000000000000000000000000000000000";
      when others =>                                                                         
        var_start_vector := "1000000000000000000000000000000000000000000000000000000000000000";
    end case;
   
    Return (var_start_vector);
   
  end function get_start_64;
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_end_64
  --
  -- Function Description:
  --   Returns the 64-bit vector filled with '1's from the lsbit
  -- of the vector to the end offset.
  --
  -------------------------------------------------------------------
  function get_end_64 (end_offset : natural) return std_logic_vector is
  
    Variable var_end_vector : std_logic_vector(63 downto 0) := (others => '0');
  
  begin
  
    case end_offset is
      when 0 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000000000000001";
      when 1 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000000000000011";
      when 2 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000000000000111";
      when 3 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000000000001111";                    
      when 4 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000000000011111";
      when 5 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000000000111111";
      when 6 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000000001111111";
      when 7 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000000011111111";
      when 8 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000000111111111";
      when 9 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000001111111111";
      when 10 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000011111111111";
      when 11 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000000111111111111";
      when 12 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000001111111111111";
      when 13 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000011111111111111";
      when 14 =>
        var_end_vector := "0000000000000000000000000000000000000000000000000111111111111111";
      when 15 =>
        var_end_vector := "0000000000000000000000000000000000000000000000001111111111111111";
      when 16 =>
        var_end_vector := "0000000000000000000000000000000000000000000000011111111111111111";
      when 17 =>
        var_end_vector := "0000000000000000000000000000000000000000000000111111111111111111";
      when 18 =>
        var_end_vector := "0000000000000000000000000000000000000000000001111111111111111111";
      when 19 =>
        var_end_vector := "0000000000000000000000000000000000000000000011111111111111111111";
      when 20 =>
        var_end_vector := "0000000000000000000000000000000000000000000111111111111111111111";
      when 21 =>
        var_end_vector := "0000000000000000000000000000000000000000001111111111111111111111";
      when 22 =>
        var_end_vector := "0000000000000000000000000000000000000000011111111111111111111111";
      when 23 =>
        var_end_vector := "0000000000000000000000000000000000000000111111111111111111111111";
      when 24 =>
        var_end_vector := "0000000000000000000000000000000000000001111111111111111111111111";
      when 25 =>
        var_end_vector := "0000000000000000000000000000000000000011111111111111111111111111";
      when 26 =>
        var_end_vector := "0000000000000000000000000000000000000111111111111111111111111111";
      when 27 =>
        var_end_vector := "0000000000000000000000000000000000001111111111111111111111111111";
      when 28 =>
        var_end_vector := "0000000000000000000000000000000000011111111111111111111111111111";
      when 29 =>
        var_end_vector := "0000000000000000000000000000000000111111111111111111111111111111";
      when 30 =>
        var_end_vector := "0000000000000000000000000000000001111111111111111111111111111111";
      when 31 =>
        var_end_vector := "0000000000000000000000000000000011111111111111111111111111111111";
      
      when 32 =>
        var_end_vector := "0000000000000000000000000000000111111111111111111111111111111111";
      when 33 =>
        var_end_vector := "0000000000000000000000000000001111111111111111111111111111111111";
      when 34 =>
        var_end_vector := "0000000000000000000000000000011111111111111111111111111111111111";
      when 35 =>
        var_end_vector := "0000000000000000000000000000111111111111111111111111111111111111";
      when 36 =>
        var_end_vector := "0000000000000000000000000001111111111111111111111111111111111111";
      when 37 =>
        var_end_vector := "0000000000000000000000000011111111111111111111111111111111111111";
      when 38 =>
        var_end_vector := "0000000000000000000000000111111111111111111111111111111111111111";
      when 39 =>
        var_end_vector := "0000000000000000000000001111111111111111111111111111111111111111";
      when 40 =>
        var_end_vector := "0000000000000000000000011111111111111111111111111111111111111111";
      when 41 =>
        var_end_vector := "0000000000000000000000111111111111111111111111111111111111111111";
      when 42 =>
        var_end_vector := "0000000000000000000001111111111111111111111111111111111111111111";
      when 43 =>
        var_end_vector := "0000000000000000000011111111111111111111111111111111111111111111";
      when 44 =>
        var_end_vector := "0000000000000000000111111111111111111111111111111111111111111111";
      when 45 =>
        var_end_vector := "0000000000000000001111111111111111111111111111111111111111111111";
      when 46 =>
        var_end_vector := "0000000000000000011111111111111111111111111111111111111111111111";
      when 47 =>
        var_end_vector := "0000000000000000111111111111111111111111111111111111111111111111";
      when 48 =>
        var_end_vector := "0000000000000001111111111111111111111111111111111111111111111111";
      when 49 =>
        var_end_vector := "0000000000000011111111111111111111111111111111111111111111111111";
      when 50 =>
        var_end_vector := "0000000000000111111111111111111111111111111111111111111111111111";
      when 51 =>
        var_end_vector := "0000000000001111111111111111111111111111111111111111111111111111";
      when 52 =>
        var_end_vector := "0000000000011111111111111111111111111111111111111111111111111111";
      when 53 =>
        var_end_vector := "0000000000111111111111111111111111111111111111111111111111111111";
      when 54 =>
        var_end_vector := "0000000001111111111111111111111111111111111111111111111111111111";
      when 55 =>
        var_end_vector := "0000000011111111111111111111111111111111111111111111111111111111";
      when 56 =>
        var_end_vector := "0000000111111111111111111111111111111111111111111111111111111111";
      when 57 =>
        var_end_vector := "0000001111111111111111111111111111111111111111111111111111111111";
      when 58 =>
        var_end_vector := "0000011111111111111111111111111111111111111111111111111111111111";
      when 59 =>
        var_end_vector := "0000111111111111111111111111111111111111111111111111111111111111";
      when 60 =>
        var_end_vector := "0001111111111111111111111111111111111111111111111111111111111111";
      when 61 =>
        var_end_vector := "0011111111111111111111111111111111111111111111111111111111111111";
      when 62 =>
        var_end_vector := "0111111111111111111111111111111111111111111111111111111111111111";
      
      when others =>
        var_end_vector := "1111111111111111111111111111111111111111111111111111111111111111";
    end case;
   
    Return (var_end_vector);
   
  end function get_end_64;
 
 
 
 
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_start_128
  --
  -- Function Description:
  --   returns the 128-bit vector filled with '1's from the start
  -- offset to the end of of the vector
  --
  -------------------------------------------------------------------
  function get_start_128 (start_offset : natural) return std_logic_vector is
  
    Variable var_start_vector : std_logic_vector(127 downto 0) := (others => '0');
  
  begin
  
    case start_offset is
      when 0 =>
        
        var_start_vector(127 downto 0) := (others => '1');
      
      when 1 =>
        
        var_start_vector(127 downto 1) := (others => '1');
        var_start_vector(  0 downto 0) := (others => '0');
      
      when 2 =>
        
        var_start_vector(127 downto 2) := (others => '1');
        var_start_vector(  1 downto 0) := (others => '0');
      
      when 3 =>
        
        var_start_vector(127 downto 3) := (others => '1');
        var_start_vector(  2 downto 0) := (others => '0');
      
      when 4 =>
        
        var_start_vector(127 downto 4) := (others => '1');
        var_start_vector(  3 downto 0) := (others => '0');
      
      when 5 =>
        
        var_start_vector(127 downto 5) := (others => '1');
        var_start_vector(  4 downto 0) := (others => '0');
      
      when 6 =>
        
        var_start_vector(127 downto 6) := (others => '1');
        var_start_vector(  5 downto 0) := (others => '0');
      
      when 7 =>
        
        var_start_vector(127 downto 7) := (others => '1');
        var_start_vector(  6 downto 0) := (others => '0');
      
      when 8 =>
        
        var_start_vector(127 downto 8) := (others => '1');
        var_start_vector(  7 downto 0) := (others => '0');
      
      when 9 =>
        
        var_start_vector(127 downto 9) := (others => '1');
        var_start_vector(  8 downto 0) := (others => '0');
      
      when 10 =>
        
        var_start_vector(127 downto 10) := (others => '1');
        var_start_vector(  9 downto 0) := (others => '0');
      
      when 11 =>
        
        var_start_vector(127 downto 11) := (others => '1');
        var_start_vector( 10 downto  0) := (others => '0');
      
      when 12 =>
        
        var_start_vector(127 downto 12) := (others => '1');
        var_start_vector( 11 downto  0) := (others => '0');
      
      when 13 =>
        
        var_start_vector(127 downto 13) := (others => '1');
        var_start_vector( 12 downto  0) := (others => '0');
      
      when 14 =>
        
        var_start_vector(127 downto 14) := (others => '1');
        var_start_vector( 13 downto  0) := (others => '0');
      
      when 15 =>
        
        var_start_vector(127 downto 15) := (others => '1');
        var_start_vector( 14 downto  0) := (others => '0');
      
      when 16 =>
        
        var_start_vector(127 downto 16) := (others => '1');
        var_start_vector( 15 downto  0) := (others => '0');
      
      when 17 =>
        
        var_start_vector(127 downto 17) := (others => '1');
        var_start_vector( 16 downto  0) := (others => '0');
      
      when 18 =>
        
        var_start_vector(127 downto 18) := (others => '1');
        var_start_vector( 17 downto  0) := (others => '0');
      
      when 19 =>
        
        var_start_vector(127 downto 19) := (others => '1');
        var_start_vector( 18 downto  0) := (others => '0');
      
      when 20 =>
        
        var_start_vector(127 downto 20) := (others => '1');
        var_start_vector( 19 downto  0) := (others => '0');
      
      when 21 =>
        
        var_start_vector(127 downto 21) := (others => '1');
        var_start_vector( 20 downto  0) := (others => '0');
      
      when 22 =>
        
        var_start_vector(127 downto 22) := (others => '1');
        var_start_vector( 21 downto  0) := (others => '0');
      
      when 23 =>
        
        var_start_vector(127 downto 23) := (others => '1');
        var_start_vector( 22 downto  0) := (others => '0');
      
      when 24 =>
        
        var_start_vector(127 downto 24) := (others => '1');
        var_start_vector( 23 downto  0) := (others => '0');
      
      when 25 =>
        
        var_start_vector(127 downto 25) := (others => '1');
        var_start_vector( 24 downto  0) := (others => '0');
      
      when 26 =>
        
        var_start_vector(127 downto 26) := (others => '1');
        var_start_vector( 25 downto  0) := (others => '0');
      
      when 27 =>
        
        var_start_vector(127 downto 27) := (others => '1');
        var_start_vector( 26 downto  0) := (others => '0');
      
      when 28 =>
        
        var_start_vector(127 downto 28) := (others => '1');
        var_start_vector( 27 downto  0) := (others => '0');
      
      when 29 =>
        
        var_start_vector(127 downto 29) := (others => '1');
        var_start_vector( 28 downto  0) := (others => '0');
      
      when 30 =>
        
        var_start_vector(127 downto 30) := (others => '1');
        var_start_vector( 29 downto  0) := (others => '0');
      
      when 31 =>
        
        var_start_vector(127 downto 31) := (others => '1');
        var_start_vector( 30 downto  0) := (others => '0');
      
      
      
      when 32 =>
        
        var_start_vector(127 downto 32) := (others => '1');
        var_start_vector( 31 downto  0) := (others => '0');
      
      when 33 =>
        
        var_start_vector(127 downto 33) := (others => '1');
        var_start_vector( 32 downto  0) := (others => '0');
      
      when 34 =>
        
        var_start_vector(127 downto 34) := (others => '1');
        var_start_vector( 33 downto  0) := (others => '0');
      
      when 35 =>
        
        var_start_vector(127 downto 35) := (others => '1');
        var_start_vector( 34 downto  0) := (others => '0');
      
      when 36 =>
        
        var_start_vector(127 downto 36) := (others => '1');
        var_start_vector( 35 downto  0) := (others => '0');
      
      when 37 =>
        
        var_start_vector(127 downto 37) := (others => '1');
        var_start_vector( 36 downto  0) := (others => '0');
      
      when 38 =>
        
        var_start_vector(127 downto 38) := (others => '1');
        var_start_vector( 37 downto  0) := (others => '0');
      
      when 39 =>
        
        var_start_vector(127 downto 39) := (others => '1');
        var_start_vector( 38 downto  0) := (others => '0');
      
      when 40 =>
        
        var_start_vector(127 downto 40) := (others => '1');
        var_start_vector( 39 downto  0) := (others => '0');
      
      when 41 =>
        
        var_start_vector(127 downto 41) := (others => '1');
        var_start_vector( 40 downto  0) := (others => '0');
      
      when 42 =>
        
        var_start_vector(127 downto 42) := (others => '1');
        var_start_vector( 41 downto  0) := (others => '0');
      
      when 43 =>
        
        var_start_vector(127 downto 43) := (others => '1');
        var_start_vector( 42 downto  0) := (others => '0');
      
      when 44 =>
        
        var_start_vector(127 downto 44) := (others => '1');
        var_start_vector( 43 downto  0) := (others => '0');
      
      when 45 =>
        
        var_start_vector(127 downto 45) := (others => '1');
        var_start_vector( 44 downto  0) := (others => '0');
      
      when 46 =>
        
        var_start_vector(127 downto 46) := (others => '1');
        var_start_vector( 45 downto  0) := (others => '0');
      
      when 47 =>
        
        var_start_vector(127 downto 47) := (others => '1');
        var_start_vector( 46 downto  0) := (others => '0');
      
      when 48 =>
        
        var_start_vector(127 downto 48) := (others => '1');
        var_start_vector( 47 downto  0) := (others => '0');
      
      when 49 =>
        
        var_start_vector(127 downto 49) := (others => '1');
        var_start_vector( 48 downto  0) := (others => '0');
      
      when 50 =>
        
        var_start_vector(127 downto 50) := (others => '1');
        var_start_vector( 49 downto  0) := (others => '0');
      
      when 51 =>
        
        var_start_vector(127 downto 51) := (others => '1');
        var_start_vector( 50 downto  0) := (others => '0');
      
      when 52 =>
        
        var_start_vector(127 downto 52) := (others => '1');
        var_start_vector( 51 downto  0) := (others => '0');
      
      when 53 =>
        
        var_start_vector(127 downto 53) := (others => '1');
        var_start_vector( 52 downto  0) := (others => '0');
      
      when 54 =>
        
        var_start_vector(127 downto 54) := (others => '1');
        var_start_vector( 53 downto  0) := (others => '0');
      
      when 55 =>
        
        var_start_vector(127 downto 55) := (others => '1');
        var_start_vector( 54 downto  0) := (others => '0');
      
      when 56 =>
        
        var_start_vector(127 downto 56) := (others => '1');
        var_start_vector( 55 downto  0) := (others => '0');
      
      when 57 =>
        
        var_start_vector(127 downto 57) := (others => '1');
        var_start_vector( 56 downto  0) := (others => '0');
      
      when 58 =>
        
        var_start_vector(127 downto 58) := (others => '1');
        var_start_vector( 57 downto  0) := (others => '0');
      
      when 59 =>
        
        var_start_vector(127 downto 59) := (others => '1');
        var_start_vector( 58 downto  0) := (others => '0');
      
      when 60 =>
        
        var_start_vector(127 downto 60) := (others => '1');
        var_start_vector( 59 downto  0) := (others => '0');
      
      when 61 =>
        
        var_start_vector(127 downto 61) := (others => '1');
        var_start_vector( 60 downto  0) := (others => '0');
      
      when 62 =>
        
        var_start_vector(127 downto 62) := (others => '1');
        var_start_vector( 61 downto  0) := (others => '0');
      
      when 63 =>
        
        var_start_vector(127 downto 63) := (others => '1');
        var_start_vector( 62 downto  0) := (others => '0');
      
      
      
      
      when 64 =>
        
        var_start_vector(127 downto 64) := (others => '1');
        var_start_vector( 63 downto  0) := (others => '0');
 
      when 65 =>
        
        var_start_vector(127 downto 65) := (others => '1');
        var_start_vector( 64 downto  0) := (others => '0');
      
      when 66 =>                                                                           
        
        var_start_vector(127 downto 66) := (others => '1');
        var_start_vector( 65 downto  0) := (others => '0');
      
      when 67 =>                                                                           
        
        var_start_vector(127 downto 67) := (others => '1');
        var_start_vector( 66 downto  0) := (others => '0');
      
      when 68 =>                                                                           
        
        var_start_vector(127 downto 68) := (others => '1');
        var_start_vector( 67 downto  0) := (others => '0');
      
      when 69 =>                                                                           
        
        var_start_vector(127 downto 69) := (others => '1');
        var_start_vector( 68 downto  0) := (others => '0');
      
      when 70 =>                                                                           
        
        var_start_vector(127 downto 70) := (others => '1');
        var_start_vector( 69 downto  0) := (others => '0');
      
      when 71 =>                                                                           
        
        var_start_vector(127 downto 71) := (others => '1');
        var_start_vector( 70 downto  0) := (others => '0');
      
      when 72 =>                                                                           
        
        var_start_vector(127 downto 72) := (others => '1');
        var_start_vector( 71 downto  0) := (others => '0');
      
      when 73 =>                                                                           
        
        var_start_vector(127 downto 73) := (others => '1');
        var_start_vector( 72 downto  0) := (others => '0');
      
      when 74 =>                                                                           
        
        var_start_vector(127 downto 74) := (others => '1');
        var_start_vector( 73 downto  0) := (others => '0');
      
      when 75 =>                                                                           
        
        var_start_vector(127 downto 75) := (others => '1');
        var_start_vector( 74 downto  0) := (others => '0');
      
      when 76 =>                                                                           
        
        var_start_vector(127 downto 76) := (others => '1');
        var_start_vector( 75 downto  0) := (others => '0');
      
      when 77 =>                                                                           
        
        var_start_vector(127 downto 77) := (others => '1');
        var_start_vector( 76 downto  0) := (others => '0');
      
      when 78 =>                                                                           
        
        var_start_vector(127 downto 78) := (others => '1');
        var_start_vector( 77 downto  0) := (others => '0');
      
      when 79 =>                                                                           
        
        var_start_vector(127 downto 79) := (others => '1');
        var_start_vector( 78 downto  0) := (others => '0');
      
      when 80 =>                                                                           
        
        var_start_vector(127 downto 80) := (others => '1');
        var_start_vector( 79 downto  0) := (others => '0');
      
      when 81 =>                                                                           
        
        var_start_vector(127 downto 81) := (others => '1');
        var_start_vector( 80 downto  0) := (others => '0');
      
      when 82 =>                                                                           
        
        var_start_vector(127 downto 82) := (others => '1');
        var_start_vector( 81 downto  0) := (others => '0');
      
      when 83 =>                                                                           
        
        var_start_vector(127 downto 83) := (others => '1');
        var_start_vector( 82 downto  0) := (others => '0');
      
      when 84 =>                                                                           
        
        var_start_vector(127 downto 84) := (others => '1');
        var_start_vector( 83 downto  0) := (others => '0');
      
      when 85 =>                                                                           
        
        var_start_vector(127 downto 85) := (others => '1');
        var_start_vector( 84 downto  0) := (others => '0');
      
      when 86 =>                                                                           
        
        var_start_vector(127 downto 86) := (others => '1');
        var_start_vector( 85 downto  0) := (others => '0');
      
      when 87 =>                                                                           
        
        var_start_vector(127 downto 87) := (others => '1');
        var_start_vector( 86 downto  0) := (others => '0');
      
      when 88 =>                                                                           
        
        var_start_vector(127 downto 88) := (others => '1');
        var_start_vector( 87 downto  0) := (others => '0');
      
      when 89 =>                                                                           
        
        var_start_vector(127 downto 89) := (others => '1');
        var_start_vector( 88 downto  0) := (others => '0');
      
      when 90 =>                                                                           
        
        var_start_vector(127 downto 90) := (others => '1');
        var_start_vector( 89 downto  0) := (others => '0');
      
      when 91 =>                                                                           
        
        var_start_vector(127 downto 91) := (others => '1');
        var_start_vector( 90 downto  0) := (others => '0');
      
      when 92 =>                                                                           
        
        var_start_vector(127 downto 92) := (others => '1');
        var_start_vector( 91 downto  0) := (others => '0');
      
      when 93 =>                                                                           
        
        var_start_vector(127 downto 93) := (others => '1');
        var_start_vector( 92 downto  0) := (others => '0');
      
      when 94 =>                                                                           
        
        var_start_vector(127 downto 94) := (others => '1');
        var_start_vector( 93 downto  0) := (others => '0');
      
      when 95 =>                                                                           
        
        var_start_vector(127 downto 95) := (others => '1');
        var_start_vector( 94 downto  0) := (others => '0');
      
      when 96 =>                                                                           
        
        var_start_vector(127 downto 96) := (others => '1');
        var_start_vector( 95 downto  0) := (others => '0');
      
      when 97 =>                                                                           
        
        var_start_vector(127 downto 97) := (others => '1');
        var_start_vector( 96 downto  0) := (others => '0');
      
      when 98 =>                                                                           
        
        var_start_vector(127 downto 98) := (others => '1');
        var_start_vector( 97 downto  0) := (others => '0');
      
      when 99 =>                                                                           
        
        var_start_vector(127 downto 99) := (others => '1');
        var_start_vector( 98 downto  0) := (others => '0');
      
      when 100 =>                                                                          
        
        var_start_vector(127 downto 100) := (others => '1');
        var_start_vector( 99 downto   0) := (others => '0');
      
      when 101 =>                                                                          
        
        var_start_vector(127 downto 101) := (others => '1');
        var_start_vector(100 downto   0) := (others => '0');
      
      when 102 =>                                                                          
        
        var_start_vector(127 downto 102) := (others => '1');
        var_start_vector(101 downto   0) := (others => '0');
      
      when 103 =>                                                                          
        
        var_start_vector(127 downto 103) := (others => '1');
        var_start_vector(102 downto   0) := (others => '0');
      
      when 104 =>                                                                          
        
        var_start_vector(127 downto 104) := (others => '1');
        var_start_vector(103 downto   0) := (others => '0');
      
      when 105 =>                                                                          
        
        var_start_vector(127 downto 105) := (others => '1');
        var_start_vector(104 downto   0) := (others => '0');
      
      when 106 =>                                                                          
        
        var_start_vector(127 downto 106) := (others => '1');
        var_start_vector(105 downto   0) := (others => '0');
      
      when 107 =>                                                                          
        
        var_start_vector(127 downto 107) := (others => '1');
        var_start_vector(106 downto   0) := (others => '0');
      
      when 108 =>                                                                          
        
        var_start_vector(127 downto 108) := (others => '1');
        var_start_vector(107 downto   0) := (others => '0');
      
      when 109 =>                                                                          
        
        var_start_vector(127 downto 109) := (others => '1');
        var_start_vector(108 downto   0) := (others => '0');
      
      when 110 =>                                                                          
        
        var_start_vector(127 downto 110) := (others => '1');
        var_start_vector(109 downto   0) := (others => '0');
      
      when 111 =>                                                                          
        
        var_start_vector(127 downto 111) := (others => '1');
        var_start_vector(110 downto   0) := (others => '0');
      
      when 112 =>                                                                          
        
        var_start_vector(127 downto 112) := (others => '1');
        var_start_vector(111 downto   0) := (others => '0');
      
      when 113 =>                                                                          
        
        var_start_vector(127 downto 113) := (others => '1');
        var_start_vector(112 downto   0) := (others => '0');
      
      when 114 =>                                                                          
        
        var_start_vector(127 downto 114) := (others => '1');
        var_start_vector(113 downto   0) := (others => '0');
      
      when 115 =>                                                                          
        
        var_start_vector(127 downto 115) := (others => '1');
        var_start_vector(114 downto   0) := (others => '0');
      
      when 116 =>                                                                          
        
        var_start_vector(127 downto 116) := (others => '1');
        var_start_vector(115 downto   0) := (others => '0');
      
      when 117 =>                                                                          
        
        var_start_vector(127 downto 117) := (others => '1');
        var_start_vector(116 downto   0) := (others => '0');
      
      when 118 =>                                                                          
        
        var_start_vector(127 downto 118) := (others => '1');
        var_start_vector(117 downto   0) := (others => '0');
      
      when 119 =>                                                                          
        
        var_start_vector(127 downto 119) := (others => '1');
        var_start_vector(118 downto   0) := (others => '0');
      
      when 120 =>                                                                          
        
        var_start_vector(127 downto 120) := (others => '1');
        var_start_vector(119 downto   0) := (others => '0');
      
      when 121 =>                                                                          
        
        var_start_vector(127 downto 121) := (others => '1');
        var_start_vector(120 downto   0) := (others => '0');
      
      when 122 =>                                                                          
        
        var_start_vector(127 downto 122) := (others => '1');
        var_start_vector(121 downto   0) := (others => '0');
      
      when 123 =>                                                                          
        
        var_start_vector(127 downto 123) := (others => '1');
        var_start_vector(122 downto   0) := (others => '0');
      
      when 124 =>                                                                          
        
        var_start_vector(127 downto 124) := (others => '1');
        var_start_vector(123 downto   0) := (others => '0');
      
      when 125 =>                                                                          
        
        var_start_vector(127 downto 125) := (others => '1');
        var_start_vector(124 downto   0) := (others => '0');
      
      when 126 =>                                                                          
        
        var_start_vector(127 downto 126) := (others => '1');
        var_start_vector(125 downto   0) := (others => '0');
      
      when others =>
        
        var_start_vector(127 downto 127) := (others => '1');
        var_start_vector(126 downto   0) := (others => '0');
      
    end case;
   
    Return (var_start_vector);
   
  end function get_start_128;
  
  
 
  
  
  
  
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: get_end_128
  --
  -- Function Description:
  --   Returns the 128-bit vector filled with '1's from the lsbit
  -- of the vector to the end offset.
  --
  -------------------------------------------------------------------
  function get_end_128 (end_offset : natural) return std_logic_vector is
  
    Variable var_end_vector : std_logic_vector(127 downto 0) := (others => '0');
  
  begin
  
    case end_offset is
      when 0 =>
        
        var_end_vector(127 downto 1) := (others => '0');
        var_end_vector(  0 downto 0) := (others => '1');
      
      when 1 =>
        
        var_end_vector(127 downto 2) := (others => '0');
        var_end_vector(  1 downto 0) := (others => '1');
      
      when 2 =>
        
        var_end_vector(127 downto 3) := (others => '0');
        var_end_vector(  2 downto 0) := (others => '1');
      
      when 3 =>
        
        var_end_vector(127 downto 4) := (others => '0');
        var_end_vector(  3 downto 0) := (others => '1');
      
      when 4 =>
        
        var_end_vector(127 downto 5) := (others => '0');
        var_end_vector(  4 downto 0) := (others => '1');
      
      when 5 =>
        
        var_end_vector(127 downto 6) := (others => '0');
        var_end_vector(  5 downto 0) := (others => '1');
      
      when 6 =>
        
        var_end_vector(127 downto 7) := (others => '0');
        var_end_vector(  6 downto 0) := (others => '1');
      
      when 7 =>
        
        var_end_vector(127 downto 8) := (others => '0');
        var_end_vector(  7 downto 0) := (others => '1');
      
      when 8 =>
        
        var_end_vector(127 downto 9) := (others => '0');
        var_end_vector(  8 downto 0) := (others => '1');
      
      when 9 =>
        
        var_end_vector(127 downto 10) := (others => '0');
        var_end_vector(  9 downto 0) := (others => '1');
      
      when 10 =>
        
        var_end_vector(127 downto 11) := (others => '0');
        var_end_vector( 10 downto  0) := (others => '1');
      
      when 11 =>
        
        var_end_vector(127 downto 12) := (others => '0');
        var_end_vector( 11 downto  0) := (others => '1');
      
      when 12 =>
        
        var_end_vector(127 downto 13) := (others => '0');
        var_end_vector( 12 downto  0) := (others => '1');
      
      when 13 =>
        
        var_end_vector(127 downto 14) := (others => '0');
        var_end_vector( 13 downto  0) := (others => '1');
      
      when 14 =>
        
        var_end_vector(127 downto 15) := (others => '0');
        var_end_vector( 14 downto  0) := (others => '1');
      
      when 15 =>
        
        var_end_vector(127 downto 16) := (others => '0');
        var_end_vector( 15 downto  0) := (others => '1');
      
      when 16 =>
        
        var_end_vector(127 downto 17) := (others => '0');
        var_end_vector( 16 downto  0) := (others => '1');
      
      when 17 =>
        
        var_end_vector(127 downto 18) := (others => '0');
        var_end_vector( 17 downto  0) := (others => '1');
      
      when 18 =>
        
        var_end_vector(127 downto 19) := (others => '0');
        var_end_vector( 18 downto  0) := (others => '1');
      
      when 19 =>
        
        var_end_vector(127 downto 20) := (others => '0');
        var_end_vector( 19 downto  0) := (others => '1');
      
      when 20 =>
        
        var_end_vector(127 downto 21) := (others => '0');
        var_end_vector( 20 downto  0) := (others => '1');
      
      when 21 =>
        
        var_end_vector(127 downto 22) := (others => '0');
        var_end_vector( 21 downto  0) := (others => '1');
      
      when 22 =>
        
        var_end_vector(127 downto 23) := (others => '0');
        var_end_vector( 22 downto  0) := (others => '1');
      
      when 23 =>
        
        var_end_vector(127 downto 24) := (others => '0');
        var_end_vector( 23 downto  0) := (others => '1');
      
      when 24 =>
        
        var_end_vector(127 downto 25) := (others => '0');
        var_end_vector( 24 downto  0) := (others => '1');
      
      when 25 =>
        
        var_end_vector(127 downto 26) := (others => '0');
        var_end_vector( 25 downto  0) := (others => '1');
      
      when 26 =>
        
        var_end_vector(127 downto 27) := (others => '0');
        var_end_vector( 26 downto  0) := (others => '1');
      
      when 27 =>
        
        var_end_vector(127 downto 28) := (others => '0');
        var_end_vector( 27 downto  0) := (others => '1');
      
      when 28 =>
        
        var_end_vector(127 downto 29) := (others => '0');
        var_end_vector( 28 downto  0) := (others => '1');
      
      when 29 =>
        
        var_end_vector(127 downto 30) := (others => '0');
        var_end_vector( 29 downto  0) := (others => '1');
      
      when 30 =>
        
        var_end_vector(127 downto 31) := (others => '0');
        var_end_vector( 30 downto  0) := (others => '1');
      
      when 31 =>
        
        var_end_vector(127 downto 32) := (others => '0');
        var_end_vector( 31 downto  0) := (others => '1');
      
      
      when 32 =>
        
        var_end_vector(127 downto 33) := (others => '0');
        var_end_vector( 32 downto  0) := (others => '1');
      
      when 33 =>
        
        var_end_vector(127 downto 34) := (others => '0');
        var_end_vector( 33 downto  0) := (others => '1');
      
      when 34 =>
        
        var_end_vector(127 downto 35) := (others => '0');
        var_end_vector( 34 downto  0) := (others => '1');
      
      when 35 =>
        
        var_end_vector(127 downto 36) := (others => '0');
        var_end_vector( 35 downto  0) := (others => '1');
      
      when 36 =>
        
        var_end_vector(127 downto 37) := (others => '0');
        var_end_vector( 36 downto  0) := (others => '1');
      
      when 37 =>
        
        var_end_vector(127 downto 38) := (others => '0');
        var_end_vector( 37 downto  0) := (others => '1');
      
      when 38 =>
        
        var_end_vector(127 downto 39) := (others => '0');
        var_end_vector( 38 downto  0) := (others => '1');
      
      when 39 =>
        
        var_end_vector(127 downto 40) := (others => '0');
        var_end_vector( 39 downto  0) := (others => '1');
      
      when 40 =>
        
        var_end_vector(127 downto 41) := (others => '0');
        var_end_vector( 40 downto  0) := (others => '1');
      
      when 41 =>
        
        var_end_vector(127 downto 42) := (others => '0');
        var_end_vector( 41 downto  0) := (others => '1');
      
      when 42 =>
        
        var_end_vector(127 downto 43) := (others => '0');
        var_end_vector( 42 downto  0) := (others => '1');
      
      when 43 =>
        
        var_end_vector(127 downto 44) := (others => '0');
        var_end_vector( 43 downto  0) := (others => '1');
      
      when 44 =>
        
        var_end_vector(127 downto 45) := (others => '0');
        var_end_vector( 44 downto  0) := (others => '1');
      
      when 45 =>
        
        var_end_vector(127 downto 46) := (others => '0');
        var_end_vector( 45 downto  0) := (others => '1');
      
      when 46 =>
        
        var_end_vector(127 downto 47) := (others => '0');
        var_end_vector( 46 downto  0) := (others => '1');
      
      when 47 =>
        
        var_end_vector(127 downto 48) := (others => '0');
        var_end_vector( 47 downto  0) := (others => '1');
      
      when 48 =>
        
        var_end_vector(127 downto 49) := (others => '0');
        var_end_vector( 48 downto  0) := (others => '1');
      
      when 49 =>
        
        var_end_vector(127 downto 50) := (others => '0');
        var_end_vector( 49 downto  0) := (others => '1');
      
      when 50 =>
        
        var_end_vector(127 downto 51) := (others => '0');
        var_end_vector( 50 downto  0) := (others => '1');
      
      when 51 =>
        
        var_end_vector(127 downto 52) := (others => '0');
        var_end_vector( 51 downto  0) := (others => '1');
      
      when 52 =>
        
        var_end_vector(127 downto 53) := (others => '0');
        var_end_vector( 52 downto  0) := (others => '1');
      
      when 53 =>
        
        var_end_vector(127 downto 54) := (others => '0');
        var_end_vector( 53 downto  0) := (others => '1');
      
      when 54 =>
        
        var_end_vector(127 downto 55) := (others => '0');
        var_end_vector( 54 downto  0) := (others => '1');
      
      when 55 =>
        
        var_end_vector(127 downto 56) := (others => '0');
        var_end_vector( 55 downto  0) := (others => '1');
      
      when 56 =>
        
        var_end_vector(127 downto 57) := (others => '0');
        var_end_vector( 56 downto  0) := (others => '1');
      
      when 57 =>
        
        var_end_vector(127 downto 58) := (others => '0');
        var_end_vector( 57 downto  0) := (others => '1');
      
      when 58 =>
        
        var_end_vector(127 downto 59) := (others => '0');
        var_end_vector( 58 downto  0) := (others => '1');
      
      when 59 =>
        
        var_end_vector(127 downto 60) := (others => '0');
        var_end_vector( 59 downto  0) := (others => '1');
      
      when 60 =>
        
        var_end_vector(127 downto 61) := (others => '0');
        var_end_vector( 60 downto  0) := (others => '1');
      
      when 61 =>
        
        var_end_vector(127 downto 62) := (others => '0');
        var_end_vector( 61 downto  0) := (others => '1');
      
      when 62 =>
        
        var_end_vector(127 downto 63) := (others => '0');
        var_end_vector( 62 downto  0) := (others => '1');
      
      when 63 =>
        
        var_end_vector(127 downto 64) := (others => '0');
        var_end_vector( 63 downto  0) := (others => '1');
      
 
 
 
 
      when 64 =>
        
        var_end_vector(127 downto 65) := (others => '0');
        var_end_vector( 64 downto  0) := (others => '1');
      
      when 65 =>
        
        var_end_vector(127 downto 66) := (others => '0');
        var_end_vector( 65 downto  0) := (others => '1');
      
      when 66 =>                                                                           
        
        var_end_vector(127 downto 67) := (others => '0');
        var_end_vector( 66 downto  0) := (others => '1');
      
      when 67 =>                                                                           
        
        var_end_vector(127 downto 68) := (others => '0');
        var_end_vector( 67 downto  0) := (others => '1');
      
      when 68 =>                                                                           
        
        var_end_vector(127 downto 69) := (others => '0');
        var_end_vector( 68 downto  0) := (others => '1');
      
      when 69 =>                                                                           
        
        var_end_vector(127 downto 70) := (others => '0');
        var_end_vector( 69 downto  0) := (others => '1');
      
      when 70 =>                                                                           
        
        var_end_vector(127 downto 71) := (others => '0');
        var_end_vector( 70 downto  0) := (others => '1');
      
      when 71 =>                                                                           
        
        var_end_vector(127 downto 72) := (others => '0');
        var_end_vector( 71 downto  0) := (others => '1');
      
      when 72 =>                                                                           
        
        var_end_vector(127 downto 73) := (others => '0');
        var_end_vector( 72 downto  0) := (others => '1');
      
      when 73 =>                                                                           
        
        var_end_vector(127 downto 74) := (others => '0');
        var_end_vector( 73 downto  0) := (others => '1');
      
      when 74 =>                                                                           
        
        var_end_vector(127 downto 75) := (others => '0');
        var_end_vector( 74 downto  0) := (others => '1');
      
      when 75 =>                                                                           
        
        var_end_vector(127 downto 76) := (others => '0');
        var_end_vector( 75 downto  0) := (others => '1');
      
      when 76 =>                                                                           
        
        var_end_vector(127 downto 77) := (others => '0');
        var_end_vector( 76 downto  0) := (others => '1');
      
      when 77 =>                                                                           
        
        var_end_vector(127 downto 78) := (others => '0');
        var_end_vector( 77 downto  0) := (others => '1');
      
      when 78 =>                                                                           
        
        var_end_vector(127 downto 79) := (others => '0');
        var_end_vector( 78 downto  0) := (others => '1');
      
      when 79 =>                                                                           
        
        var_end_vector(127 downto 80) := (others => '0');
        var_end_vector( 79 downto  0) := (others => '1');
      
      when 80 =>                                                                           
        
        var_end_vector(127 downto 81) := (others => '0');
        var_end_vector( 80 downto  0) := (others => '1');
      
      when 81 =>                                                                           
        
        var_end_vector(127 downto 82) := (others => '0');
        var_end_vector( 81 downto  0) := (others => '1');
      
      when 82 =>                                                                           
        
        var_end_vector(127 downto 83) := (others => '0');
        var_end_vector( 82 downto  0) := (others => '1');
      
      when 83 =>                                                                           
        
        var_end_vector(127 downto 84) := (others => '0');
        var_end_vector( 83 downto  0) := (others => '1');
      
      when 84 =>                                                                           
        
        var_end_vector(127 downto 85) := (others => '0');
        var_end_vector( 84 downto  0) := (others => '1');
      
      when 85 =>                                                                           
        
        var_end_vector(127 downto 86) := (others => '0');
        var_end_vector( 85 downto  0) := (others => '1');
      
      when 86 =>                                                                           
        
        var_end_vector(127 downto 87) := (others => '0');
        var_end_vector( 86 downto  0) := (others => '1');
      
      when 87 =>                                                                           
        
        var_end_vector(127 downto 88) := (others => '0');
        var_end_vector( 87 downto  0) := (others => '1');
      
      when 88 =>                                                                           
        
        var_end_vector(127 downto 89) := (others => '0');
        var_end_vector( 88 downto  0) := (others => '1');
      
      when 89 =>                                                                           
        
        var_end_vector(127 downto 90) := (others => '0');
        var_end_vector( 89 downto  0) := (others => '1');
      
      when 90 =>                                                                           
        
        var_end_vector(127 downto 91) := (others => '0');
        var_end_vector( 90 downto  0) := (others => '1');
      
      when 91 =>                                                                           
        
        var_end_vector(127 downto 92) := (others => '0');
        var_end_vector( 91 downto  0) := (others => '1');
      
      when 92 =>                                                                           
        
        var_end_vector(127 downto 93) := (others => '0');
        var_end_vector( 92 downto  0) := (others => '1');
      
      when 93 =>                                                                           
        
        var_end_vector(127 downto 94) := (others => '0');
        var_end_vector( 93 downto  0) := (others => '1');
      
      when 94 =>                                                                           
        
        var_end_vector(127 downto 95) := (others => '0');
        var_end_vector( 94 downto  0) := (others => '1');
      
      when 95 =>                                                                           
        
        var_end_vector(127 downto 96) := (others => '0');
        var_end_vector( 95 downto  0) := (others => '1');
      
      when 96 =>                                                                           
        
        var_end_vector(127 downto 97) := (others => '0');
        var_end_vector( 96 downto  0) := (others => '1');
      
      when 97 =>                                                                           
        
        var_end_vector(127 downto 98) := (others => '0');
        var_end_vector( 97 downto  0) := (others => '1');
      
      when 98 =>                                                                           
        
        var_end_vector(127 downto 99) := (others => '0');
        var_end_vector( 98 downto  0) := (others => '1');
      
      when 99 =>                                                                           
        
        var_end_vector(127 downto 100) := (others => '0');
        var_end_vector( 99 downto   0) := (others => '1');
      
      when 100 =>                                                                          
        
        var_end_vector(127 downto 101) := (others => '0');
        var_end_vector(100 downto   0) := (others => '1');
      
      when 101 =>                                                                          
        
        var_end_vector(127 downto 102) := (others => '0');
        var_end_vector(101 downto   0) := (others => '1');
      
      when 102 =>                                                                          
        
        var_end_vector(127 downto 103) := (others => '0');
        var_end_vector(102 downto   0) := (others => '1');
      
      when 103 =>                                                                          
        
        var_end_vector(127 downto 104) := (others => '0');
        var_end_vector(103 downto   0) := (others => '1');
      
      when 104 =>                                                                          
        
        var_end_vector(127 downto 105) := (others => '0');
        var_end_vector(104 downto   0) := (others => '1');
      
      when 105 =>                                                                          
        
        var_end_vector(127 downto 106) := (others => '0');
        var_end_vector(105 downto   0) := (others => '1');
      
      when 106 =>                                                                          
        
        var_end_vector(127 downto 107) := (others => '0');
        var_end_vector(106 downto   0) := (others => '1');
      
      when 107 =>                                                                          
        
        var_end_vector(127 downto 108) := (others => '0');
        var_end_vector(107 downto   0) := (others => '1');
      
      when 108 =>                                                                          
        
        var_end_vector(127 downto 109) := (others => '0');
        var_end_vector(108 downto   0) := (others => '1');
      
      when 109 =>                                                                          
        
        var_end_vector(127 downto 110) := (others => '0');
        var_end_vector(109 downto   0) := (others => '1');
      
      when 110 =>                                                                          
        
        var_end_vector(127 downto 111) := (others => '0');
        var_end_vector(110 downto   0) := (others => '1');
      
      when 111 =>                                                                          
        
        var_end_vector(127 downto 112) := (others => '0');
        var_end_vector(111 downto   0) := (others => '1');
      
      when 112 =>                                                                          
        
        var_end_vector(127 downto 113) := (others => '0');
        var_end_vector(112 downto   0) := (others => '1');
      
      when 113 =>                                                                          
        
        var_end_vector(127 downto 114) := (others => '0');
        var_end_vector(113 downto   0) := (others => '1');
      
      when 114 =>                                                                          
        
        var_end_vector(127 downto 115) := (others => '0');
        var_end_vector(114 downto   0) := (others => '1');
      
      when 115 =>                                                                          
        
        var_end_vector(127 downto 116) := (others => '0');
        var_end_vector(115 downto   0) := (others => '1');
      
      when 116 =>                                                                          
        
        var_end_vector(127 downto 117) := (others => '0');
        var_end_vector(116 downto   0) := (others => '1');
      
      when 117 =>                                                                          
        
        var_end_vector(127 downto 118) := (others => '0');
        var_end_vector(117 downto   0) := (others => '1');
      
      when 118 =>                                                                          
        
        var_end_vector(127 downto 119) := (others => '0');
        var_end_vector(118 downto   0) := (others => '1');
      
      when 119 =>                                                                          
        
        var_end_vector(127 downto 120) := (others => '0');
        var_end_vector(119 downto   0) := (others => '1');
      
      when 120 =>                                                                          
        
        var_end_vector(127 downto 121) := (others => '0');
        var_end_vector(120 downto   0) := (others => '1');
      
      when 121 =>                                                                          
        
        var_end_vector(127 downto 122) := (others => '0');
        var_end_vector(121 downto   0) := (others => '1');
      
      when 122 =>                                                                          
        
        var_end_vector(127 downto 123) := (others => '0');
        var_end_vector(122 downto   0) := (others => '1');
      
      when 123 =>                                                                          
        
        var_end_vector(127 downto 124) := (others => '0');
        var_end_vector(123 downto   0) := (others => '1');
      
      when 124 =>                                                                          
        
        var_end_vector(127 downto 125) := (others => '0');
        var_end_vector(124 downto   0) := (others => '1');
      
      when 125 =>                                                                          
        
        var_end_vector(127 downto 126) := (others => '0');
        var_end_vector(125 downto   0) := (others => '1');
      
      when 126 =>                                                                          
        
        var_end_vector(127 downto 127) := (others => '0');
        var_end_vector(126 downto   0) := (others => '1');
      
      when others =>
        
        var_end_vector(127 downto   0) := (others => '1');
      
    end case;
   
    Return (var_end_vector);
   
  end function get_end_128;
  
  
 
  
  
  
  
  
  
  
  
  
  
 
   
   
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: funct_clip_value
  --
  -- Function Description:
  --   Returns a value that cannot exceed a clip value.
  --
  -------------------------------------------------------------------
  function funct_clip_value (input_value : natural;
                             max_value   : natural) return natural is
  
    Variable temp_value : Natural := 0;
  
  begin
  
    If (input_value <= max_value) Then
    
      temp_value := input_value;
    
    Else 
    
      temp_value := max_value;
    
    End if;
   
    Return (temp_value);
   
   
  end function funct_clip_value;
  
   
   
  -- Constants
  Constant INTERNAL_CALC_WIDTH   : integer  := C_NUM_BYTES_WIDTH+(C_OP_MODE*2); -- Add 2 bits of math headroom
                                                                                -- if op Mode = 1
  
  
  -- Signals
  signal sig_ouput_stbs       : std_logic_vector(C_STRB_WIDTH-1 downto 0) := (others => '0');
  signal sig_start_offset_un  : unsigned(INTERNAL_CALC_WIDTH-1 downto 0)  := (others => '0');
  signal sig_end_offset_un    : unsigned(INTERNAL_CALC_WIDTH-1 downto 0)  := (others => '0');
 
  

begin --(architecture implementation)

 
  -- Assign the output strobe value
  strb_out <= sig_ouput_stbs ;
  
  
  
  
  
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_OFF_OFF_CASE
  --
  -- If Generate Description:
  --  Calculates the internal start and end offsets for the 
  -- case when start and end offsets are being provided.
  --
  --
  ------------------------------------------------------------
  GEN_OFF_OFF_CASE : if (C_OP_MODE = 1) generate
 
 
    begin
     
      sig_start_offset_un <= RESIZE(UNSIGNED(start_addr_offset), INTERNAL_CALC_WIDTH);
     
      sig_end_offset_un   <= RESIZE(UNSIGNED(end_addr_offset), INTERNAL_CALC_WIDTH);
     
     
   
    end generate GEN_OFF_OFF_CASE;
  
  
  
  
  
  
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_OFF_LEN_CASE
  --
  -- If Generate Description:
  --  Calculates the internal start and end offsets for the 
  -- case when start offset and length are being provided.
  --
  ------------------------------------------------------------
  GEN_OFF_LEN_CASE : if (C_OP_MODE = 0) generate

   -- Local Constants Declarations
    Constant L_INTERNAL_CALC_WIDTH   : integer  := INTERNAL_CALC_WIDTH;
    Constant L_ONE                   : unsigned := TO_UNSIGNED(1, L_INTERNAL_CALC_WIDTH);
    Constant L_ZERO                  : unsigned := TO_UNSIGNED(0, L_INTERNAL_CALC_WIDTH);
    Constant MAX_VALUE               : natural  := C_STRB_WIDTH-1;
    
    
    
   -- local signals
    signal lsig_addr_offset_us       : unsigned(L_INTERNAL_CALC_WIDTH-1 downto 0) := (others => '0');
    signal lsig_num_valid_bytes_us   : unsigned(L_INTERNAL_CALC_WIDTH-1 downto 0) := (others => '0');
    signal lsig_length_adjust_us     : unsigned(L_INTERNAL_CALC_WIDTH-1 downto 0) := (others => '0');
    signal lsig_incr_offset_bytes_us : unsigned(L_INTERNAL_CALC_WIDTH-1 downto 0) := (others => '0');
    signal lsig_end_addr_us          : unsigned(L_INTERNAL_CALC_WIDTH-1 downto 0) := (others => '0');
    signal lsig_end_addr_int         : integer := 0;
    signal lsig_strt_addr_int        : integer := 0;
   
    

    begin
  
      lsig_addr_offset_us       <= RESIZE(UNSIGNED(start_addr_offset), L_INTERNAL_CALC_WIDTH); 
                                                                                            
      lsig_num_valid_bytes_us   <= RESIZE(UNSIGNED(num_valid_bytes)  , L_INTERNAL_CALC_WIDTH); 
      
      lsig_length_adjust_us     <= L_ZERO
        When (lsig_num_valid_bytes_us = L_ZERO)
        Else L_ONE;
      
      lsig_incr_offset_bytes_us <= lsig_num_valid_bytes_us - lsig_length_adjust_us;                             
                                                                                            
      lsig_end_addr_us          <= lsig_addr_offset_us + lsig_incr_offset_bytes_us;            
                                                                                            
      lsig_strt_addr_int        <= TO_INTEGER(lsig_addr_offset_us);                           
                                                                                            
      lsig_end_addr_int         <= TO_INTEGER(lsig_end_addr_us);                              
      
   
   
   
      
      sig_start_offset_un       <= TO_UNSIGNED(funct_clip_value(lsig_strt_addr_int, MAX_VALUE), INTERNAL_CALC_WIDTH);
                                  
      sig_end_offset_un         <= TO_UNSIGNED(funct_clip_value(lsig_end_addr_int, MAX_VALUE), INTERNAL_CALC_WIDTH) ;
      
    
    
    
    
                                                                                            
   
    end generate GEN_OFF_LEN_CASE;
  
  
  
  
  
  
  
 
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_1BIT_CASE
  --
  -- If Generate Description:
  --  Generates the strobes for the 1-bit strobe width case.
  --
  --
  ------------------------------------------------------------
  GEN_1BIT_CASE : if (C_STRB_WIDTH = 1) generate
 
 
    begin
  
      sig_ouput_stbs    <=  (others => '1') ;
      
   
    end generate GEN_1BIT_CASE;
  
  
  
  
  
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_2BIT_CASE
  --
  -- If Generate Description:
  --  Generates the strobes for the 2-bit strobe width case.
  --
  --
  ------------------------------------------------------------
  GEN_2BIT_CASE : if (C_STRB_WIDTH = 2) generate
 
    -- local signals
    Signal lsig_start_offset : Natural :=  0;
    Signal lsig_end_offset   : Natural :=  1;
    
    Signal lsig_start_vect   : std_logic_vector(1 downto 0) := (others => '0');
    Signal lsig_end_vect     : std_logic_vector(1 downto 0) := (others => '0');
    Signal lsig_cmplt_vect   : std_logic_vector(1 downto 0) := (others => '0');
    
 
    begin
 
  
      lsig_start_offset <=  TO_INTEGER(sig_start_offset_un) ;
      lsig_end_offset   <=  TO_INTEGER(sig_end_offset_un  ) ;
      
  
      lsig_start_vect   <=  get_start_2(lsig_start_offset);
      lsig_end_vect     <=  get_end_2(lsig_end_offset)    ;
  
  
      lsig_cmplt_vect   <=  lsig_start_vect and
                            lsig_end_vect; 
  
      sig_ouput_stbs    <=  lsig_cmplt_vect ;
      
   
    end generate GEN_2BIT_CASE;
  
  
  
  
  
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_4BIT_CASE
  --
  -- If Generate Description:
  --  Generates the strobes for the 4-bit strobe width case.
  --
  --
  ------------------------------------------------------------
  GEN_4BIT_CASE : if (C_STRB_WIDTH = 4) generate
 
    -- local signals
    Signal lsig_start_offset : Natural :=  0;
    Signal lsig_end_offset   : Natural :=  3;
    
    Signal lsig_start_vect   : std_logic_vector(3 downto 0) := (others => '0');
    Signal lsig_end_vect     : std_logic_vector(3 downto 0) := (others => '0');
    Signal lsig_cmplt_vect   : std_logic_vector(3 downto 0) := (others => '0');
    
 
    begin
 
  
      lsig_start_offset <=  TO_INTEGER(sig_start_offset_un) ;
      lsig_end_offset   <=  TO_INTEGER(sig_end_offset_un  ) ;
      
  
      lsig_start_vect   <=  get_start_4(lsig_start_offset);
      lsig_end_vect     <=  get_end_4(lsig_end_offset)    ;
  
  
      lsig_cmplt_vect   <=  lsig_start_vect and
                            lsig_end_vect; 
  
      sig_ouput_stbs    <=  lsig_cmplt_vect ;
      
   
    end generate GEN_4BIT_CASE;
  
  
  
  
  
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_8BIT_CASE
  --
  -- If Generate Description:
  --  Generates the strobes for the 8-bit strobe width case.
  --
  --
  ------------------------------------------------------------
  GEN_8BIT_CASE : if (C_STRB_WIDTH = 8) generate
 
    -- local signals
    Signal lsig_start_offset : Natural :=  0;
    Signal lsig_end_offset   : Natural :=  7;
    
    Signal lsig_start_vect   : std_logic_vector(7 downto 0) := (others => '0');
    Signal lsig_end_vect     : std_logic_vector(7 downto 0) := (others => '0');
    Signal lsig_cmplt_vect   : std_logic_vector(7 downto 0) := (others => '0');
    
 
    begin
 
  
      lsig_start_offset <=  TO_INTEGER(sig_start_offset_un) ;
      lsig_end_offset   <=  TO_INTEGER(sig_end_offset_un  ) ;
      
  
      lsig_start_vect   <=  get_start_8(lsig_start_offset);
      lsig_end_vect     <=  get_end_8(lsig_end_offset)    ;
  
  
      lsig_cmplt_vect   <=  lsig_start_vect and
                            lsig_end_vect; 
  
      sig_ouput_stbs    <=  lsig_cmplt_vect ;
      
   
    end generate GEN_8BIT_CASE;
  
  
  
  
  
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_16BIT_CASE
  --
  -- If Generate Description:
  --  Generates the strobes for the 16-bit strobe width case.
  --
  --
  ------------------------------------------------------------
  GEN_16BIT_CASE : if (C_STRB_WIDTH = 16) generate
 
    -- local signals
    Signal lsig_start_offset : Natural :=  0;
    Signal lsig_end_offset   : Natural := 15;
    
    Signal lsig_start_vect   : std_logic_vector(15 downto 0) := (others => '0');
    Signal lsig_end_vect     : std_logic_vector(15 downto 0) := (others => '0');
    Signal lsig_cmplt_vect   : std_logic_vector(15 downto 0) := (others => '0');
    
 
    begin
 
  
      lsig_start_offset <=  TO_INTEGER(sig_start_offset_un) ;
      lsig_end_offset   <=  TO_INTEGER(sig_end_offset_un  ) ;
      
  
      lsig_start_vect   <=  get_start_16(lsig_start_offset);
      lsig_end_vect     <=  get_end_16(lsig_end_offset)    ;
  
  
      lsig_cmplt_vect   <=  lsig_start_vect and
                            lsig_end_vect; 
  
      sig_ouput_stbs    <=  lsig_cmplt_vect ;
      
   
    end generate GEN_16BIT_CASE;
  
  
  
  
  
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_32BIT_CASE
  --
  -- If Generate Description:
  --  Generates the strobes for the 32-bit strobe width case.
  --
  --
  ------------------------------------------------------------
  GEN_32BIT_CASE : if (C_STRB_WIDTH = 32) generate
 
    -- local signals
    Signal lsig_start_offset : Natural :=  0;
    Signal lsig_end_offset   : Natural := 31;
    
    Signal lsig_start_vect   : std_logic_vector(31 downto 0) := (others => '0');
    Signal lsig_end_vect     : std_logic_vector(31 downto 0) := (others => '0');
    Signal lsig_cmplt_vect   : std_logic_vector(31 downto 0) := (others => '0');
    
 
    begin
 
  
      lsig_start_offset <=  TO_INTEGER(sig_start_offset_un) ;
      lsig_end_offset   <=  TO_INTEGER(sig_end_offset_un  ) ;
      
  
      lsig_start_vect   <=  get_start_32(lsig_start_offset);
      lsig_end_vect     <=  get_end_32(lsig_end_offset)    ;
  
  
      lsig_cmplt_vect   <=  lsig_start_vect and
                            lsig_end_vect; 
  
      sig_ouput_stbs    <=  lsig_cmplt_vect ;
      
   
    end generate GEN_32BIT_CASE;
  
  
  
  
  
 
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_64BIT_CASE
  --
  -- If Generate Description:
  --  Generates the strobes for the 64-bit strobe width case.
  --
  --
  ------------------------------------------------------------
  GEN_64BIT_CASE : if (C_STRB_WIDTH = 64) generate
 
    -- local signals
    Signal lsig_start_offset : Natural :=  0;
    Signal lsig_end_offset   : Natural := 63;
    
    Signal lsig_start_vect   : std_logic_vector(63 downto 0) := (others => '0');
    Signal lsig_end_vect     : std_logic_vector(63 downto 0) := (others => '0');
    Signal lsig_cmplt_vect   : std_logic_vector(63 downto 0) := (others => '0');
    
 
    begin
 
  
      lsig_start_offset <=  TO_INTEGER(sig_start_offset_un) ;
      lsig_end_offset   <=  TO_INTEGER(sig_end_offset_un  ) ;
      
  
      lsig_start_vect   <=  get_start_64(lsig_start_offset);
      lsig_end_vect     <=  get_end_64(lsig_end_offset)    ;
  
  
      lsig_cmplt_vect   <=  lsig_start_vect and
                            lsig_end_vect; 
  
      sig_ouput_stbs    <=  lsig_cmplt_vect ;
      
   
    end generate GEN_64BIT_CASE;
  
  
  
 
 
  
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_128BIT_CASE
  --
  -- If Generate Description:
  --  Generates the strobes for the 64-bit strobe width case.
  --
  --
  ------------------------------------------------------------
  GEN_128BIT_CASE : if (C_STRB_WIDTH = 128) generate
 
    -- local signals
    Signal lsig_start_offset : Natural :=   0;
    Signal lsig_end_offset   : Natural := 127;
    
    Signal lsig_start_vect   : std_logic_vector(127 downto 0) := (others => '0');
    Signal lsig_end_vect     : std_logic_vector(127 downto 0) := (others => '0');
    Signal lsig_cmplt_vect   : std_logic_vector(127 downto 0) := (others => '0');
    
 
    begin
 
  
      lsig_start_offset <=  TO_INTEGER(sig_start_offset_un) ;
      lsig_end_offset   <=  TO_INTEGER(sig_end_offset_un  ) ;
      
  
      lsig_start_vect   <=  get_start_128(lsig_start_offset);
      lsig_end_vect     <=  get_end_128(lsig_end_offset)    ;
  
  
      lsig_cmplt_vect   <=  lsig_start_vect and
                            lsig_end_vect; 
  
      sig_ouput_stbs    <=  lsig_cmplt_vect ;
      
   
    end generate GEN_128BIT_CASE;
  
  
  
  
  
 
 
 
 
 

end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_pcc.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_pcc.vhd
  --
  -- Description:
  --    This file implements the DataMover Predictive Command Calculator (PCC).
  --
  --
  --
  --
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;

  library axi_datamover_v5_1_14;
  use axi_datamover_v5_1_14.axi_datamover_strb_gen2;

  -------------------------------------------------------------------------------

  entity axi_datamover_pcc is
    generic (

      C_IS_MM2S              : Integer range  0 to   1 :=  0;
        -- This parameter tells the PCC module if it is a MM2S
        -- instance or a S2MM instance.
        -- 0 = S2MM Instance
        -- 1 = MM2S Instance
      
      C_DRE_ALIGN_WIDTH      : Integer range  1 to   6 :=  2;
        -- Sets the width of the DRE Aligment output ports
        
      C_SEL_ADDR_WIDTH       : Integer range  1 to   8 :=  5;
        -- Sets the width of the LS address bus used for 
        -- Muxing/Demuxing data to/from a wider AXI4 data bus
      
      C_ADDR_WIDTH           : Integer range 32 to  64 := 32;
        -- Sets the width of the AXi Address Channel 
      
      C_STREAM_DWIDTH        : Integer range  8 to 1024 := 32;
        -- Sets the width of the Stream Data width that
        -- is being supported by the PCC
      
      C_MAX_BURST_LEN        : Integer range 2 to 256 := 16;
        -- Indicates the max allowed burst length to use for
        -- AXI4 transfer calculations
      
      C_CMD_WIDTH            : Integer                 := 68;
        -- Sets the width of the input command port
      
      C_TAG_WIDTH            : Integer range  1 to   8 :=  4;
        -- Sets the width of the Tag field in the input command
      
      C_BTT_USED             : Integer range  8 to  23 := 16;
        -- Sets the width of the used portion of the BTT field
        -- of the input command
      
      C_SUPPORT_INDET_BTT    : Integer range  0 to   1 :=  0;
        -- Indicates if the Indeterminate BTT mode is enabled
      
      C_NATIVE_XFER_WIDTH    : Integer range  8 to 1024 :=  32;
        -- Indicates the Native transfer width to use for all
        -- transfer calculations. This will either be the DataMover
        -- input Stream width or the AXI4 MMap data width depending
        -- on DataMover parameterization.
      
      C_STRT_SF_OFFSET_WIDTH : Integer range  1 to   7 :=  1
        -- Indicates the width of the starting address offset
        -- bus passed to Store and Forward functions                                                    

      );
    port (

      -- Clock and Reset input ----------------------------------------
      primary_aclk           : in  std_logic;                        --
         -- Primary synchronization clock for the Master side        --
         -- interface and internal logic. It is also used            --
         -- for the User interface synchronization when              --
         -- C_STSCMD_IS_ASYNC = 0.                                   --
                                                                     --
      -- Reset input                                                 --
      mmap_reset             : in  std_logic;                        --
        -- Reset used for the internal master logic                  --
      -----------------------------------------------------------------




      -- Master Command FIFO/Register Interface --------------------------------------------
                                                                                          --
      cmd2mstr_command      : in std_logic_vector(C_CMD_WIDTH-1 downto 0);                --
         -- The next command value available from the Command FIFO/Register               --
                                                                                          --
      cache2mstr_command      : in std_logic_vector(7 downto 0);                --
         -- The next command value available from the Command FIFO/Register               --
      cmd2mstr_cmd_valid    : in std_logic;                                               --
         -- Handshake bit indicating if the Command FIFO/Register has at leasdt 1 entry   --
                                                                                          --
      mst2cmd_cmd_ready     : out  std_logic;                                             --
         -- Handshake bit indicating the Command Calculator is ready to accept            --
         -- another command                                                               --
      --------------------------------------------------------------------------------------



      -- Address Channel Controller Interface -----------------------------------
                                                                               --
      mstr2addr_tag       : out std_logic_vector(C_TAG_WIDTH-1 downto 0);      --
         -- The next command tag                                               --
                                                                               --
      mstr2addr_addr      : out std_logic_vector(C_ADDR_WIDTH-1 downto 0);     --
         -- The next command address to put on the AXI MMap ADDR               --
                                                                               --
      mstr2addr_len       : out std_logic_vector(7 downto 0);                  --
         -- The next command length to put on the AXI MMap LEN                 --
                                                                               --
      mstr2addr_size      : out std_logic_vector(2 downto 0);                  --
         -- The next command size to put on the AXI MMap SIZE                  --
                                                                               --
      mstr2addr_burst     : out std_logic_vector(1 downto 0);                  --
         -- The next command burst type to put on the AXI MMap BURST           --
                                                                               --
      mstr2addr_cache     : out std_logic_vector(3 downto 0);                  --
         -- The next command burst type to put on the AXI MMap BURST           --

      mstr2addr_user     : out std_logic_vector(3 downto 0);                  --
         -- The next command burst type to put on the AXI MMap BURST           --

      mstr2addr_cmd_cmplt : out std_logic;                                     --
         -- The indication to the Address Channel that the current             --
         -- sub-command output is the last one compiled from the               --
         -- parent command pulled from the Command FIFO                        --
                                                                               --
      mstr2addr_calc_error : out std_logic;                                    --
         -- Indication if the next command in the calculation pipe             --
         -- has a calcualtion error                                            --
                                                                               --
      mstr2addr_cmd_valid : out std_logic;                                     --
         -- The next command valid indication to the Address Channel           --
         -- Controller for the AXI MMap                                        --
                                                                               --
      addr2mstr_cmd_ready : In std_logic;                                      --
         -- Indication from the Address Channel Controller that the            --
         -- command is being accepted                                          --
      ---------------------------------------------------------------------------
      



      -- Data Channel Controller Interface ------------------------------------------------
                                                                                         --
      mstr2data_tag        : out std_logic_vector(C_TAG_WIDTH-1 downto 0);               --
         -- The next command tag                                                         --
                                                                                         --
      mstr2data_saddr_lsb  : out std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0);          --
         -- The next command start address LSbs to use for the read data                 --
         -- mux (only used if Stream data width is less than the MMap data               --
         -- width).                                                                      --
                                                                                         --
      mstr2data_len        : out std_logic_vector(7 downto 0);                           --
         -- The LEN value output to the Address Channel                                  --
                                                                                         --
      mstr2data_strt_strb  : out std_logic_vector((C_NATIVE_XFER_WIDTH/8)-1 downto 0);   --
         -- The starting strobe value to use for the data transfer                       --
                                                                                         --
      mstr2data_last_strb  : out std_logic_vector((C_NATIVE_XFER_WIDTH/8)-1 downto 0);   --
         -- The endiing (LAST) strobe value to use for the data transfer                 --
                                                                                         --
      mstr2data_drr        : out std_logic;                                              --
         -- The starting tranfer of a sequence of transfers                              --
                                                                                         --
      mstr2data_eof        : out std_logic;                                              --
         -- The endiing tranfer of a sequence of parent transfer commands                --
                                                                                         --
      mstr2data_sequential : Out std_logic;                                              --
         -- The next sequential tranfer of a sequence of transfers                       --
         -- spawned from a single parent command                                         --
                                                                                         --
      mstr2data_calc_error : out std_logic;                                              --
         -- Indication if the next command in the calculation pipe                       --
         -- has a calculation error                                                      --
                                                                                         --
      mstr2data_cmd_cmplt  : out std_logic;                                              --
         -- The indication to the Data Channel that the current                          --
         -- sub-command output is the last one compiled from the                         --
         -- parent command pulled from the Command FIFO                                  --
                                                                                         --
      mstr2data_cmd_valid  : out std_logic;                                              --
         -- The next command valid indication to the Data Channel                        --
         -- Controller for the AXI MMap                                                  --
                                                                                         --
      data2mstr_cmd_ready  : In std_logic ;                                              --
         -- Indication from the Data Channel Controller that the                         --
         -- command is being accepted on the AXI Address                                 --
         -- Channel                                                                      --
                                                                                         --
      mstr2data_dre_src_align  : Out std_logic_vector(C_DRE_ALIGN_WIDTH-1 downto 0);     --
         -- The source (input) alignment for the MM2S DRE                                --
                                                                                         --
      mstr2data_dre_dest_align : Out std_logic_vector(C_DRE_ALIGN_WIDTH-1 downto 0);     --
         -- The destinstion (output) alignment for the MM2S DRE                          --
      -------------------------------------------------------------------------------------
                                                                                         
                                                                                         
                                                                                         
                                                                                         
      -- Output flag indicating that a calculation error has occured ----------------------
                                                                                         --
      calc_error               : Out std_logic;                                          --
         -- Indication from the Command Calculator that a calculation                    --
         -- error has occured.                                                           --
      -------------------------------------------------------------------------------------





      -- Special DRE Controller Interface      --------------------------------------------
                                                                                         --
      dre2mstr_cmd_ready        : In std_logic ;                                         --
         -- Indication from the S2MM DRE Controller that it can                          --
         -- accept another command.                                                      --
                                                                                         --
      mstr2dre_cmd_valid        : out std_logic ;                                        --
         -- The next command valid indication to the S2MM DRE                            --
         -- Controller.                                                                  --
                                                                                         --
      mstr2dre_tag              : out std_logic_vector(C_TAG_WIDTH-1 downto 0);          --
         -- The next command tag                                                         --
                                                                                         --
      mstr2dre_dre_src_align    : Out std_logic_vector(C_DRE_ALIGN_WIDTH-1 downto 0) ;   --
         -- The source (S2MM Stream) alignment for the S2MM DRE                          --
                                                                                         --
      mstr2dre_dre_dest_align   : Out std_logic_vector(C_DRE_ALIGN_WIDTH-1 downto 0) ;   --
         -- The destinstion (S2MM MMap) alignment for the S2MM DRE                       --
                                                                                         --
      mstr2dre_btt              : out std_logic_vector(C_BTT_USED-1 downto 0) ;          --
         -- The BTT value output to the S2MM DRE. This is needed for                     --
         -- Scatter operations.                                                          --
                                                                                         --
      mstr2dre_drr              : out std_logic ;                                        --
         -- The starting tranfer of a sequence of transfers                              --
                                                                                         --
      mstr2dre_eof              : out std_logic ;                                        --
         -- The endiing tranfer of a sequence of parent transfer commands                --
                                                                                         --
      mstr2dre_cmd_cmplt        : Out std_logic ;                                        --
         -- The last child tranfer of a sequence of transfers                            --
         -- spawned from a single parent command                                         --
                                                                                         --
      mstr2dre_calc_error       : out std_logic ;                                        --
         -- Indication if the next command in the calculation pipe                       --
         -- has a calculation error                                                      --
      -------------------------------------------------------------------------------------

     
     
     -- Store and Forward Support Start Offset --------------------------------------------- 
                                                                                          -- 
     mstr2dre_strt_offset       : out std_logic_vector(C_STRT_SF_OFFSET_WIDTH-1 downto 0) -- 
       -- Relays the starting address offset for a transfer to the Store and Forward      -- 
       -- functions incorporating upsizer/downsizer logic                                 -- 
     ---------------------------------------------------------------------------------------    


      );

  end entity axi_datamover_pcc;


  architecture implementation of axi_datamover_pcc is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


  -- Function declarations  -------------------

    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_dbeat_residue_width
    --
    -- Function Description:
    --  Calculates the number of Least significant bits of the BTT field
    -- that are unused for the LEN calculation
    --
    -------------------------------------------------------------------
    function funct_get_dbeat_residue_width (bytes_per_beat : integer) return integer is

      Variable temp_dbeat_residue_width : Integer := 0; -- 8-bit stream

    begin

      case bytes_per_beat is
        when 1 =>  
            temp_dbeat_residue_width := 0;
        when 2 =>
            temp_dbeat_residue_width := 1;
        when 4 =>
            temp_dbeat_residue_width := 2;
        when 8 =>
            temp_dbeat_residue_width := 3;
        when 16 =>
            temp_dbeat_residue_width := 4;
        when 32 =>
            temp_dbeat_residue_width := 5;
        when 64 =>
            temp_dbeat_residue_width := 6;
        
        when others =>  -- 128-byte transfers
            temp_dbeat_residue_width := 7;
      end case;

      Return (temp_dbeat_residue_width);

    end function funct_get_dbeat_residue_width;




    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_burstcnt_offset
    --
    -- Function Description:
    -- Calculates the bit offset from the residue bits needed to detirmine
    -- the load value for the burst counter.
    --
    -------------------------------------------------------------------
    function funct_get_burst_residue_width (max_burst_len : integer) return integer is


      Variable temp_burst_residue_width : Integer := 0;

    begin

      case max_burst_len is

        when 256 =>
          temp_burst_residue_width := 8;
        when 128 =>
          temp_burst_residue_width := 7;
        when 64 =>
          temp_burst_residue_width := 6;
        when 32 =>
          temp_burst_residue_width := 5;
        when 16 =>
          temp_burst_residue_width := 4;
        when 8 =>
          temp_burst_residue_width := 3;
        when 4 =>
          temp_burst_residue_width := 2;
        when others =>   -- assume 2 dbeats
          temp_burst_residue_width := 1;
      end case;

      Return (temp_burst_residue_width);

    end function funct_get_burst_residue_width;


    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_get_axi_size
    --
    -- Function Description:
    -- Calculates the AXI SIZE Qualifier based on the data width.
    --
    -------------------------------------------------------------------
    function func_get_axi_size (native_dwidth : integer) return std_logic_vector is

      Constant AXI_SIZE_1BYTE    : std_logic_vector(2 downto 0) := "000";
      Constant AXI_SIZE_2BYTE    : std_logic_vector(2 downto 0) := "001";
      Constant AXI_SIZE_4BYTE    : std_logic_vector(2 downto 0) := "010";
      Constant AXI_SIZE_8BYTE    : std_logic_vector(2 downto 0) := "011";
      Constant AXI_SIZE_16BYTE   : std_logic_vector(2 downto 0) := "100";
      Constant AXI_SIZE_32BYTE   : std_logic_vector(2 downto 0) := "101";
      Constant AXI_SIZE_64BYTE   : std_logic_vector(2 downto 0) := "110";
      Constant AXI_SIZE_128BYTE  : std_logic_vector(2 downto 0) := "111";

      Variable temp_size : std_logic_vector(2 downto 0) := (others => '0');

    begin

       case native_dwidth is
         when 8 =>
           temp_size := AXI_SIZE_1BYTE;
         when 16 =>
           temp_size := AXI_SIZE_2BYTE;
         when 32 =>
           temp_size := AXI_SIZE_4BYTE;
         when 64 =>
           temp_size := AXI_SIZE_8BYTE;
         when 128 =>
           temp_size := AXI_SIZE_16BYTE;
         when 256 =>
           temp_size := AXI_SIZE_32BYTE;
         when 512 =>
           temp_size := AXI_SIZE_64BYTE;
         
         when others => -- 1024 bit dwidth
           temp_size := AXI_SIZE_128BYTE;
       end case;

       Return (temp_size);


    end function func_get_axi_size;



    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_sf_offset_ls_index
    --
    -- Function Description:
    --   Calculates the Ls index of the Store and Forward
    --  starting offset bus based on the User Stream Width.
    --
    -------------------------------------------------------------------
    function funct_get_sf_offset_ls_index (stream_width : integer) return integer is
    
      Variable lvar_temp_ls_index : Integer := 0;
    
    begin
    
      case stream_width is
        when 8 =>
          lvar_temp_ls_index := 0;
        when 16 =>
          lvar_temp_ls_index := 1;
        when 32 =>
          lvar_temp_ls_index := 2;
        when 64 =>
          lvar_temp_ls_index := 3;
        when 128 =>
          lvar_temp_ls_index := 4;
        when 256 =>
          lvar_temp_ls_index := 5;
        when 512 =>
          lvar_temp_ls_index := 6;
        when others => -- 1024
          lvar_temp_ls_index := 7;
      end case;
      
      Return (lvar_temp_ls_index);
    
    
    end function funct_get_sf_offset_ls_index;
    


    -- Constant Declarations  ----------------------------------------

    Constant BASE_CMD_WIDTH    : integer := 32; -- Bit Width of Command LS (no address)
    Constant CMD_BTT_WIDTH     : integer := C_BTT_USED;
    Constant CMD_BTT_LS_INDEX  : integer := 0;
    Constant CMD_BTT_MS_INDEX  : integer := CMD_BTT_WIDTH-1;
    Constant CMD_TYPE_INDEX    : integer := 23;
    Constant CMD_DRR_INDEX     : integer := BASE_CMD_WIDTH-1;
    Constant CMD_EOF_INDEX     : integer := BASE_CMD_WIDTH-2;
    Constant CMD_DSA_WIDTH     : integer := 6;
    Constant CMD_DSA_LS_INDEX  : integer := CMD_TYPE_INDEX+1;
    Constant CMD_DSA_MS_INDEX  : integer := (CMD_DSA_LS_INDEX+CMD_DSA_WIDTH)-1;
    Constant CMD_ADDR_LS_INDEX : integer := BASE_CMD_WIDTH;
    Constant CMD_ADDR_MS_INDEX : integer := (C_ADDR_WIDTH+BASE_CMD_WIDTH)-1;
    Constant CMD_TAG_WIDTH     : integer := C_TAG_WIDTH;
    Constant CMD_TAG_LS_INDEX  : integer := C_ADDR_WIDTH+BASE_CMD_WIDTH;
    Constant CMD_TAG_MS_INDEX  : integer := (CMD_TAG_LS_INDEX+CMD_TAG_WIDTH)-1;


    ----------------------------------------------------------------------------------------
    -- Command calculation constants

    Constant SIZE_TO_USE               : std_logic_vector(2 downto 0) := func_get_axi_size(C_NATIVE_XFER_WIDTH);
    Constant BYTES_PER_DBEAT           : integer := C_NATIVE_XFER_WIDTH/8;
    Constant DBEATS_PER_BURST          : integer := C_MAX_BURST_LEN;
    Constant BYTES_PER_MAX_BURST       : integer := DBEATS_PER_BURST*BYTES_PER_DBEAT;
    Constant LEN_WIDTH                 : integer  := 8;  -- 8 bits fixed
    Constant MAX_LEN_VALUE             : integer  := DBEATS_PER_BURST-1;
    Constant XFER_LEN_ZERO             : std_logic_vector(LEN_WIDTH-1 downto 0) := (others => '0');
    Constant DBEAT_RESIDUE_WIDTH       : integer  := funct_get_dbeat_residue_width(BYTES_PER_DBEAT);
    Constant BURST_RESIDUE_WIDTH       : integer  := funct_get_burst_residue_width(C_MAX_BURST_LEN);
    Constant BURST_RESIDUE_LS_INDEX    : integer  := DBEAT_RESIDUE_WIDTH;
    Constant BTT_RESIDUE_WIDTH         : integer  := DBEAT_RESIDUE_WIDTH+BURST_RESIDUE_WIDTH;
    Constant BTT_ZEROS                 : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    Constant BTT_RESIDUE_1             : unsigned := TO_UNSIGNED( 1, BTT_RESIDUE_WIDTH);
    Constant BTT_RESIDUE_0             : unsigned := TO_UNSIGNED( 0, BTT_RESIDUE_WIDTH);
    Constant BURST_CNT_LS_INDEX        : integer  := DBEAT_RESIDUE_WIDTH+BURST_RESIDUE_WIDTH;
    Constant BURST_CNTR_WIDTH          : integer  := CMD_BTT_WIDTH - (DBEAT_RESIDUE_WIDTH+BURST_RESIDUE_WIDTH); 
    Constant BRST_CNT_1                : unsigned := TO_UNSIGNED( 1, BURST_CNTR_WIDTH);
    Constant BRST_CNT_0                : unsigned := TO_UNSIGNED( 0, BURST_CNTR_WIDTH);
    Constant BRST_RESIDUE_0            : std_logic_vector(BURST_RESIDUE_WIDTH-1 downto 0) := (others => '0');
    Constant DBEAT_RESIDUE_0           : std_logic_vector(DBEAT_RESIDUE_WIDTH-1 downto 0) := (others => '0');
    Constant ADDR_CNTR_WIDTH           : integer  := 16;  -- Addres Counter slice
    Constant ADDR_MS_SLICE_WIDTH       : integer  := C_ADDR_WIDTH-ADDR_CNTR_WIDTH;
    Constant ADDR_CNTR_MAX_VALUE       : unsigned := TO_UNSIGNED((2**ADDR_CNTR_WIDTH)-1, ADDR_CNTR_WIDTH);
    Constant ADDR_CNTR_ONE             : unsigned := TO_UNSIGNED(1, ADDR_CNTR_WIDTH);
    Constant MBAA_ADDR_SLICE_WIDTH     : integer := BTT_RESIDUE_WIDTH;
    Constant STRBGEN_ADDR_SLICE_WIDTH  : integer := DBEAT_RESIDUE_WIDTH;
    Constant STRBGEN_ADDR_0            : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
    Constant STRBGEN_ADDR_SLICE_1      : unsigned := TO_UNSIGNED( 1, STRBGEN_ADDR_SLICE_WIDTH);

    Constant SF_OFFSET_LS_INDEX        : integer := funct_get_sf_offset_ls_index(C_STREAM_DWIDTH);
    Constant SF_OFFSET_MS_INDEX        : integer := (SF_OFFSET_LS_INDEX + C_STRT_SF_OFFSET_WIDTH)-1;
    
    

    -- Type Declarations  --------------------------------------------

    type PCC_SM_STATE_TYPE is (
                INIT,
                WAIT_FOR_CMD,
                CALC_1,
                CALC_2,
                CALC_3,
                WAIT_ON_XFER_PUSH,
                CHK_IF_DONE,
                ERROR_TRAP
                );





    -- Signal Declarations  --------------------------------------------
    Signal sig_pcc_sm_state                 : PCC_SM_STATE_TYPE := INIT;
    Signal sig_pcc_sm_state_ns              : PCC_SM_STATE_TYPE := INIT;
    signal sig_sm_halt_ns                   : std_logic := '0';
    signal sig_sm_halt_reg                  : std_logic := '0';
    signal sig_sm_ld_xfer_reg_ns            : std_logic := '0';
    signal sig_sm_ld_xfer_reg_ns_tmp            : std_logic := '0';
    signal sig_sm_pop_input_reg_ns          : std_logic := '0';
    signal sig_sm_pop_input_reg             : std_logic := '0';
    signal sig_sm_ld_calc1_reg_ns           : std_logic := '0';
    signal sig_sm_ld_calc1_reg              : std_logic := '0';
    signal sig_sm_ld_calc2_reg_ns           : std_logic := '0';
    signal sig_sm_ld_calc2_reg              : std_logic := '0';
    
    signal sig_sm_ld_calc3_reg_ns           : std_logic := '0';
    signal sig_sm_ld_calc3_reg              : std_logic := '0';
    
    
    signal sig_parent_done                  : std_logic := '0';
    signal sig_ld_xfer_reg                  : std_logic := '0';
    signal sig_ld_xfer_reg_tmp                  : std_logic := '0';
    signal sig_btt_raw                      : std_logic := '0';
    signal sig_btt_is_zero                  : std_logic := '0';
    signal sig_btt_is_zero_reg              : std_logic := '0';
    
    -- unused signal sig_next_tag                     : std_logic_vector(CMD_TAG_WIDTH-1 downto 0) := (others => '0');
    -- unused signal sig_next_addr                    : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    -- unused signal sig_next_len                     : std_logic_vector(LEN_WIDTH-1 downto 0) := (others => '0');
    -- unused signal sig_next_size                    : std_logic_vector(2 downto 0) := (others => '0');
    -- unused signal sig_next_burst                   : std_logic_vector(1 downto 0) := (others => '0');
    -- unused signal sig_next_strt_strb               : std_logic_vector((C_NATIVE_XFER_WIDTH/8)-1 downto 0) := (others => '0');
    -- unused signal sig_next_end_strb                : std_logic_vector((C_NATIVE_XFER_WIDTH/8)-1 downto 0) := (others => '0');

    ----------------------------------------------------------------------------------------
    -- Burst Buster signals
    signal sig_burst_cnt_slice_im0          : unsigned(BURST_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_last_xfer_valid_im1          : std_logic := '0';
    signal sig_brst_cnt_eq_zero_im0         : std_logic := '0';
    signal sig_brst_cnt_eq_zero_ireg1       : std_logic := '0';
    signal sig_brst_cnt_eq_one_im0          : std_logic := '0';
    signal sig_brst_cnt_eq_one_ireg1        : std_logic := '0';
    signal sig_brst_residue_eq_zero         : std_logic := '0';
    signal sig_brst_residue_eq_zero_reg     : std_logic := '0';
    signal sig_no_btt_residue_im0           : std_logic := '0';
    signal sig_no_btt_residue_ireg1         : std_logic := '0';
    signal sig_btt_residue_slice_im0        : Unsigned(BTT_RESIDUE_WIDTH-1 downto 0) := (others => '0');

    -- Input command register
    signal sig_push_input_reg               : std_logic := '0';
    signal sig_pop_input_reg                : std_logic := '0';
    signal sig_input_burst_type_reg         : std_logic := '0';
    signal sig_input_cache_type_reg         : std_logic_vector (3 downto 0) := "0000";
    signal sig_input_user_type_reg          : std_logic_vector (3 downto 0) := "0000";
    signal sig_input_btt_residue_minus1_reg : std_logic_vector(BTT_RESIDUE_WIDTH-1 downto 0) := (others => '0');
    signal sig_input_dsa_reg                : std_logic_vector(CMD_DSA_WIDTH-1 downto 0) := (others => '0');
    signal sig_input_drr_reg                : std_logic := '0';
    signal sig_input_eof_reg                : std_logic := '0';
    signal sig_input_tag_reg                : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_input_reg_empty              : std_logic := '0';
    signal sig_input_reg_full               : std_logic := '0';

    -- Output qualifier Register
    --  signal sig_ld_output                    : std_logic := '0';
    signal sig_push_xfer_reg                : std_logic := '0';
    signal sig_pop_xfer_reg                 : std_logic := '0';
    signal sig_xfer_addr_reg                : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_xfer_type_reg                : std_logic := '0';
    signal sig_xfer_cache_reg               : std_logic_vector (3 downto 0) := "0000";
    signal sig_xfer_user_reg                : std_logic_vector (3 downto 0) := "0000";
    signal sig_xfer_len_reg                 : std_logic_vector(LEN_WIDTH-1 downto 0) := (others => '0');
    signal sig_xfer_tag_reg                 : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_xfer_dsa_reg                 : std_logic_vector(CMD_DSA_WIDTH-1 downto 0) := (others => '0');
    signal sig_xfer_drr_reg                 : std_logic := '0';
    signal sig_xfer_eof_reg                 : std_logic := '0';
    signal sig_xfer_strt_strb_reg           : std_logic_vector(BYTES_PER_DBEAT-1 downto 0) := (others => '0');
    signal sig_xfer_end_strb_reg            : std_logic_vector(BYTES_PER_DBEAT-1 downto 0) := (others => '0');
    signal sig_xfer_is_seq_reg              : std_logic := '0';
    signal sig_xfer_cmd_cmplt_reg           : std_logic := '0';
    signal sig_xfer_calc_err_reg            : std_logic := '0';
    signal sig_xfer_reg_empty               : std_logic := '0';
    signal sig_xfer_reg_full                : std_logic := '0';
                                            
    -- Address Counter                      
    signal sig_ld_addr_cntr                 : std_logic := '0';
    signal sig_incr_addr_cntr               : std_logic := '0';
    signal sig_addr_cntr_incr_im1           : Unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_byte_change_minus1_im2       : Unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');

    -- misc
    signal sig_xfer_len_im2                 : std_logic_vector(LEN_WIDTH-1 downto 0);
    signal sig_xfer_strt_strb_im2           : std_logic_vector(BYTES_PER_DBEAT-1 downto 0) := (others => '0');
    signal sig_xfer_strt_strb2use_im3       : std_logic_vector(BYTES_PER_DBEAT-1 downto 0) := (others => '0');
    signal sig_xfer_end_strb_im2            : std_logic_vector(BYTES_PER_DBEAT-1 downto 0) := (others => '0');
    signal sig_xfer_end_strb2use_im3        : std_logic_vector(BYTES_PER_DBEAT-1 downto 0) := (others => '0');
    signal sig_xfer_address_im0             : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_xfer_size                    : std_logic_vector(2 downto 0) := (others => '0');
    signal sig_cmd_addr_slice               : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd_btt_slice                : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd_type_slice               : std_logic := '0';
    signal sig_cmd_cache_slice              : std_logic_vector (3 downto 0) := "0000";
    signal sig_cmd_user_slice               : std_logic_vector (3 downto 0) := "0000";
    signal sig_cmd_tag_slice                : std_logic_vector(CMD_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd_dsa_slice                : std_logic_vector(CMD_DSA_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd_drr_slice                : std_logic := '0';
    signal sig_cmd_eof_slice                : std_logic := '0';
    signal sig_calc_error_reg               : std_logic := '0';
    signal sig_calc_error_pushed            : std_logic := '0';

   -- PCC2 stuff
    signal sig_finish_addr_offset_im1       : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
    signal sig_xfer_len_eq_0_im2            : std_logic := '0';
    signal sig_first_xfer_im0               : std_logic := '0';
    signal sig_bytes_to_mbaa_im0            : unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_bytes_to_mbaa_ireg1          : unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_lsh_rollover            : std_logic := '0';
    signal sig_predict_addr_lsh_slv         : std_logic_vector(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_predict_addr_lsh_im1         : unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_cntr_lsh_im0            : unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_cntr_lsh_kh            : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_cntr_lsh_im0_slv        : std_logic_vector(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_cntr_im0_msh            : unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_strbgen_addr_im0             : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
    signal sig_strbgen_bytes_im1            : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH   downto 0) := (others => '0');
    signal sig_ld_btt_cntr                  : std_logic := '0';
    signal sig_decr_btt_cntr                : std_logic := '0';
    signal sig_btt_cntr_im0                 : unsigned(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd2data_valid               : std_logic := '0';
    signal sig_clr_cmd2data_valid           : std_logic := '0';
    signal sig_cmd2addr_valid               : std_logic := '0';
    signal sig_clr_cmd2addr_valid           : std_logic := '0';
    signal sig_btt_lt_b2mbaa_im0            : std_logic := '0';
    signal sig_btt_lt_b2mbaa_ireg1          : std_logic := '0';
    signal sig_btt_eq_b2mbaa_im0            : std_logic := '0';
    signal sig_btt_eq_b2mbaa_ireg1          : std_logic := '0';
    signal sig_addr_incr_ge_bpdb_im1        : std_logic := '0';

    -- Unaligned start address support
    signal sig_adjusted_addr_incr_im1       : Unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_adjusted_addr_incr_ireg2     : Unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_start_addr_offset_slice_im0  : Unsigned(DBEAT_RESIDUE_WIDTH-1 downto 0) := (others => '0');
    signal sig_mbaa_addr_cntr_slice_im0     : Unsigned(MBAA_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_aligned_im0             : std_logic := '0';
    signal sig_addr_aligned_ireg1           : std_logic := '0';

    -- S2MM DRE Support
    signal sig_cmd2dre_valid                : std_logic := '0';
    signal sig_clr_cmd2dre_valid            : std_logic := '0';
    signal sig_input_xfer_btt_im0           : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    signal sig_xfer_btt_reg                 : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    signal sig_xfer_dre_eof_reg             : std_logic := '0';

    -- Long Timing path breakup intermediate registers
    signal sig_strbgen_addr_ireg2           : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
    signal sig_strbgen_bytes_ireg2          : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH   downto 0) := (others => '0');
    signal sig_finish_addr_offset_ireg2     : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
    signal sig_last_addr_offset_im2         : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
    
    signal sig_xfer_strt_strb_ireg3         : std_logic_vector(BYTES_PER_DBEAT-1 downto 0) := (others => '0');
    signal sig_xfer_end_strb_ireg3          : std_logic_vector(BYTES_PER_DBEAT-1 downto 0) := (others => '0');
    signal sig_xfer_len_eq_0_ireg3          : std_logic := '0';

    signal sig_addr_cntr_incr_ireg2         : Unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_predict_addr_lsh_im3_slv     : std_logic_vector(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_predict_addr_lsh_im2         : unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_predict_addr_lsh_ireg3       : unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_lsh_rollover_im3        : std_logic := '0';
    
    signal sig_mmap_reset_reg               : std_logic := '0';
    
    ----------------------------------------------------------
 




  begin --(architecture implementation)



     -- Assign calculation error output
     calc_error               <= sig_calc_error_reg;

     -- Assign the ready output to the Command FIFO
     mst2cmd_cmd_ready        <= not(sig_sm_halt_reg) and
                                 sig_input_reg_empty  and
                                 not(sig_calc_error_pushed);

     -- Assign the Address Channel Controller Qualifiers
     mstr2addr_tag            <=  sig_xfer_tag_reg ;
     mstr2addr_addr           <=  sig_xfer_addr_reg;
     mstr2addr_len            <=  sig_xfer_len_reg ;
     mstr2addr_size           <=  sig_xfer_size    ;
     mstr2addr_burst          <=  '0' & sig_xfer_type_reg; -- only fixed or increment supported
     mstr2addr_cache          <=  sig_xfer_cache_reg; -- only fixed or increment supported
     mstr2addr_user           <=  sig_xfer_user_reg; -- only fixed or increment supported
     mstr2addr_cmd_valid      <=  sig_cmd2addr_valid;
     mstr2addr_calc_error     <=  sig_xfer_calc_err_reg;
     mstr2addr_cmd_cmplt      <=  sig_xfer_cmd_cmplt_reg;
     

     -- Assign the Data Channel Controller Qualifiers
     mstr2data_tag            <= sig_xfer_tag_reg      ;
     mstr2data_saddr_lsb      <= sig_xfer_addr_reg(C_SEL_ADDR_WIDTH-1 downto 0);
     mstr2data_len            <= sig_xfer_len_reg      ;
     mstr2data_strt_strb      <= sig_xfer_strt_strb_reg;
     mstr2data_last_strb      <= sig_xfer_end_strb_reg ;
     mstr2data_drr            <= sig_xfer_drr_reg      ;
     mstr2data_eof            <= sig_xfer_eof_reg      ;
     mstr2data_sequential     <= sig_xfer_is_seq_reg   ;
     mstr2data_cmd_cmplt      <= sig_xfer_cmd_cmplt_reg;
     mstr2data_cmd_valid      <= sig_cmd2data_valid    ;

     mstr2data_dre_src_align  <= sig_xfer_addr_reg(C_DRE_ALIGN_WIDTH-1 downto 0);  -- Used by MM2S DRE
     mstr2data_dre_dest_align <= sig_xfer_dsa_reg(C_DRE_ALIGN_WIDTH-1 downto 0);   -- Used by MM2S DRE

     mstr2data_calc_error     <= sig_xfer_calc_err_reg ;


     -- Assign the DRE Controller Qualifiers
     mstr2dre_cmd_valid       <= sig_cmd2dre_valid     ;                            -- Used by DRE
     mstr2dre_tag             <= sig_xfer_tag_reg      ;                            -- Used by DRE
     mstr2dre_btt             <= sig_xfer_btt_reg      ;                            -- Used by DRE
     mstr2dre_drr             <= sig_xfer_drr_reg      ;                            -- Used by DRE
     mstr2dre_eof             <= sig_xfer_dre_eof_reg  ;                            -- Used by DRE
     mstr2dre_cmd_cmplt       <= sig_xfer_cmd_cmplt_reg;                            -- Used by DRE
     mstr2dre_calc_error      <= sig_xfer_calc_err_reg ;                            -- Used by DRE


     
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: DO_MM2S_CASE
     --
     -- If Generate Description:
     --  Assigns the auxillary DRE Control Source and Destination
     -- ports for the MM2S use case.
     --
     ------------------------------------------------------------
     DO_MM2S_CASE : if (C_IS_MM2S = 1) generate
    
       begin
      
         mstr2dre_dre_src_align   <= sig_xfer_addr_reg(C_DRE_ALIGN_WIDTH-1 downto 0);   -- Used by DRE
         mstr2dre_dre_dest_align  <= sig_xfer_dsa_reg(C_DRE_ALIGN_WIDTH-1 downto 0) ;   -- Used by DRE
      
       end generate DO_MM2S_CASE;



     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: DO_S2MM_CASE
     --
     -- If Generate Description:
     --  Assigns the auxillary DRE Control Source and Destination
     -- ports for the S2MM use case.
     --
     ------------------------------------------------------------
     DO_S2MM_CASE : if (C_IS_MM2S = 0) generate
    
       begin
      
         mstr2dre_dre_src_align   <= sig_xfer_dsa_reg(C_DRE_ALIGN_WIDTH-1 downto 0) ;   -- Used by DRE
         mstr2dre_dre_dest_align  <= sig_xfer_addr_reg(C_DRE_ALIGN_WIDTH-1 downto 0);   -- Used by DRE
      
       end generate DO_S2MM_CASE;







     -- Store and Forward Support Start Offset (used by Packer/Unpacker logic)
     mstr2dre_strt_offset     <= sig_xfer_addr_reg(SF_OFFSET_MS_INDEX downto SF_OFFSET_LS_INDEX);  
     
     

   -- Start internal logic.

--    sig_cmd_type_slice        <=  '1';  -- always incrementing (per Interface_X guidelines)
    sig_cmd_user_slice       <= cache2mstr_command(7 downto 4);
    sig_cmd_cache_slice        <= cache2mstr_command(3 downto 0);
 
    sig_cmd_type_slice        <=  cmd2mstr_command(CMD_TYPE_INDEX);
    sig_cmd_addr_slice        <=  cmd2mstr_command(CMD_ADDR_MS_INDEX downto CMD_ADDR_LS_INDEX);
    sig_cmd_tag_slice         <=  cmd2mstr_command(CMD_TAG_MS_INDEX downto CMD_TAG_LS_INDEX);
    sig_cmd_btt_slice         <=  cmd2mstr_command(CMD_BTT_MS_INDEX downto CMD_BTT_LS_INDEX);

    sig_cmd_dsa_slice         <=  cmd2mstr_command(CMD_DSA_MS_INDEX downto CMD_DSA_LS_INDEX);
    sig_cmd_drr_slice         <=  cmd2mstr_command(CMD_DRR_INDEX);
    sig_cmd_eof_slice         <=  cmd2mstr_command(CMD_EOF_INDEX);



    -- Check for a zero length BTT (error condition)
    sig_btt_is_zero  <= '1'
      when  (sig_cmd_btt_slice = BTT_ZEROS)
      Else '0';

    sig_xfer_size <= SIZE_TO_USE;



    -----------------------------------------------------------------
    -- Reset fanout control
    -----------------------------------------------------------------
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_RESET_REG
    --
    -- Process Description:
    --  Registers the input reset to reduce fanout. This module
    --  has a high number of register bits to reset.
    --
    -------------------------------------------------------------
    IMP_RESET_REG : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
        
          sig_mmap_reset_reg <= mmap_reset;
        
        end if;       
      end process IMP_RESET_REG; 

    
    
    
    
    
    
    

    -----------------------------------------------------------------
    -- Input xfer register design

    sig_push_input_reg  <=  not(sig_sm_halt_reg) and
                            cmd2mstr_cmd_valid   and
                            sig_input_reg_empty  and
                            not(sig_calc_error_reg);

    sig_pop_input_reg   <= sig_sm_pop_input_reg;



    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_INPUT_QUAL
    --
    -- Process Description:
    --  Implements the input command qualifier holding register
    --
    -------------------------------------------------------------
    REG_INPUT_QUAL : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_mmap_reset_reg    = '1' or
                sig_pop_input_reg     = '1' or
                sig_calc_error_pushed = '1') then

              sig_input_cache_type_reg          <=  (others => '0');
              sig_input_user_type_reg           <=  (others => '0');
              sig_input_burst_type_reg          <=  '0';
              sig_input_tag_reg                 <=  (others => '0');
              sig_input_dsa_reg                 <=  (others => '0');
              sig_input_drr_reg                 <=  '0';
              sig_input_eof_reg                 <=  '0';

              sig_input_reg_empty               <=  '1';
              sig_input_reg_full                <=  '0';

            elsif (sig_push_input_reg = '1') then
              sig_input_cache_type_reg          <=  sig_cmd_cache_slice;
              sig_input_user_type_reg           <=  sig_cmd_user_slice;
              sig_input_burst_type_reg          <=  sig_cmd_type_slice;
              sig_input_tag_reg                 <=  sig_cmd_tag_slice;
              sig_input_dsa_reg                 <=  sig_cmd_dsa_slice;
              sig_input_drr_reg                 <=  sig_cmd_drr_slice;
              sig_input_eof_reg                 <=  sig_cmd_eof_slice;

              sig_input_reg_empty               <=  '0';
              sig_input_reg_full                <=  '1';

            else
              null; -- Hold current State
            end if;
         end if;
       end process REG_INPUT_QUAL;





  ----------------------------------------------------------------------
  -- Calculation Error Logic


  -------------------------------------------------------------
  -- Synchronous Process with Sync Reset
  --
  -- Label: IMP_CALC_ERROR_FLOP
  --
  -- Process Description:
  --   Implements the flop for the Calc Error flag, Once set,
  -- the flag cannot be cleared until a reset is issued.
  --
  -------------------------------------------------------------
  IMP_CALC_ERROR_FLOP : process (primary_aclk)
     begin
       if (primary_aclk'event and primary_aclk = '1') then
          if (sig_mmap_reset_reg = '1') then
            sig_calc_error_reg <= '0';
          elsif (sig_push_input_reg = '1' and
                 sig_calc_error_reg = '0') then
            sig_calc_error_reg <= sig_btt_is_zero;
          else
            Null;  -- hold the current state
          end if;
       end if;
     end process IMP_CALC_ERROR_FLOP;



  -------------------------------------------------------------
  -- Synchronous Process with Sync Reset
  --
  -- Label: IMP_CALC_ERROR_PUSHED
  --
  -- Process Description:
  --   Implements the flop for generating a flag indicating the
  -- calculation error flag has been pushed to the addr and data
  -- controllers.
  --
  -------------------------------------------------------------
  IMP_CALC_ERROR_PUSHED : process (primary_aclk)
     begin
       if (primary_aclk'event and primary_aclk = '1') then
          if (sig_mmap_reset_reg = '1') then
            sig_calc_error_pushed <= '0';
          elsif (sig_push_xfer_reg = '1' and
                 sig_calc_error_pushed = '0') then
            sig_calc_error_pushed <= sig_calc_error_reg;
          else
            Null;  -- hold the current state
          end if;
       end if;
     end process IMP_CALC_ERROR_PUSHED;









  ---------------------------------------------------------------------
  -- Strobe Generator Logic

   sig_xfer_strt_strb2use_im3 <=  sig_xfer_strt_strb_ireg3
     When (sig_first_xfer_im0 = '1')
     Else (others => '1');

   sig_xfer_end_strb2use_im3 <= sig_xfer_strt_strb2use_im3
     When (sig_xfer_len_eq_0_ireg3 = '1' and
           sig_first_xfer_im0      = '1')
     else sig_xfer_end_strb_ireg3
     When (sig_last_xfer_valid_im1 = '1')
     Else (others => '1');


   
   
   
    ----------------------------------------------------------
    -- Intermediate registers for STBGEN Fmax path
   
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_IM_STBGEN_REGS
    --
    -- Process Description:
    --  Intermediate registers for Strobegen inputs to break
    -- long timing paths.
    --
    -------------------------------------------------------------
    IMP_IM_STBGEN_REGS : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (sig_mmap_reset_reg = '1') then

             sig_strbgen_addr_ireg2       <= (others => '0');
             sig_strbgen_bytes_ireg2      <= (others => '0');
             sig_finish_addr_offset_ireg2 <= (others => '0');
    
           elsif (sig_sm_ld_calc2_reg = '1') then

             sig_strbgen_addr_ireg2       <= sig_strbgen_addr_im0      ;
             sig_strbgen_bytes_ireg2      <= sig_strbgen_bytes_im1     ;
             sig_finish_addr_offset_ireg2 <= sig_finish_addr_offset_im1;
    
           else
    
             null;  -- hold state
    
           end if; 
        end if;       
      end process IMP_IM_STBGEN_REGS; 
   
   
   
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_IM_STBGEN_OUT_REGS
    --
    -- Process Description:
    --  Intermediate registers for Strobegen outputs to break
    -- long timing paths.
    --
    -------------------------------------------------------------
    IMP_IM_STBGEN_OUT_REGS : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (sig_mmap_reset_reg = '1') then

             sig_xfer_strt_strb_ireg3     <= (others => '0');
             sig_xfer_end_strb_ireg3      <= (others => '0');
             sig_xfer_len_eq_0_ireg3      <= '0';
    
           elsif (sig_sm_ld_calc3_reg = '1') then

             sig_xfer_strt_strb_ireg3     <= sig_xfer_strt_strb_im2;
             sig_xfer_end_strb_ireg3      <= sig_xfer_end_strb_im2 ;
             sig_xfer_len_eq_0_ireg3      <= sig_xfer_len_eq_0_im2 ;
    
           else
    
             null;  -- hold state
    
           end if; 
        end if;       
      end process IMP_IM_STBGEN_OUT_REGS; 
   
   
   
   
   
   
   
   
    ------------------------------------------------------------
    -- Instance: I_STRT_STRB_GEN 
    --
    -- Description:
    --  Strobe generator instance. Generates strobe bits for
    -- a designated starting byte lane and the number of bytes
    -- to be transfered (for that data beat).    
    --
    ------------------------------------------------------------
    I_STRT_STRB_GEN : entity axi_datamover_v5_1_14.axi_datamover_strb_gen2
    generic map (
                          
      C_OP_MODE            =>  0                         , -- 0 = Offset/Length mode
      C_STRB_WIDTH         =>  BYTES_PER_DBEAT           ,   
      C_OFFSET_WIDTH       =>  STRBGEN_ADDR_SLICE_WIDTH  ,   
      C_NUM_BYTES_WIDTH    =>  STRBGEN_ADDR_SLICE_WIDTH+1           
  
      )
    port map (
      
      start_addr_offset    =>  sig_strbgen_addr_ireg2  , 
      end_addr_offset      =>  STRBGEN_ADDR_0          , -- not used in op mode 0
      num_valid_bytes      =>  sig_strbgen_bytes_ireg2 , 
      strb_out             =>  sig_xfer_strt_strb_im2   
  
      );
                               
   

    
    
    -- The ending address offset is 1 less than the calculated 
    -- starting address for the next sequential transfer.
    sig_last_addr_offset_im2 <=  STD_LOGIC_VECTOR(UNSIGNED(sig_finish_addr_offset_ireg2) - 
                                 STRBGEN_ADDR_SLICE_1);
   

    ------------------------------------------------------------
    -- Instance: I_END_STRB_GEN 
    --
    -- Description:
    --  End Strobe generator instance. Generates asserted strobe 
    -- bits from byte offset 0 to the ending byte offset.    
    --
    ------------------------------------------------------------
    I_END_STRB_GEN : entity axi_datamover_v5_1_14.axi_datamover_strb_gen2
    generic map (
                          
      C_OP_MODE            =>  1                        , -- 0 = Offset/Length mode
      C_STRB_WIDTH         =>  BYTES_PER_DBEAT          ,   
      C_OFFSET_WIDTH       =>  STRBGEN_ADDR_SLICE_WIDTH ,   
      C_NUM_BYTES_WIDTH    =>  STRBGEN_ADDR_SLICE_WIDTH          
  
      )
    port map (
      
      start_addr_offset    =>  STRBGEN_ADDR_0               , 
      end_addr_offset      =>  sig_last_addr_offset_im2     , 
      num_valid_bytes      =>  STRBGEN_ADDR_0               , -- not used in op mode 1
      strb_out             =>  sig_xfer_end_strb_im2   
  
      );
                               
  


   


    -----------------------------------------------------------------
    -- Output xfer register design

    sig_push_xfer_reg <= (sig_ld_xfer_reg and sig_xfer_reg_empty);

                         -- Data taking xfer after Addr and DRE
    sig_pop_xfer_reg  <= (sig_clr_cmd2data_valid and not(sig_cmd2addr_valid) and not(sig_cmd2dre_valid))  or
                         -- Addr taking xfer after Data and DRE
                         (sig_clr_cmd2addr_valid and not(sig_cmd2data_valid) and not(sig_cmd2dre_valid))  or 
                         -- DRE taking xfer after Data and ADDR
                         (sig_clr_cmd2dre_valid  and not(sig_cmd2data_valid) and not(sig_cmd2addr_valid)) or  
                         
                         -- data and Addr taking xfer after DRE
                         (sig_clr_cmd2data_valid and sig_clr_cmd2addr_valid and not(sig_cmd2dre_valid))   or 
                         -- Addr and DRE taking xfer after Data
                         (sig_clr_cmd2addr_valid and sig_clr_cmd2dre_valid  and not(sig_cmd2data_valid))  or 
                         -- Data and DRE taking xfer after Addr
                         (sig_clr_cmd2data_valid and sig_clr_cmd2dre_valid  and not(sig_cmd2addr_valid))  or  
                         
                         -- Addr, Data,  and DRE all taking xfer
                         (sig_clr_cmd2data_valid and sig_clr_cmd2addr_valid and sig_clr_cmd2dre_valid);       



    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_OUTPUT_QUAL
    --
    -- Process Description:
    --  Implements the output xfer qualifier holding register
    --
    -------------------------------------------------------------
    REG_OUTPUT_QUAL : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_mmap_reset_reg = '1' or
               (sig_pop_xfer_reg   = '1' and
                sig_push_xfer_reg  = '0')) then

--              sig_xfer_cache_reg      <=  (others => '0');
--              sig_xfer_user_reg       <=  (others => '0');
--              sig_xfer_addr_reg       <=  (others => '0');
--              sig_xfer_type_reg       <=  '0';
 --             sig_xfer_len_reg        <=  (others => '0');
 --             sig_xfer_tag_reg        <=  (others => '0');
 --             sig_xfer_dsa_reg        <=  (others => '0');
 --             sig_xfer_drr_reg        <=  '0';
 --             sig_xfer_eof_reg        <=  '0';
 --             sig_xfer_strt_strb_reg  <=  (others => '0');
 --             sig_xfer_end_strb_reg   <=  (others => '0');
 --             sig_xfer_is_seq_reg     <=  '0';
 --             sig_xfer_cmd_cmplt_reg  <=  '0';
 --             sig_xfer_calc_err_reg   <=  '0';
 --             sig_xfer_btt_reg        <=  (others => '0');
 --             sig_xfer_dre_eof_reg    <=  '0';

              sig_xfer_reg_empty      <=  '1';
              sig_xfer_reg_full       <=  '0';

            elsif (sig_push_xfer_reg = '1') then
 --             if (sig_input_burst_type_reg = '0') then
 --                 sig_xfer_addr_reg       <=  sig_addr_cntr_lsh_kh; 
 --             else
 --                 sig_xfer_addr_reg       <=  sig_xfer_address_im0         ;
 --             end if;
 --             sig_xfer_type_reg       <=  sig_input_burst_type_reg     ;
 --             sig_xfer_cache_reg      <=  sig_input_cache_type_reg     ;
 --             sig_xfer_user_reg       <=  sig_input_user_type_reg     ;
 --             sig_xfer_len_reg        <=  sig_xfer_len_im2             ;
 --             sig_xfer_tag_reg        <=  sig_input_tag_reg            ;
 --             sig_xfer_dsa_reg        <=  sig_input_dsa_reg            ;
 --             sig_xfer_drr_reg        <=  sig_input_drr_reg and
 --                                         sig_first_xfer_im0           ;
 --             sig_xfer_eof_reg        <=  sig_input_eof_reg and
 --                                         sig_last_xfer_valid_im1      ;
 --             sig_xfer_strt_strb_reg  <=  sig_xfer_strt_strb2use_im3   ;
 --             sig_xfer_end_strb_reg   <=  sig_xfer_end_strb2use_im3    ;
 --             sig_xfer_is_seq_reg     <=  not(sig_last_xfer_valid_im1) ;
 --             sig_xfer_cmd_cmplt_reg  <=  sig_last_xfer_valid_im1 or
 --                                         sig_calc_error_reg           ;
 --             sig_xfer_calc_err_reg   <=  sig_calc_error_reg           ;
 --             sig_xfer_btt_reg        <=  sig_input_xfer_btt_im0       ;
 --             sig_xfer_dre_eof_reg    <=  sig_input_eof_reg            ;

              sig_xfer_reg_empty      <=  '0';
              sig_xfer_reg_full       <=  '1';

            else
              null; -- Hold current State
            end if;
         end if;
       end process REG_OUTPUT_QUAL;


 --             if (sig_input_burst_type_reg = '0') then
 --                 sig_xfer_addr_reg       <=  sig_addr_cntr_lsh_kh; 
 --             else
                  sig_xfer_addr_reg       <=  sig_xfer_address_im0 when (sig_input_burst_type_reg = '1') else
                                              sig_addr_cntr_lsh_kh      ;
 --             end if;
              sig_xfer_type_reg       <=  sig_input_burst_type_reg     ;
              sig_xfer_cache_reg      <=  sig_input_cache_type_reg     ;
              sig_xfer_user_reg       <=  sig_input_user_type_reg     ;
              sig_xfer_len_reg        <=  sig_xfer_len_im2             ;
              sig_xfer_tag_reg        <=  sig_input_tag_reg            ;
              sig_xfer_dsa_reg        <=  sig_input_dsa_reg            ;
              sig_xfer_drr_reg        <=  sig_input_drr_reg and
                                          sig_first_xfer_im0           ;
              sig_xfer_eof_reg        <=  sig_input_eof_reg and
                                          sig_last_xfer_valid_im1      ;
              sig_xfer_strt_strb_reg  <=  sig_xfer_strt_strb2use_im3   ;
              sig_xfer_end_strb_reg   <=  sig_xfer_end_strb2use_im3    ;
              sig_xfer_is_seq_reg     <=  not(sig_last_xfer_valid_im1) ;
              sig_xfer_cmd_cmplt_reg  <=  sig_last_xfer_valid_im1 or
                                          sig_calc_error_reg           ;
              sig_xfer_calc_err_reg   <=  sig_calc_error_reg           ;
              sig_xfer_btt_reg        <=  sig_input_xfer_btt_im0       ;
              sig_xfer_dre_eof_reg    <=  sig_input_eof_reg            ;



    --------------------------------------------------------------
    -- BTT Counter Logic


    sig_ld_btt_cntr   <= sig_ld_addr_cntr;

--    sig_decr_btt_cntr <= sig_incr_addr_cntr;
--    above signal is using the incr_addr_cntr signal and hence cannot be 
--    used if burst type is Fixed

    sig_decr_btt_cntr <= sig_incr_addr_cntr; --sig_push_xfer_reg;


    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_BTT_CNTR
    --
    -- Process Description:
    -- Bytes to transfer counter implementation.
    --
    -------------------------------------------------------------
    IMP_BTT_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_mmap_reset_reg = '1') then

              sig_btt_cntr_im0 <= (others => '0');

            elsif (sig_ld_btt_cntr = '1') then

              sig_btt_cntr_im0 <= UNSIGNED(sig_cmd_btt_slice);

            Elsif (sig_decr_btt_cntr = '1') Then

              sig_btt_cntr_im0 <= sig_btt_cntr_im0-RESIZE(sig_addr_cntr_incr_ireg2, CMD_BTT_WIDTH);

            else
              null;  -- hold current state
            end if;
         end if;
       end process IMP_BTT_CNTR;



    -- Convert to logic vector for the S2MM DRE use
    -- The DRE will only use this value prior to the first
    -- decrement of the BTT Counter. Using this saves a separate
    -- BTT register.
    sig_input_xfer_btt_im0 <= STD_LOGIC_VECTOR(sig_btt_cntr_im0);


    -- Rip the Burst Count slice from BTT counter value
    sig_burst_cnt_slice_im0 <= sig_btt_cntr_im0(CMD_BTT_WIDTH-1 downto BURST_CNT_LS_INDEX);



    sig_brst_cnt_eq_zero_im0 <= '1'
      When (sig_burst_cnt_slice_im0 = BRST_CNT_0)
      Else '0';

    sig_brst_cnt_eq_one_im0 <= '1'
      When (sig_burst_cnt_slice_im0 = BRST_CNT_1)
      Else '0';


    -- Rip the BTT residue field from the BTT counter value
    sig_btt_residue_slice_im0   <=  sig_btt_cntr_im0(BTT_RESIDUE_WIDTH-1 downto 0);



    -- Check for transfer length residue of zero prior to subtracting 1
    sig_no_btt_residue_im0 <= '1'
       when (sig_btt_residue_slice_im0 = BTT_RESIDUE_0)
       Else '0';


    -- Unaligned address compensation
    -- Add the number of starting address offset byte positions to the
    -- final byte change value needed to calculate the AXI LEN field

    sig_start_addr_offset_slice_im0 <=  sig_addr_cntr_lsh_im0(DBEAT_RESIDUE_WIDTH-1 downto 0);

    sig_adjusted_addr_incr_im1      <=  sig_addr_cntr_incr_im1 +
                                        RESIZE(sig_start_addr_offset_slice_im0, ADDR_CNTR_WIDTH);







    -- adjust the address increment down by 1 byte to compensate
    -- for the LEN requirement of being N-1 data beats

    sig_byte_change_minus1_im2 <=  sig_adjusted_addr_incr_ireg2-ADDR_CNTR_ONE;



    -- Rip the new transfer length value
    sig_xfer_len_im2 <=  STD_LOGIC_VECTOR(
                         RESIZE(
                            sig_byte_change_minus1_im2(BTT_RESIDUE_WIDTH-1 downto
                                                       DBEAT_RESIDUE_WIDTH),
                         LEN_WIDTH)
                     );


    -- Check to see if the new xfer length is zero (1 data beat)
    sig_xfer_len_eq_0_im2 <= '1'
      when (sig_xfer_len_im2 = XFER_LEN_ZERO)
      Else '0';



    -- Check for Last transfer condition
    --sig_last_xfer_valid_im1  <=  (sig_brst_cnt_eq_one_im0 and
    sig_last_xfer_valid_im1  <=  (sig_brst_cnt_eq_one_ireg1 and
                                  --sig_no_btt_residue_im0  and
                                  sig_no_btt_residue_ireg1  and
                                  -- sig_addr_aligned_im0)   or  -- always the last databeat case
                                  sig_addr_aligned_ireg1)   or  -- always the last databeat case

                                 -- ((sig_btt_lt_b2mbaa_im0 or sig_btt_eq_b2mbaa_im0) and  -- less than a full burst remaining
                                 ((sig_btt_lt_b2mbaa_ireg1 or sig_btt_eq_b2mbaa_ireg1) and  -- less than a full burst remaining
                                 -- (sig_brst_cnt_eq_zero_im0 and not(sig_no_btt_residue_im0)));
                                 (sig_brst_cnt_eq_zero_ireg1 and not(sig_no_btt_residue_ireg1)));



 
    
    
    
    
    
    
    
    
    ----------------------------------------------------------------------------------------------------
    ----------------------------------------------------------------------------------------------------
    --
    -- General Address Counter Logic (applies to any address width of 32 or greater
    -- The address counter is divided into 2 16-bit segements for 32-bit address support. As the
    -- address gets wider, up to 2 more segements will be added via IfGens to provide for 64-bit 
    -- addressing.
    --
    ----------------------------------------------------------------------------------------------------
    


    -- Rip the LS bits of the LS Address Counter for the StrobeGen
    -- starting address offset
    sig_strbgen_addr_im0  <=  STD_LOGIC_VECTOR(sig_addr_cntr_lsh_im0(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0));



    -- Check if the calcualted address increment (in bytes) is greater than the
    -- number of bytes that can be transfered per data beat
    sig_addr_incr_ge_bpdb_im1 <= '1'
      When (sig_addr_cntr_incr_im1 >= TO_UNSIGNED(BYTES_PER_DBEAT, ADDR_CNTR_WIDTH))
      Else '0';


    -- If the calculated address increment (in bytes) is greater than the
    -- number of bytes that can be transfered per data beat, then clip the
    -- strobegen byte value to the number of bytes per data beat, else use the
    -- increment value.
    sig_strbgen_bytes_im1 <=  STD_LOGIC_VECTOR(TO_UNSIGNED(BYTES_PER_DBEAT, STRBGEN_ADDR_SLICE_WIDTH+1))
      when (sig_addr_incr_ge_bpdb_im1 = '1')
      else STD_LOGIC_VECTOR(sig_addr_cntr_incr_im1(STRBGEN_ADDR_SLICE_WIDTH downto 0));




    --------------------------------------------------------------------------
    -- Address Counter logic

    sig_ld_addr_cntr   <= sig_push_input_reg;

    -- don't increment address cntr if type is '0' (non-incrementing)
    sig_incr_addr_cntr <= sig_pop_xfer_reg;-- and
                     --     sig_input_burst_type_reg;



    sig_mbaa_addr_cntr_slice_im0 <= sig_addr_cntr_lsh_im0(MBAA_ADDR_SLICE_WIDTH-1 downto 0);




    sig_bytes_to_mbaa_im0 <=  TO_UNSIGNED(BYTES_PER_MAX_BURST, ADDR_CNTR_WIDTH) -
                              RESIZE(sig_mbaa_addr_cntr_slice_im0,ADDR_CNTR_WIDTH);



    sig_addr_aligned_im0 <= '1'
      when (sig_mbaa_addr_cntr_slice_im0 = BTT_RESIDUE_0)
      Else '0';


    -- Check to see if the jump to the Max Burst Aligned Address (mbaa) is less
    -- than or equal to the remaining bytes to transfer. If it is, then at least
    -- two tranfers have to be scheduled.
    sig_btt_lt_b2mbaa_im0 <= '1'
      when ((RESIZE(sig_btt_residue_slice_im0, ADDR_CNTR_WIDTH) < sig_bytes_to_mbaa_im0) and
            (sig_brst_cnt_eq_zero_im0 = '1'))

      Else '0';


    sig_btt_eq_b2mbaa_im0 <= '1'
      when ((RESIZE(sig_btt_residue_slice_im0, ADDR_CNTR_WIDTH) = sig_bytes_to_mbaa_im0) and
             (sig_brst_cnt_eq_zero_im0 = '1'))
      Else '0';





    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_IM_REG1
    --
    -- Process Description:
    --  Intermediate register stage 1 for Address Counter 
    -- derivative calculations.
    --
    -------------------------------------------------------------
    IMP_IM_REG1 : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (sig_mmap_reset_reg = '1') then

             sig_bytes_to_mbaa_ireg1      <= (others => '0');
             sig_addr_aligned_ireg1       <= '0'            ;
             sig_btt_lt_b2mbaa_ireg1      <= '0'            ;
             sig_btt_eq_b2mbaa_ireg1      <= '0'            ;
             
             
             sig_brst_cnt_eq_zero_ireg1   <= '0'            ;
             sig_brst_cnt_eq_one_ireg1    <= '0'            ;
             sig_no_btt_residue_ireg1     <= '0'            ;
             
             
           elsif (sig_sm_ld_calc1_reg = '1') then

             sig_bytes_to_mbaa_ireg1      <= sig_bytes_to_mbaa_im0   ;
             sig_addr_aligned_ireg1       <= sig_addr_aligned_im0    ;
             sig_btt_lt_b2mbaa_ireg1      <= sig_btt_lt_b2mbaa_im0   ;
             sig_btt_eq_b2mbaa_ireg1      <= sig_btt_eq_b2mbaa_im0   ;
    
             sig_brst_cnt_eq_zero_ireg1   <= sig_brst_cnt_eq_zero_im0;
             sig_brst_cnt_eq_one_ireg1    <= sig_brst_cnt_eq_one_im0 ;
             sig_no_btt_residue_ireg1     <= sig_no_btt_residue_im0  ;
             
           else
    
             null;  -- hold state
    
           end if; 
        end if;       
      end process IMP_IM_REG1; 
   
   
   


    -- Select the address counter increment value to use
    sig_addr_cntr_incr_im1 <= RESIZE(sig_btt_residue_slice_im0, ADDR_CNTR_WIDTH)
      --When (sig_btt_lt_b2mbaa_im0 = '1')
      When (sig_btt_lt_b2mbaa_ireg1 = '1')
      --else sig_bytes_to_mbaa_im0
      else  sig_bytes_to_mbaa_ireg1
      when (sig_first_xfer_im0 = '1')
      else TO_UNSIGNED(BYTES_PER_MAX_BURST, ADDR_CNTR_WIDTH);



    -- calculate the next starting address after the current
    -- xfer completes
    sig_predict_addr_lsh_im1    <=  sig_addr_cntr_lsh_im0 + sig_addr_cntr_incr_im1;


    -- Predict next transfer's address offset for the Strobe Generator
    sig_finish_addr_offset_im1  <= STD_LOGIC_VECTOR(sig_predict_addr_lsh_im1(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0));


    sig_addr_cntr_lsh_im0_slv   <= STD_LOGIC_VECTOR(sig_addr_cntr_lsh_im0);



    -- Determine if an address count lsh rollover is going to occur when
    -- jumping to the next starting address by comparing the MS bit of the
    -- current address lsh to the MS bit of the predicted address lsh .
    -- A transition of a '1' to a '0' is a rollover.
    sig_addr_lsh_rollover_im3 <= '1'
      when (
            (sig_addr_cntr_lsh_im0_slv(ADDR_CNTR_WIDTH-1)    = '1') and
            (sig_predict_addr_lsh_im3_slv(ADDR_CNTR_WIDTH-1) = '0')
           )
      Else '0';




       
     
          
          
    ----------------------------------------------------------
    -- Intermediate registers for reducing the Address Counter 
    -- Increment timing path
    ----------------------------------------------------------
   

    -- calculate the next starting address after the current
    -- xfer completes using intermediate register values
    sig_predict_addr_lsh_im2      <= sig_addr_cntr_lsh_im0 + sig_addr_cntr_incr_ireg2;
    
    sig_predict_addr_lsh_im3_slv  <= STD_LOGIC_VECTOR(sig_predict_addr_lsh_ireg3);
   
   
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_IM_ADDRINC_REG
    --
    -- Process Description:
    --  Intermediate registers for address counter increment to 
    -- break long timing paths.
    --
    -------------------------------------------------------------
    IMP_IM_ADDRINC_REG : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (sig_mmap_reset_reg = '1') then

             sig_addr_cntr_incr_ireg2   <= (others => '0');
    
           elsif (sig_sm_ld_calc2_reg = '1') then

             sig_addr_cntr_incr_ireg2   <= sig_addr_cntr_incr_im1;
    
           else
    
             null;  -- hold state
    
           end if; 
        end if;       
      end process IMP_IM_ADDRINC_REG; 
   
   
   
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_IM_PREDICT_ADDR_REG
    --
    -- Process Description:
    --  Intermediate register for predicted address to break up
    -- long timing paths.
    --
    -------------------------------------------------------------
    IMP_IM_PREDICT_ADDR_REG : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (sig_mmap_reset_reg = '1') then

             sig_predict_addr_lsh_ireg3     <= (others => '0');
    
           elsif (sig_sm_ld_calc3_reg = '1') then

             sig_predict_addr_lsh_ireg3     <= sig_predict_addr_lsh_im2;
    
           else
    
             null;  -- hold state
    
           end if; 
        end if;       
      end process IMP_IM_PREDICT_ADDR_REG; 
   
   
   
     
     
     

    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_ADDR_STUFF
    --
    -- Process Description:
    --  Implements a general register for address counter related
    -- things.
    --
    -------------------------------------------------------------
    REG_ADDR_STUFF : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_mmap_reset_reg = '1') then

              sig_adjusted_addr_incr_ireg2 <= (others => '0');

           elsif (sig_sm_ld_calc2_reg = '1') then
              
              sig_adjusted_addr_incr_ireg2 <= sig_adjusted_addr_incr_im1;

            else
    
             null;  -- hold state
    
            end if;
         end if;
       end process REG_ADDR_STUFF;




    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_LSH_ADDR_CNTR
    --
    -- Process Description:
    -- Least Significant Half Address counter implementation.
    --
    -------------------------------------------------------------
    IMP_LSH_ADDR_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_mmap_reset_reg = '1') then

              sig_addr_cntr_lsh_im0 <= (others => '0');
              sig_addr_cntr_lsh_kh <= (others => '0');

            elsif (sig_ld_addr_cntr = '1') then

              sig_addr_cntr_lsh_im0 <= UNSIGNED(sig_cmd_addr_slice(ADDR_CNTR_WIDTH-1 downto 0));
              sig_addr_cntr_lsh_kh <= sig_cmd_addr_slice;

            Elsif (sig_incr_addr_cntr = '1') then -- and sig_input_burst_type_reg = '1') Then

              sig_addr_cntr_lsh_im0 <= sig_predict_addr_lsh_ireg3;

            else
              null;  -- hold current state
            end if;
         end if;
       end process IMP_LSH_ADDR_CNTR;




    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_MSH_ADDR_CNTR
    --
    -- Process Description:
    -- Least Significant Half Address counter implementation.
    --
    -------------------------------------------------------------
    IMP_MSH_ADDR_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_mmap_reset_reg = '1') then

              sig_addr_cntr_im0_msh <= (others => '0');

            elsif (sig_ld_addr_cntr = '1') then

              sig_addr_cntr_im0_msh <= UNSIGNED(sig_cmd_addr_slice((2*ADDR_CNTR_WIDTH)-1 downto ADDR_CNTR_WIDTH));

            Elsif (sig_incr_addr_cntr        = '1' and 
                   sig_addr_lsh_rollover_im3 = '1') then

              sig_addr_cntr_im0_msh <= sig_addr_cntr_im0_msh+ADDR_CNTR_ONE;

            else
              null;  -- hold current state
            end if;
         end if;
       end process IMP_MSH_ADDR_CNTR;





   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_FIRST_XFER_FLOP
   --
   -- Process Description:
   --  Implements the register flop for the first transfer flag.
   --
   -------------------------------------------------------------
   IMP_FIRST_XFER_FLOP : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (sig_mmap_reset_reg  = '1' or
               sig_incr_addr_cntr  = '1') then

             sig_first_xfer_im0 <= '0';

           elsif (sig_ld_addr_cntr = '1') then

             sig_first_xfer_im0 <= '1';

           else
             null;  -- hold current state
           end if;
        end if;
      end process IMP_FIRST_XFER_FLOP;




 
 
 
 
 
 
 
 
 
 
 
 
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ADDR_32
    --
    -- If Generate Description:
    -- Implements the Address segment merge logic for the 32-bit
    -- address width case. The address counter segments are split 
    -- into two 16-bit sections to improve Fmax convergence.
    --
    --
    ------------------------------------------------------------
    GEN_ADDR_32 : if (C_ADDR_WIDTH = 32) generate


       begin


         -- Populate the transfer address value by concatonating the
         -- address counter segments
         sig_xfer_address_im0 <= STD_LOGIC_VECTOR(sig_addr_cntr_im0_msh) &
                                 STD_LOGIC_VECTOR(sig_addr_cntr_lsh_im0);


       end generate GEN_ADDR_32;









    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ADDR_GT_32_LE_48
    --
    -- If Generate Description:
    -- Implements the additional Address Counter logic for the case 
    -- when the address width is greater than 32 bits and less than
    -- or equal to 48 bits. In this case, an additional counter segment
    -- is implemented (segment 3) that is variable width of 1
    -- to 16 bits.
    --
    ------------------------------------------------------------
    GEN_ADDR_GT_32_LE_48 : if (C_ADDR_WIDTH  > 32 and
                               C_ADDR_WIDTH <= 48) generate


      -- Local constants
      Constant ACNTR_SEG3_WIDTH    : integer  := C_ADDR_WIDTH-32;
      Constant ACNTR_SEG3_ONE      : unsigned := TO_UNSIGNED(1, ACNTR_SEG3_WIDTH);
      Constant ACNTR_MSH_MAX       : unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '1');
      
      Constant SEG3_ADDR_RIP_MS_INDEX : integer := C_ADDR_WIDTH-1;
      Constant SEG3_ADDR_RIP_LS_INDEX : integer := 32;
      
      
      -- Local Signals
      signal lsig_seg3_addr_cntr       : unsigned(ACNTR_SEG3_WIDTH-1 downto 0) := (others => '0');
      signal lsig_acntr_msh_eq_max     : std_logic := '0';
      signal lsig_acntr_msh_eq_max_reg : std_logic := '0';
       
       
      begin

      
      
        -- Populate the transfer address value by concatonating the
        -- 3 address counter segments
        sig_xfer_address_im0 <= STD_LOGIC_VECTOR(lsig_seg3_addr_cntr  ) &
                                STD_LOGIC_VECTOR(sig_addr_cntr_im0_msh) &
                                STD_LOGIC_VECTOR(sig_addr_cntr_lsh_im0);


       
        
        -- See if the MSH (Segment 2) of the Adress Counter is at a max value
        lsig_acntr_msh_eq_max <= '1'
          when (sig_addr_cntr_im0_msh = ACNTR_MSH_MAX)
          Else '0';


  
        
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_SEG2_EQ_MAX_REG
        --
        -- Process Description:
        --  Implements a register for the flag indicating the address
        -- counter MSH (Segment 2) is at max value and will rollover
        -- at the next increment interval for the counter. Registering 
        -- this signal and using it for the Seg 3 increment logic only
        -- works because there is always at least a 1 clock time gap 
        -- between the increment causing the segment 2 counter to go to 
        -- max and the next increment operation that can bump segment 3. 
        --
        -------------------------------------------------------------
        IMP_SEG2_EQ_MAX_REG : process (primary_aclk)
          begin
            if (primary_aclk'event and primary_aclk = '1') then
               if (sig_mmap_reset_reg = '1') then
        
                 lsig_acntr_msh_eq_max_reg <= '0';
        
               else
        
                 lsig_acntr_msh_eq_max_reg <= lsig_acntr_msh_eq_max;
        
               end if; 
            end if;       
          end process IMP_SEG2_EQ_MAX_REG; 
  
        
        
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_SEG3_ADDR_CNTR
        --
        -- Process Description:
        -- Segment 3 of the Address counter implementation.
        --
        -------------------------------------------------------------
        IMP_SEG3_ADDR_CNTR : process (primary_aclk)
           begin
             if (primary_aclk'event and primary_aclk = '1') then
                if (sig_mmap_reset_reg = '1') then

                  lsig_seg3_addr_cntr <= (others => '0');

                elsif (sig_ld_addr_cntr = '1') then

                  lsig_seg3_addr_cntr <= UNSIGNED(sig_cmd_addr_slice(SEG3_ADDR_RIP_MS_INDEX downto 
                                                                     SEG3_ADDR_RIP_LS_INDEX));

                Elsif (sig_incr_addr_cntr        = '1' and --sig_input_burst_type_reg = '1' and 
                       sig_addr_lsh_rollover_im3 = '1' and
                       lsig_acntr_msh_eq_max_reg = '1') then

                  lsig_seg3_addr_cntr <= lsig_seg3_addr_cntr+ACNTR_SEG3_ONE;

                else
                  null;  -- hold current state
                end if;
             end if;
           end process IMP_SEG3_ADDR_CNTR;



  
  
      end generate GEN_ADDR_GT_32_LE_48;



     
     
     
     
     
     
     
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ADDR_GT_48
    --
    -- If Generate Description:
    -- Implements the additional Address Counter logic for the case 
    -- when the address width is greater than 48 bits and less than
    -- or equal to 64. In this case, an additional 2 counter segments
    -- are implemented (segment 3 and 4). Segment 3 is a fixed 16-bits
    -- and segment 4 is variable width of 1 to 16 bits.
    --
    ------------------------------------------------------------
    GEN_ADDR_GT_48 : if (C_ADDR_WIDTH  > 48) generate


      -- Local constants
      Constant ACNTR_SEG3_WIDTH    : integer  := ADDR_CNTR_WIDTH;
      Constant ACNTR_SEG3_ONE      : unsigned := TO_UNSIGNED(1, ACNTR_SEG3_WIDTH);
      Constant ACNTR_SEG3_MAX      : unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '1');
      
      Constant ACNTR_MSH_MAX       : unsigned(ADDR_CNTR_WIDTH-1 downto 0) := (others => '1');
      
      
      Constant ACNTR_SEG4_WIDTH    : integer  := C_ADDR_WIDTH-48;
      Constant ACNTR_SEG4_ONE      : unsigned := TO_UNSIGNED(1, ACNTR_SEG4_WIDTH);
      
      
      Constant SEG3_ADDR_RIP_MS_INDEX : integer := 47;
      Constant SEG3_ADDR_RIP_LS_INDEX : integer := 32;
      
      Constant SEG4_ADDR_RIP_MS_INDEX : integer := C_ADDR_WIDTH-1;
      Constant SEG4_ADDR_RIP_LS_INDEX : integer := 48;
      
      
      
      
      -- Local Signals
      signal lsig_seg3_addr_cntr        : unsigned(ACNTR_SEG3_WIDTH-1 downto 0) := (others => '0');
      signal lsig_acntr_msh_eq_max      : std_logic := '0';
      signal lsig_acntr_msh_eq_max_reg  : std_logic := '0';
 
      signal lsig_acntr_seg3_eq_max     : std_logic := '0';
      signal lsig_acntr_seg3_eq_max_reg : std_logic := '0';
      
      
      signal lsig_seg4_addr_cntr        : unsigned(ACNTR_SEG4_WIDTH-1 downto 0) := (others => '0');
 
       
       
      begin

      
      
        -- Populate the transfer address value by concatonating the
        -- 4 address counter segments
        sig_xfer_address_im0 <= STD_LOGIC_VECTOR(lsig_seg4_addr_cntr  ) &
                                STD_LOGIC_VECTOR(lsig_seg3_addr_cntr  ) &
                                STD_LOGIC_VECTOR(sig_addr_cntr_im0_msh) &
                                STD_LOGIC_VECTOR(sig_addr_cntr_lsh_im0);


       
        
        -- See if the MSH (Segment 2) of the Address Counter is at a max value
        lsig_acntr_msh_eq_max <= '1'
          when (sig_addr_cntr_im0_msh = ACNTR_MSH_MAX)
          Else '0';

        -- See if the Segment 3 of the Address Counter is at a max value
        lsig_acntr_seg3_eq_max <= '1'
          when (lsig_seg3_addr_cntr = ACNTR_SEG3_MAX)
          Else '0';


        
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_SEG2_3_EQ_MAX_REG
        --
        -- Process Description:
        --  Implements a register for the flag indicating the address
        -- counter segments 2 and 3 are at max value and will rollover
        -- at the next increment interval for the counter. Registering 
        -- these signals and using themt for the Seg 3/4 increment logic 
        -- only works because there is always at least a 1 clock time gap 
        -- between the increment causing the segment 2 or 3 counter to go  
        -- to max and the next increment operation. 
        --
        -------------------------------------------------------------
        IMP_SEG2_3_EQ_MAX_REG : process (primary_aclk)
          begin
            if (primary_aclk'event and primary_aclk = '1') then
               if (sig_mmap_reset_reg = '1') then
        
                 lsig_acntr_msh_eq_max_reg  <= '0';
                 lsig_acntr_seg3_eq_max_reg <= '0';
                 
               else
        
                 lsig_acntr_msh_eq_max_reg  <= lsig_acntr_msh_eq_max;
                 lsig_acntr_seg3_eq_max_reg <= lsig_acntr_seg3_eq_max;
        
               end if; 
            end if;       
          end process IMP_SEG2_3_EQ_MAX_REG; 
  
        
        
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_SEG3_ADDR_CNTR
        --
        -- Process Description:
        -- Segment 3 of the Address counter implementation.
        --
        -------------------------------------------------------------
        IMP_SEG3_ADDR_CNTR : process (primary_aclk)
           begin
             if (primary_aclk'event and primary_aclk = '1') then
                if (sig_mmap_reset_reg = '1') then

                  lsig_seg3_addr_cntr <= (others => '0');

                elsif (sig_ld_addr_cntr = '1') then

                  lsig_seg3_addr_cntr <= UNSIGNED(sig_cmd_addr_slice(SEG3_ADDR_RIP_MS_INDEX downto 
                                                                     SEG3_ADDR_RIP_LS_INDEX));

                Elsif (sig_incr_addr_cntr        = '1' and
                       sig_addr_lsh_rollover_im3 = '1' and
                       lsig_acntr_msh_eq_max_reg = '1') then

                  lsig_seg3_addr_cntr <= lsig_seg3_addr_cntr+ACNTR_SEG3_ONE;

                else
                  null;  -- hold current state
                end if;
             end if;
           end process IMP_SEG3_ADDR_CNTR;



        
  
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_SEG4_ADDR_CNTR
        --
        -- Process Description:
        -- Segment 4 of the Address counter implementation.
        --
        -------------------------------------------------------------
        IMP_SEG4_ADDR_CNTR : process (primary_aclk)
           begin
             if (primary_aclk'event and primary_aclk = '1') then
                if (sig_mmap_reset_reg = '1') then

                  lsig_seg4_addr_cntr <= (others => '0');

                elsif (sig_ld_addr_cntr = '1') then

                  lsig_seg4_addr_cntr <= UNSIGNED(sig_cmd_addr_slice(SEG4_ADDR_RIP_MS_INDEX downto 
                                                                     SEG4_ADDR_RIP_LS_INDEX));

                Elsif (sig_incr_addr_cntr         = '1' and
                       sig_addr_lsh_rollover_im3  = '1' and
                       lsig_acntr_msh_eq_max_reg  = '1' and
                       lsig_acntr_seg3_eq_max_reg = '1') then

                  lsig_seg4_addr_cntr <= lsig_seg4_addr_cntr+ACNTR_SEG4_ONE;

                else
                  null;  -- hold current state
                end if;
             end if;
           end process IMP_SEG4_ADDR_CNTR;



  
  
  
      end generate GEN_ADDR_GT_48;



     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     




    -- Addr and data Cntlr FIFO interface handshake logic ------------------------------

     sig_clr_cmd2data_valid    <= sig_cmd2data_valid and data2mstr_cmd_ready;

     sig_clr_cmd2addr_valid    <= sig_cmd2addr_valid and addr2mstr_cmd_ready;

     sig_clr_cmd2dre_valid     <= sig_cmd2dre_valid  and dre2mstr_cmd_ready;



     -------------------------------------------------------------
     -- Synchronous Process with Sync Reset
     --
     -- Label: CMD2DATA_VALID_FLOP
     --
     -- Process Description:
     --  Implements the set/reset flop for the Command Valid control
     -- to the Data Controller Module.
     --
     -------------------------------------------------------------
     CMD2DATA_VALID_FLOP : process (primary_aclk)
        begin
          if (primary_aclk'event and primary_aclk = '1') then
             if (sig_mmap_reset_reg     = '1' or
                 sig_clr_cmd2data_valid = '1') then

               sig_cmd2data_valid <= '0';

             elsif (sig_sm_ld_xfer_reg_ns = '1') then

               sig_cmd2data_valid <= '1';

             else
               null; -- hold current state
             end if;
          end if;
        end process CMD2DATA_VALID_FLOP;



     -------------------------------------------------------------
     -- Synchronous Process with Sync Reset
     --
     -- Label: CMD2ADDR_VALID_FLOP
     --
     -- Process Description:
     --  Implements the set/reset flop for the Command Valid control
     -- to the Address Controller Module.
     --
     -------------------------------------------------------------
     CMD2ADDR_VALID_FLOP : process (primary_aclk)
        begin
          if (primary_aclk'event and primary_aclk = '1') then
             if (sig_mmap_reset_reg     = '1' or
                 sig_clr_cmd2addr_valid = '1') then

               sig_cmd2addr_valid <= '0';

             elsif (sig_sm_ld_xfer_reg_ns = '1') then

               sig_cmd2addr_valid <= '1';

             else
               null; -- hold current state
             end if;
          end if;
        end process CMD2ADDR_VALID_FLOP;







     -------------------------------------------------------------
     -- Synchronous Process with Sync Reset
     --
     -- Label: CMD2DRE_VALID_FLOP
     --
     -- Process Description:
     --  Implements the set/reset flop for the Command Valid control
     -- to the DRE Module (S2MM DRE Only).
     --
     -- Note that the S2MM DRE only needs to be loaded with a command
     -- for each parent command, not every child command.
     --
     -------------------------------------------------------------
     CMD2DRE_VALID_FLOP : process (primary_aclk)
        begin
          if (primary_aclk'event and primary_aclk = '1') then
             if (sig_mmap_reset_reg    = '1' or
                 sig_clr_cmd2dre_valid = '1') then

               sig_cmd2dre_valid <= '0';

             elsif (sig_sm_ld_xfer_reg_ns = '1' and
                    sig_first_xfer_im0    = '1') then

               sig_cmd2dre_valid <= '1';

             else
               null; -- hold current state
             end if;
          end if;
        end process CMD2DRE_VALID_FLOP;








    -------------------------------------------------------------------------
    -- PCC State machine Logic





    -------------------------------------------------------------
    -- Combinational Process
    --
    -- Label: PCC_SM_COMBINATIONAL
    --
    -- Process Description:
    -- PCC State Machine combinational implementation
    --
    -------------------------------------------------------------
    PCC_SM_COMBINATIONAL : process (sig_pcc_sm_state     ,
                                    sig_parent_done      ,
                                    sig_push_input_reg   ,
                                    sig_pop_xfer_reg    ,
                                    sig_calc_error_pushed)
       begin

         -- SM Defaults
         sig_pcc_sm_state_ns     <=  INIT;
         sig_sm_halt_ns          <=  '0';
         sig_sm_ld_xfer_reg_ns   <=  '0';
         sig_sm_pop_input_reg_ns <=  '0';
         sig_sm_ld_calc1_reg_ns  <=  '0';
         sig_sm_ld_calc2_reg_ns  <=  '0';
         sig_sm_ld_calc3_reg_ns  <=  '0';


         case sig_pcc_sm_state is

           --------------------------------------------
           when INIT =>

             sig_pcc_sm_state_ns  <=  WAIT_FOR_CMD;
             sig_sm_halt_ns       <=  '1';

           --------------------------------------------
           when WAIT_FOR_CMD =>

             If (sig_push_input_reg = '1') Then

               sig_pcc_sm_state_ns     <=  CALC_1;
               sig_sm_ld_calc1_reg_ns  <=  '1';


             else

               sig_pcc_sm_state_ns <=  WAIT_FOR_CMD;

             End if;

           --------------------------------------------
           when CALC_1 =>

             sig_pcc_sm_state_ns     <=  CALC_2;
             sig_sm_ld_calc2_reg_ns  <=  '1';
             

           --------------------------------------------
           when CALC_2 =>

             sig_pcc_sm_state_ns    <=  CALC_3;
             sig_sm_ld_calc3_reg_ns <= '1';


           --------------------------------------------
           when CALC_3 =>

             sig_pcc_sm_state_ns    <=  WAIT_ON_XFER_PUSH;
             sig_sm_ld_xfer_reg_ns  <= '1';


           --------------------------------------------
           when WAIT_ON_XFER_PUSH =>

             if (sig_pop_xfer_reg = '1') then

               sig_pcc_sm_state_ns <=  CHK_IF_DONE;

             else  -- wait until output register is loaded

               sig_pcc_sm_state_ns <=  WAIT_ON_XFER_PUSH;


             end if;


           --------------------------------------------
           when CHK_IF_DONE =>

             If (sig_calc_error_pushed = '1') then -- Internal error, go to trap

               sig_pcc_sm_state_ns <=  ERROR_TRAP;
               sig_sm_halt_ns      <=  '1';

             elsif (sig_parent_done = '1') Then  -- done with parent command

               sig_pcc_sm_state_ns     <=  WAIT_FOR_CMD;
               sig_sm_pop_input_reg_ns <= '1';

             else  -- Still breaking up parent command

               sig_pcc_sm_state_ns     <=  CALC_1;
               sig_sm_ld_calc1_reg_ns  <=  '1';

             end if;


           --------------------------------------------
           when ERROR_TRAP =>

             sig_pcc_sm_state_ns <=  ERROR_TRAP;
             sig_sm_halt_ns      <=  '1';

           --------------------------------------------
           when others =>

             sig_pcc_sm_state_ns <=  INIT;

         end case;



       end process PCC_SM_COMBINATIONAL;




    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: PCC_SM_REGISTERED
    --
    -- Process Description:
    -- PCC State Machine registered implementation
    --
    -------------------------------------------------------------
    PCC_SM_REGISTERED : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_mmap_reset_reg = '1') then

              sig_pcc_sm_state     <= INIT;
              sig_sm_halt_reg      <= '1' ;
              sig_sm_pop_input_reg <= '0' ;
              sig_sm_ld_calc1_reg  <= '0' ;
              sig_sm_ld_calc2_reg  <= '0' ;
              sig_sm_ld_calc3_reg  <= '0' ;

            else

              sig_pcc_sm_state     <=  sig_pcc_sm_state_ns    ;
              sig_sm_halt_reg      <=  sig_sm_halt_ns         ;
              sig_sm_pop_input_reg <=  sig_sm_pop_input_reg_ns;
              sig_sm_ld_calc1_reg  <=  sig_sm_ld_calc1_reg_ns ;
              sig_sm_ld_calc2_reg  <=  sig_sm_ld_calc2_reg_ns ;
              sig_sm_ld_calc3_reg  <=  sig_sm_ld_calc3_reg_ns ;

            end if;
         end if;
       end process PCC_SM_REGISTERED;









    ------------------------------------------------------------------
    -- Transfer Register Load Enable logic


    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: LD_XFER_REG_FLOP
    --
    -- Process Description:
    -- Sample and Hold FLOP for signaling a load of the output
    -- xfer register.
    --
    -------------------------------------------------------------
    LD_XFER_REG_FLOP : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_mmap_reset_reg = '1' or
                sig_push_xfer_reg  = '1') then

              sig_ld_xfer_reg <=  '0';

            Elsif (sig_sm_ld_xfer_reg_ns = '1') Then

              sig_ld_xfer_reg <=  '1';

            else

              null;   -- hold current state

            end if;
         end if;
       end process LD_XFER_REG_FLOP;


    LD_XFER_REG_FLOP1 : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_mmap_reset_reg = '1' or
                sig_pop_xfer_reg  = '1') then

              sig_ld_xfer_reg_tmp <=  '0';

            Elsif (sig_sm_ld_xfer_reg_ns = '1') Then

              sig_ld_xfer_reg_tmp <=  '1';

            else

              null;   -- hold current state

            end if;
         end if;
       end process LD_XFER_REG_FLOP1;




    ------------------------------------------------------------------
    -- Parent Done flag logic


    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: PARENT_DONE_FLOP
    --
    -- Process Description:
    -- Sample and Hold FLOP for signaling a load of the output
    -- xfer register.
    --
    -------------------------------------------------------------
    PARENT_DONE_FLOP : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_mmap_reset_reg = '1' or
                sig_push_input_reg = '1') then

              sig_parent_done <=  '0';

            Elsif (sig_ld_xfer_reg_tmp = '1') Then

              sig_parent_done <=  sig_last_xfer_valid_im1;

            else

              null;   -- hold current state

            end if;
         end if;
       end process PARENT_DONE_FLOP;











  end implementation;


  ----------------------------------------------------------------------------
  -- axi_datamover_addr_cntl.vhd
  ----------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_addr_cntl.vhd
  --
  -- Description:     
  --    This file implements the axi_datamover Master Address Controller.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library axi_datamover_v5_1_14;
  Use axi_datamover_v5_1_14.axi_datamover_fifo;
  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_addr_cntl is
    generic (
      C_ADDR_FIFO_DEPTH      : Integer range  1 to  32 :=  4;
        -- sets the depth of the Command Queue FIFO
        
      C_ADDR_WIDTH           : Integer range 32 to  64 := 32;
        -- Sets the address bus width
      
      C_ADDR_ID              : Integer range  0 to 255 :=  0;
        -- Sets the value to be on the AxID output
      
      C_ADDR_ID_WIDTH        : Integer range  1 to   8 :=  4;
        -- Sets the width of the AxID output
      
      C_TAG_WIDTH            : Integer range  1 to   8 :=  4;
        -- Sets the width of the Command Tag field width
      
      C_FAMILY               : String  := "virtex7"
        -- Specifies the target FPGA family
        
      );
    port (
      
      -- Clock input ---------------------------------------------
      primary_aclk         : in  std_logic;                     --
         -- Primary synchronization clock for the Master side   --
         -- interface and internal logic. It is also used       --
         -- for the User interface synchronization when         --
         -- C_STSCMD_IS_ASYNC = 0.                              --
                                                                --
      -- Reset input                                            --
      mmap_reset           : in  std_logic;                     --
         -- Reset used for the internal master logic            --
      ------------------------------------------------------------   
      
     
      -- AXI Address Channel I/O  --------------------------------------------
      addr2axi_aid     : out std_logic_vector(C_ADDR_ID_WIDTH-1 downto 0);  --
         -- AXI Address Channel ID output                                   --
                                                                            --
      addr2axi_aaddr   : out std_logic_vector(C_ADDR_WIDTH-1 downto 0);     --
         -- AXI Address Channel Address output                              --
                                                                            --
      addr2axi_alen    : out std_logic_vector(7 downto 0);                  --
         -- AXI Address Channel LEN output                                  --
         -- Sized to support 256 data beat bursts                           --
                                                                            --
      addr2axi_asize   : out std_logic_vector(2 downto 0);                  --
         -- AXI Address Channel SIZE output                                 --
                                                                            --
      addr2axi_aburst  : out std_logic_vector(1 downto 0);                  --
         -- AXI Address Channel BURST output                                --
                                                                            --
      addr2axi_acache  : out std_logic_vector(3 downto 0);                  --
         -- AXI Address Channel BURST output                                --
                                                                            --
      addr2axi_auser  : out std_logic_vector(3 downto 0);                  --
         -- AXI Address Channel BURST output                                --
                                                                            --
      addr2axi_aprot   : out std_logic_vector(2 downto 0);                  --
         -- AXI Address Channel PROT output                                 --
                                                                            --
      addr2axi_avalid  : out std_logic;                                     --
         -- AXI Address Channel VALID output                                --

                                                                            --
      axi2addr_aready  : in  std_logic;                                     --
         -- AXI Address Channel READY input                                 --
      ------------------------------------------------------------------------   
         
         
        
      -- Currently unsupported AXI Address Channel output signals -------
        -- addr2axi_alock   : out std_logic_vector(2 downto 0);        --
        -- addr2axi_acache  : out std_logic_vector(4 downto 0);        --
        -- addr2axi_aqos    : out std_logic_vector(3 downto 0);        --
        -- addr2axi_aregion : out std_logic_vector(3 downto 0);        --
      -------------------------------------------------------------------
      
      
      
      -- Command Calculation Interface -----------------------------------------
      mstr2addr_tag         : In std_logic_vector(C_TAG_WIDTH-1 downto 0);    --
         -- The next command tag                                              --
                                                                              --
      mstr2addr_addr        : In  std_logic_vector(C_ADDR_WIDTH-1 downto 0);  --
         -- The next command address to put on the AXI MMap ADDR              --
                                                                              --
      mstr2addr_len         : In  std_logic_vector(7 downto 0);               --
         -- The next command length to put on the AXI MMap LEN                --
         -- Sized to support 256 data beat bursts                             --
                                                                              --
      mstr2addr_size        : In  std_logic_vector(2 downto 0);               --
         -- The next command size to put on the AXI MMap SIZE                 --
                                                                              --
      mstr2addr_burst       : In  std_logic_vector(1 downto 0);               --
         -- The next command burst type to put on the AXI MMap BURST          --
                                                                              --
      mstr2addr_cache        : In  std_logic_vector(3 downto 0);               --
         -- The next command burst type to put on the AXI MMap BURST          --
                                                                              --
      mstr2addr_user        : In  std_logic_vector(3 downto 0);               --
         -- The next command burst type to put on the AXI MMap BURST          --
                                                                              --
      mstr2addr_cmd_cmplt   : In std_logic;                                   --
         -- The indication to the Address Channel that the current            --
         -- sub-command output is the last one compiled from the              --
         -- parent command pulled from the Command FIFO                       --
                                                                              --
      mstr2addr_calc_error  : In std_logic;                                   --
         -- Indication if the next command in the calculation pipe            --
         -- has a calculation error                                           --
                                                                              --
      mstr2addr_cmd_valid   : in std_logic;                                   --
         -- The next command valid indication to the Address Channel          --
         -- Controller for the AXI MMap                                       --
  
                                                                              --
      addr2mstr_cmd_ready   : out std_logic;                                  --
         -- Indication to the Command Calculator that the                     --
         -- command is being accepted                                         --
      --------------------------------------------------------------------------
     
     
      
      -- Halted Indication to Reset Module ------------------------------
      addr2rst_stop_cmplt     : out std_logic;                         --
         -- Output flag indicating the address controller has stopped  --
         -- posting commands to the Address Channel due to a stop      --
         -- request vai the data2addr_stop_req input port              --
       ------------------------------------------------------------------
       
      
      
      -- Address Generation Control ---------------------------------------
      allow_addr_req    : in std_logic;                                  --
         -- Input used to enable/stall the posting of address requests.  --
         -- 0 = stall address request generation.                        --
         -- 1 = Enable Address request geneartion                        --
                                                                         --
      addr_req_posted   : out std_logic;                                 --
         -- Indication from the Address Channel Controller to external   --
         -- User logic that an address has been posted to the            --
         -- AXI Address Channel.                                         --
      ---------------------------------------------------------------------
    
      
                                                                        
                                                                        
      -- Data Channel Interface ---------------------------------------------
      addr2data_addr_posted    : Out std_logic;                            --
         -- Indication from the Address Channel Controller to the          --
         -- Data Controller that an address has been posted to the         --
         -- AXI Address Channel.                                           --
                                                                           --
      data2addr_data_rdy       : In std_logic;                             --
         -- Indication that the Data Channel is ready to send the first    --
         -- databeat of the next command on the write data channel.        --
         -- This is used for the "wait for data" feature which keeps the   --
         -- address controller from issuing a transfer requset until the   --
         -- corresponding data is ready. This is expected to be held in    --
         -- the asserted state until the addr2data_addr_posted signal is   --
         -- asserted.                                                      --
                                                                           --
      data2addr_stop_req       : In std_logic;                             --
         -- Indication that the Data Channel has encountered an error      --
         -- or a soft shutdown request and needs the Address Controller    --
         -- to stop posting commands to the AXI Address channel            --
      -----------------------------------------------------------------------

       
      -- Status Module Interface ---------------------------------------
      addr2stat_calc_error     : out std_logic;                       --
         -- Indication to the Status Module that the Addr Cntl FIFO   --
         -- is loaded with a Calc error                               --
                                                                      --
      addr2stat_cmd_fifo_empty : out std_logic                        --
         -- Indication to the Status Module that the Addr Cntl FIFO   --
         -- is empty                                                  --
      ------------------------------------------------------------------   
      
     
     
         
      );
  
  end entity axi_datamover_addr_cntl;
  
  
  architecture implementation of axi_datamover_addr_cntl is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    -- Constant Declarations  --------------------------------------------
    Constant APROT_VALUE      : std_logic_vector(2 downto 0) := (others => '0');
                                --'0' & -- bit 2, Normal Access
                                --'0' & -- bit 1, Nonsecure Access
                                --'0';  -- bit 0, Data Access
    
    Constant LEN_WIDTH        : integer := 8;
    Constant SIZE_WIDTH       : integer := 3;
    Constant BURST_WIDTH      : integer := 2;
    Constant CMD_CMPLT_WIDTH  : integer := 1;
    Constant CALC_ERROR_WIDTH : integer := 1;
    
    Constant ADDR_QUAL_WIDTH  : integer := C_TAG_WIDTH     +  -- Cmd Tag field width
                                           C_ADDR_WIDTH    +  -- Cmd Address field width
                                           LEN_WIDTH       +  -- Cmd Len field width
                                           SIZE_WIDTH      +  -- Cmd Size field width
                                           BURST_WIDTH     +  -- Cmd Burst field width
                                           CMD_CMPLT_WIDTH +  -- Cmd Cmplt filed width
                                           CALC_ERROR_WIDTH +  -- Cmd Calc Error flag
                                           8; -- Cmd Cache, user fields
    
    Constant USE_SYNC_FIFO    : integer := 0;
    Constant REG_FIFO_PRIM    : integer := 0; 
    Constant BRAM_FIFO_PRIM   : integer := 1; 
    Constant SRL_FIFO_PRIM    : integer := 2; 
    Constant FIFO_PRIM_TYPE   : integer := SRL_FIFO_PRIM; 
    
    
    
    
    
    -- Signal Declarations  --------------------------------------------
    signal sig_axi_addr               : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_axi_alen               : std_logic_vector(7 downto 0) := (others => '0');             
    signal sig_axi_asize              : std_logic_vector(2 downto 0) := (others => '0');             
    signal sig_axi_aburst             : std_logic_vector(1 downto 0) := (others => '0');             
    signal sig_axi_acache             : std_logic_vector(3 downto 0) := (others => '0');             
    signal sig_axi_auser              : std_logic_vector(3 downto 0) := (others => '0');             
    signal sig_axi_avalid             : std_logic := '0';
    signal sig_axi_aready             : std_logic := '0';
    signal sig_addr_posted            : std_logic := '0';
    signal sig_calc_error             : std_logic := '0';
    signal sig_cmd_fifo_empty         : std_logic := '0';
    Signal sig_aq_fifo_data_in        : std_logic_vector(ADDR_QUAL_WIDTH-1 downto 0) := (others => '0');
    Signal sig_aq_fifo_data_out       : std_logic_vector(ADDR_QUAL_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_tag          : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_addr         : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_len          : std_logic_vector(7 downto 0) := (others => '0');             
    signal sig_fifo_next_size         : std_logic_vector(2 downto 0) := (others => '0');             
    signal sig_fifo_next_burst        : std_logic_vector(1 downto 0) := (others => '0');
    signal sig_fifo_next_user         : std_logic_vector(3 downto 0) := (others => '0');
    signal sig_fifo_next_cache        : std_logic_vector(3 downto 0) := (others => '0');
    signal sig_fifo_next_cmd_cmplt    : std_logic := '0';
    signal sig_fifo_calc_error        : std_logic := '0';
    signal sig_fifo_wr_cmd_valid      : std_logic := '0';
    signal sig_fifo_wr_cmd_ready      : std_logic := '0';
    signal sig_fifo_rd_cmd_valid      : std_logic := '0';
    signal sig_fifo_rd_cmd_ready      : std_logic := '0';
    signal sig_next_tag_reg           : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_addr_reg          : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_len_reg           : std_logic_vector(7 downto 0) := (others => '0');             
    signal sig_next_size_reg          : std_logic_vector(2 downto 0) := (others => '0');             
    signal sig_next_burst_reg         : std_logic_vector(1 downto 0) := (others => '0');             
    signal sig_next_cache_reg         : std_logic_vector(3 downto 0) := (others => '0');             
    signal sig_next_user_reg          : std_logic_vector(3 downto 0) := (others => '0');             
    signal sig_next_cmd_cmplt_reg     : std_logic := '0';             
    signal sig_addr_valid_reg         : std_logic := '0';
    signal sig_calc_error_reg         : std_logic := '0';
    signal sig_pop_addr_reg           : std_logic := '0';
    signal sig_push_addr_reg          : std_logic := '0';
    signal sig_addr_reg_empty         : std_logic := '0';
    signal sig_addr_reg_full          : std_logic := '0';
    signal sig_posted_to_axi          : std_logic := '0';
    -- obsoleted   signal sig_set_wfd_flop           : std_logic := '0';
    -- obsoleted   signal sig_clr_wfd_flop           : std_logic := '0';
    -- obsoleted   signal sig_wait_for_data          : std_logic := '0';
    -- obsoleted   signal sig_data2addr_data_rdy_reg : std_logic := '0';
    signal sig_allow_addr_req         : std_logic := '0';
    signal sig_posted_to_axi_2        : std_logic := '0';
 
    signal new_cmd_in : std_logic;
    signal first_addr_valid : std_logic;
    signal first_addr_valid_del : std_logic;
    signal first_addr_int : std_logic_vector (C_ADDR_WIDTH-1 downto 0);
    signal last_addr_int : std_logic_vector (C_ADDR_WIDTH-1 downto 0);
    signal addr2axi_cache_int  : std_logic_vector (7 downto 0); 
    signal addr2axi_cache_int1  : std_logic_vector (7 downto 0); 

    signal last_one : std_logic;
    signal latch : std_logic;
    signal first_one : std_logic;
    signal latch_n : std_logic;
    signal latch_n_del : std_logic;
    signal mstr2addr_cache_info_int : std_logic_vector (7 downto 0);
-- Register duplication attribute assignments to control fanout
-- on handshake output signals  
  
  Attribute KEEP : string; -- declaration
  Attribute EQUIVALENT_REGISTER_REMOVAL : string; -- declaration
  
  Attribute KEEP of sig_posted_to_axi   : signal is "TRUE"; -- definition
  Attribute KEEP of sig_posted_to_axi_2 : signal is "TRUE"; -- definition
  
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_posted_to_axi   : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_posted_to_axi_2 : signal is "no"; 
  
  
  
 
    
  begin --(architecture implementation)
  
    -- AXI I/O Port assignments
    addr2axi_aid        <= STD_LOGIC_VECTOR(TO_UNSIGNED(C_ADDR_ID, C_ADDR_ID_WIDTH));
    addr2axi_aaddr      <= sig_axi_addr  ;
    addr2axi_alen       <= sig_axi_alen  ;
    addr2axi_asize      <= sig_axi_asize ;
    addr2axi_aburst     <= sig_axi_aburst;
    addr2axi_acache     <= sig_axi_acache;
    addr2axi_auser      <= sig_axi_auser;
    addr2axi_aprot      <= APROT_VALUE   ; 
    addr2axi_avalid     <= sig_axi_avalid;
    sig_axi_aready      <= axi2addr_aready;
    
    
    -- Command Calculator Handshake output
    sig_fifo_wr_cmd_valid <= mstr2addr_cmd_valid  ;
    addr2mstr_cmd_ready   <= sig_fifo_wr_cmd_ready;
    
    -- Data Channel Controller synchro pulse output                  
    addr2data_addr_posted <= sig_addr_posted;
     
     
    -- Status Module Interface outputs
    addr2stat_calc_error     <= sig_calc_error    ;
    addr2stat_cmd_fifo_empty <= sig_addr_reg_empty and
                                sig_cmd_fifo_empty; 
     
    

   
    -- Flag Indicating the Address Controller has completed a Stop
    addr2rst_stop_cmplt   <=  (data2addr_stop_req and  -- normal shutdown case
                               sig_addr_reg_empty) or
                              (data2addr_stop_req and  -- shutdown after error trap
                               sig_calc_error);
   
    -- Assign the address posting control and status
    sig_allow_addr_req    <=  allow_addr_req      ;
    addr_req_posted       <=  sig_posted_to_axi_2 ;
   
   
    
    -- Internal logic ------------------------------





    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ADDR_FIFO
    --
    -- If Generate Description:
    --   Implements the case where the cmd qualifier depth is
    -- greater than 1.
    --
    ------------------------------------------------------------
    GEN_ADDR_FIFO : if (C_ADDR_FIFO_DEPTH > 1) generate

      begin

        -- Format the input FIFO data word 
        
        sig_aq_fifo_data_in <= mstr2addr_cache      &
                               mstr2addr_user      &
                               mstr2addr_calc_error &
                               mstr2addr_cmd_cmplt  &
                               mstr2addr_burst      &
                               mstr2addr_size       &
                               mstr2addr_len        &
                               mstr2addr_addr       &
                               mstr2addr_tag ;
        
    
    
        -- Rip fields from FIFO output data word
        sig_fifo_next_cache     <=  sig_aq_fifo_data_out((C_ADDR_WIDTH  +
                                                        C_TAG_WIDTH     +
                                                        LEN_WIDTH       +
                                                        SIZE_WIDTH      +
                                                        BURST_WIDTH     +
                                                        CMD_CMPLT_WIDTH + 
                                                        CALC_ERROR_WIDTH + 7)
                                                    downto
                                                        (C_ADDR_WIDTH  +
                                                        C_TAG_WIDTH     +
                                                        LEN_WIDTH       +
                                                        SIZE_WIDTH      +
                                                        BURST_WIDTH     +
                                                        CMD_CMPLT_WIDTH +
                                                        CALC_ERROR_WIDTH + 4)
                                                    ); 

        sig_fifo_next_user     <=  sig_aq_fifo_data_out((C_ADDR_WIDTH  +
                                                        C_TAG_WIDTH     +
                                                        LEN_WIDTH       +
                                                        SIZE_WIDTH      +
                                                        BURST_WIDTH     +
                                                        CMD_CMPLT_WIDTH + 
                                                        CALC_ERROR_WIDTH + 3)
                                                    downto
                                                        (C_ADDR_WIDTH  +
                                                        C_TAG_WIDTH     +
                                                        LEN_WIDTH       +
                                                        SIZE_WIDTH      +
                                                        BURST_WIDTH     +
                                                        CMD_CMPLT_WIDTH +
                                                        CALC_ERROR_WIDTH)
                                                    ); 


        sig_fifo_calc_error     <=  sig_aq_fifo_data_out((C_ADDR_WIDTH  +
                                                        C_TAG_WIDTH     +
                                                        LEN_WIDTH       +
                                                        SIZE_WIDTH      +
                                                        BURST_WIDTH     +
                                                        CMD_CMPLT_WIDTH + 
                                                        CALC_ERROR_WIDTH)-1); 
                                                        
        
        sig_fifo_next_cmd_cmplt <=  sig_aq_fifo_data_out((C_ADDR_WIDTH  +
                                                        C_TAG_WIDTH     +
                                                        LEN_WIDTH       +
                                                        SIZE_WIDTH      +
                                                        BURST_WIDTH     +
                                                        CMD_CMPLT_WIDTH)-1); 
                                                        
        
        sig_fifo_next_burst     <=  sig_aq_fifo_data_out((C_ADDR_WIDTH +
                                                         C_TAG_WIDTH   +
                                                         LEN_WIDTH     +
                                                         SIZE_WIDTH    +
                                                         BURST_WIDTH)-1 
                                                downto 
                                                         C_ADDR_WIDTH  +
                                                         C_TAG_WIDTH   +
                                                         LEN_WIDTH     +
                                                         SIZE_WIDTH) ;
                                                          
        sig_fifo_next_size      <=  sig_aq_fifo_data_out((C_ADDR_WIDTH +
                                                         C_TAG_WIDTH   +
                                                         LEN_WIDTH     +
                                                         SIZE_WIDTH)-1
                                                downto 
                                                         C_ADDR_WIDTH  +
                                                         C_TAG_WIDTH   +
                                                         LEN_WIDTH) ; 
                                                         
        sig_fifo_next_len       <=  sig_aq_fifo_data_out((C_ADDR_WIDTH +
                                                         C_TAG_WIDTH   +
                                                         LEN_WIDTH)-1 
                                                downto 
                                                        C_ADDR_WIDTH   +
                                                        C_TAG_WIDTH) ;
                                                         
        sig_fifo_next_addr      <=  sig_aq_fifo_data_out((C_ADDR_WIDTH +
                                                         C_TAG_WIDTH)-1 
                                                downto 
                                                         C_TAG_WIDTH) ;
         
        sig_fifo_next_tag       <=  sig_aq_fifo_data_out(C_TAG_WIDTH-1 downto 0); 
      
      
      
        ------------------------------------------------------------
        -- Instance: I_ADDR_QUAL_FIFO 
        --
        -- Description:
        -- Instance for the Address/Qualifier FIFO
        --
        ------------------------------------------------------------
         I_ADDR_QUAL_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
         generic map (
       
           C_DWIDTH            =>  ADDR_QUAL_WIDTH         ,  
           C_DEPTH             =>  C_ADDR_FIFO_DEPTH       ,  
           C_IS_ASYNC          =>  USE_SYNC_FIFO           ,  
           C_PRIM_TYPE         =>  FIFO_PRIM_TYPE          ,  
           C_FAMILY            =>  C_FAMILY                   
          
           )
         port map (
           
           -- Write Clock and reset
           fifo_wr_reset        =>   mmap_reset            ,  
           fifo_wr_clk          =>   primary_aclk          ,  
           
           -- Write Side
           fifo_wr_tvalid       =>   sig_fifo_wr_cmd_valid ,  
           fifo_wr_tready       =>   sig_fifo_wr_cmd_ready ,  
           fifo_wr_tdata        =>   sig_aq_fifo_data_in   ,  
           fifo_wr_full         =>   open                  ,  
          
          
           -- Read Clock and reset
           fifo_async_rd_reset  =>   mmap_reset            ,     
           fifo_async_rd_clk    =>   primary_aclk          ,   
           
           -- Read Side
           fifo_rd_tvalid       =>   sig_fifo_rd_cmd_valid ,  
           fifo_rd_tready       =>   sig_fifo_rd_cmd_ready ,  
           fifo_rd_tdata        =>   sig_aq_fifo_data_out  ,  
           fifo_rd_empty        =>   sig_cmd_fifo_empty       
          
           );
       
      
      
      end generate GEN_ADDR_FIFO;
       
   

    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_NO_ADDR_FIFO
    --
    -- If Generate Description:
    --   Implements the case where no additional FIFOing is needed
    -- on the input command address/qualifiers.
    --
    ------------------------------------------------------------
    GEN_NO_ADDR_FIFO : if (C_ADDR_FIFO_DEPTH = 1) generate

      begin

        -- Bypass FIFO
        sig_fifo_next_tag       <=  mstr2addr_tag         ; 
        sig_fifo_next_addr      <=  mstr2addr_addr        ; 
        sig_fifo_next_len       <=  mstr2addr_len         ; 
        sig_fifo_next_size      <=  mstr2addr_size        ; 
        sig_fifo_next_burst     <=  mstr2addr_burst       ; 
        sig_fifo_next_cache     <=  mstr2addr_cache       ; 
        sig_fifo_next_user      <=  mstr2addr_user       ; 
        sig_fifo_next_cmd_cmplt <=  mstr2addr_cmd_cmplt   ;
        sig_fifo_calc_error     <=  mstr2addr_calc_error  ; 
        sig_cmd_fifo_empty      <=  sig_addr_reg_empty    ;
        sig_fifo_wr_cmd_ready   <=  sig_fifo_rd_cmd_ready ;
        sig_fifo_rd_cmd_valid   <=  sig_fifo_wr_cmd_valid ;
        
      
      end generate GEN_NO_ADDR_FIFO;
       
    
    
    -- Output Register Logic ------------------------------------------- 
    
    sig_axi_addr           <= sig_next_addr_reg  ;
    sig_axi_alen           <= sig_next_len_reg   ;
    sig_axi_asize          <= sig_next_size_reg  ;
    sig_axi_aburst         <= sig_next_burst_reg ;
    sig_axi_acache         <= sig_next_cache_reg ;
    sig_axi_auser          <= sig_next_user_reg ;
    sig_axi_avalid         <= sig_addr_valid_reg ;
    
    sig_calc_error         <= sig_calc_error_reg ;
    
    sig_fifo_rd_cmd_ready  <= sig_addr_reg_empty  and
                              sig_allow_addr_req  and   
                              -- obsoleted   not(sig_wait_for_data) and
                              not(data2addr_stop_req);
    
    sig_addr_posted        <= sig_posted_to_axi  ;
    
    
    -- Internal signals
    sig_push_addr_reg      <= sig_addr_reg_empty     and 
                              sig_fifo_rd_cmd_valid  and
                              sig_allow_addr_req     and   
                              -- obsoleted   not(sig_wait_for_data) and
                              not(data2addr_stop_req);
    
    sig_pop_addr_reg       <= not(sig_calc_error_reg) and 
                              sig_axi_aready          and 
                              sig_addr_reg_full;
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ADDR_FIFO_REG
    --
    -- Process Description:
    --    This process implements a register for the Address 
    -- Control FIFO that operates like a 1 deep Sync FIFO.
    --
    -------------------------------------------------------------
    IMP_ADDR_FIFO_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset       = '1' or
                sig_pop_addr_reg = '1') then
              
              sig_next_tag_reg        <= (others => '0')         ;
              sig_next_addr_reg       <= (others => '0')         ;
              sig_next_len_reg        <= (others => '0')         ;
              sig_next_size_reg       <= (others => '0')         ;
              sig_next_burst_reg      <= (others => '0')         ;
              sig_next_cache_reg      <= (others => '0')         ;
              sig_next_user_reg      <= (others => '0')         ;
              sig_next_cmd_cmplt_reg  <= '0'                     ;
              sig_addr_valid_reg      <= '0'                     ;
              sig_calc_error_reg      <= '0'                     ;
                                                                 
              sig_addr_reg_empty      <= '1'                     ;
              sig_addr_reg_full       <= '0'                     ;
              
            elsif (sig_push_addr_reg = '1') then
              
              sig_next_tag_reg        <= sig_fifo_next_tag       ;
              sig_next_addr_reg       <= sig_fifo_next_addr      ;
              sig_next_len_reg        <= sig_fifo_next_len       ;
              sig_next_size_reg       <= sig_fifo_next_size      ;
              sig_next_burst_reg      <= sig_fifo_next_burst     ;
              sig_next_cache_reg      <= sig_fifo_next_cache     ;
              sig_next_user_reg      <= sig_fifo_next_user     ;
              sig_next_cmd_cmplt_reg  <= sig_fifo_next_cmd_cmplt ;
              sig_addr_valid_reg      <= not(sig_fifo_calc_error);
              sig_calc_error_reg      <= sig_fifo_calc_error     ;
              
              sig_addr_reg_empty      <= '0'                     ;
              sig_addr_reg_full       <= '1'                     ;
              
            else
              null;  -- don't change state
            end if; 
         end if;       
       end process IMP_ADDR_FIFO_REG; 
     

    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_POSTED_FLAG
    --
    -- Process Description:
    --   This implements a FLOP that creates a 1 clock wide pulse
    -- indicating a new address/qualifier set has been posted to
    -- the AXI Addres Channel outputs. This is used to synchronize 
    -- the Data Channel Controller.
    --
    -------------------------------------------------------------
    IMP_POSTED_FLAG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then

              sig_posted_to_axi   <= '0';
              sig_posted_to_axi_2 <= '0';
              
            elsif (sig_push_addr_reg = '1') then

              sig_posted_to_axi   <= '1';
              sig_posted_to_axi_2 <= '1';
              
            else
              
              sig_posted_to_axi   <= '0';
              sig_posted_to_axi_2 <= '0';
              
            end if; 
         end if;       
       end process IMP_POSTED_FLAG; 
  
--     PROC_CMD_DETECT :  process (primary_aclk)
--                     begin
--                     if (mmap_reset = '1') then
--                        first_addr_valid_del <= '0';
--                     elsif (primary_aclk'event and primary_aclk = '1') then
--                        first_addr_valid_del <= first_addr_valid;
--                     end if;
--     end process PROC_CMD_DETECT;
--
--     PROC_ADDR_DET : process (primary_aclk)
--            begin
--              if (mmap_reset = '1') then
--                 first_addr_valid <= '0';
--                 first_addr_int <= (others => '0');
--                 last_addr_int <= (others => '0');
--              elsif (primary_aclk'event and primary_aclk = '1') then
--                if (mstr2addr_cmd_valid = '1' and first_addr_valid = '0') then
--                   first_addr_valid <= '1';
--                   first_addr_int <= mstr2addr_addr;
--                   last_addr_int <= last_addr_int;
--                elsif (mstr2addr_cmd_cmplt = '1') then
--                   first_addr_valid <= '0';
--                   first_addr_int <= first_addr_int;
--                   last_addr_int <= mstr2addr_addr;
--                end if;
--              end if;
--           end process PROC_ADDR_DET;
--
--     latch <= first_addr_valid and (not first_addr_valid_del);
--     latch_n <= (not first_addr_valid) and first_addr_valid_del;
--
--     PROC_CACHE1 : process (primary_aclk)
--          begin
--              if (mmap_reset = '1') then
--                 mstr2addr_cache_info_int <= (others => '0');
--                 latch_n_del <= '0';
--              elsif (primary_aclk'event and primary_aclk = '1') then
--                 if (latch_n = '1') then 
--                   mstr2addr_cache_info_int <= mstr2addr_cache_info;
--                 end if;
--                 latch_n_del <= latch_n;
--              end if;
--          end process PROC_CACHE1;
--       
--
--     PROC_CACHE : process (primary_aclk)
--          begin
--              if (mmap_reset = '1') then
--                 addr2axi_cache_int1 <= (others => '0');
--                 first_one <= '0'; 
--              elsif (primary_aclk'event and primary_aclk = '1') then
--                 first_one <= '0';
----                 if (latch = '1' and first_one = '0') then  -- first one
--                 if (sig_addr_valid_reg = '0' and first_addr_valid = '0') then
--                    addr2axi_cache_int1 <= mstr2addr_cache_info;
----                    first_one <= '1';
----                 elsif (latch_n_del = '1') then
----                    addr2axi_cache_int <= mstr2addr_cache_info_int;
--                 elsif ((first_addr_int = sig_next_addr_reg) and (sig_addr_valid_reg = '1')) then
--                    addr2axi_cache_int1 <= addr2axi_cache_int1; --mstr2addr_cache_info (7 downto 4);
--                 elsif ((last_addr_int >= sig_next_addr_reg) and (sig_addr_valid_reg = '1')) then 
--                    addr2axi_cache_int1 <= addr2axi_cache_int1; --mstr2addr_cache_info (7 downto 4);
--                 end if; 
--              end if;
--          end process PROC_CACHE;
--
--
--     PROC_CACHE2 : process (primary_aclk)
--          begin
--              if (mmap_reset = '1') then
--                 addr2axi_cache_int <= (others => '0');
--              elsif (primary_aclk'event and primary_aclk = '1') then
--                 addr2axi_cache_int <= addr2axi_cache_int1;
--              end if;
--          end process PROC_CACHE2;
-- 
--addr2axi_cache <= addr2axi_cache_int (3 downto 0);
--addr2axi_user  <= addr2axi_cache_int (7 downto 4);
-- 
  end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_rdmux.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_rdmux.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Read Data Multiplexer.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_rdmux is
    generic (
      
      C_SEL_ADDR_WIDTH     : Integer range 1  to   8 :=  5;
        -- Sets the width of the select control bus
      
      C_MMAP_DWIDTH        : Integer range 32 to 1024 := 32;
        -- Indicates the width of the AXI4 Data Channel
      
      C_STREAM_DWIDTH      : Integer range  8 to 1024 := 32
        -- Indicates the width of the AXI Stream Data Channel
      
      );
    port (
      
     
      -- AXI MMap Data Channel Input  -----------------------------------------------
                                                                                   --
      mmap_read_data_in         : In  std_logic_vector(C_MMAP_DWIDTH-1 downto 0);  --
        -- AXI Read data input                                                     --
      -------------------------------------------------------------------------------
      
      
      -- AXI Master Stream  ---------------------------------------------------------
                                                                                   --
      mux_data_out    : Out std_logic_vector(C_STREAM_DWIDTH-1 downto 0);          --
        --Mux data output                                                          --
      -------------------------------------------------------------------------------         
                
                
      -- Command Calculator Interface -----------------------------------------------
                                                                                   --
      mstr2data_saddr_lsb : In std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0)       --
         -- The next command start address LSbs to use for the read data           --
         -- mux (only used if Stream data width is less than the MMap Data         --
         -- Width).                                                                --
      -------------------------------------------------------------------------------
         
      );
  
  end entity axi_datamover_rdmux;
  
  
  architecture implementation of axi_datamover_rdmux is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    -- Function Decalarations -------------------------------------------------
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_mux_sel_width
    --
    -- Function Description:
    --   Calculates the number of needed bits for the Mux Select control
    -- based on the number of input channels to the mux.
    --
    -- Note that the number of input mux channels are always a 
    -- power of 2.
    --
    -------------------------------------------------------------------
    function func_mux_sel_width (num_channels : integer) return integer is
    
     Variable var_sel_width : integer := 0;
    
    begin
    
       case num_channels is
         when 2 =>
             var_sel_width := 1;
         when 4 =>
             var_sel_width := 2;
         when 8 =>
             var_sel_width := 3;
         when 16 =>
             var_sel_width := 4;
         when 32 =>
             var_sel_width := 5;
         when 64 =>
             var_sel_width := 6;
         when 128 =>
             var_sel_width := 7;
         
         when others => 
             var_sel_width := 0; 
       end case;
       
       Return (var_sel_width);
        
        
    end function func_mux_sel_width;
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_sel_ls_index
    --
    -- Function Description:
    --   Calculates the LS index of the select field to rip from the
    -- input select bus.
    --
    -- Note that the number of input mux channels are always a 
    -- power of 2.
    --
    -------------------------------------------------------------------
    function func_sel_ls_index (channel_width : integer) return integer is
    
     Variable var_sel_ls_index : integer := 0;
    
    begin
    
       case channel_width is
         when 8 => 
             var_sel_ls_index := 0;
         when 16 =>
             var_sel_ls_index := 1;
         when 32 =>
             var_sel_ls_index := 2;
         when 64 =>
             var_sel_ls_index := 3;
         when 128 =>
             var_sel_ls_index := 4;
         when 256 =>
             var_sel_ls_index := 5;
         when 512 =>
             var_sel_ls_index := 6;
         
         when others => -- 1024-bit channel case
             var_sel_ls_index := 7;
       end case;
       
       Return (var_sel_ls_index);
        
        
    end function func_sel_ls_index;
    
    
    
    
    
    -- Constant Decalarations -------------------------------------------------
    
    Constant CHANNEL_DWIDTH   : integer := C_STREAM_DWIDTH;
    Constant NUM_MUX_CHANNELS : integer := C_MMAP_DWIDTH/CHANNEL_DWIDTH;
    Constant MUX_SEL_WIDTH    : integer := func_mux_sel_width(NUM_MUX_CHANNELS);
    Constant MUX_SEL_LS_INDEX : integer := func_sel_ls_index(CHANNEL_DWIDTH);
    
    
    
    
    -- Signal Declarations  --------------------------------------------
 
    signal sig_rdmux_dout     : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');



    
  begin --(architecture implementation)
  
  
  
  
   -- Assign the Output data port 
    mux_data_out        <= sig_rdmux_dout;
  
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_STRM_EQ_MMAP
    --
    -- If Generate Description:
    --   This IfGen implements the case where the Stream Data Width is 
    -- the same as the Memory Map read Data width.
    --
    --
    ------------------------------------------------------------
    GEN_STRM_EQ_MMAP : if (NUM_MUX_CHANNELS = 1) generate
        
       begin
    
          sig_rdmux_dout <= mmap_read_data_in;
        
       end generate GEN_STRM_EQ_MMAP;
   
   
    
    
    
     
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_2XN
    --
    -- If Generate Description:
    --  2 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_2XN : if (NUM_MUX_CHANNELS = 2) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice   <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd  <=  UNSIGNED(sig_mux_sel_slice);  -- convert to unsigned
        
         sig_mux_sel_int     <=  TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                 -- with locally static subtype error in each of the
                                                                 -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_2XN_NUX
         --
         -- Process Description:
         --  Implement the 2XN Mux
         --
         -------------------------------------------------------------
         DO_2XN_NUX : process (sig_mux_sel_int_local,
                               mmap_read_data_in)
            begin
              
              case sig_mux_sel_int_local is
                when 0 =>
                    sig_mux_dout <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0);
                
                when others => -- 1 case
                    sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1);
              end case;
              
            end process DO_2XN_NUX; 
 
         
       end generate GEN_2XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_4XN
    --
    -- If Generate Description:
    --  4 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_4XN : if (NUM_MUX_CHANNELS = 4) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_4XN_NUX
         --
         -- Process Description:
         --  Implement the 4XN Mux
         --
         -------------------------------------------------------------
         DO_4XN_NUX : process (sig_mux_sel_int_local,
                               mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
               when 0 =>
                   sig_mux_dout <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0);
               when 1 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1);
               when 2 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2);
               
               when others => -- 3 case
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3);
             end case;
             
           end process DO_4XN_NUX; 
  
         
       end generate GEN_4XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_8XN
    --
    -- If Generate Description:
    --  8 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_8XN : if (NUM_MUX_CHANNELS = 8) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_8XN_NUX
         --
         -- Process Description:
         --  Implement the 8XN Mux
         --
         -------------------------------------------------------------
         DO_8XN_NUX : process (sig_mux_sel_int_local,
                               mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
               when 0 =>
                   sig_mux_dout <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0);
               when 1 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1);
               when 2 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2);
               when 3 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3);
               when 4 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*5)-1 downto CHANNEL_DWIDTH*4);
               when 5 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*6)-1 downto CHANNEL_DWIDTH*5);
               when 6 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*7)-1 downto CHANNEL_DWIDTH*6);
               
               when others =>  -- 7 case
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*8)-1 downto CHANNEL_DWIDTH*7);
             end case;
                 
           end process DO_8XN_NUX; 
 
         
       end generate GEN_8XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_16XN
    --
    -- If Generate Description:
    --  16 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_16XN : if (NUM_MUX_CHANNELS = 16) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_16XN_NUX
         --
         -- Process Description:
         --  Implement the 16XN Mux
         --
         -------------------------------------------------------------
         DO_16XN_NUX : process (sig_mux_sel_int_local,
                                mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
               when 0 =>
                   sig_mux_dout <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0);
               when 1 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1);
               when 2 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2);
               when 3 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3);
               when 4 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*5)-1 downto CHANNEL_DWIDTH*4);
               when 5 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*6)-1 downto CHANNEL_DWIDTH*5);
               when 6 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*7)-1 downto CHANNEL_DWIDTH*6);
               when 7 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*8)-1 downto CHANNEL_DWIDTH*7);
               when 8 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*9)-1 downto CHANNEL_DWIDTH*8);
               when 9 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*10)-1 downto CHANNEL_DWIDTH*9);
               when 10 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*11)-1 downto CHANNEL_DWIDTH*10);
               when 11 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*12)-1 downto CHANNEL_DWIDTH*11);
               when 12 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*13)-1 downto CHANNEL_DWIDTH*12);
               when 13 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*14)-1 downto CHANNEL_DWIDTH*13);
               when 14 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*15)-1 downto CHANNEL_DWIDTH*14);
               
               when others => -- 15 case
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*16)-1 downto CHANNEL_DWIDTH*15);
             end case;
          
           end process DO_16XN_NUX; 
 
         
       end generate GEN_16XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_32XN
    --
    -- If Generate Description:
    --  32 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_32XN : if (NUM_MUX_CHANNELS = 32) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_32XN_NUX
         --
         -- Process Description:
         --  Implement the 32XN Mux
         --
         -------------------------------------------------------------
         DO_32XN_NUX : process (sig_mux_sel_int_local,
                                mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
               
               when 0 =>
                   sig_mux_dout <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0);
               when 1 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1);
               when 2 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2);
               when 3 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3);
               when 4 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*5)-1 downto CHANNEL_DWIDTH*4);
               when 5 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*6)-1 downto CHANNEL_DWIDTH*5);
               when 6 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*7)-1 downto CHANNEL_DWIDTH*6);
               when 7 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*8)-1 downto CHANNEL_DWIDTH*7);
               when 8 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*9)-1 downto CHANNEL_DWIDTH*8);
               when 9 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*10)-1 downto CHANNEL_DWIDTH*9);
               when 10 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*11)-1 downto CHANNEL_DWIDTH*10);
               when 11 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*12)-1 downto CHANNEL_DWIDTH*11);
               when 12 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*13)-1 downto CHANNEL_DWIDTH*12);
               when 13 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*14)-1 downto CHANNEL_DWIDTH*13);
               when 14 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*15)-1 downto CHANNEL_DWIDTH*14);
               when 15 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*16)-1 downto CHANNEL_DWIDTH*15);
               when 16 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*17)-1 downto CHANNEL_DWIDTH*16);
               when 17 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*18)-1 downto CHANNEL_DWIDTH*17);
               when 18 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*19)-1 downto CHANNEL_DWIDTH*18);
               when 19 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*20)-1 downto CHANNEL_DWIDTH*19);
               when 20 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*21)-1 downto CHANNEL_DWIDTH*20);
               when 21 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*22)-1 downto CHANNEL_DWIDTH*21);
               when 22 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*23)-1 downto CHANNEL_DWIDTH*22);
               when 23 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*24)-1 downto CHANNEL_DWIDTH*23);
               when 24 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*25)-1 downto CHANNEL_DWIDTH*24);
               when 25 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*26)-1 downto CHANNEL_DWIDTH*25);
               when 26 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*27)-1 downto CHANNEL_DWIDTH*26);
               when 27 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*28)-1 downto CHANNEL_DWIDTH*27);
               when 28 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*29)-1 downto CHANNEL_DWIDTH*28);
               when 29 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*30)-1 downto CHANNEL_DWIDTH*29);
               when 30 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*31)-1 downto CHANNEL_DWIDTH*30);
               
               when others => -- 31 case
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*32)-1 downto CHANNEL_DWIDTH*31);
             end case;
          
           end process DO_32XN_NUX; 
 
         
       end generate GEN_32XN;
  
 
 
 
 
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_64XN
    --
    -- If Generate Description:
    --  64 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_64XN : if (NUM_MUX_CHANNELS = 64) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0)          := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_64XN_NUX
         --
         -- Process Description:
         --  Implement the 64XN Mux
         --
         -------------------------------------------------------------
         DO_64XN_NUX : process (sig_mux_sel_int_local,
                                mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
             
               when 0 =>
                   sig_mux_dout  <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0)                     ;
               when 1 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1)  ;
               when 2 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2)  ;
               when 3 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3)  ;
               when 4 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*5)-1 downto CHANNEL_DWIDTH*4)  ;
               when 5 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*6)-1 downto CHANNEL_DWIDTH*5)  ;
               when 6 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*7)-1 downto CHANNEL_DWIDTH*6)  ;
               when 7 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*8)-1 downto CHANNEL_DWIDTH*7)  ;
               when 8 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*9)-1 downto CHANNEL_DWIDTH*8)  ;
               when 9 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*10)-1 downto CHANNEL_DWIDTH*9) ;
               when 10 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*11)-1 downto CHANNEL_DWIDTH*10);
               when 11 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*12)-1 downto CHANNEL_DWIDTH*11);
               when 12 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*13)-1 downto CHANNEL_DWIDTH*12);
               when 13 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*14)-1 downto CHANNEL_DWIDTH*13);
               when 14 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*15)-1 downto CHANNEL_DWIDTH*14);
               when 15 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*16)-1 downto CHANNEL_DWIDTH*15);
               when 16 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*17)-1 downto CHANNEL_DWIDTH*16);
               when 17 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*18)-1 downto CHANNEL_DWIDTH*17);
               when 18 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*19)-1 downto CHANNEL_DWIDTH*18);
               when 19 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*20)-1 downto CHANNEL_DWIDTH*19);
               when 20 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*21)-1 downto CHANNEL_DWIDTH*20);
               when 21 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*22)-1 downto CHANNEL_DWIDTH*21);
               when 22 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*23)-1 downto CHANNEL_DWIDTH*22);
               when 23 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*24)-1 downto CHANNEL_DWIDTH*23);
               when 24 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*25)-1 downto CHANNEL_DWIDTH*24);
               when 25 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*26)-1 downto CHANNEL_DWIDTH*25);
               when 26 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*27)-1 downto CHANNEL_DWIDTH*26);
               when 27 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*28)-1 downto CHANNEL_DWIDTH*27);
               when 28 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*29)-1 downto CHANNEL_DWIDTH*28);
               when 29 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*30)-1 downto CHANNEL_DWIDTH*29);
               when 30 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*31)-1 downto CHANNEL_DWIDTH*30);
               when 31 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*32)-1 downto CHANNEL_DWIDTH*31);
             
             
               when 32 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*33)-1 downto CHANNEL_DWIDTH*32);
               when 33 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*34)-1 downto CHANNEL_DWIDTH*33);
               when 34 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*35)-1 downto CHANNEL_DWIDTH*34);
               when 35 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*36)-1 downto CHANNEL_DWIDTH*35);
               when 36 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*37)-1 downto CHANNEL_DWIDTH*36);
               when 37 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*38)-1 downto CHANNEL_DWIDTH*37);
               when 38 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*39)-1 downto CHANNEL_DWIDTH*38);
               when 39 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*40)-1 downto CHANNEL_DWIDTH*39);
               when 40 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*41)-1 downto CHANNEL_DWIDTH*40);
               when 41 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*42)-1 downto CHANNEL_DWIDTH*41);
               when 42 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*43)-1 downto CHANNEL_DWIDTH*42);
               when 43 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*44)-1 downto CHANNEL_DWIDTH*43);
               when 44 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*45)-1 downto CHANNEL_DWIDTH*44);
               when 45 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*46)-1 downto CHANNEL_DWIDTH*45);
               when 46 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*47)-1 downto CHANNEL_DWIDTH*46);
               when 47 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*48)-1 downto CHANNEL_DWIDTH*47);
               when 48 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*49)-1 downto CHANNEL_DWIDTH*48);
               when 49 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*50)-1 downto CHANNEL_DWIDTH*49);
               when 50 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*51)-1 downto CHANNEL_DWIDTH*50);
               when 51 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*52)-1 downto CHANNEL_DWIDTH*51);
               when 52 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*53)-1 downto CHANNEL_DWIDTH*52);
               when 53 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*54)-1 downto CHANNEL_DWIDTH*53);
               when 54 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*55)-1 downto CHANNEL_DWIDTH*54);
               when 55 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*56)-1 downto CHANNEL_DWIDTH*55);
               when 56 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*57)-1 downto CHANNEL_DWIDTH*56);
               when 57 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*58)-1 downto CHANNEL_DWIDTH*57);
               when 58 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*59)-1 downto CHANNEL_DWIDTH*58);
               when 59 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*60)-1 downto CHANNEL_DWIDTH*59);
               when 60 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*61)-1 downto CHANNEL_DWIDTH*60);
               when 61 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*62)-1 downto CHANNEL_DWIDTH*61);
               when 62 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*63)-1 downto CHANNEL_DWIDTH*62);
               
               when others => -- 63 case
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*64)-1 downto CHANNEL_DWIDTH*63);
             
             end case;
          
           end process DO_64XN_NUX; 
 
         
       end generate GEN_64XN;
  
 
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_128XN
    --
    -- If Generate Description:
    --  128 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_128XN : if (NUM_MUX_CHANNELS = 128) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0)          := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_128XN_NUX
         --
         -- Process Description:
         --  Implement the 64XN Mux
         --
         -------------------------------------------------------------
         DO_128XN_NUX : process (sig_mux_sel_int_local,
                                 mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
               
               when 0 =>
                   sig_mux_dout  <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0)                     ;
               when 1 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1)  ;
               when 2 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2)  ;
               when 3 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3)  ;
               when 4 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*5)-1 downto CHANNEL_DWIDTH*4)  ;
               when 5 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*6)-1 downto CHANNEL_DWIDTH*5)  ;
               when 6 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*7)-1 downto CHANNEL_DWIDTH*6)  ;
               when 7 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*8)-1 downto CHANNEL_DWIDTH*7)  ;
               when 8 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*9)-1 downto CHANNEL_DWIDTH*8)  ;
               when 9 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*10)-1 downto CHANNEL_DWIDTH*9) ;
               when 10 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*11)-1 downto CHANNEL_DWIDTH*10);
               when 11 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*12)-1 downto CHANNEL_DWIDTH*11);
               when 12 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*13)-1 downto CHANNEL_DWIDTH*12);
               when 13 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*14)-1 downto CHANNEL_DWIDTH*13);
               when 14 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*15)-1 downto CHANNEL_DWIDTH*14);
               when 15 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*16)-1 downto CHANNEL_DWIDTH*15);
               when 16 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*17)-1 downto CHANNEL_DWIDTH*16);
               when 17 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*18)-1 downto CHANNEL_DWIDTH*17);
               when 18 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*19)-1 downto CHANNEL_DWIDTH*18);
               when 19 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*20)-1 downto CHANNEL_DWIDTH*19);
               when 20 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*21)-1 downto CHANNEL_DWIDTH*20);
               when 21 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*22)-1 downto CHANNEL_DWIDTH*21);
               when 22 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*23)-1 downto CHANNEL_DWIDTH*22);
               when 23 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*24)-1 downto CHANNEL_DWIDTH*23);
               when 24 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*25)-1 downto CHANNEL_DWIDTH*24);
               when 25 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*26)-1 downto CHANNEL_DWIDTH*25);
               when 26 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*27)-1 downto CHANNEL_DWIDTH*26);
               when 27 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*28)-1 downto CHANNEL_DWIDTH*27);
               when 28 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*29)-1 downto CHANNEL_DWIDTH*28);
               when 29 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*30)-1 downto CHANNEL_DWIDTH*29);
               when 30 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*31)-1 downto CHANNEL_DWIDTH*30);
               when 31 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*32)-1 downto CHANNEL_DWIDTH*31);
             
             
               when 32 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*33)-1 downto CHANNEL_DWIDTH*32);
               when 33 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*34)-1 downto CHANNEL_DWIDTH*33);
               when 34 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*35)-1 downto CHANNEL_DWIDTH*34);
               when 35 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*36)-1 downto CHANNEL_DWIDTH*35);
               when 36 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*37)-1 downto CHANNEL_DWIDTH*36);
               when 37 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*38)-1 downto CHANNEL_DWIDTH*37);
               when 38 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*39)-1 downto CHANNEL_DWIDTH*38);
               when 39 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*40)-1 downto CHANNEL_DWIDTH*39);
               when 40 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*41)-1 downto CHANNEL_DWIDTH*40);
               when 41 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*42)-1 downto CHANNEL_DWIDTH*41);
               when 42 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*43)-1 downto CHANNEL_DWIDTH*42);
               when 43 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*44)-1 downto CHANNEL_DWIDTH*43);
               when 44 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*45)-1 downto CHANNEL_DWIDTH*44);
               when 45 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*46)-1 downto CHANNEL_DWIDTH*45);
               when 46 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*47)-1 downto CHANNEL_DWIDTH*46);
               when 47 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*48)-1 downto CHANNEL_DWIDTH*47);
               when 48 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*49)-1 downto CHANNEL_DWIDTH*48);
               when 49 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*50)-1 downto CHANNEL_DWIDTH*49);
               when 50 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*51)-1 downto CHANNEL_DWIDTH*50);
               when 51 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*52)-1 downto CHANNEL_DWIDTH*51);
               when 52 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*53)-1 downto CHANNEL_DWIDTH*52);
               when 53 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*54)-1 downto CHANNEL_DWIDTH*53);
               when 54 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*55)-1 downto CHANNEL_DWIDTH*54);
               when 55 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*56)-1 downto CHANNEL_DWIDTH*55);
               when 56 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*57)-1 downto CHANNEL_DWIDTH*56);
               when 57 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*58)-1 downto CHANNEL_DWIDTH*57);
               when 58 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*59)-1 downto CHANNEL_DWIDTH*58);
               when 59 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*60)-1 downto CHANNEL_DWIDTH*59);
               when 60 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*61)-1 downto CHANNEL_DWIDTH*60);
               when 61 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*62)-1 downto CHANNEL_DWIDTH*61);
               when 62 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*63)-1 downto CHANNEL_DWIDTH*62);
               when 63 => 
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*64)-1 downto CHANNEL_DWIDTH*63);
             
             
               when 64 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*65)-1 downto CHANNEL_DWIDTH*64) ;
               when 65 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*66)-1 downto CHANNEL_DWIDTH*65) ;
               when 66 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*67)-1 downto CHANNEL_DWIDTH*66) ;
               when 67 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*68)-1 downto CHANNEL_DWIDTH*67) ;
               when 68 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*69)-1 downto CHANNEL_DWIDTH*68) ;
               when 69 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*70)-1 downto CHANNEL_DWIDTH*69) ;
               when 70 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*71)-1 downto CHANNEL_DWIDTH*70) ;
               when 71 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*72)-1 downto CHANNEL_DWIDTH*71) ;
               when 72 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*73)-1 downto CHANNEL_DWIDTH*72) ;
               when 73 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*74)-1 downto CHANNEL_DWIDTH*73) ;
               when 74 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*75)-1 downto CHANNEL_DWIDTH*74) ;
               when 75 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*76)-1 downto CHANNEL_DWIDTH*75) ;
               when 76 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*77)-1 downto CHANNEL_DWIDTH*76) ;
               when 77 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*78)-1 downto CHANNEL_DWIDTH*77) ;
               when 78 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*79)-1 downto CHANNEL_DWIDTH*78) ;
               when 79 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*80)-1 downto CHANNEL_DWIDTH*79) ;
               when 80 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*81)-1 downto CHANNEL_DWIDTH*80) ;
               when 81 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*82)-1 downto CHANNEL_DWIDTH*81) ;
               when 82 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*83)-1 downto CHANNEL_DWIDTH*82) ;
               when 83 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*84)-1 downto CHANNEL_DWIDTH*83) ;
               when 84 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*85)-1 downto CHANNEL_DWIDTH*84) ;
               when 85 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*86)-1 downto CHANNEL_DWIDTH*85) ;
               when 86 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*87)-1 downto CHANNEL_DWIDTH*86) ;
               when 87 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*88)-1 downto CHANNEL_DWIDTH*87) ;
               when 88 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*89)-1 downto CHANNEL_DWIDTH*88) ;
               when 89 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*90)-1 downto CHANNEL_DWIDTH*89) ;
               when 90 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*91)-1 downto CHANNEL_DWIDTH*90) ;
               when 91 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*92)-1 downto CHANNEL_DWIDTH*91) ;
               when 92 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*93)-1 downto CHANNEL_DWIDTH*92) ;
               when 93 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*94)-1 downto CHANNEL_DWIDTH*93) ;
               when 94 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*95)-1 downto CHANNEL_DWIDTH*94) ;
               when 95 =>                                                                 
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*96)-1 downto CHANNEL_DWIDTH*95) ;
             
             
               when 96 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*97 )-1 downto CHANNEL_DWIDTH*96 ) ;
               when 97 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*98 )-1 downto CHANNEL_DWIDTH*97 ) ;
               when 98 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*99 )-1 downto CHANNEL_DWIDTH*98 ) ;
               when 99 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*100)-1 downto CHANNEL_DWIDTH*99 ) ;
               when 100 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*101)-1 downto CHANNEL_DWIDTH*100) ;
               when 101 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*102)-1 downto CHANNEL_DWIDTH*101) ;
               when 102 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*103)-1 downto CHANNEL_DWIDTH*102) ;
               when 103 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*104)-1 downto CHANNEL_DWIDTH*103) ;
               when 104 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*105)-1 downto CHANNEL_DWIDTH*104) ;
               when 105 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*106)-1 downto CHANNEL_DWIDTH*105) ;
               when 106 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*107)-1 downto CHANNEL_DWIDTH*106) ;
               when 107 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*108)-1 downto CHANNEL_DWIDTH*107) ;
               when 108 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*109)-1 downto CHANNEL_DWIDTH*108) ;
               when 109 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*110)-1 downto CHANNEL_DWIDTH*109) ;
               when 110 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*111)-1 downto CHANNEL_DWIDTH*110) ;
               when 111 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*112)-1 downto CHANNEL_DWIDTH*111) ;
               when 112 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*113)-1 downto CHANNEL_DWIDTH*112) ;
               when 113 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*114)-1 downto CHANNEL_DWIDTH*113) ;
               when 114 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*115)-1 downto CHANNEL_DWIDTH*114) ;
               when 115 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*116)-1 downto CHANNEL_DWIDTH*115) ;
               when 116 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*117)-1 downto CHANNEL_DWIDTH*116) ;
               when 117 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*118)-1 downto CHANNEL_DWIDTH*117) ;
               when 118 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*119)-1 downto CHANNEL_DWIDTH*118) ;
               when 119 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*120)-1 downto CHANNEL_DWIDTH*119) ;
               when 120 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*121)-1 downto CHANNEL_DWIDTH*120) ;
               when 121 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*122)-1 downto CHANNEL_DWIDTH*121) ;
               when 122 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*123)-1 downto CHANNEL_DWIDTH*122) ;
               when 123 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*124)-1 downto CHANNEL_DWIDTH*123) ;
               when 124 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*125)-1 downto CHANNEL_DWIDTH*124) ;
               when 125 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*126)-1 downto CHANNEL_DWIDTH*125) ;
               when 126 =>                                                                 
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*127)-1 downto CHANNEL_DWIDTH*126) ;
               
               when others => -- 127 case
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*128)-1 downto CHANNEL_DWIDTH*127) ;
             
             
             end case;
          
           end process DO_128XN_NUX; 
 
         
       end generate GEN_128XN;
  
 
  
  
  
  
  
  
  end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_rddata_cntl.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_rddata_cntl.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Read Data Controller.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  library axi_datamover_v5_1_14; 
  use axi_datamover_v5_1_14.axi_datamover_rdmux;  
  
  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_rddata_cntl is
    generic (
      
      C_INCLUDE_DRE          : Integer range  0 to   1 :=  0;
        -- Indicates if the DRE interface is used
        
      C_ALIGN_WIDTH          : Integer range  1 to   6 :=  3;
        -- Sets the width of the DRE Alignment controls
        
      C_SEL_ADDR_WIDTH       : Integer range  1 to   8 :=  5;
        -- Sets the width of the LS bits of the transfer address that
        -- are being used to Mux read data from a wider AXI4 Read
        -- Data Bus
        
      C_DATA_CNTL_FIFO_DEPTH : Integer range  1 to  32 :=  4;
        -- Sets the depth of the internal command fifo used for the
        -- command queue
        
      C_MMAP_DWIDTH          : Integer range 32 to 1024 := 32;
        -- Indicates the native data width of the Read Data port
        
      C_STREAM_DWIDTH        : Integer range  8 to 1024 := 32;
        -- Sets the width of the Stream output data port
        
      C_TAG_WIDTH            : Integer range  1 to   8 :=  4;
        -- Indicates the width of the Tag field of the input command

    C_ENABLE_MM2S_TKEEP             : integer range 0 to 1 := 1; 
        
      C_FAMILY               : String                  := "virtex7"
        -- Indicates the device family of the target FPGA
        
      
      );
    port (
      
      -- Clock and Reset inputs ----------------------------------------
                                                                      --
      primary_aclk          : in  std_logic;                          --
         -- Primary synchronization clock for the Master side         --
         -- interface and internal logic. It is also used             --
         -- for the User interface synchronization when               --
         -- C_STSCMD_IS_ASYNC = 0.                                    --
                                                                      --
      -- Reset input                                                  --
      mmap_reset            : in  std_logic;                          --
         -- Reset used for the internal master logic                  --
      ------------------------------------------------------------------
      
     
      
      -- Soft Shutdown internal interface -----------------------------------
                                                                           --
      rst2data_stop_request : in  std_logic;                               --
         -- Active high soft stop request to modules                       --
                                                                           --
      data2addr_stop_req    : Out std_logic;                               --
        -- Active high signal requesting the Address Controller            --
        -- to stop posting commands to the AXI Read Address Channel        --
                                                                           --
      data2rst_stop_cmplt   : Out std_logic;                               --
        -- Active high indication that the Data Controller has completed   --
        -- any pending transfers committed by the Address Controller       --
        -- after a stop has been requested by the Reset module.            --
      -----------------------------------------------------------------------
   
   
        
      -- External Address Pipelining Contol support -------------------------
                                                                           --
      mm2s_rd_xfer_cmplt    : out std_logic;                               --
        -- Active high indication that the Data Controller has completed   --
        -- a single read data transfer on the AXI4 Read Data Channel.      --
        -- This signal escentially echos the assertion of rlast received   --
        -- from the AXI4.                                                  --
      -----------------------------------------------------------------------
      
      
      
        
     -- AXI Read Data Channel I/O  ---------------------------------------------
                                                                              --
      mm2s_rdata            : In  std_logic_vector(C_MMAP_DWIDTH-1 downto 0); --
        -- AXI Read data input                                                --
                                                                              --
      mm2s_rresp            : In  std_logic_vector(1 downto 0);               --
        -- AXI Read response input                                            --
                                                                              --
      mm2s_rlast            : In  std_logic;                                  --
        -- AXI Read LAST input                                                --
                                                                              --
      mm2s_rvalid           : In  std_logic;                                  --
        -- AXI Read VALID input                                               --
                                                                              --
      mm2s_rready           : Out std_logic;                                  --
        -- AXI Read data READY output                                         --
      --------------------------------------------------------------------------
               
                
                
                
     -- MM2S DRE Control  -------------------------------------------------------------
                                                                                     --
      mm2s_dre_new_align      : Out std_logic;                                       --
        -- Active high signal indicating new DRE aligment required                   --
                                                                                     --
      mm2s_dre_use_autodest   : Out std_logic;                                       --
        -- Active high signal indicating to the DRE to use an auto-                  --
        -- calculated desination alignment based on the last transfer                --
                                                                                     --
      mm2s_dre_src_align      : Out std_logic_vector(C_ALIGN_WIDTH-1 downto 0);      --
        -- Bit field indicating the byte lane of the first valid data byte           --
        -- being sent to the DRE                                                     --
                                                                                     --
      mm2s_dre_dest_align     : Out std_logic_vector(C_ALIGN_WIDTH-1 downto 0);      --
        -- Bit field indicating the desired byte lane of the first valid data byte   --
        -- to be output by the DRE                                                   --
                                                                                     --
      mm2s_dre_flush          : Out std_logic;                                       --
        -- Active high signal indicating to the DRE to flush the current             --
        -- contents to the output register in preparation of a new alignment         --
        -- that will be comming on the next transfer input                           --
      ---------------------------------------------------------------------------------
               
                
                
                
     -- AXI Master Stream Channel------------------------------------------------------
                                                                                     --
      mm2s_strm_wvalid   : Out std_logic;                                            --
        -- AXI Stream VALID Output                                                   --
                                                                                     --
      mm2s_strm_wready   : In  Std_logic;                                            --
        -- AXI Stream READY input                                                    --
                                                                                     --
      mm2s_strm_wdata    : Out std_logic_vector(C_STREAM_DWIDTH-1 downto 0);         --
        -- AXI Stream data output                                                    --
                                                                                     --
      mm2s_strm_wstrb    : Out std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);     --    
        -- AXI Stream STRB output                                                    --
                                                                                     --
      mm2s_strm_wlast    : Out std_logic;                                            --
        -- AXI Stream LAST output                                                    --
      ---------------------------------------------------------------------------------
               
                
      
      -- MM2S Store and Forward Supplimental Control   --------------------------------
      -- This output is time aligned and qualified with the AXI Master Stream Channel--
                                                                                     --
      mm2s_data2sf_cmd_cmplt   : out std_logic;                                      --
                                                                                     --
      ---------------------------------------------------------------------------------
                                                                                     
                                                                                     
                                                                                     
                                                                                     
                                                                                     
                
                
      -- Command Calculator Interface -------------------------------------------------
                                                                                     --
      mstr2data_tag        : In std_logic_vector(C_TAG_WIDTH-1 downto 0);            --
         -- The next command tag                                                     --
                                                                                     --
      mstr2data_saddr_lsb  : In std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0);       --
         -- The next command start address LSbs to use for the read data             --
         -- mux (only used if Stream data width is 8 or 16 bits).                    --
                                                                                     --
      mstr2data_len        : In std_logic_vector(7 downto 0);                        --
         -- The LEN value output to the Address Channel                              --
                                                                                     --
      mstr2data_strt_strb  : In std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);    --
         -- The starting strobe value to use for the first stream data beat          --
                                                                                     --
      mstr2data_last_strb  : In std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);    --
         -- The endiing (LAST) strobe value to use for the last stream               --
         -- data beat                                                                --
                                                                                     --
      mstr2data_drr        : In std_logic;                                           --
         -- The starting tranfer of a sequence of transfers                          --
                                                                                     --
      mstr2data_eof        : In std_logic;                                           --
         -- The endiing tranfer of a sequence of transfers                           --
                                                                                     --
      mstr2data_sequential : In std_logic;                                           --
         -- The next sequential tranfer of a sequence of transfers                   --
         -- spawned from a single parent command                                     --
                                                                                     --
      mstr2data_calc_error : In std_logic;                                           --
         -- Indication if the next command in the calculation pipe                   --
         -- has a calculation error                                                  --
                                                                                     --
      mstr2data_cmd_cmplt  : In std_logic;                                           --
         -- The indication to the Data Channel that the current                      --
         -- sub-command output is the last one compiled from the                     --
         -- parent command pulled from the Command FIFO                              --
                                                                                     --
      mstr2data_cmd_valid  : In std_logic;                                           --
         -- The next command valid indication to the Data Channel                    --
         -- Controller for the AXI MMap                                              --
                                                                                     --
      data2mstr_cmd_ready  : Out std_logic ;                                         --
         -- Indication from the Data Channel Controller that the                     --
         -- command is being accepted on the AXI Address Channel                     --
                                                                                     --
      mstr2data_dre_src_align   : In std_logic_vector(C_ALIGN_WIDTH-1 downto 0);     --
         -- The source (input) alignment for the DRE                                 --
                                                                                     --
      mstr2data_dre_dest_align  : In std_logic_vector(C_ALIGN_WIDTH-1 downto 0);     --
         -- The destinstion (output) alignment for the DRE                           --
      ---------------------------------------------------------------------------------
     
      
      
      
        
      -- Address Controller Interface -------------------------------------------------
                                                                                     --
      addr2data_addr_posted : In std_logic ;                                         --
         -- Indication from the Address Channel Controller to the                    --
         -- Data Controller that an address has been posted to the                   --
         -- AXI Address Channel                                                      --
      ---------------------------------------------------------------------------------


      
      -- Data Controller General Halted Status ----------------------------------------
                                                                                     --
      data2all_dcntlr_halted : Out std_logic;                                        --
         -- When asserted, this indicates the data controller has satisfied          --
         -- all pending transfers queued by the Address Controller and is halted.    --
      ---------------------------------------------------------------------------------
      
       
 
      -- Output Stream Skid Buffer Halt control ---------------------------------------
                                                                                     --
      data2skid_halt : Out std_logic;                                                --
         -- The data controller asserts this output for 1 primary clock period       --
         -- The pulse commands the MM2S Stream skid buffer to tun off outputs        --
         -- at the next tlast transmission.                                          --
      ---------------------------------------------------------------------------------
      
       
 
       
      -- Read Status Controller Interface ------------------------------------------------
                                                                                        --
      data2rsc_tag       : Out std_logic_vector(C_TAG_WIDTH-1 downto 0);                --
         -- The propagated command tag from the Command Calculator                      --
                                                                                        --
      data2rsc_calc_err  : Out std_logic ;                                              --
         -- Indication that the current command out from the Cntl FIFO                  --
         -- has a propagated calculation error from the Command Calculator              --
                                                                                        --
      data2rsc_okay      : Out std_logic ;                                              --
         -- Indication that the AXI Read transfer completed with OK status              --
                                                                                        --
      data2rsc_decerr    : Out std_logic ;                                              --
         -- Indication that the AXI Read transfer completed with decode error status    --
                                                                                        --
      data2rsc_slverr    : Out std_logic ;                                              --
         -- Indication that the AXI Read transfer completed with slave error status     --
                                                                                        --
      data2rsc_cmd_cmplt : Out std_logic ;                                              --
         -- Indication by the Data Channel Controller that the                          --
         -- corresponding status is the last status for a parent command                --
         -- pulled from the command FIFO                                                --
                                                                                        --
      rsc2data_ready     : in  std_logic;                                               --
         -- Handshake bit from the Read Status Controller Module indicating             --
         -- that the it is ready to accept a new Read status transfer                   --
                                                                                        --
      data2rsc_valid     : Out  std_logic ;                                             --
         -- Handshake bit output to the Read Status Controller Module                   --
         -- indicating that the Data Controller has valid tag and status                --
         -- indicators to transfer                                                      --
                                                                                        --
      rsc2mstr_halt_pipe : In std_logic                                                 --
         -- Status Flag indicating the Status Controller needs to stall the command     --
         -- execution pipe due to a Status flow issue or internal error. Generally      --
         -- this will occur if the Status FIFO is not being serviced fast enough to     --
         -- keep ahead of the command execution.                                        --
      ------------------------------------------------------------------------------------
      
      );
  
  end entity axi_datamover_rddata_cntl;
  
  
  architecture implementation of axi_datamover_rddata_cntl is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    
    -- Function declaration   ----------------------------------------
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_set_cnt_width
    --
    -- Function Description:
    --    Sets a count width based on a fifo depth. A depth of 4 or less
    -- is a special case which requires a minimum count width of 3 bits.
    --
    -------------------------------------------------------------------
    function funct_set_cnt_width (fifo_depth : integer) return integer is
    
      Variable temp_cnt_width : Integer := 4;
    
    begin
    
      
      if (fifo_depth <= 4) then
      
         temp_cnt_width := 3;
      
      elsif (fifo_depth <= 8) then
      
         temp_cnt_width := 4;
      
      elsif (fifo_depth <= 16) then
      
         temp_cnt_width := 5;
      
      elsif (fifo_depth <= 32) then
      
         temp_cnt_width := 6;
      
      else  -- fifo depth <= 64
      
         temp_cnt_width := 7;
      
      end if;
      
      Return (temp_cnt_width);
       
       
    end function funct_set_cnt_width;
    
 
 
 
  
    -- Constant Declarations  --------------------------------------------
    
    Constant OKAY                   : std_logic_vector(1 downto 0) := "00";
    Constant EXOKAY                 : std_logic_vector(1 downto 0) := "01";
    Constant SLVERR                 : std_logic_vector(1 downto 0) := "10";
    Constant DECERR                 : std_logic_vector(1 downto 0) := "11";
                                    
    Constant STRM_STRB_WIDTH        : integer := C_STREAM_DWIDTH/8;
    Constant LEN_OF_ZERO            : std_logic_vector(7 downto 0) := (others => '0');
    Constant USE_SYNC_FIFO          : integer := 0;
    Constant REG_FIFO_PRIM          : integer := 0; 
    Constant BRAM_FIFO_PRIM         : integer := 1; 
    Constant SRL_FIFO_PRIM          : integer := 2; 
    Constant FIFO_PRIM_TYPE         : integer := SRL_FIFO_PRIM; 
    Constant TAG_WIDTH              : integer := C_TAG_WIDTH;
    Constant SADDR_LSB_WIDTH        : integer := C_SEL_ADDR_WIDTH;
    Constant LEN_WIDTH              : integer := 8;
    Constant STRB_WIDTH             : integer := C_STREAM_DWIDTH/8;
    Constant SOF_WIDTH              : integer := 1;
    Constant EOF_WIDTH              : integer := 1;
    Constant CMD_CMPLT_WIDTH        : integer := 1;
    Constant SEQUENTIAL_WIDTH       : integer := 1;
    Constant CALC_ERR_WIDTH         : integer := 1;
    Constant DRE_ALIGN_WIDTH        : integer := C_ALIGN_WIDTH;
                                    
    Constant DCTL_FIFO_WIDTH        : Integer := TAG_WIDTH        +  -- Tag field
                                                 SADDR_LSB_WIDTH  +  -- LS Address field width
                                                 LEN_WIDTH        +  -- LEN field
                                                 STRB_WIDTH       +  -- Starting Strobe field
                                                 STRB_WIDTH       +  -- Ending Strobe field
                                                 SOF_WIDTH        +  -- SOF Flag Field
                                                 EOF_WIDTH        +  -- EOF flag field
                                                 SEQUENTIAL_WIDTH +  -- Calc error flag
                                                 CMD_CMPLT_WIDTH  +  -- Sequential command flag
                                                 CALC_ERR_WIDTH   +  -- Command Complete Flag
                                                 DRE_ALIGN_WIDTH  +  -- DRE Source Align width
                                                 DRE_ALIGN_WIDTH ;   -- DRE Dest Align width
                                                 
                                    
    -- Caution, the INDEX calculations are order dependent so don't rearrange
    Constant TAG_STRT_INDEX         : integer := 0;
    Constant SADDR_LSB_STRT_INDEX   : integer := TAG_STRT_INDEX + TAG_WIDTH;
    Constant LEN_STRT_INDEX         : integer := SADDR_LSB_STRT_INDEX + SADDR_LSB_WIDTH;
    Constant STRT_STRB_STRT_INDEX   : integer := LEN_STRT_INDEX + LEN_WIDTH;
    Constant LAST_STRB_STRT_INDEX   : integer := STRT_STRB_STRT_INDEX + STRB_WIDTH;
    Constant SOF_STRT_INDEX         : integer := LAST_STRB_STRT_INDEX + STRB_WIDTH;
    Constant EOF_STRT_INDEX         : integer := SOF_STRT_INDEX + SOF_WIDTH;
    Constant SEQUENTIAL_STRT_INDEX  : integer := EOF_STRT_INDEX + EOF_WIDTH;
    Constant CMD_CMPLT_STRT_INDEX   : integer := SEQUENTIAL_STRT_INDEX + SEQUENTIAL_WIDTH;
    Constant CALC_ERR_STRT_INDEX    : integer := CMD_CMPLT_STRT_INDEX + CMD_CMPLT_WIDTH;
    Constant DRE_SRC_STRT_INDEX     : integer := CALC_ERR_STRT_INDEX + CALC_ERR_WIDTH;
    Constant DRE_DEST_STRT_INDEX    : integer := DRE_SRC_STRT_INDEX + DRE_ALIGN_WIDTH;
    
    Constant ADDR_INCR_VALUE        : integer := C_STREAM_DWIDTH/8;
    
    --Constant ADDR_POSTED_CNTR_WIDTH : integer := 5; -- allows up to 32 entry address queue
    Constant ADDR_POSTED_CNTR_WIDTH : integer := funct_set_cnt_width(C_DATA_CNTL_FIFO_DEPTH); 
    
    
    Constant ADDR_POSTED_ZERO       : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := (others => '0');
    Constant ADDR_POSTED_ONE        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := TO_UNSIGNED(1, ADDR_POSTED_CNTR_WIDTH);
    Constant ADDR_POSTED_MAX        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                       := (others => '1');
                    
    
    
    
    -- Signal Declarations  --------------------------------------------
    
    signal sig_good_dbeat               : std_logic := '0';
    signal sig_get_next_dqual           : std_logic := '0';
    signal sig_last_mmap_dbeat          : std_logic := '0';
    signal sig_last_mmap_dbeat_reg      : std_logic := '0';
    signal sig_data2mmap_ready          : std_logic := '0';
    signal sig_mmap2data_valid          : std_logic := '0';
    signal sig_mmap2data_last           : std_logic := '0';
    signal sig_aposted_cntr_ready       : std_logic := '0';
    signal sig_ld_new_cmd               : std_logic := '0';
    signal sig_ld_new_cmd_reg           : std_logic := '0';
    signal sig_cmd_cmplt_reg            : std_logic := '0';
    signal sig_tag_reg                  : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_lsb_reg             : std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_strt_strb_reg            : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_last_strb_reg            : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_addr_posted              : std_logic := '0';
    signal sig_addr_chan_rdy            : std_logic := '0';
    signal sig_dqual_rdy                : std_logic := '0';
    signal sig_good_mmap_dbeat          : std_logic := '0';
    signal sig_first_dbeat              : std_logic := '0';
    signal sig_last_dbeat               : std_logic := '0';
    signal sig_new_len_eq_0             : std_logic := '0';
    signal sig_dbeat_cntr               : unsigned(7 downto 0) := (others => '0');
    Signal sig_dbeat_cntr_int           : Integer range 0 to 255 := 0;
    signal sig_dbeat_cntr_eq_0          : std_logic := '0';
    signal sig_dbeat_cntr_eq_1          : std_logic := '0';
    signal sig_calc_error_reg           : std_logic := '0';
    signal sig_decerr                   : std_logic := '0';
    signal sig_slverr                   : std_logic := '0';
    signal sig_coelsc_okay_reg          : std_logic := '0';
    signal sig_coelsc_interr_reg        : std_logic := '0';
    signal sig_coelsc_decerr_reg        : std_logic := '0';
    signal sig_coelsc_slverr_reg        : std_logic := '0';
    signal sig_coelsc_cmd_cmplt_reg     : std_logic := '0';
    signal sig_coelsc_tag_reg           : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_pop_coelsc_reg           : std_logic := '0';
    signal sig_push_coelsc_reg          : std_logic := '0';
    signal sig_coelsc_reg_empty         : std_logic := '0';
    signal sig_coelsc_reg_full          : std_logic := '0';
    signal sig_rsc2data_ready           : std_logic := '0';
    signal sig_cmd_cmplt_last_dbeat     : std_logic := '0';
    signal sig_next_tag_reg             : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_strt_strb_reg       : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_next_last_strb_reg       : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_next_eof_reg             : std_logic := '0';
    signal sig_next_sequential_reg      : std_logic := '0';
    signal sig_next_cmd_cmplt_reg       : std_logic := '0';
    signal sig_next_calc_error_reg      : std_logic := '0';
    signal sig_next_dre_src_align_reg   : std_logic_vector(C_ALIGN_WIDTH-1 downto 0) := (others => '0');  
    signal sig_next_dre_dest_align_reg  : std_logic_vector(C_ALIGN_WIDTH-1 downto 0) := (others => '0'); 
    signal sig_pop_dqual_reg            : std_logic := '0';
    signal sig_push_dqual_reg           : std_logic := '0';
    signal sig_dqual_reg_empty          : std_logic := '0';
    signal sig_dqual_reg_full           : std_logic := '0';
    signal sig_addr_posted_cntr         : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_posted_cntr_eq_0    : std_logic := '0';
    signal sig_addr_posted_cntr_max     : std_logic := '0';
    signal sig_decr_addr_posted_cntr    : std_logic := '0';
    signal sig_incr_addr_posted_cntr    : std_logic := '0';
    signal sig_ls_addr_cntr             : unsigned(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_incr_ls_addr_cntr        : std_logic := '0';
    signal sig_addr_incr_unsgnd         : unsigned(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_no_posted_cmds           : std_logic := '0';
    Signal sig_cmd_fifo_data_in         : std_logic_vector(DCTL_FIFO_WIDTH-1 downto 0);
    Signal sig_cmd_fifo_data_out        : std_logic_vector(DCTL_FIFO_WIDTH-1 downto 0);
    signal sig_fifo_next_tag            : std_logic_vector(TAG_WIDTH-1 downto 0);
    signal sig_fifo_next_sadddr_lsb     : std_logic_vector(SADDR_LSB_WIDTH-1 downto 0);             
    signal sig_fifo_next_len            : std_logic_vector(LEN_WIDTH-1 downto 0);             
    signal sig_fifo_next_strt_strb      : std_logic_vector(STRB_WIDTH-1 downto 0);             
    signal sig_fifo_next_last_strb      : std_logic_vector(STRB_WIDTH-1 downto 0);             
    signal sig_fifo_next_drr            : std_logic := '0';
    signal sig_fifo_next_eof            : std_logic := '0';
    signal sig_fifo_next_cmd_cmplt      : std_logic := '0';
    signal sig_fifo_next_calc_error     : std_logic := '0';
    signal sig_fifo_next_sequential     : std_logic := '0';
    signal sig_fifo_next_dre_src_align  : std_logic_vector(C_ALIGN_WIDTH-1 downto 0) := (others => '0');  
    signal sig_fifo_next_dre_dest_align : std_logic_vector(C_ALIGN_WIDTH-1 downto 0) := (others => '0'); 
    signal sig_cmd_fifo_empty           : std_logic := '0';
    signal sig_fifo_wr_cmd_valid        : std_logic := '0';
    signal sig_fifo_wr_cmd_ready        : std_logic := '0';
    signal sig_fifo_rd_cmd_valid        : std_logic := '0';
    signal sig_fifo_rd_cmd_ready        : std_logic := '0';
    signal sig_sequential_push          : std_logic := '0';
    signal sig_clr_dqual_reg            : std_logic := '0';
    signal sig_advance_pipe             : std_logic := '0';
    signal sig_halt_reg                 : std_logic := '0';
    signal sig_halt_reg_dly1            : std_logic := '0';
    signal sig_halt_reg_dly2            : std_logic := '0';
    signal sig_halt_reg_dly3            : std_logic := '0';
    signal sig_data2skid_halt           : std_logic := '0';
    signal sig_rd_xfer_cmplt            : std_logic := '0';
    
                              
    
  begin --(architecture implementation)
  
    -- AXI MMap Data Channel Port assignments
    mm2s_rready          <= sig_data2mmap_ready;
    sig_mmap2data_valid  <= mm2s_rvalid        ;
    sig_mmap2data_last   <= mm2s_rlast         ;
    
    -- Read Status Block interface
    data2rsc_valid       <= sig_coelsc_reg_full      ;
    sig_rsc2data_ready   <= rsc2data_ready           ;
    
    data2rsc_tag         <= sig_coelsc_tag_reg       ;
    data2rsc_calc_err    <= sig_coelsc_interr_reg    ;
    data2rsc_okay        <= sig_coelsc_okay_reg      ;
    data2rsc_decerr      <= sig_coelsc_decerr_reg    ;
    data2rsc_slverr      <= sig_coelsc_slverr_reg    ;
    data2rsc_cmd_cmplt   <= sig_coelsc_cmd_cmplt_reg ;
    
    
                                                    
    -- AXI MM2S Stream Channel Port assignments               
    mm2s_strm_wvalid     <= (mm2s_rvalid             and         
                             sig_advance_pipe)       or
                            (sig_halt_reg            and  -- Force tvalid high on a Halt and
                             sig_dqual_reg_full      and  -- a transfer is scheduled and
                             not(sig_no_posted_cmds) and  -- there are cmds posted to AXi and
                             not(sig_calc_error_reg));    -- not a calc error       
    
            
                                                    
    mm2s_strm_wlast      <= (mm2s_rlast              and
                            sig_next_eof_reg)        or
                            (sig_halt_reg            and  -- Force tvalid high on a Halt and
                             sig_dqual_reg_full      and  -- a transfer is scheduled and
                             not(sig_no_posted_cmds) and  -- there are cmds posted to AXi and
                             not(sig_calc_error_reg));    -- not a calc error;        
    
    
     
    

GEN_MM2S_TKEEP_ENABLE5 : if C_ENABLE_MM2S_TKEEP = 1 generate
begin
   -- Generate the Write Strobes for the Stream interface
    mm2s_strm_wstrb <= (others => '1')
      When (sig_halt_reg = '1')        -- Force tstrb high on a Halt
      else sig_strt_strb_reg
      When (sig_first_dbeat = '1')
      Else sig_last_strb_reg
      When (sig_last_dbeat = '1')
      Else (others => '1');

 
end generate GEN_MM2S_TKEEP_ENABLE5;

GEN_MM2S_TKEEP_DISABLE5 : if C_ENABLE_MM2S_TKEEP = 0 generate
begin
   -- Generate the Write Strobes for the Stream interface
    mm2s_strm_wstrb <= (others => '1');


end generate GEN_MM2S_TKEEP_DISABLE5;

    
    
    
    -- MM2S Supplimental Controls
    mm2s_data2sf_cmd_cmplt <= (mm2s_rlast              and
                               sig_next_cmd_cmplt_reg) or
                              (sig_halt_reg            and  
                               sig_dqual_reg_full      and  
                               not(sig_no_posted_cmds) and  
                               not(sig_calc_error_reg));    
    
    
    
    
    
    
    -- Address Channel Controller synchro pulse input                  
    sig_addr_posted      <= addr2data_addr_posted;
                                                        
 
 
    -- Request to halt the Address Channel Controller                  
    data2addr_stop_req   <= sig_halt_reg;
 
    
    -- Halted flag to the reset module                  
    data2rst_stop_cmplt  <= (sig_halt_reg_dly3 and   -- Normal Mode shutdown
                            sig_no_posted_cmds and 
                            not(sig_calc_error_reg)) or
                            (sig_halt_reg_dly3 and   -- Shutdown after error trap
                             sig_calc_error_reg);
    
     
    
    -- Read Transfer Completed Status output
    mm2s_rd_xfer_cmplt <=  sig_rd_xfer_cmplt;                     
    
    
     
    -- Internal logic ------------------------------
 
 
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_RD_CMPLT_FLAG
    --
    -- Process Description:
    --   Implements the status flag indicating that a read data 
    -- transfer has completed. This is an echo of a rlast assertion
    -- and a qualified data beat on the AXI4 Read Data Channel 
    -- inputs.
    --
    -------------------------------------------------------------
    IMP_RD_CMPLT_FLAG : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset = '1') then
    
             sig_rd_xfer_cmplt <= '0';
    
           else
    
             sig_rd_xfer_cmplt <= sig_mmap2data_last and 
                                  sig_good_mmap_dbeat;
                                  
           end if; 
        end if;       
      end process IMP_RD_CMPLT_FLAG; 
     
    
 
  
    
    -- General flag for advancing the MMap Read and the Stream
    -- data pipelines
    sig_advance_pipe     <=  sig_addr_chan_rdy        and                                  
                             sig_dqual_rdy            and                                  
                             not(sig_coelsc_reg_full) and  -- new status back-pressure term
                             not(sig_calc_error_reg);                                      
    
                                      
    -- test for Kevin's status throttle case
    sig_data2mmap_ready  <= (mm2s_strm_wready or 
                             sig_halt_reg)    and    -- Ignore the Stream ready on a Halt request                              
                             sig_advance_pipe;          
    
     
     
    sig_good_mmap_dbeat  <= sig_data2mmap_ready and 
                            sig_mmap2data_valid;
    
    
    sig_last_mmap_dbeat  <= sig_good_mmap_dbeat and 
                            sig_mmap2data_last;
     
     
    sig_get_next_dqual   <= sig_last_mmap_dbeat; 
    
    
    
    
    
    
         
    ------------------------------------------------------------
    -- Instance: I_READ_MUX 
    --
    -- Description:
    --  Instance of the MM2S Read Data Channel Read Mux   
    --
    ------------------------------------------------------------
    I_READ_MUX : entity axi_datamover_v5_1_14.axi_datamover_rdmux
    generic map (
  
      C_SEL_ADDR_WIDTH     =>  C_SEL_ADDR_WIDTH ,   
      C_MMAP_DWIDTH        =>  C_MMAP_DWIDTH    ,   
      C_STREAM_DWIDTH      =>  C_STREAM_DWIDTH      
  
      )
    port map (
  
      mmap_read_data_in    =>  mm2s_rdata       ,   
      mux_data_out         =>  mm2s_strm_wdata  ,            
      mstr2data_saddr_lsb  =>  sig_addr_lsb_reg     
    
      );
   
   
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_LAST_DBEAT
    --
    -- Process Description:
    --   This implements a FLOP that creates a pulse
    -- indicating the LAST signal for an incoming read data channel
    -- has been received. Note that it is possible to have back to 
    -- back LAST databeats.
    --
    -------------------------------------------------------------
    REG_LAST_DBEAT : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset = '1') then

             sig_last_mmap_dbeat_reg <= '0';
             
           else
             
             sig_last_mmap_dbeat_reg <= sig_last_mmap_dbeat;
             
           end if; 
         end if;       
       end process REG_LAST_DBEAT; 
  
    
    
    
    
    

     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_NO_DATA_CNTL_FIFO
     --
     -- If Generate Description:
     --   Omits the input data control FIFO if the requested FIFO
     -- depth is 1. The Data Qualifier Register serves as a 
     -- 1 deep FIFO by itself.
     --
     ------------------------------------------------------------
     GEN_NO_DATA_CNTL_FIFO : if (C_DATA_CNTL_FIFO_DEPTH = 1) generate

       begin

         -- Command Calculator Handshake output
         data2mstr_cmd_ready    <= sig_fifo_wr_cmd_ready;           
        
         sig_fifo_rd_cmd_valid  <= mstr2data_cmd_valid ;
         
         
         
         -- pre 13.1 sig_fifo_wr_cmd_ready  <= sig_dqual_reg_empty     and                                         
         -- pre 13.1                           sig_aposted_cntr_ready  and                                         
         -- pre 13.1                           not(rsc2mstr_halt_pipe) and  -- The Rd Status Controller is not stalling
         -- pre 13.1                           not(sig_calc_error_reg);     -- the command execution pipe and there is  
         -- pre 13.1                                                        -- no calculation error being propagated

         sig_fifo_wr_cmd_ready  <= sig_push_dqual_reg;


                                                                    
         
         sig_fifo_next_tag             <= mstr2data_tag        ;    
         sig_fifo_next_sadddr_lsb      <= mstr2data_saddr_lsb  ;    
         sig_fifo_next_len             <= mstr2data_len        ;    
         sig_fifo_next_strt_strb       <= mstr2data_strt_strb  ;    
         sig_fifo_next_last_strb       <= mstr2data_last_strb  ;    
         sig_fifo_next_drr             <= mstr2data_drr        ;    
         sig_fifo_next_eof             <= mstr2data_eof        ;    
         sig_fifo_next_sequential      <= mstr2data_sequential ;    
         sig_fifo_next_cmd_cmplt       <= mstr2data_cmd_cmplt  ;    
         sig_fifo_next_calc_error      <= mstr2data_calc_error ; 
            
         sig_fifo_next_dre_src_align   <= mstr2data_dre_src_align  ; 
         sig_fifo_next_dre_dest_align  <= mstr2data_dre_dest_align ; 
                                                              
             
   
       end generate GEN_NO_DATA_CNTL_FIFO;
         
    
    
    
    
      
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_DATA_CNTL_FIFO
     --
     -- If Generate Description:
     --   Includes the input data control FIFO if the requested 
     -- FIFO depth is more than 1.
     --
     ------------------------------------------------------------
     GEN_DATA_CNTL_FIFO : if (C_DATA_CNTL_FIFO_DEPTH > 1) generate

       begin

       
         -- Command Calculator Handshake output
         data2mstr_cmd_ready    <= sig_fifo_wr_cmd_ready;           
         
         sig_fifo_wr_cmd_valid  <= mstr2data_cmd_valid  ;
         

         sig_fifo_rd_cmd_ready    <= sig_push_dqual_reg;  -- pop the fifo when dqual reg is pushed   
                                                               

         
         
                              
         -- Format the input fifo data word
         sig_cmd_fifo_data_in  <=   mstr2data_dre_dest_align &
                                    mstr2data_dre_src_align  &
                                    mstr2data_calc_error     &
                                    mstr2data_cmd_cmplt      &
                                    mstr2data_sequential     &
                                    mstr2data_eof            &
                                    mstr2data_drr            &
                                    mstr2data_last_strb      &
                                    mstr2data_strt_strb      &
                                    mstr2data_len            &
                                    mstr2data_saddr_lsb      &
                                    mstr2data_tag ;
         
          
         -- Rip the output fifo data word
         sig_fifo_next_tag        <= sig_cmd_fifo_data_out((TAG_STRT_INDEX+TAG_WIDTH)-1 downto 
                                                            TAG_STRT_INDEX);                   
         sig_fifo_next_sadddr_lsb <= sig_cmd_fifo_data_out((SADDR_LSB_STRT_INDEX+SADDR_LSB_WIDTH)-1 downto 
                                                            SADDR_LSB_STRT_INDEX);
         sig_fifo_next_len        <= sig_cmd_fifo_data_out((LEN_STRT_INDEX+LEN_WIDTH)-1 downto 
                                                            LEN_STRT_INDEX);
         sig_fifo_next_strt_strb  <= sig_cmd_fifo_data_out((STRT_STRB_STRT_INDEX+STRB_WIDTH)-1 downto 
                                                            STRT_STRB_STRT_INDEX);
         sig_fifo_next_last_strb  <= sig_cmd_fifo_data_out((LAST_STRB_STRT_INDEX+STRB_WIDTH)-1 downto 
                                                            LAST_STRB_STRT_INDEX);
         sig_fifo_next_drr        <= sig_cmd_fifo_data_out(SOF_STRT_INDEX);
         sig_fifo_next_eof        <= sig_cmd_fifo_data_out(EOF_STRT_INDEX);
         sig_fifo_next_sequential <= sig_cmd_fifo_data_out(SEQUENTIAL_STRT_INDEX);
         sig_fifo_next_cmd_cmplt  <= sig_cmd_fifo_data_out(CMD_CMPLT_STRT_INDEX);
         sig_fifo_next_calc_error <= sig_cmd_fifo_data_out(CALC_ERR_STRT_INDEX);

         sig_fifo_next_dre_src_align   <= sig_cmd_fifo_data_out((DRE_SRC_STRT_INDEX+DRE_ALIGN_WIDTH)-1 downto 
                                                                 DRE_SRC_STRT_INDEX);
         sig_fifo_next_dre_dest_align  <= sig_cmd_fifo_data_out((DRE_DEST_STRT_INDEX+DRE_ALIGN_WIDTH)-1 downto 
                                                                 DRE_DEST_STRT_INDEX);
         
                                           
                                           
                                           
         ------------------------------------------------------------
         -- Instance: I_DATA_CNTL_FIFO 
         --
         -- Description:
         -- Instance for the Command Qualifier FIFO
         --
         ------------------------------------------------------------
          I_DATA_CNTL_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
          generic map (
        
            C_DWIDTH             =>  DCTL_FIFO_WIDTH        ,  
            C_DEPTH              =>  C_DATA_CNTL_FIFO_DEPTH ,  
            C_IS_ASYNC           =>  USE_SYNC_FIFO          ,  
            C_PRIM_TYPE          =>  FIFO_PRIM_TYPE         ,  
            C_FAMILY             =>  C_FAMILY                  
           
            )
          port map (
            
            -- Write Clock and reset
            fifo_wr_reset        =>   mmap_reset            ,  
            fifo_wr_clk          =>   primary_aclk          ,  
            
            -- Write Side
            fifo_wr_tvalid       =>   sig_fifo_wr_cmd_valid ,  
            fifo_wr_tready       =>   sig_fifo_wr_cmd_ready ,  
            fifo_wr_tdata        =>   sig_cmd_fifo_data_in  ,  
            fifo_wr_full         =>   open                  ,  
           
            -- Read Clock and reset
            fifo_async_rd_reset  =>   mmap_reset            ,   
            fifo_async_rd_clk    =>   primary_aclk          ,  
            
            -- Read Side
            fifo_rd_tvalid       =>   sig_fifo_rd_cmd_valid ,  
            fifo_rd_tready       =>   sig_fifo_rd_cmd_ready ,  
            fifo_rd_tdata        =>   sig_cmd_fifo_data_out ,  
            fifo_rd_empty        =>   sig_cmd_fifo_empty       
           
            );
        

       end generate GEN_DATA_CNTL_FIFO;
         
          
    
   
  
  
  
  
  
    -- Data Qualifier Register ------------------------------------
    
    sig_ld_new_cmd           <= sig_push_dqual_reg      ;
    sig_addr_chan_rdy        <= not(sig_addr_posted_cntr_eq_0);
    sig_dqual_rdy            <= sig_dqual_reg_full      ;
    sig_strt_strb_reg        <= sig_next_strt_strb_reg  ;
    sig_last_strb_reg        <= sig_next_last_strb_reg  ;
    sig_tag_reg              <= sig_next_tag_reg        ;
    sig_cmd_cmplt_reg        <= sig_next_cmd_cmplt_reg  ;
    sig_calc_error_reg       <= sig_next_calc_error_reg ;
    
    
    -- Flag indicating that there are no posted commands to AXI
    sig_no_posted_cmds       <= sig_addr_posted_cntr_eq_0;
    
    
    
    -- new for no bubbles between child requests
    sig_sequential_push      <= sig_good_mmap_dbeat and -- MMap handshake qualified
                                sig_last_dbeat      and -- last data beat of transfer
                                sig_next_sequential_reg;-- next queued command is sequential 
                                                        -- to the current command
    
    
    -- pre 13.1 sig_push_dqual_reg       <= (sig_sequential_push   or
    -- pre 13.1                              sig_dqual_reg_empty)  and 
    -- pre 13.1                             sig_fifo_rd_cmd_valid  and
    -- pre 13.1                             sig_aposted_cntr_ready and 
    -- pre 13.1                             not(rsc2mstr_halt_pipe);  -- The Rd Status Controller is not  
                                                                      -- stalling the command execution pipe
    
    sig_push_dqual_reg       <= (sig_sequential_push    or
                                 sig_dqual_reg_empty)   and 
                                sig_fifo_rd_cmd_valid   and
                                sig_aposted_cntr_ready  and 
                                not(sig_calc_error_reg) and -- 13.1 addition => An error has not been propagated
                                not(rsc2mstr_halt_pipe);    -- The Rd Status Controller is not  
                                                            -- stalling the command execution pipe
                                                        
                                                        
    sig_pop_dqual_reg        <= not(sig_next_calc_error_reg) and 
                                sig_get_next_dqual and 
                                sig_dqual_reg_full  ; 
    
  
    -- new for no bubbles between child requests
    sig_clr_dqual_reg        <=  mmap_reset         or
                                 (sig_pop_dqual_reg and
                                 not(sig_push_dqual_reg));
  
    
  -------------------------------------------------------------
  -- Synchronous Process with Sync Reset
  --
  -- Label: IMP_DQUAL_REG
  --
  -- Process Description:
  --    This process implements a register for the Data 
  -- Control and qualifiers. It operates like a 1 deep Sync FIFO.
  --
  -------------------------------------------------------------
  IMP_DQUAL_REG : process (primary_aclk)
     begin
       if (primary_aclk'event and primary_aclk = '1') then
          if (sig_clr_dqual_reg = '1') then
            
            sig_next_tag_reg             <= (others => '0');
            sig_next_strt_strb_reg       <= (others => '0');
            sig_next_last_strb_reg       <= (others => '0');
            sig_next_eof_reg             <= '0';
            sig_next_cmd_cmplt_reg       <= '0';
            sig_next_sequential_reg      <= '0';
            sig_next_calc_error_reg      <= '0';
            sig_next_dre_src_align_reg   <= (others => '0');
            sig_next_dre_dest_align_reg  <= (others => '0');
            
            sig_dqual_reg_empty          <= '1';
            sig_dqual_reg_full           <= '0';
            
          elsif (sig_push_dqual_reg = '1') then
            
            sig_next_tag_reg             <= sig_fifo_next_tag            ;
            sig_next_strt_strb_reg       <= sig_fifo_next_strt_strb      ;
            sig_next_last_strb_reg       <= sig_fifo_next_last_strb      ;
            sig_next_eof_reg             <= sig_fifo_next_eof            ;
            sig_next_cmd_cmplt_reg       <= sig_fifo_next_cmd_cmplt      ;
            sig_next_sequential_reg      <= sig_fifo_next_sequential     ;
            sig_next_calc_error_reg      <= sig_fifo_next_calc_error     ;
            sig_next_dre_src_align_reg   <= sig_fifo_next_dre_src_align  ;  
            sig_next_dre_dest_align_reg  <= sig_fifo_next_dre_dest_align ;  
            
            sig_dqual_reg_empty          <= '0';
            sig_dqual_reg_full           <= '1';
            
          else
            null;  -- don't change state
          end if; 
       end if;       
     end process IMP_DQUAL_REG; 
     

  
  
  
    
    
    -- Address LS Cntr logic  --------------------------
    
    sig_addr_lsb_reg         <= STD_LOGIC_VECTOR(sig_ls_addr_cntr);
    sig_addr_incr_unsgnd     <= TO_UNSIGNED(ADDR_INCR_VALUE, C_SEL_ADDR_WIDTH);
    sig_incr_ls_addr_cntr    <= sig_good_mmap_dbeat;
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: DO_ADDR_LSB_CNTR
    --
    -- Process Description:
    --  Implements the LS Address Counter used for controlling
    -- the Read Data Mux during Burst transfers
    --
    -------------------------------------------------------------
    DO_ADDR_LSB_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset         = '1'  or
               (sig_pop_dqual_reg  = '1'  and
                sig_push_dqual_reg = '0')) then                 -- Clear the Counter
              
              sig_ls_addr_cntr <= (others => '0');
            
            elsif (sig_push_dqual_reg = '1') then               -- Load the Counter
              
              sig_ls_addr_cntr <= unsigned(sig_fifo_next_sadddr_lsb);
            
            elsif (sig_incr_ls_addr_cntr = '1') then            -- Increment the Counter
              
              sig_ls_addr_cntr <= sig_ls_addr_cntr + sig_addr_incr_unsgnd;
            
            else
              null;  -- Hold Current value
            end if; 
         end if;       
       end process DO_ADDR_LSB_CNTR; 
    
    
    
    
    
    
    
    
    
    
    
    
    ----- Address posted Counter logic --------------------------------
    
    sig_incr_addr_posted_cntr <= sig_addr_posted              ;
    
    
    sig_decr_addr_posted_cntr <= sig_last_mmap_dbeat_reg      ;
    
    
    sig_aposted_cntr_ready    <= not(sig_addr_posted_cntr_max);
    
    sig_addr_posted_cntr_eq_0 <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_ZERO)
      Else '0';
    
    sig_addr_posted_cntr_max <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_MAX)
      Else '0';
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ADDR_POSTED_FIFO_CNTR
    --
    -- Process Description:
    --    This process implements a register for the Address 
    -- Posted FIFO that operates like a 1 deep Sync FIFO.
    --
    -------------------------------------------------------------
    IMP_ADDR_POSTED_FIFO_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_addr_posted_cntr <= ADDR_POSTED_ZERO;
              
            elsif (sig_incr_addr_posted_cntr = '1' and
                   sig_decr_addr_posted_cntr  = '0' and
                   sig_addr_posted_cntr_max = '0') then
              
              sig_addr_posted_cntr <= sig_addr_posted_cntr + ADDR_POSTED_ONE ;
              
            elsif (sig_incr_addr_posted_cntr  = '0' and
                   sig_decr_addr_posted_cntr   = '1' and
                   sig_addr_posted_cntr_eq_0 = '0') then
              
              sig_addr_posted_cntr <= sig_addr_posted_cntr - ADDR_POSTED_ONE ;
              
            else
              null;  -- don't change state
            end if; 
         end if;       
       end process IMP_ADDR_POSTED_FIFO_CNTR; 
     
         
      
      
      
      
      
      
    ------- First/Middle/Last Dbeat detirmination -------------------
     
     sig_new_len_eq_0 <= '1'
       When  (sig_fifo_next_len = LEN_OF_ZERO)
       else '0';
     
     
      
      
     -------------------------------------------------------------
     -- Synchronous Process with Sync Reset
     --
     -- Label: DO_FIRST_MID_LAST
     --
     -- Process Description:
     --  Implements the detection of the First/Mid/Last databeat of
     -- a transfer.
     --
     -------------------------------------------------------------
     DO_FIRST_MID_LAST : process (primary_aclk)
        begin
          if (primary_aclk'event and primary_aclk = '1') then
             if (mmap_reset = '1') then
               
               sig_first_dbeat   <= '0';
               sig_last_dbeat    <= '0';
             
             elsif (sig_ld_new_cmd = '1') then
               
               sig_first_dbeat   <= not(sig_new_len_eq_0);
               sig_last_dbeat    <= sig_new_len_eq_0;
             
             Elsif (sig_dbeat_cntr_eq_1 = '1' and
                    sig_good_mmap_dbeat = '1') Then
             
               sig_first_dbeat   <= '0';
               sig_last_dbeat    <= '1';
             
             Elsif (sig_dbeat_cntr_eq_0 = '0' and
                    sig_dbeat_cntr_eq_1 = '0' and
                    sig_good_mmap_dbeat = '1') Then
             
               sig_first_dbeat   <= '0';
               sig_last_dbeat    <= '0';
             
             else
               null; -- hols current state
             end if; 
          end if;       
        end process DO_FIRST_MID_LAST; 
    
    
   
   
   
   -------  Data Controller Halted Indication ------------------------------- 
    
 
    data2all_dcntlr_halted <= sig_no_posted_cmds  and
                              (sig_calc_error_reg or
                               rst2data_stop_request);
 
 
    
    
   -------  Data Beat counter logic ------------------------------- 
    sig_dbeat_cntr_int  <= TO_INTEGER(sig_dbeat_cntr);
    
    sig_dbeat_cntr_eq_0 <= '1'
      when (sig_dbeat_cntr_int = 0)
      Else '0';
    
    sig_dbeat_cntr_eq_1 <= '1'
      when (sig_dbeat_cntr_int = 1)
      Else '0';
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: DO_DBEAT_CNTR
    --
    -- Process Description:
    --
    --
    -------------------------------------------------------------
    DO_DBEAT_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              sig_dbeat_cntr <= (others => '0');
            elsif (sig_ld_new_cmd = '1') then
              sig_dbeat_cntr <= unsigned(sig_fifo_next_len);
            Elsif (sig_good_mmap_dbeat = '1' and
                   sig_dbeat_cntr_eq_0 = '0') Then
              sig_dbeat_cntr <= sig_dbeat_cntr-1;
            else
              null; -- Hold current state
            end if; 
         end if;       
       end process DO_DBEAT_CNTR; 
      
  
 
 
  
  
   ------  Read Response Status Logic  ------------------------------
  
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: LD_NEW_CMD_PULSE
   --
   -- Process Description:
   -- Generate a 1 Clock wide pulse when a new command has been
   -- loaded into the Command Register
   --
   -------------------------------------------------------------
   LD_NEW_CMD_PULSE : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset         = '1' or
               sig_ld_new_cmd_reg = '1') then
             sig_ld_new_cmd_reg <= '0';
           elsif (sig_ld_new_cmd = '1') then
             sig_ld_new_cmd_reg <= '1';
           else
             null; -- hold State
           end if; 
        end if;       
      end process LD_NEW_CMD_PULSE; 
  
                               
                               
   sig_pop_coelsc_reg  <= sig_coelsc_reg_full and
                          sig_rsc2data_ready ; 
                          
   sig_push_coelsc_reg <= (sig_good_mmap_dbeat and  
                           not(sig_coelsc_reg_full)) or
                          (sig_ld_new_cmd_reg and 
                           sig_calc_error_reg) ; 
   
   sig_cmd_cmplt_last_dbeat <= (sig_cmd_cmplt_reg and sig_mmap2data_last) or
                                sig_calc_error_reg;
   
   
      
  -------  Read Response Decode   
    
   -- Decode the AXI MMap Read Response       
   sig_decerr  <= '1'
     When (mm2s_rresp = DECERR and mm2s_rvalid = '1')
     Else '0'; 
          
   sig_slverr  <= '1'
     When (mm2s_rresp = SLVERR and mm2s_rvalid = '1')
     Else '0';      
   
   
   
          
          
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: RD_RESP_COELESC_REG
    --
    -- Process Description:
    --   Implement the Read error/status coelescing register. 
    -- Once a bit is set it will remain set until the overall 
    -- status is written to the Status Controller. 
    -- Tag bits are just registered at each valid dbeat.
    --
    -------------------------------------------------------------
    STATUS_COELESC_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset          = '1' or
               (sig_pop_coelsc_reg  = '1' and        -- Added more qualification here for simultaneus
                sig_push_coelsc_reg = '0')) then     -- push and pop condition per CR590244
                                                                                                
              sig_coelsc_tag_reg       <= (others => '0');
              sig_coelsc_cmd_cmplt_reg <= '0';
              sig_coelsc_interr_reg    <= '0';
              sig_coelsc_decerr_reg    <= '0';
              sig_coelsc_slverr_reg    <= '0';
              sig_coelsc_okay_reg      <= '1';       -- set back to default of "OKAY"
  
              sig_coelsc_reg_full      <= '0';
              sig_coelsc_reg_empty     <= '1';
  
  
              
            Elsif (sig_push_coelsc_reg = '1') Then
            
              sig_coelsc_tag_reg       <= sig_tag_reg;                             
              sig_coelsc_cmd_cmplt_reg <= sig_cmd_cmplt_last_dbeat;                  
              sig_coelsc_interr_reg    <= sig_calc_error_reg or 
                                          sig_coelsc_interr_reg;
              sig_coelsc_decerr_reg    <= sig_decerr or sig_coelsc_decerr_reg;
              sig_coelsc_slverr_reg    <= sig_slverr or sig_coelsc_slverr_reg;
              sig_coelsc_okay_reg      <= not(sig_decerr       or 
                                              sig_slverr       or 
                                              sig_calc_error_reg );
              
              sig_coelsc_reg_full      <= sig_cmd_cmplt_last_dbeat;
              sig_coelsc_reg_empty     <= not(sig_cmd_cmplt_last_dbeat);
                                            
              
            else
              
              null;  -- hold current state
              
            end if; 
         end if;       
       end process STATUS_COELESC_REG; 
   
   
   
   
   
   
   
   
   
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_NO_DRE
    --
    -- If Generate Description:
    --  Ties off DRE Control signals to logic low when DRE is
    -- omitted from the MM2S functionality.
    --
    --
    ------------------------------------------------------------
    GEN_NO_DRE : if (C_INCLUDE_DRE = 0) generate
    
       begin
    
         mm2s_dre_new_align     <= '0';
         mm2s_dre_use_autodest  <= '0';
         mm2s_dre_src_align     <= (others => '0');
         mm2s_dre_dest_align    <= (others => '0');
         mm2s_dre_flush         <= '0';
        
       end generate GEN_NO_DRE;
   
    
    
    
    
    
    
    
    
    
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_INCLUDE_DRE_CNTLS
    --
    -- If Generate Description:
    -- Implements the DRE Control logic when MM2S DRE is enabled.
    --
    --  - The DRE needs to have forced alignment at a SOF assertion
    --
    --
    ------------------------------------------------------------
    GEN_INCLUDE_DRE_CNTLS : if (C_INCLUDE_DRE = 1) generate
    
       -- local signals
       signal lsig_s_h_dre_autodest  : std_logic := '0';
       signal lsig_s_h_dre_new_align : std_logic := '0';
    
       begin
    
        
         mm2s_dre_new_align     <= lsig_s_h_dre_new_align;
         
         
         
         
         -- Autodest is asserted on a new parent command and the 
         -- previous parent command was not delimited with a EOF
         mm2s_dre_use_autodest  <= lsig_s_h_dre_autodest;
         
         
         
         
         -- Assign the DRE Source and Destination Alignments
         -- Only used when mm2s_dre_new_align is asserted 
         mm2s_dre_src_align     <= sig_next_dre_src_align_reg ;
         mm2s_dre_dest_align    <= sig_next_dre_dest_align_reg;
         
         
         -- Assert the Flush flag when the MMap Tlast input of the current transfer is
         -- asserted and the next transfer is not sequential and not the last 
         -- transfer of a packet.
         mm2s_dre_flush         <= mm2s_rlast and
                                   not(sig_next_sequential_reg) and   
                                   not(sig_next_eof_reg);
        
        
        
         
         
         -------------------------------------------------------------
         -- Synchronous Process with Sync Reset
         --
         -- Label: IMP_S_H_NEW_ALIGN
         --
         -- Process Description:
         --   Generates the new alignment command flag to the DRE.
         --
         -------------------------------------------------------------
         IMP_S_H_NEW_ALIGN : process (primary_aclk)
            begin
              if (primary_aclk'event and primary_aclk = '1') then
                 if (mmap_reset   = '1') then
                   
                   lsig_s_h_dre_new_align <= '0';
                 
                 
                 Elsif (sig_push_dqual_reg = '1' and
                        sig_fifo_next_drr  = '1') Then
                 
                   lsig_s_h_dre_new_align <= '1';
                 
                 elsif (sig_pop_dqual_reg = '1') then
                   
                   lsig_s_h_dre_new_align <=  sig_next_cmd_cmplt_reg and
                                             not(sig_next_sequential_reg) and
                                             not(sig_next_eof_reg);
                 
                 Elsif (sig_good_mmap_dbeat = '1') Then
                 
                   lsig_s_h_dre_new_align <= '0';
                 
                 
                 else
                   
                   null; -- hold current state
                 
                 end if; 
              end if;       
            end process IMP_S_H_NEW_ALIGN; 
        
        
        
         
         
        
         -------------------------------------------------------------
         -- Synchronous Process with Sync Reset
         --
         -- Label: IMP_S_H_AUTODEST
         --
         -- Process Description:
         --   Generates the control for the DRE indicating whether the
         -- DRE destination alignment should be derived from the write
         -- strobe stat of the last completed data-beat to the AXI 
         -- stream output.
         --
         -------------------------------------------------------------
         IMP_S_H_AUTODEST : process (primary_aclk)
            begin
              if (primary_aclk'event and primary_aclk = '1') then
                 if (mmap_reset   = '1') then
                   
                   lsig_s_h_dre_autodest <= '0';
                 
                 
                 Elsif (sig_push_dqual_reg = '1' and
                        sig_fifo_next_drr  = '1') Then
                   
                   lsig_s_h_dre_autodest <= '0';
                 
                 elsif (sig_pop_dqual_reg = '1') then
                   
                   lsig_s_h_dre_autodest <=  sig_next_cmd_cmplt_reg and
                                             not(sig_next_sequential_reg) and
                                             not(sig_next_eof_reg);
                 
                 Elsif (lsig_s_h_dre_new_align = '1' and
                        sig_good_mmap_dbeat    = '1') Then
                 
                   lsig_s_h_dre_autodest <= '0';
                 
                 
                 else
                   
                   null; -- hold current state
                 
                 end if; 
              end if;       
            end process IMP_S_H_AUTODEST; 
        
        
        
        
       end generate GEN_INCLUDE_DRE_CNTLS;
   
   
   
   
   
   
    
    
    
    
    
    
    
    
    
    
    
    
    
    
   -------  Soft Shutdown Logic ------------------------------- 
    
    
    -- Assign the output port skid buf control
    data2skid_halt      <= sig_data2skid_halt;
    
    -- Create a 1 clock wide pulse to tell the output
    -- stream skid buffer to shut down its outputs
    sig_data2skid_halt  <=  sig_halt_reg_dly2 and 
                            not(sig_halt_reg_dly3);
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_REQ_REG
    --
    -- Process Description:
    --   Implements the flop for capturing the Halt request from 
    -- the Reset module.
    --
    -------------------------------------------------------------
    IMP_HALT_REQ_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg      <= '0';
            
            elsif (rst2data_stop_request = '1') then
              
              sig_halt_reg <= '1';
            
            else
              null;  -- Hold current State
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG; 
  
  
   
   
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_REQ_REG_DLY
    --
    -- Process Description:
    --   Implements the flops for delaying the halt request by 3
    -- clocks to allow the Address Controller to halt before the
    -- Data Contoller can safely indicate it has exhausted all
    -- transfers committed to the AXI Address Channel by the Address
    -- Controller.
    --
    -------------------------------------------------------------
    IMP_HALT_REQ_REG_DLY : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg_dly1 <= '0';
              sig_halt_reg_dly2 <= '0';
              sig_halt_reg_dly3 <= '0';
            
            else
              
              sig_halt_reg_dly1 <= sig_halt_reg;
              sig_halt_reg_dly2 <= sig_halt_reg_dly1;
              sig_halt_reg_dly3 <= sig_halt_reg_dly2;
            
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG_DLY; 
  
  
   
   
   
   
   
   
   
   
   
   
      
      
  end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_rd_status_cntl.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_rd_status_cntl.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Read Status Controller.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_rd_status_cntl is
    generic (
      
      C_STS_WIDTH          : Integer               := 8;
        -- sets the width of the Status ports
      
      C_TAG_WIDTH          : Integer range  1 to 8 := 4
        -- Sets the width of the Tag field in the Status reply
      
      );
    port (
      
      -- Clock and Reset input --------------------------------------
                                                                   --
      primary_aclk           : in  std_logic;                      --
         -- Primary synchronization clock for the Master side      --
         -- interface and internal logic. It is also used          --
         -- for the User interface synchronization when            --
         -- C_STSCMD_IS_ASYNC = 0.                                 --
                                                                   --
      -- Reset input                                               --
      mmap_reset             : in  std_logic;                      --
         -- Reset used for the internal master logic               --
      ---------------------------------------------------------------
      
      
      
                
                
      -- Command Calculator Status Interface  ---------------------------
                                                                       --
      calc2rsc_calc_error    : in std_logic ;                          --
         -- Indication from the Command Calculator that a calculation  --
         -- error has occured.                                         --
      -------------------------------------------------------------------
     
     
     
      
        
      -- Address Controller Status Interface ----------------------------
                                                                       --
      addr2rsc_calc_error    : In std_logic ;                          --
         -- Indication from the Data Channel Controller FIFO that it   --
         -- is empty (no commands pending)                             --
                                                                       --
      addr2rsc_fifo_empty    : In std_logic ;                          --
         -- Indication from the Address Controller FIFO that it        --
         -- is empty (no commands pending)                             --
      -------------------------------------------------------------------

                    
                    
                    
      --  Data Controller Status Interface ---------------------------------------------
                                                                                      --
      data2rsc_tag           : In std_logic_vector(C_TAG_WIDTH-1 downto 0);           --
         -- The command tag                                                           --
                                                                                      --
      data2rsc_calc_error    : In std_logic ;                                         --
         -- Indication from the Data Channel Controller FIFO that it                  --
         -- is empty (no commands pending)                                            --
                                                                                      --
      data2rsc_okay          : In std_logic ;                                         --
         -- Indication that the AXI Read transfer completed with OK status            --
                                                                                      --
      data2rsc_decerr        : In std_logic ;                                         --
         -- Indication that the AXI Read transfer completed with decode error status  --
                                                                                      --
      data2rsc_slverr        : In std_logic ;                                         --
         -- Indication that the AXI Read transfer completed with slave error status   --
                                                                                      --
      data2rsc_cmd_cmplt     : In std_logic ;                                         --
         -- Indication by the Data Channel Controller that the                        --
         -- corresponding status is the last status for a parent command              --
         -- pulled from the command FIFO                                              --
                                                                                      --
      rsc2data_ready         : Out  std_logic;                                        --
         -- Handshake bit from the Read Status Controller Module indicating           --
         -- that the it is ready to accept a new Read status transfer                 --
                                                                                      --
      data2rsc_valid         : in  std_logic ;                                        --
         -- Handshake bit output to the Read Status Controller Module                 --
         -- indicating that the Data Controller has valid tag and status              --
         -- indicators to transfer                                                    --
      ----------------------------------------------------------------------------------


      
      -- Command/Status Module Interface ----------------------------------------------
                                                                                     --
      rsc2stat_status        : Out std_logic_vector(C_STS_WIDTH-1 downto 0);         --
         -- Read Status value collected during a Read Data transfer                  --
         -- Output to the Command/Status Module                                      --
                                                                                     --
      stat2rsc_status_ready  : In  std_logic;                                        --
         -- Input from the Command/Status Module indicating that the                 --
         -- Status Reg/FIFO is ready to accept a transfer                            --
                                                                                     --
      rsc2stat_status_valid  : Out std_logic ;                                       --
         -- Control Signal to the Status Reg/FIFO indicating a new status            --
         -- output value is valid and ready for transfer                             --
      ---------------------------------------------------------------------------------

  
    
    
      -- Address and Data Controller Pipe halt ----------------------------------
                                                                               --
      rsc2mstr_halt_pipe     : Out std_logic                                   --
         -- Indication to Halt the Data and Address Command pipeline due       --
         -- to the Status FIFO going full or an internal error being logged    --
      ---------------------------------------------------------------------------

  
      );
  
  end entity axi_datamover_rd_status_cntl;
  
  
  architecture implementation of axi_datamover_rd_status_cntl is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    -- Constant Declarations  --------------------------------------------
    
    Constant OKAY               : std_logic_vector(1 downto 0) := "00";
    Constant EXOKAY             : std_logic_vector(1 downto 0) := "01";
    Constant SLVERR             : std_logic_vector(1 downto 0) := "10";
    Constant DECERR             : std_logic_vector(1 downto 0) := "11";
    Constant STAT_RSVD          : std_logic_vector(3 downto 0) := "0000";
    Constant TAG_WIDTH          : integer := C_TAG_WIDTH;
    Constant STAT_REG_TAG_WIDTH : integer := 4;
    
    
    -- Signal Declarations  --------------------------------------------
    
    signal sig_tag2status            : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_rsc2status_valid      : std_logic := '0';
    signal sig_rsc2data_ready        : std_logic := '0';
    signal sig_rd_sts_okay_reg       : std_logic := '0';
    signal sig_rd_sts_interr_reg     : std_logic := '0';
    signal sig_rd_sts_decerr_reg     : std_logic := '0';
    signal sig_rd_sts_slverr_reg     : std_logic := '0';
    signal sig_rd_sts_tag_reg        : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_pop_rd_sts_reg        : std_logic := '0';
    signal sig_push_rd_sts_reg       : std_logic := '0';
    Signal sig_rd_sts_push_ok        : std_logic := '0';
    signal sig_rd_sts_reg_empty      : std_logic := '0';
    signal sig_rd_sts_reg_full       : std_logic := '0';
    
    
    
    
    
            
  begin --(architecture implementation)
  
    -- Assign the status write output control
    rsc2stat_status_valid  <= sig_rsc2status_valid ;
    
    sig_rsc2status_valid   <= sig_rd_sts_reg_full;
    
                                           
    -- Formulate the status outout value (assumes an 8-bit status width)
    rsc2stat_status        <=  sig_rd_sts_okay_reg    &   
                               sig_rd_sts_slverr_reg  &
                               sig_rd_sts_decerr_reg  & 
                               sig_rd_sts_interr_reg  &
                               sig_tag2status;
    
    -- Detect that a push of a new status word is completing
    sig_rd_sts_push_ok   <= sig_rsc2status_valid and 
                            stat2rsc_status_ready;
    
    -- Signal a halt to the execution pipe if new status
    -- is valid but the Status FIFO is not accepting it
    rsc2mstr_halt_pipe   <=  sig_rsc2status_valid and
                             (not(stat2rsc_status_ready) ); 
   
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_TAG_LE_STAT
    --
    -- If Generate Description:
    -- Populates the TAG bits into the availble Status bits when
    -- the TAG width is less than or equal to the available number
    -- of bits in the Status word. 
    --
    ------------------------------------------------------------
    GEN_TAG_LE_STAT : if (TAG_WIDTH <= STAT_REG_TAG_WIDTH) generate
    
       -- local signals
         signal lsig_temp_tag_small : std_logic_vector(STAT_REG_TAG_WIDTH-1 downto 0) := (others => '0');
         
         
       begin
    
         sig_tag2status <= lsig_temp_tag_small;
         
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: POPULATE_SMALL_TAG
         --
         -- Process Description:
         --
         --
         -------------------------------------------------------------
         POPULATE_SMALL_TAG : process (sig_rd_sts_tag_reg)
            begin
         
              -- Set default value
              lsig_temp_tag_small <= (others => '0');
          
              -- Now overload actual TAG bits
              lsig_temp_tag_small(TAG_WIDTH-1 downto 0) <= sig_rd_sts_tag_reg;
          
         
            end process POPULATE_SMALL_TAG; 
         
         
       end generate GEN_TAG_LE_STAT;
     
     
     
     
     
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_TAG_GT_STAT
    --
    -- If Generate Description:
    -- Populates the TAG bits into the availble Status bits when
    -- the TAG width is greater than the available number of 
    -- bits in the Status word. The upper bits of the TAG are 
    -- clipped off (discarded). 
    --
    ------------------------------------------------------------
    GEN_TAG_GT_STAT : if (TAG_WIDTH > STAT_REG_TAG_WIDTH) generate
    
       -- local signals
         signal lsig_temp_tag_big : std_logic_vector(STAT_REG_TAG_WIDTH-1 downto 0);
         
         
       begin
    
         
         sig_tag2status <= lsig_temp_tag_big;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: POPULATE_BIG_TAG
         --
         -- Process Description:
         --
         --
         -------------------------------------------------------------
         POPULATE_SMALL_TAG : process (sig_rd_sts_tag_reg)
            begin
         
              -- Set default value
              lsig_temp_tag_big <= (others => '0');
          
              -- Now overload actual TAG bits
              lsig_temp_tag_big <= sig_rd_sts_tag_reg(STAT_REG_TAG_WIDTH-1 downto 0);
          
         
            end process POPULATE_SMALL_TAG; 
         
         
       end generate GEN_TAG_GT_STAT;
     
     
     
     
     
          
   ------- Read Status Collection Logic --------------------------------       
    
    rsc2data_ready      <=  sig_rsc2data_ready ;
    
    sig_rsc2data_ready  <= sig_rd_sts_reg_empty;
    
    
    sig_push_rd_sts_reg <= data2rsc_valid and
                           sig_rsc2data_ready;
          
    sig_pop_rd_sts_reg  <= sig_rd_sts_push_ok;
    
    
          
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: RD_STATUS_FIFO_REG
    --
    -- Process Description:
    --   Implement Read status FIFO register. 
    -- This register holds the Read status from the Data Controller
    -- until it is transfered to the Status FIFO.
    --
    -------------------------------------------------------------
    RD_STATUS_FIFO_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset         = '1' or
                sig_pop_rd_sts_reg = '1') then
              
              sig_rd_sts_tag_reg       <= (others => '0');
              sig_rd_sts_interr_reg    <= '0';
              sig_rd_sts_decerr_reg    <= '0';
              sig_rd_sts_slverr_reg    <= '0';
              sig_rd_sts_okay_reg      <= '1'; -- set back to default of "OKAY"
  
              sig_rd_sts_reg_full      <= '0';
              sig_rd_sts_reg_empty     <= '1';
  
  
              
            Elsif (sig_push_rd_sts_reg = '1') Then
            
              sig_rd_sts_tag_reg       <= data2rsc_tag;                             
              sig_rd_sts_interr_reg    <= data2rsc_calc_error or 
                                          sig_rd_sts_interr_reg;
              sig_rd_sts_decerr_reg    <= data2rsc_decerr or sig_rd_sts_decerr_reg;
              sig_rd_sts_slverr_reg    <= data2rsc_slverr or sig_rd_sts_slverr_reg;
              sig_rd_sts_okay_reg      <= data2rsc_okay and 
                                          not(data2rsc_decerr          or 
                                              sig_rd_sts_decerr_reg    or
                                              data2rsc_slverr          or 
                                              sig_rd_sts_slverr_reg    or
                                              data2rsc_calc_error      or
                                              sig_rd_sts_interr_reg      
                                              );
              
              sig_rd_sts_reg_full      <= data2rsc_cmd_cmplt or
                                          data2rsc_calc_error;
              sig_rd_sts_reg_empty     <= not(data2rsc_cmd_cmplt or
                                              data2rsc_calc_error);
                                            
            else
              
              null;  -- hold current state
              
            end if; 
         end if;       
       end process RD_STATUS_FIFO_REG; 
      
    
    
    
    
          
          
          
  
  end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_wr_demux.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_wr_demux.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Write Strobe De-Multiplexer.                 
  --  This is needed when the native data width of the DataMover is narrower 
  --  than the AXI4 Write Data Channel.                
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_wr_demux is
    generic (
      
      C_SEL_ADDR_WIDTH     : Integer range  1  to  8 :=  5;
        -- Sets the width of the select control bus
      
      C_MMAP_DWIDTH        : Integer range 32 to 1024 := 32;
        -- Indicates the width of the AXI4 Write Data Channel
      
      C_STREAM_DWIDTH      : Integer range  8 to 1024 := 32
        -- Indicates the native data width of the DataMover S2MM. If 
        -- S2MM Store and Forward with upsizer is enabled, the width is 
        -- the AXi4 Write Data Channel, else it is the S2MM Stream data width.
      
      );
    port (
      
     
      -- AXI MMap Data Channel Input  --------------------------------------------
                                                                                --
      wstrb_in         : In  std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);  --
        -- data input                                                           --
      ----------------------------------------------------------------------------
     
      
      
      -- AXI Master Stream  ------------------------------------------------------
                                                                                --
      demux_wstrb_out    : Out std_logic_vector((C_MMAP_DWIDTH/8)-1 downto 0);  --       
        --De-Mux strb output                                                    --
      ----------------------------------------------------------------------------
               
                
                
      -- Command Calculator Interface --------------------------------------------
                                                                                --
      debeat_saddr_lsb : In std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0)       --
         -- The next command start address LSbs to use for the read data        --
         -- mux (only used if Stream data width is less than the MMap Data      --
         -- Width).                                                             --
      ----------------------------------------------------------------------------
      
         
      );
  
  end entity axi_datamover_wr_demux;
  
  
  architecture implementation of axi_datamover_wr_demux is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    -- Function Decalarations -------------------------------------------------
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_mux_sel_width
    --
    -- Function Description:
    --   Calculates the number of needed bits for the Mux Select control
    -- based on the number of input channels to the mux.
    --
    -- Note that the number of input mux channels are always a 
    -- power of 2.
    --
    -------------------------------------------------------------------
    function func_mux_sel_width (num_channels : integer) return integer is
    
     Variable var_sel_width : integer := 0;
    
    begin
    
       case num_channels is
         --when 2 =>
         --    var_sel_width := 1;
         when 4 =>
             var_sel_width := 2;
         when 8 =>
             var_sel_width := 3;
         when 16 =>
             var_sel_width := 4;
         when 32 =>
             var_sel_width := 5;
         when 64 =>
             var_sel_width := 6;
         when 128 =>
             var_sel_width := 7;
         when others =>
             var_sel_width := 1; 
       end case;
       
       Return (var_sel_width);
        
        
    end function func_mux_sel_width;
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_sel_ls_index
    --
    -- Function Description:
    --   Calculates the LS index of the select field to rip from the
    -- input select bus.
    --
    -- Note that the number of input mux channels are always a 
    -- power of 2.
    --
    -------------------------------------------------------------------
    function func_sel_ls_index (stream_width : integer) return integer is
    
     Variable var_sel_ls_index : integer := 0;
    
    begin
    
       case stream_width is
         when 8 =>
             var_sel_ls_index := 0;
         when 16 =>
             var_sel_ls_index := 1;
         when 32 =>
             var_sel_ls_index := 2;
         when 64 =>
             var_sel_ls_index := 3;
         when 128 =>
             var_sel_ls_index := 4;
         when 256 =>
             var_sel_ls_index := 5;
         when 512 =>
             var_sel_ls_index := 6;
         when others =>  -- assume 1024 bit width
             var_sel_ls_index := 7;
       end case;
       
       Return (var_sel_ls_index);
        
        
    end function func_sel_ls_index;
    
    
    
    
    
    -- Constant Decalarations -------------------------------------------------
    
    Constant OMIT_DEMUX    : boolean := (C_STREAM_DWIDTH = C_MMAP_DWIDTH);
    Constant INCLUDE_DEMUX : boolean := not(OMIT_DEMUX);
    
    
    
    
    Constant STREAM_WSTB_WIDTH   : integer := C_STREAM_DWIDTH/8;
    Constant MMAP_WSTB_WIDTH     : integer := C_MMAP_DWIDTH/8;
    Constant NUM_MUX_CHANNELS    : integer := MMAP_WSTB_WIDTH/STREAM_WSTB_WIDTH;
    Constant MUX_SEL_WIDTH       : integer := func_mux_sel_width(NUM_MUX_CHANNELS);
    Constant MUX_SEL_LS_INDEX    : integer := func_sel_ls_index(C_STREAM_DWIDTH);
    
    
    -- Signal Declarations  --------------------------------------------
 
    signal sig_demux_wstrb_out   : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');



    
  begin --(architecture implementation)
  
  
  
  
   -- Assign the Output data port 
    demux_wstrb_out        <= sig_demux_wstrb_out;
  


    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_STRM_EQ_MMAP
    --
    -- If Generate Description:
    --   This IfGen implements the case where the Stream Data Width is 
    -- the same as the Memeory Map read Data width.
    --
    --
    ------------------------------------------------------------
    GEN_STRM_EQ_MMAP : if (OMIT_DEMUX) generate
        
       begin
        
          sig_demux_wstrb_out <= wstrb_in;
        
        
       end generate GEN_STRM_EQ_MMAP;
   
   
    
    
    
     
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_2XN
    --
    -- If Generate Description:
    --  2 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_2XN : if (INCLUDE_DEMUX and 
                  NUM_MUX_CHANNELS = 2) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);  -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                     -- with locally static subtype error in each of the
                                                                     -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_2XN_DEMUX
         --
         -- Process Description:
         --  Implement the 2XN DeMux
         --
         -------------------------------------------------------------
         DO_2XN_DEMUX : process (lsig_demux_sel_int_local,
                                  wstrb_in)
            begin
              
              -- Set default value
              lsig_demux_wstrb_out <=  (others => '0');
              
              case lsig_demux_sel_int_local is
                when 0 =>
                    lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
                
                when others => -- 1 case
                    lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1) <=  wstrb_in;
              end case;
              
            end process DO_2XN_DEMUX; 
 
         
       end generate GEN_2XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_4XN
    --
    -- If Generate Description:
    --  4 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_4XN : if (INCLUDE_DEMUX and 
                  NUM_MUX_CHANNELS = 4) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);  -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                 -- with locally static subtype error in each of the
                                                                 -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_4XN_DEMUX
         --
         -- Process Description:
         --  Implement the 4XN DeMux
         --
         -------------------------------------------------------------
         DO_4XN_DEMUX : process (lsig_demux_sel_int_local,
                                 wstrb_in)
           begin
              
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1) <=  wstrb_in;
               when 2 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2) <=  wstrb_in;
               
               when others =>  -- 3 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3) <=  wstrb_in;
             end case;
             
           end process DO_4XN_DEMUX; 
  
         
       end generate GEN_4XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_8XN
    --
    -- If Generate Description:
    --  8 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_8XN : if (INCLUDE_DEMUX and 
                  NUM_MUX_CHANNELS = 8) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);    -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                     -- with locally static subtype error in each of the
                                                                     -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_8XN_DEMUX
         --
         -- Process Description:
         --  Implement the 8XN DeMux
         --
         -------------------------------------------------------------
         DO_8XN_DEMUX : process (lsig_demux_sel_int_local,
                                 wstrb_in)
           begin
             
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1) <=  wstrb_in;
               when 2 =>                                                             
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2) <=  wstrb_in;
               when 3 =>                                                             
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3) <=  wstrb_in;
               when 4 =>                                                             
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*5)-1 downto STREAM_WSTB_WIDTH*4) <=  wstrb_in;
               when 5 =>                                                             
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*6)-1 downto STREAM_WSTB_WIDTH*5) <=  wstrb_in;
               when 6 =>                                                             
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*7)-1 downto STREAM_WSTB_WIDTH*6) <=  wstrb_in;
               
               when others => -- 7 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*8)-1 downto STREAM_WSTB_WIDTH*7) <=  wstrb_in;
             end case;
                 
           end process DO_8XN_DEMUX; 
 
         
       end generate GEN_8XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_16XN
    --
    -- If Generate Description:
    --  16 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_16XN : if (INCLUDE_DEMUX and 
                   NUM_MUX_CHANNELS = 16) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);  -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                 -- with locally static subtype error in each of the
                                                                 -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_16XN_DEMUX
         --
         -- Process Description:
         --  Implement the 16XN DeMux
         --
         -------------------------------------------------------------
         DO_16XN_DEMUX : process (lsig_demux_sel_int_local,
                                  wstrb_in)
           begin
             
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1)   <=  wstrb_in;
               when 2 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2)   <=  wstrb_in;
               when 3 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3)   <=  wstrb_in;
               when 4 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*5)-1 downto STREAM_WSTB_WIDTH*4)   <=  wstrb_in;
               when 5 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*6)-1 downto STREAM_WSTB_WIDTH*5)   <=  wstrb_in;
               when 6 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*7)-1 downto STREAM_WSTB_WIDTH*6)   <=  wstrb_in;
               when 7 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*8)-1 downto STREAM_WSTB_WIDTH*7)   <=  wstrb_in;
               when 8 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*9)-1 downto STREAM_WSTB_WIDTH*8)   <=  wstrb_in;
               when 9 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*10)-1 downto STREAM_WSTB_WIDTH*9)  <=  wstrb_in;
               when 10 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*11)-1 downto STREAM_WSTB_WIDTH*10) <=  wstrb_in;
               when 11 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*12)-1 downto STREAM_WSTB_WIDTH*11) <=  wstrb_in;
               when 12 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*13)-1 downto STREAM_WSTB_WIDTH*12) <=  wstrb_in;
               when 13 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*14)-1 downto STREAM_WSTB_WIDTH*13) <=  wstrb_in;
               when 14 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*15)-1 downto STREAM_WSTB_WIDTH*14) <=  wstrb_in;
               
               when others => -- 15 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*16)-1 downto STREAM_WSTB_WIDTH*15) <=  wstrb_in;
             end case;
          
           end process DO_16XN_DEMUX; 
 
         
       end generate GEN_16XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_32XN
    --
    -- If Generate Description:
    --  32 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_32XN : if (INCLUDE_DEMUX and 
                   NUM_MUX_CHANNELS = 32) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);  -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                 -- with locally static subtype error in each of the
                                                                 -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_32XN_DEMUX
         --
         -- Process Description:
         --  Implement the 32XN DeMux
         --
         -------------------------------------------------------------
         DO_32XN_DEMUX : process (lsig_demux_sel_int_local,
                                  wstrb_in)
           begin
             
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1)   <=  wstrb_in;
               when 2 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2)   <=  wstrb_in;
               when 3 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3)   <=  wstrb_in;
               when 4 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*5)-1 downto STREAM_WSTB_WIDTH*4)   <=  wstrb_in;
               when 5 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*6)-1 downto STREAM_WSTB_WIDTH*5)   <=  wstrb_in;
               when 6 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*7)-1 downto STREAM_WSTB_WIDTH*6)   <=  wstrb_in;
               when 7 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*8)-1 downto STREAM_WSTB_WIDTH*7)   <=  wstrb_in;
               when 8 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*9)-1 downto STREAM_WSTB_WIDTH*8)   <=  wstrb_in;
               when 9 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*10)-1 downto STREAM_WSTB_WIDTH*9)  <=  wstrb_in;
               when 10 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*11)-1 downto STREAM_WSTB_WIDTH*10) <=  wstrb_in;
               when 11 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*12)-1 downto STREAM_WSTB_WIDTH*11) <=  wstrb_in;
               when 12 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*13)-1 downto STREAM_WSTB_WIDTH*12) <=  wstrb_in;
               when 13 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*14)-1 downto STREAM_WSTB_WIDTH*13) <=  wstrb_in;
               when 14 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*15)-1 downto STREAM_WSTB_WIDTH*14) <=  wstrb_in;
               when 15 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*16)-1 downto STREAM_WSTB_WIDTH*15) <=  wstrb_in;
               when 16 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*17)-1 downto STREAM_WSTB_WIDTH*16) <=  wstrb_in;
               when 17 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*18)-1 downto STREAM_WSTB_WIDTH*17) <=  wstrb_in;
               when 18 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*19)-1 downto STREAM_WSTB_WIDTH*18) <=  wstrb_in;
               when 19 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*20)-1 downto STREAM_WSTB_WIDTH*19) <=  wstrb_in;
               when 20 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*21)-1 downto STREAM_WSTB_WIDTH*20) <=  wstrb_in;
               when 21 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*22)-1 downto STREAM_WSTB_WIDTH*21) <=  wstrb_in;
               when 22 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*23)-1 downto STREAM_WSTB_WIDTH*22) <=  wstrb_in;
               when 23 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*24)-1 downto STREAM_WSTB_WIDTH*23) <=  wstrb_in;
               when 24 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*25)-1 downto STREAM_WSTB_WIDTH*24) <=  wstrb_in;
               when 25 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*26)-1 downto STREAM_WSTB_WIDTH*25) <=  wstrb_in;
               when 26 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*27)-1 downto STREAM_WSTB_WIDTH*26) <=  wstrb_in;
               when 27 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*28)-1 downto STREAM_WSTB_WIDTH*27) <=  wstrb_in;
               when 28 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*29)-1 downto STREAM_WSTB_WIDTH*28) <=  wstrb_in;
               when 29 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*30)-1 downto STREAM_WSTB_WIDTH*29) <=  wstrb_in;
               when 30 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*31)-1 downto STREAM_WSTB_WIDTH*30) <=  wstrb_in;
               
               when others => -- 31 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*32)-1 downto STREAM_WSTB_WIDTH*31) <=  wstrb_in;
             end case;
          
           end process DO_32XN_DEMUX; 
 
         
       end generate GEN_32XN;
  
 
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_64XN
    --
    -- If Generate Description:
    --  64 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_64XN : if (INCLUDE_DEMUX and 
                   NUM_MUX_CHANNELS = 64) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);  -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                 -- with locally static subtype error in each of the
                                                                 -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_64XN_DEMUX
         --
         -- Process Description:
         --  Implement the 32XN DeMux
         --
         -------------------------------------------------------------
         DO_64XN_DEMUX : process (lsig_demux_sel_int_local,
                                  wstrb_in)
           begin
             
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1)   <=  wstrb_in;
               when 2 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2)   <=  wstrb_in;
               when 3 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3)   <=  wstrb_in;
               when 4 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*5)-1 downto STREAM_WSTB_WIDTH*4)   <=  wstrb_in;
               when 5 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*6)-1 downto STREAM_WSTB_WIDTH*5)   <=  wstrb_in;
               when 6 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*7)-1 downto STREAM_WSTB_WIDTH*6)   <=  wstrb_in;
               when 7 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*8)-1 downto STREAM_WSTB_WIDTH*7)   <=  wstrb_in;
               when 8 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*9)-1 downto STREAM_WSTB_WIDTH*8)   <=  wstrb_in;
               when 9 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*10)-1 downto STREAM_WSTB_WIDTH*9)  <=  wstrb_in;
               when 10 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*11)-1 downto STREAM_WSTB_WIDTH*10) <=  wstrb_in;
               when 11 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*12)-1 downto STREAM_WSTB_WIDTH*11) <=  wstrb_in;
               when 12 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*13)-1 downto STREAM_WSTB_WIDTH*12) <=  wstrb_in;
               when 13 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*14)-1 downto STREAM_WSTB_WIDTH*13) <=  wstrb_in;
               when 14 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*15)-1 downto STREAM_WSTB_WIDTH*14) <=  wstrb_in;
               when 15 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*16)-1 downto STREAM_WSTB_WIDTH*15) <=  wstrb_in;
               when 16 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*17)-1 downto STREAM_WSTB_WIDTH*16) <=  wstrb_in;
               when 17 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*18)-1 downto STREAM_WSTB_WIDTH*17) <=  wstrb_in;
               when 18 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*19)-1 downto STREAM_WSTB_WIDTH*18) <=  wstrb_in;
               when 19 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*20)-1 downto STREAM_WSTB_WIDTH*19) <=  wstrb_in;
               when 20 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*21)-1 downto STREAM_WSTB_WIDTH*20) <=  wstrb_in;
               when 21 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*22)-1 downto STREAM_WSTB_WIDTH*21) <=  wstrb_in;
               when 22 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*23)-1 downto STREAM_WSTB_WIDTH*22) <=  wstrb_in;
               when 23 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*24)-1 downto STREAM_WSTB_WIDTH*23) <=  wstrb_in;
               when 24 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*25)-1 downto STREAM_WSTB_WIDTH*24) <=  wstrb_in;
               when 25 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*26)-1 downto STREAM_WSTB_WIDTH*25) <=  wstrb_in;
               when 26 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*27)-1 downto STREAM_WSTB_WIDTH*26) <=  wstrb_in;
               when 27 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*28)-1 downto STREAM_WSTB_WIDTH*27) <=  wstrb_in;
               when 28 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*29)-1 downto STREAM_WSTB_WIDTH*28) <=  wstrb_in;
               when 29 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*30)-1 downto STREAM_WSTB_WIDTH*29) <=  wstrb_in;
               when 30 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*31)-1 downto STREAM_WSTB_WIDTH*30) <=  wstrb_in;
               when 31 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*32)-1 downto STREAM_WSTB_WIDTH*31) <=  wstrb_in;
             
             
               when 32 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*33)-1 downto STREAM_WSTB_WIDTH*32) <=  wstrb_in;
               when 33 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*34)-1 downto STREAM_WSTB_WIDTH*33) <=  wstrb_in;
               when 34 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*35)-1 downto STREAM_WSTB_WIDTH*34) <=  wstrb_in;
               when 35 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*36)-1 downto STREAM_WSTB_WIDTH*35) <=  wstrb_in;
               when 36 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*37)-1 downto STREAM_WSTB_WIDTH*36) <=  wstrb_in;
               when 37 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*38)-1 downto STREAM_WSTB_WIDTH*37) <=  wstrb_in;
               when 38 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*39)-1 downto STREAM_WSTB_WIDTH*38) <=  wstrb_in;
               when 39 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*40)-1 downto STREAM_WSTB_WIDTH*39) <=  wstrb_in;
               when 40 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*41)-1 downto STREAM_WSTB_WIDTH*40) <=  wstrb_in;
               when 41 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*42)-1 downto STREAM_WSTB_WIDTH*41) <=  wstrb_in;
               when 42 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*43)-1 downto STREAM_WSTB_WIDTH*42) <=  wstrb_in;
               when 43 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*44)-1 downto STREAM_WSTB_WIDTH*43) <=  wstrb_in;
               when 44 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*45)-1 downto STREAM_WSTB_WIDTH*44) <=  wstrb_in;
               when 45 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*46)-1 downto STREAM_WSTB_WIDTH*45) <=  wstrb_in;
               when 46 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*47)-1 downto STREAM_WSTB_WIDTH*46) <=  wstrb_in;
               when 47 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*48)-1 downto STREAM_WSTB_WIDTH*47) <=  wstrb_in;
               when 48 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*49)-1 downto STREAM_WSTB_WIDTH*48) <=  wstrb_in;
               when 49 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*50)-1 downto STREAM_WSTB_WIDTH*49) <=  wstrb_in;
               when 50 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*51)-1 downto STREAM_WSTB_WIDTH*50) <=  wstrb_in;
               when 51 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*52)-1 downto STREAM_WSTB_WIDTH*51) <=  wstrb_in;
               when 52 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*53)-1 downto STREAM_WSTB_WIDTH*52) <=  wstrb_in;
               when 53 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*54)-1 downto STREAM_WSTB_WIDTH*53) <=  wstrb_in;
               when 54 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*55)-1 downto STREAM_WSTB_WIDTH*54) <=  wstrb_in;
               when 55 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*56)-1 downto STREAM_WSTB_WIDTH*55) <=  wstrb_in;
               when 56 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*57)-1 downto STREAM_WSTB_WIDTH*56) <=  wstrb_in;
               when 57 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*58)-1 downto STREAM_WSTB_WIDTH*57) <=  wstrb_in;
               when 58 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*59)-1 downto STREAM_WSTB_WIDTH*58) <=  wstrb_in;
               when 59 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*60)-1 downto STREAM_WSTB_WIDTH*59) <=  wstrb_in;
               when 60 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*61)-1 downto STREAM_WSTB_WIDTH*60) <=  wstrb_in;
               when 61 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*62)-1 downto STREAM_WSTB_WIDTH*61) <=  wstrb_in;
               when 62 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*63)-1 downto STREAM_WSTB_WIDTH*62) <=  wstrb_in;
               
               when others => -- 63 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*64)-1 downto STREAM_WSTB_WIDTH*63) <=  wstrb_in;
             
             
             end case;
          
           end process DO_64XN_DEMUX; 
 
         
       end generate GEN_64XN;
  
 
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_128XN
    --
    -- If Generate Description:
    --  128 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_128XN : if (INCLUDE_DEMUX and 
                    NUM_MUX_CHANNELS = 128) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);    -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                     -- with locally static subtype error in each of the
                                                                     -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_128XN_DEMUX
         --
         -- Process Description:
         --  Implement the 32XN DeMux
         --
         -------------------------------------------------------------
         DO_128XN_DEMUX : process (lsig_demux_sel_int_local,
                                  wstrb_in)
           begin
             
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1)   <=  wstrb_in;
               when 2 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2)   <=  wstrb_in;
               when 3 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3)   <=  wstrb_in;
               when 4 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*5)-1 downto STREAM_WSTB_WIDTH*4)   <=  wstrb_in;
               when 5 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*6)-1 downto STREAM_WSTB_WIDTH*5)   <=  wstrb_in;
               when 6 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*7)-1 downto STREAM_WSTB_WIDTH*6)   <=  wstrb_in;
               when 7 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*8)-1 downto STREAM_WSTB_WIDTH*7)   <=  wstrb_in;
               when 8 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*9)-1 downto STREAM_WSTB_WIDTH*8)   <=  wstrb_in;
               when 9 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*10)-1 downto STREAM_WSTB_WIDTH*9)  <=  wstrb_in;
               when 10 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*11)-1 downto STREAM_WSTB_WIDTH*10) <=  wstrb_in;
               when 11 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*12)-1 downto STREAM_WSTB_WIDTH*11) <=  wstrb_in;
               when 12 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*13)-1 downto STREAM_WSTB_WIDTH*12) <=  wstrb_in;
               when 13 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*14)-1 downto STREAM_WSTB_WIDTH*13) <=  wstrb_in;
               when 14 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*15)-1 downto STREAM_WSTB_WIDTH*14) <=  wstrb_in;
               when 15 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*16)-1 downto STREAM_WSTB_WIDTH*15) <=  wstrb_in;
               when 16 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*17)-1 downto STREAM_WSTB_WIDTH*16) <=  wstrb_in;
               when 17 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*18)-1 downto STREAM_WSTB_WIDTH*17) <=  wstrb_in;
               when 18 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*19)-1 downto STREAM_WSTB_WIDTH*18) <=  wstrb_in;
               when 19 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*20)-1 downto STREAM_WSTB_WIDTH*19) <=  wstrb_in;
               when 20 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*21)-1 downto STREAM_WSTB_WIDTH*20) <=  wstrb_in;
               when 21 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*22)-1 downto STREAM_WSTB_WIDTH*21) <=  wstrb_in;
               when 22 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*23)-1 downto STREAM_WSTB_WIDTH*22) <=  wstrb_in;
               when 23 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*24)-1 downto STREAM_WSTB_WIDTH*23) <=  wstrb_in;
               when 24 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*25)-1 downto STREAM_WSTB_WIDTH*24) <=  wstrb_in;
               when 25 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*26)-1 downto STREAM_WSTB_WIDTH*25) <=  wstrb_in;
               when 26 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*27)-1 downto STREAM_WSTB_WIDTH*26) <=  wstrb_in;
               when 27 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*28)-1 downto STREAM_WSTB_WIDTH*27) <=  wstrb_in;
               when 28 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*29)-1 downto STREAM_WSTB_WIDTH*28) <=  wstrb_in;
               when 29 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*30)-1 downto STREAM_WSTB_WIDTH*29) <=  wstrb_in;
               when 30 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*31)-1 downto STREAM_WSTB_WIDTH*30) <=  wstrb_in;
               when 31 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*32)-1 downto STREAM_WSTB_WIDTH*31) <=  wstrb_in;
             
             
               when 32 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*33)-1 downto STREAM_WSTB_WIDTH*32) <=  wstrb_in;
               when 33 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*34)-1 downto STREAM_WSTB_WIDTH*33) <=  wstrb_in;
               when 34 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*35)-1 downto STREAM_WSTB_WIDTH*34) <=  wstrb_in;
               when 35 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*36)-1 downto STREAM_WSTB_WIDTH*35) <=  wstrb_in;
               when 36 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*37)-1 downto STREAM_WSTB_WIDTH*36) <=  wstrb_in;
               when 37 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*38)-1 downto STREAM_WSTB_WIDTH*37) <=  wstrb_in;
               when 38 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*39)-1 downto STREAM_WSTB_WIDTH*38) <=  wstrb_in;
               when 39 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*40)-1 downto STREAM_WSTB_WIDTH*39) <=  wstrb_in;
               when 40 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*41)-1 downto STREAM_WSTB_WIDTH*40) <=  wstrb_in;
               when 41 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*42)-1 downto STREAM_WSTB_WIDTH*41) <=  wstrb_in;
               when 42 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*43)-1 downto STREAM_WSTB_WIDTH*42) <=  wstrb_in;
               when 43 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*44)-1 downto STREAM_WSTB_WIDTH*43) <=  wstrb_in;
               when 44 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*45)-1 downto STREAM_WSTB_WIDTH*44) <=  wstrb_in;
               when 45 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*46)-1 downto STREAM_WSTB_WIDTH*45) <=  wstrb_in;
               when 46 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*47)-1 downto STREAM_WSTB_WIDTH*46) <=  wstrb_in;
               when 47 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*48)-1 downto STREAM_WSTB_WIDTH*47) <=  wstrb_in;
               when 48 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*49)-1 downto STREAM_WSTB_WIDTH*48) <=  wstrb_in;
               when 49 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*50)-1 downto STREAM_WSTB_WIDTH*49) <=  wstrb_in;
               when 50 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*51)-1 downto STREAM_WSTB_WIDTH*50) <=  wstrb_in;
               when 51 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*52)-1 downto STREAM_WSTB_WIDTH*51) <=  wstrb_in;
               when 52 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*53)-1 downto STREAM_WSTB_WIDTH*52) <=  wstrb_in;
               when 53 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*54)-1 downto STREAM_WSTB_WIDTH*53) <=  wstrb_in;
               when 54 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*55)-1 downto STREAM_WSTB_WIDTH*54) <=  wstrb_in;
               when 55 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*56)-1 downto STREAM_WSTB_WIDTH*55) <=  wstrb_in;
               when 56 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*57)-1 downto STREAM_WSTB_WIDTH*56) <=  wstrb_in;
               when 57 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*58)-1 downto STREAM_WSTB_WIDTH*57) <=  wstrb_in;
               when 58 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*59)-1 downto STREAM_WSTB_WIDTH*58) <=  wstrb_in;
               when 59 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*60)-1 downto STREAM_WSTB_WIDTH*59) <=  wstrb_in;
               when 60 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*61)-1 downto STREAM_WSTB_WIDTH*60) <=  wstrb_in;
               when 61 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*62)-1 downto STREAM_WSTB_WIDTH*61) <=  wstrb_in;
               when 62 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*63)-1 downto STREAM_WSTB_WIDTH*62) <=  wstrb_in;
               when 63 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*64)-1 downto STREAM_WSTB_WIDTH*63) <=  wstrb_in;
              
               
               when 64 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*65)-1 downto STREAM_WSTB_WIDTH*64) <=  wstrb_in;
               when 65 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*66)-1 downto STREAM_WSTB_WIDTH*65) <=  wstrb_in;
               when 66 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*67)-1 downto STREAM_WSTB_WIDTH*66) <=  wstrb_in;
               when 67 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*68)-1 downto STREAM_WSTB_WIDTH*67) <=  wstrb_in;
               when 68 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*69)-1 downto STREAM_WSTB_WIDTH*68) <=  wstrb_in;
               when 69 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*70)-1 downto STREAM_WSTB_WIDTH*69) <=  wstrb_in;
               when 70 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*71)-1 downto STREAM_WSTB_WIDTH*70) <=  wstrb_in;
               when 71 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*72)-1 downto STREAM_WSTB_WIDTH*71) <=  wstrb_in;
               when 72 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*73)-1 downto STREAM_WSTB_WIDTH*72) <=  wstrb_in;
               when 73 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*74)-1 downto STREAM_WSTB_WIDTH*73) <=  wstrb_in;
               when 74 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*75)-1 downto STREAM_WSTB_WIDTH*74) <=  wstrb_in;
               when 75 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*76)-1 downto STREAM_WSTB_WIDTH*75) <=  wstrb_in;
               when 76 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*77)-1 downto STREAM_WSTB_WIDTH*76) <=  wstrb_in;
               when 77 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*78)-1 downto STREAM_WSTB_WIDTH*77) <=  wstrb_in;
               when 78 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*79)-1 downto STREAM_WSTB_WIDTH*78) <=  wstrb_in;
               when 79 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*80)-1 downto STREAM_WSTB_WIDTH*79) <=  wstrb_in;
               when 80 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*81)-1 downto STREAM_WSTB_WIDTH*80) <=  wstrb_in;
               when 81 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*82)-1 downto STREAM_WSTB_WIDTH*81) <=  wstrb_in;
               when 82 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*83)-1 downto STREAM_WSTB_WIDTH*82) <=  wstrb_in;
               when 83 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*84)-1 downto STREAM_WSTB_WIDTH*83) <=  wstrb_in;
               when 84 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*85)-1 downto STREAM_WSTB_WIDTH*84) <=  wstrb_in;
               when 85 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*86)-1 downto STREAM_WSTB_WIDTH*85) <=  wstrb_in;
               when 86 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*87)-1 downto STREAM_WSTB_WIDTH*86) <=  wstrb_in;
               when 87 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*88)-1 downto STREAM_WSTB_WIDTH*87) <=  wstrb_in;
               when 88 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*89)-1 downto STREAM_WSTB_WIDTH*88) <=  wstrb_in;
               when 89 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*90)-1 downto STREAM_WSTB_WIDTH*89) <=  wstrb_in;
               when 90 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*91)-1 downto STREAM_WSTB_WIDTH*90) <=  wstrb_in;
               when 91 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*92)-1 downto STREAM_WSTB_WIDTH*91) <=  wstrb_in;
               when 92 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*93)-1 downto STREAM_WSTB_WIDTH*92) <=  wstrb_in;
               when 93 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*94)-1 downto STREAM_WSTB_WIDTH*93) <=  wstrb_in;
               when 94 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*95)-1 downto STREAM_WSTB_WIDTH*94) <=  wstrb_in;
               when 95 =>                                                                 
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*96)-1 downto STREAM_WSTB_WIDTH*95) <=  wstrb_in;
             
             
               when 96 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*97 )-1 downto STREAM_WSTB_WIDTH*96 ) <=  wstrb_in;
               when 97 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*98 )-1 downto STREAM_WSTB_WIDTH*97 ) <=  wstrb_in;
               when 98 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*99 )-1 downto STREAM_WSTB_WIDTH*98 ) <=  wstrb_in;
               when 99 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*100)-1 downto STREAM_WSTB_WIDTH*99 ) <=  wstrb_in;
               when 100 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*101)-1 downto STREAM_WSTB_WIDTH*100) <=  wstrb_in;
               when 101 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*102)-1 downto STREAM_WSTB_WIDTH*101) <=  wstrb_in;
               when 102 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*103)-1 downto STREAM_WSTB_WIDTH*102) <=  wstrb_in;
               when 103 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*104)-1 downto STREAM_WSTB_WIDTH*103) <=  wstrb_in;
               when 104 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*105)-1 downto STREAM_WSTB_WIDTH*104) <=  wstrb_in;
               when 105 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*106)-1 downto STREAM_WSTB_WIDTH*105) <=  wstrb_in;
               when 106 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*107)-1 downto STREAM_WSTB_WIDTH*106) <=  wstrb_in;
               when 107 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*108)-1 downto STREAM_WSTB_WIDTH*107) <=  wstrb_in;
               when 108 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*109)-1 downto STREAM_WSTB_WIDTH*108) <=  wstrb_in;
               when 109 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*110)-1 downto STREAM_WSTB_WIDTH*109) <=  wstrb_in;
               when 110 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*111)-1 downto STREAM_WSTB_WIDTH*110) <=  wstrb_in;
               when 111 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*112)-1 downto STREAM_WSTB_WIDTH*111) <=  wstrb_in;
               when 112 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*113)-1 downto STREAM_WSTB_WIDTH*112) <=  wstrb_in;
               when 113 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*114)-1 downto STREAM_WSTB_WIDTH*113) <=  wstrb_in;
               when 114 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*115)-1 downto STREAM_WSTB_WIDTH*114) <=  wstrb_in;
               when 115 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*116)-1 downto STREAM_WSTB_WIDTH*115) <=  wstrb_in;
               when 116 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*117)-1 downto STREAM_WSTB_WIDTH*116) <=  wstrb_in;
               when 117 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*118)-1 downto STREAM_WSTB_WIDTH*117) <=  wstrb_in;
               when 118 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*119)-1 downto STREAM_WSTB_WIDTH*118) <=  wstrb_in;
               when 119 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*120)-1 downto STREAM_WSTB_WIDTH*119) <=  wstrb_in;
               when 120 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*121)-1 downto STREAM_WSTB_WIDTH*120) <=  wstrb_in;
               when 121 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*122)-1 downto STREAM_WSTB_WIDTH*121) <=  wstrb_in;
               when 122 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*123)-1 downto STREAM_WSTB_WIDTH*122) <=  wstrb_in;
               when 123 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*124)-1 downto STREAM_WSTB_WIDTH*123) <=  wstrb_in;
               when 124 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*125)-1 downto STREAM_WSTB_WIDTH*124) <=  wstrb_in;
               when 125 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*126)-1 downto STREAM_WSTB_WIDTH*125) <=  wstrb_in;
               when 126 =>                                                                 
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*127)-1 downto STREAM_WSTB_WIDTH*126) <=  wstrb_in;
               
               when others => -- 127 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*128)-1 downto STREAM_WSTB_WIDTH*127) <=  wstrb_in;
             
             
             
             end case;
          
           end process DO_128XN_DEMUX; 
 
         
       end generate GEN_128XN;
  
 
  
  
  
  
  
  
  
  
  end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_wrdata_cntl.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_wrdata_cntl.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Write Data Controller.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library axi_datamover_v5_1_14;
  use axi_datamover_v5_1_14.axi_datamover_fifo;
  use axi_datamover_v5_1_14.axi_datamover_strb_gen2;
  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_wrdata_cntl is
    generic (
      
      C_REALIGNER_INCLUDED   : Integer range  0 to   1 :=  0;
        -- Indicates the Data Realignment function is included (external
        -- to this module)
      
      C_ENABLE_INDET_BTT     : Integer range  0 to   1 :=  0;
        -- Indicates the INDET BTT function is included (external
        -- to this module)
            
      C_SF_BYTES_RCVD_WIDTH  : Integer range  1 to  23 :=  1;
        -- Sets the width of the data2wsc_bytes_rcvd port used for 
        -- relaying the actual number of bytes received when Idet BTT is 
        -- enabled (C_ENABLE_INDET_BTT = 1)
      
      C_SEL_ADDR_WIDTH       : Integer range  1 to   8 :=  5;
        -- Sets the width of the LS bits of the transfer address that
        -- are being used to Demux write data to a wider AXI4 Write
        -- Data Bus
        
      C_DATA_CNTL_FIFO_DEPTH : Integer range  1 to  32 :=  4;
        -- Sets the depth of the internal command fifo used for the
        -- command queue
        
      C_MMAP_DWIDTH          : Integer range 32 to 1024 := 32;
        -- Indicates the native data width of the Read Data port
        
      C_STREAM_DWIDTH        : Integer range  8 to 1024 := 32;
        -- Sets the width of the Stream output data port
        
      C_TAG_WIDTH            : Integer range  1 to   8 :=  4;
        -- Indicates the width of the Tag field of the input command
        
      C_FAMILY               : String                  := "virtex7"
        -- Indicates the device family of the target FPGA
        
      
      );
    port (
      
      -- Clock and Reset inputs ----------------------------------------------
                                                                            --
      primary_aclk         : in  std_logic;                                 --
         -- Primary synchronization clock for the Master side               --
         -- interface and internal logic. It is also used                   --
         -- for the User interface synchronization when                     --
         -- C_STSCMD_IS_ASYNC = 0.                                          --
                                                                            --
      -- Reset input                                                        --
      mmap_reset           : in  std_logic;                                 --
         -- Reset used for the internal master logic                        --
      ------------------------------------------------------------------------
      

     
      
      -- Soft Shutdown internal interface ------------------------------------
                                                                            --
      rst2data_stop_request : in  std_logic;                                --
         -- Active high soft stop request to modules                        --
                                                                            --
      data2addr_stop_req    : Out std_logic;                                --
        -- Active high signal requesting the Address Controller             --
        -- to stop posting commands to the AXI Read Address Channel         --
                                                                            --
      data2rst_stop_cmplt   : Out std_logic;                                --
        -- Active high indication that the Data Controller has completed    --
        -- any pending transfers committed by the Address Controller        --
        -- after a stop has been requested by the Reset module.             --
      ------------------------------------------------------------------------
      
      
        
  
      -- Store and Forward support signals for external User logic ------------
                                                                             --
      wr_xfer_cmplt         : Out std_logic;                                 --
        -- Active high indication that the Data Controller has completed     --
        -- a single write data transfer on the AXI4 Write Data Channel.      --
        -- This signal is escentially echos the assertion of wlast sent      --
        -- to the AXI4.                                                      --
                                                                             --
      s2mm_ld_nxt_len        : out std_logic;                                --
        -- Active high pulse indicating a new xfer length has been queued    --
        -- to the WDC Cmd FIFO                                               --
                                                                             --
      s2mm_wr_len            : out std_logic_vector(7 downto 0);             --
        -- Bus indicating the AXI LEN value associated with the xfer command --
        -- loaded into the WDC Command FIFO.                                 --
      -------------------------------------------------------------------------
     
     
     
      -- AXI Write Data Channel Skid buffer I/O  ---------------------------------------
                                                                                      --
      data2skid_saddr_lsb   : out std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0);      --
        -- Write DATA output to skid buffer                                           --
                                                                                      --
      data2skid_wdata       : Out  std_logic_vector(C_STREAM_DWIDTH-1 downto 0);      --
        -- Write DATA output to skid buffer                                           --
                                                                                      --
      data2skid_wstrb       : Out  std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);  --
        -- Write DATA output to skid buffer                                           --
                                                                                      --
      data2skid_wlast       : Out  std_logic;                                         --
        -- Write LAST output to skid buffer                                           --
                                                                                      --
      data2skid_wvalid      : Out  std_logic;                                         --
        -- Write VALID output to skid buffer                                          --
                                                                                      --
      skid2data_wready      : In  std_logic;                                          --
        -- Write READY input from skid buffer                                         --
      ----------------------------------------------------------------------------------
     
 
 
      -- AXI Slave Stream In -----------------------------------------------------------
                                                                                      --
      s2mm_strm_wvalid   : In  std_logic;                                             --
        -- AXI Stream VALID input                                                     --
                                                                                      --
      s2mm_strm_wready   : Out  Std_logic;                                            --
        -- AXI Stream READY Output                                                    --
                                                                                      --
      s2mm_strm_wdata    : In  std_logic_vector(C_STREAM_DWIDTH-1 downto 0);          --
        -- AXI Stream data input                                                      --
                                                                                      --
      s2mm_strm_wstrb    : In std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);       --  
        -- AXI Stream STRB input                                                      --
                                                                                      --
      s2mm_strm_wlast    : In std_logic;                                              --
        -- AXI Stream LAST input                                                      --
      ----------------------------------------------------------------------------------
               
      
      
      -- Stream input sideband signal from Indeterminate BTT and/or DRE ----------------
                                                                                      --
      s2mm_strm_eop      : In std_logic;                                              --
        -- Stream End of Packet marker input. This is only used when Indeterminate    --
        -- BTT mode is enable. Otherwise it is ignored                                --
                                                                                      --
                                                                                      --
      s2mm_stbs_asserted : in  std_logic_vector(7 downto 0);                          --
        -- Indicates the number of asserted WSTRB bits for the                        --
        -- associated input stream data beat                                          --
                                                                                      --
                                                                                      --
      -- Realigner Underrun/overrun error flag used in non Indeterminate BTT          --
      -- Mode                                                                         --
      realign2wdc_eop_error  : In  std_logic ;                                        --
        -- Asserted active high and will only clear with reset. It is only used       --
        -- when Indeterminate BTT is not enabled and the Realigner Module is          --
        -- instantiated upstream from the WDC. The Realigner will detect overrun      --
        -- underrun conditions and will will relay these conditions via this signal.  --
      ----------------------------------------------------------------------------------        
      
      
                
                
                
      -- Command Calculator Interface --------------------------------------------------
                                                                                      --
      mstr2data_tag        : In std_logic_vector(C_TAG_WIDTH-1 downto 0);             --
         -- The next command tag                                                      --
                                                                                      --
      mstr2data_saddr_lsb  : In std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0);        --
         -- The next command start address LSbs to use for the write strb             --
         -- demux (only used if Stream data width is less than the MMap Dwidth).      --
                                                                                      --
      mstr2data_len        : In std_logic_vector(7 downto 0);                         --
         -- The LEN value output to the Address Channel                               --
                                                                                      --
      mstr2data_strt_strb  : In std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);     --
         -- The starting strobe value to use for the first stream data beat           --
                                                                                      --
      mstr2data_last_strb  : In std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);     --
         -- The endiing (LAST) strobe value to use for the last stream                --
         -- data beat                                                                 --
                                                                                      --
      mstr2data_drr        : In std_logic;                                            --
         -- The starting tranfer of a sequence of transfers                           --
                                                                                      --
      mstr2data_eof        : In std_logic;                                            --
         -- The endiing tranfer of a sequence of transfers                            --
                                                                                      --
      mstr2data_sequential : In std_logic;                                            --
         -- The next sequential tranfer of a sequence of transfers                    --
         -- spawned from a single parent command                                      --
                                                                                      --
      mstr2data_calc_error : In std_logic;                                            --
         -- Indication if the next command in the calculation pipe                    --
         -- has a calculation error                                                   --
                                                                                      --
      mstr2data_cmd_cmplt  : In std_logic;                                            --
         -- The final child tranfer of a parent command fetched from                  --
         -- the Command FIFO (not necessarily an EOF command)                         --
                                                                                      --
      mstr2data_cmd_valid  : In std_logic;                                            --
         -- The next command valid indication to the Data Channel                     --
         -- Controller for the AXI MMap                                               --
                                                                                      --
      data2mstr_cmd_ready  : Out std_logic ;                                          --
         -- Indication from the Data Channel Controller that the                      --
         -- command is being accepted on the AXI Address                              --
         -- Channel                                                                   --
      ----------------------------------------------------------------------------------
      
      
      
        
      -- Address Controller Interface --------------------------------------------------
                                                                                      --
      addr2data_addr_posted    : In std_logic ;                                       --
         -- Indication from the Address Channel Controller to the                     --
         -- Data Controller that an address has been posted to the                    --
         -- AXI Address Channel                                                       --
                                                                                      --
                                                                                      --
      data2addr_data_rdy       : out std_logic;                                       --
         -- Indication that the Data Channel is ready to send the first               --
         -- databeat of the next command on the write data channel.                   --
         -- This is used for the "wait for data" feature which keeps the              --
         -- address controller from issuing a transfer request until the              --
         -- corresponding data valid is asserted on the stream input. The             --
         -- WDC will continue to assert the output until an assertion on              --
         -- the addr2data_addr_posted is received.                                    --
       ---------------------------------------------------------------------------------
  
  
      
      -- Premature TLAST assertion error flag ------------------------------------------
                                                                                      --
      data2all_tlast_error : Out std_logic;                                           --
         -- When asserted, this indicates the data controller detected                --
         -- a premature TLAST assertion on the incoming data stream.                  --
       ---------------------------------------------------------------------------------     
      
      
      -- Data Controller Halted Status -------------------------------------------------
                                                                                      --
      data2all_dcntlr_halted : Out std_logic;                                         --
         -- When asserted, this indicates the data controller has satisfied           --
         -- all pending transfers queued by the Address Controller and is halted.     --
      ----------------------------------------------------------------------------------
      
       
 
      -- Input Stream Skid Buffer Halt control -----------------------------------------
                                                                                      --
      data2skid_halt       : Out std_logic;                                           --
         -- The data controller asserts this output for 1 primary clock period        --
         -- The pulse commands the MM2S Stream skid buffer to tun off outputs         --
         -- at the next tlast transmission.                                           --
      ----------------------------------------------------------------------------------
      
       
 
       
      -- Write Status Controller Interface ---------------------------------------------
                                                                                      --
      data2wsc_tag         : Out std_logic_vector(C_TAG_WIDTH-1 downto 0);            --
         -- The command tag                                                           --
                                                                                      --
      data2wsc_calc_err    : Out std_logic ;                                          --
         -- Indication that the current command out from the Cntl FIFO                --
         -- has a calculation error                                                   --
                                                                                      --
      data2wsc_last_err    : Out std_logic ;                                          --
        -- Indication that the current write transfer encountered a premature         --
        -- TLAST assertion on the incoming Stream Channel                             --
                                                                                      --
      data2wsc_cmd_cmplt   : Out std_logic ;                                          --
         -- Indication by the Data Channel Controller that the                        --
         -- corresponding status is the last status for a command                     --
         -- pulled from the command FIFO                                              --
                                                                                      --
      wsc2data_ready       : in  std_logic;                                           --
         -- Input from the Write Status Module indicating that the                    --
         -- Status Reg/FIFO is ready to accept data                                   --
                                                                                      --
      data2wsc_valid       : Out  std_logic;                                          --
         -- Output to the Command/Status Module indicating that the                   --
         -- Data Controller has valid tag and err indicators to write                 --
         -- to the Status module                                                      --
                                                                                      --
      data2wsc_eop         : Out  std_logic;                                          --
         -- Output to the Write Status Controller indicating that the                 --
         -- associated command status also corresponds to a End of Packet             --
         -- marker for the input Stream. This is only used when Inderminate           --
         -- BTT is enabled in the S2MM.                                               --
                                                                                      --
      data2wsc_bytes_rcvd  : Out std_logic_vector(C_SF_BYTES_RCVD_WIDTH-1 downto 0);  --
         -- Output to the Write Status Controller indicating the actual               --
         -- number of bytes received from the Stream input for the                    --
         -- corresponding command status. This is only used when Inderminate          --
         -- BTT is enabled in the S2MM.                                               --
                                                                                      --
      wsc2mstr_halt_pipe   : In  std_logic                                            --
         -- Indication to Halt the Data and Address Command pipeline due              --
         -- to the Status FIFO going full or an internal error being logged           --
      ----------------------------------------------------------------------------------

    
      
      );
  
  end entity axi_datamover_wrdata_cntl;
  
  
  architecture implementation of axi_datamover_wrdata_cntl  is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    
    -- Function declaration   ----------------------------------------
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_dbeat_residue_width
    --
    -- Function Description:
    --  Calculates the number of Least significant bits of the BTT field
    -- that are unused for the LEN calculation
    --
    -------------------------------------------------------------------
    function funct_get_dbeat_residue_width (bytes_per_beat : integer) return integer is

      Variable temp_dbeat_residue_width : Integer := 0; -- 8-bit stream

    begin

      case bytes_per_beat is
        
        when 128 =>  -- 1024 bits          -- Added per Per CR616409
            temp_dbeat_residue_width := 7; -- Added per Per CR616409
        when 64 =>   -- 512 bits           -- Added per Per CR616409
            temp_dbeat_residue_width := 6; -- Added per Per CR616409
        when 32 =>   -- 256 bits           
            temp_dbeat_residue_width := 5;
        when 16 =>   -- 128 bits
            temp_dbeat_residue_width := 4;
        when 8 =>    -- 64 bits
            temp_dbeat_residue_width := 3;
        when 4 =>    -- 32 bits
            temp_dbeat_residue_width := 2;
        when 2 =>    -- 16 bits
            temp_dbeat_residue_width := 1;
        when others =>  -- assume 1-byte transfers
            temp_dbeat_residue_width := 0;
      end case;

      Return (temp_dbeat_residue_width);

    end function funct_get_dbeat_residue_width;




    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_set_cnt_width
    --
    -- Function Description:
    --    Sets a count width based on a fifo depth. A depth of 4 or less
    -- is a special case which requires a minimum count width of 3 bits.
    --
    -------------------------------------------------------------------
    function funct_set_cnt_width (fifo_depth : integer) return integer is
    
      Variable temp_cnt_width : Integer := 4;
    
    begin
    
      
      if (fifo_depth <= 4) then
      
         temp_cnt_width := 3;
      
      elsif (fifo_depth <= 8) then
      
         temp_cnt_width := 4;
      
      elsif (fifo_depth <= 16) then
      
         temp_cnt_width := 5;
      
      elsif (fifo_depth <= 32) then
      
         temp_cnt_width := 6;
      
      else  -- fifo depth <= 64
      
         temp_cnt_width := 7;
      
      end if;
      
      Return (temp_cnt_width);
       
       
    end function funct_set_cnt_width;
    
 
 
 
  
    -- Constant Declarations  --------------------------------------------
    
    Constant STRM_STRB_WIDTH        : integer := C_STREAM_DWIDTH/8;
    Constant LEN_OF_ZERO            : std_logic_vector(7 downto 0) := (others => '0');
    Constant USE_SYNC_FIFO          : integer := 0;
    Constant REG_FIFO_PRIM          : integer := 0; 
    Constant BRAM_FIFO_PRIM         : integer := 1; 
    Constant SRL_FIFO_PRIM          : integer := 2; 
    Constant FIFO_PRIM_TYPE         : integer := SRL_FIFO_PRIM; 
    Constant TAG_WIDTH              : integer := C_TAG_WIDTH;
    Constant SADDR_LSB_WIDTH        : integer := C_SEL_ADDR_WIDTH;
    Constant LEN_WIDTH              : integer := 8;
    Constant STRB_WIDTH             : integer := C_STREAM_DWIDTH/8;
    Constant DRR_WIDTH              : integer := 1;
    Constant EOF_WIDTH              : integer := 1;
    Constant CALC_ERR_WIDTH         : integer := 1;
    Constant CMD_CMPLT_WIDTH        : integer := 1;
    Constant SEQUENTIAL_WIDTH       : integer := 1;
    Constant DCTL_FIFO_WIDTH        : Integer := TAG_WIDTH        +  -- Tag field
                                                 SADDR_LSB_WIDTH  +  -- LS Address field width
                                                 LEN_WIDTH        +  -- LEN field
                                                 STRB_WIDTH       +  -- Starting Strobe field
                                                 STRB_WIDTH       +  -- Ending Strobe field
                                                 DRR_WIDTH        +  -- DRE Re-alignment Request Flag Field
                                                 EOF_WIDTH        +  -- EOF flag field
                                                 SEQUENTIAL_WIDTH +  -- Sequential command flag
                                                 CMD_CMPLT_WIDTH  +  -- Command Complete Flag
                                                 CALC_ERR_WIDTH;     -- Calc error flag
    
    Constant TAG_STRT_INDEX         : integer := 0;
    Constant SADDR_LSB_STRT_INDEX   : integer := TAG_STRT_INDEX + TAG_WIDTH;
    Constant LEN_STRT_INDEX         : integer := SADDR_LSB_STRT_INDEX + SADDR_LSB_WIDTH;
    Constant STRT_STRB_STRT_INDEX   : integer := LEN_STRT_INDEX + LEN_WIDTH;
    Constant LAST_STRB_STRT_INDEX   : integer := STRT_STRB_STRT_INDEX + STRB_WIDTH;
    Constant DRR_STRT_INDEX         : integer := LAST_STRB_STRT_INDEX + STRB_WIDTH;
    Constant EOF_STRT_INDEX         : integer := DRR_STRT_INDEX + DRR_WIDTH;
    Constant SEQUENTIAL_STRT_INDEX  : integer := EOF_STRT_INDEX + EOF_WIDTH;
    Constant CMD_CMPLT_STRT_INDEX   : integer := SEQUENTIAL_STRT_INDEX+SEQUENTIAL_WIDTH;
    Constant CALC_ERR_STRT_INDEX    : integer := CMD_CMPLT_STRT_INDEX+CMD_CMPLT_WIDTH;        
    Constant ADDR_INCR_VALUE        : integer := C_STREAM_DWIDTH/8;
    
    Constant ADDR_POSTED_CNTR_WIDTH : integer := funct_set_cnt_width(C_DATA_CNTL_FIFO_DEPTH); 
    
    
    
    Constant ADDR_POSTED_ZERO       : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := (others => '0');
    Constant ADDR_POSTED_ONE        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := TO_UNSIGNED(1, ADDR_POSTED_CNTR_WIDTH);
    Constant ADDR_POSTED_MAX        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := (others => '1');
                    
    
    
    
    
    -- Signal Declarations  --------------------------------------------
    
    signal sig_get_next_dqual        : std_logic := '0';
    signal sig_last_mmap_dbeat       : std_logic := '0';
    signal sig_last_mmap_dbeat_reg   : std_logic := '0';
    signal sig_mmap2data_ready       : std_logic := '0';
    signal sig_data2mmap_valid       : std_logic := '0';
    signal sig_data2mmap_last        : std_logic := '0';
    signal sig_data2mmap_data        : std_logic_vector(C_STREAM_DWIDTH-1 downto 0) := (others => '0');
    signal sig_ld_new_cmd            : std_logic := '0';
    signal sig_ld_new_cmd_reg        : std_logic := '0';
    signal sig_cmd_cmplt_reg         : std_logic := '0';
    signal sig_calc_error_reg        : std_logic := '0';
    signal sig_tag_reg               : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_lsb_reg          : std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_strt_strb_reg         : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_last_strb_reg         : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_addr_posted           : std_logic := '0';
    signal sig_dqual_rdy             : std_logic := '0';
    signal sig_good_mmap_dbeat       : std_logic := '0';
    signal sig_first_dbeat           : std_logic := '0';
    signal sig_last_dbeat            : std_logic := '0';
    signal sig_single_dbeat          : std_logic := '0';
    signal sig_new_len_eq_0          : std_logic := '0';
    signal sig_dbeat_cntr            : unsigned(7 downto 0) := (others => '0');
    Signal sig_dbeat_cntr_int        : Integer range 0 to 255 := 0;
    signal sig_dbeat_cntr_eq_0       : std_logic := '0';
    signal sig_dbeat_cntr_eq_1       : std_logic := '0';
    signal sig_wsc_ready             : std_logic := '0';
    signal sig_push_to_wsc           : std_logic := '0';
    signal sig_push_to_wsc_cmplt     : std_logic := '0';
    signal sig_set_push2wsc          : std_logic := '0';
    signal sig_data2wsc_tag          : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_data2wsc_calc_err     : std_logic := '0';
    signal sig_data2wsc_last_err     : std_logic := '0';
    signal sig_data2wsc_cmd_cmplt    : std_logic := '0';
    signal sig_tlast_error           : std_logic := '0';
    signal sig_tlast_error_strbs     : std_logic := '0';
    signal sig_end_stbs_match_err    : std_logic := '0';
    signal sig_tlast_error_reg       : std_logic := '0';
    signal sig_cmd_is_eof            : std_logic := '0';
    signal sig_push_err2wsc          : std_logic := '0';
    signal sig_tlast_error_ovrrun    : std_logic := '0';
    signal sig_tlast_error_undrrun   : std_logic := '0';
    signal sig_next_tag_reg          : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_strt_strb_reg    : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_next_last_strb_reg    : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_next_eof_reg          : std_logic := '0';
    signal sig_next_sequential_reg   : std_logic := '0';
    signal sig_next_cmd_cmplt_reg    : std_logic := '0';
    signal sig_next_calc_error_reg   : std_logic := '0';
    signal sig_pop_dqual_reg         : std_logic := '0';
    signal sig_push_dqual_reg        : std_logic := '0';
    signal sig_dqual_reg_empty       : std_logic := '0';
    signal sig_dqual_reg_full        : std_logic := '0';
    signal sig_addr_posted_cntr      : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_posted_cntr_eq_0 : std_logic := '0';
    signal sig_addr_posted_cntr_max  : std_logic := '0';
    signal sig_decr_addr_posted_cntr : std_logic := '0';
    signal sig_incr_addr_posted_cntr : std_logic := '0';
    signal sig_addr_posted_cntr_eq_1 : std_logic := '0';
    signal sig_apc_going2zero        : std_logic := '0';
    signal sig_aposted_cntr_ready    : std_logic := '0';
    signal sig_addr_chan_rdy         : std_logic := '0';
    Signal sig_no_posted_cmds        : std_logic := '0';
    signal sig_ls_addr_cntr          : unsigned(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_incr_ls_addr_cntr     : std_logic := '0';
    signal sig_addr_incr_unsgnd      : unsigned(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    Signal sig_cmd_fifo_data_in      : std_logic_vector(DCTL_FIFO_WIDTH-1 downto 0) := (others => '0');
    Signal sig_cmd_fifo_data_out     : std_logic_vector(DCTL_FIFO_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_tag         : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_sadddr_lsb  : std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_len         : std_logic_vector(7 downto 0) := (others => '0');
    signal sig_fifo_next_strt_strb   : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_fifo_next_last_strb   : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_fifo_next_drr         : std_logic := '0';
    signal sig_fifo_next_eof         : std_logic := '0';
    signal sig_fifo_next_cmd_cmplt   : std_logic := '0';
    signal sig_fifo_next_sequential  : std_logic := '0';
    signal sig_fifo_next_calc_error  : std_logic := '0';
    signal sig_cmd_fifo_empty        : std_logic := '0';
    signal sig_fifo_wr_cmd_valid     : std_logic := '0';
    signal sig_fifo_wr_cmd_ready     : std_logic := '0';
    signal sig_fifo_rd_cmd_valid     : std_logic := '0';
    signal sig_fifo_rd_cmd_ready     : std_logic := '0';
    signal sig_sequential_push       : std_logic := '0';
    signal sig_clr_dqual_reg         : std_logic := '0';
    signal sig_tlast_err_stop        : std_logic := '0';
    signal sig_halt_reg              : std_logic := '0';
    signal sig_halt_reg_dly1         : std_logic := '0';
    signal sig_halt_reg_dly2         : std_logic := '0';
    signal sig_halt_reg_dly3         : std_logic := '0';
    signal sig_data2skid_halt        : std_logic := '0';
    signal sig_stop_wvalid           : std_logic := '0';
    signal sig_data2rst_stop_cmplt   : std_logic := '0';
    signal sig_s2mm_strm_wready      : std_logic := '0';
    signal sig_good_strm_dbeat       : std_logic := '0';
    signal sig_halt_strb             : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_sfhalt_next_strt_strb : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_wfd_simult_clr_set    : std_logic := '0';
    signal sig_wr_xfer_cmplt         : std_logic := '0';
    signal sig_s2mm_ld_nxt_len       : std_logic := '0';
    signal sig_s2mm_wr_len           : std_logic_vector(7 downto 0) := (others => '0');
    signal sig_data2mstr_cmd_ready   : std_logic := '0';
    signal sig_spcl_push_err2wsc     : std_logic := '0';
    
    
                               
  begin --(architecture implementation)
  
    -- Command calculator handshake
    data2mstr_cmd_ready <= sig_data2mstr_cmd_ready;
    
    
    -- Write Data Channel Skid Buffer Port assignments
    sig_mmap2data_ready  <= skid2data_wready     ;
    data2skid_wvalid     <= sig_data2mmap_valid  ;
    data2skid_wlast      <= sig_data2mmap_last   ;
    data2skid_wdata      <= sig_data2mmap_data   ;
    data2skid_saddr_lsb  <= sig_addr_lsb_reg     ;
    
    -- AXI MM2S Stream Channel Port assignments           
    sig_data2mmap_data   <= s2mm_strm_wdata      ;

    
    -- Premature TLAST assertion indication
    data2all_tlast_error <= sig_tlast_error_reg  ;
    
    
    
    
   
    -- Stream Input Ready Handshake
    s2mm_strm_wready     <= sig_s2mm_strm_wready ;                                           
    
    
    
    sig_good_strm_dbeat  <= s2mm_strm_wvalid and
                            sig_s2mm_strm_wready;
    
    
    sig_data2mmap_last   <= sig_dbeat_cntr_eq_0 and
                            sig_dqual_rdy;

                      

    -- Write Status Block interface signals
    data2wsc_valid       <= sig_push_to_wsc and
                            not(sig_tlast_err_stop) ; -- only allow 1 status write on TLAST errror
    sig_wsc_ready        <= wsc2data_ready          ;
    data2wsc_tag         <= sig_data2wsc_tag        ;   
    data2wsc_calc_err    <= sig_data2wsc_calc_err   ; 
    data2wsc_last_err    <= sig_data2wsc_last_err   ; 
    data2wsc_cmd_cmplt   <= sig_data2wsc_cmd_cmplt  ;   
    
                                      
    -- Address Channel Controller synchro pulse input                  
    sig_addr_posted      <= addr2data_addr_posted;
                                                        
    
    
    -- Request to halt the Address Channel Controller                  
    data2addr_stop_req   <= sig_halt_reg or
                            sig_tlast_error_reg;
 
    
    -- Halted flag to the reset module                  
    data2rst_stop_cmplt  <= sig_data2rst_stop_cmplt;
    
    
    -- Indicate the Write Data Controller is always ready
    data2addr_data_rdy   <= '1'; 
    
    
    
    -- Write Transfer Completed Status output 
    wr_xfer_cmplt        <= sig_wr_xfer_cmplt ;
    
    -- New LEN value is being loaded 
    s2mm_ld_nxt_len      <= sig_s2mm_ld_nxt_len;
    
    -- The new LEN value
    s2mm_wr_len          <= sig_s2mm_wr_len;
    
    
     
     
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_WR_CMPLT_FLAG
    --
    -- Process Description:
    --   Implements the status flag indicating that a write data 
    -- transfer has completed. This is an echo of a wlast assertion
    -- and a qualified data beat on the AXI4 Write Data Channel.
    --
    -------------------------------------------------------------
    IMP_WR_CMPLT_FLAG : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset = '1') then
    
             sig_wr_xfer_cmplt <= '0';
    
           else
    
             sig_wr_xfer_cmplt <= sig_data2mmap_last and 
                                  sig_good_strm_dbeat;
                                  
           end if; 
        end if;       
      end process IMP_WR_CMPLT_FLAG; 
     
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_OMIT_INDET_BTT
    --
    -- If Generate Description:
    --   Omits any Indeterminate BTT Support logic and includes
    -- any error detection needed in Non Indeterminate BTT mode.
    --
    ------------------------------------------------------------
    GEN_OMIT_INDET_BTT : if (C_ENABLE_INDET_BTT = 0) generate
    
      begin
   
       
        
        
        sig_sfhalt_next_strt_strb <= sig_fifo_next_strt_strb;
        
        
        
        
        -- Just housekeep the output port signals
        
        data2wsc_eop         <= '0';
   
        data2wsc_bytes_rcvd  <= (others => '0');
   
       
        
        
        
        -- WRSTRB logic ------------------------------
                  

        -- Generate the Write Strobes for the MMap Write Data Channel
        -- for the non Indeterminate BTT Case
        data2skid_wstrb  <= sig_strt_strb_reg 
          When (sig_first_dbeat  = '1')
          Else  sig_last_strb_reg
          When  (sig_last_dbeat  = '1')
          Else (others => '1');

        
                 
        -- Generate the Stream Ready for the Stream input side
        sig_s2mm_strm_wready <= sig_halt_reg            or  -- force tready if a halt requested
                                (sig_mmap2data_ready    and
                                sig_addr_chan_rdy       and -- This puts combinational logic in the stream WREADY path
                                sig_dqual_rdy           and
                                not(sig_calc_error_reg) and
                                not(sig_tlast_error_reg));   -- Stop the stream channel at a overrun/underrun detection
        
         
        
        -- MMap Write Data Channel Valid Handshaking
        sig_data2mmap_valid <= (s2mm_strm_wvalid       or
                               sig_tlast_error_reg     or  -- force valid if TLAST error 
                               sig_halt_reg       )    and -- force valid if halt requested       
                               sig_addr_chan_rdy       and -- xfers are commited on the address channel and       
                               sig_dqual_rdy           and -- there are commands in the command fifo        
                               not(sig_calc_error_reg) and
                               not(sig_stop_wvalid);       -- gate off wvalid immediately after a wlast for 1 clk
                                                           -- or when the soft shutdown has completed
              
      
        
        
        
        
        
        ------------------------------------------------------------
        -- If Generate
        --
        -- Label: GEN_LOCAL_ERR_DETECT
        --
        -- If Generate Description:
        --  Implements the local overrun and underrun detection when
        -- the S2MM Realigner is not included.
        --
        --
        ------------------------------------------------------------
        GEN_LOCAL_ERR_DETECT : if (C_REALIGNER_INCLUDED = 0) generate
        
        
           begin
        
             -------  Input Stream TLAST assertion error ------------------------------- 
             
             
             sig_tlast_error_ovrrun <= sig_cmd_is_eof       and
                                       sig_dbeat_cntr_eq_0  and
                                       sig_good_mmap_dbeat  and
                                       not(s2mm_strm_wlast);
             
             
             
             sig_tlast_error_undrrun <= s2mm_strm_wlast     and
                                        sig_good_mmap_dbeat and
                                        (not(sig_dbeat_cntr_eq_0) or
                                         not(sig_cmd_is_eof));
             
                    
                    
             sig_end_stbs_match_err  <=  '1'                            -- Set flag if the calculated end strobe value
               When ((s2mm_strm_wstrb    /= sig_next_last_strb_reg) and -- does not match the received strobe value 
                    (s2mm_strm_wlast     = '1') and                     -- at TLAST assertion
                    (sig_good_mmap_dbeat = '1'))                        -- Qualified databeat
               Else '0';
                    
                                 
             sig_tlast_error <=  (sig_tlast_error_ovrrun  or
                                  sig_tlast_error_undrrun or
                                  sig_end_stbs_match_err) and
                                  not(sig_halt_reg);          -- Suppress TLAST error when in soft shutdown 
                                 
                                 
             
             -- Just housekeep this when local TLAST error detection is used
             sig_spcl_push_err2wsc <= '0';
            
            
            
           
           end generate GEN_LOCAL_ERR_DETECT;
        
        
 
 
 
        ------------------------------------------------------------
        -- If Generate
        --
        -- Label: GEN_EXTERN_ERR_DETECT
        --
        -- If Generate Description:
        --  Omits the local overrun and underrun detection and relies
        -- on the S2MM Realigner for the detection.
        --
        ------------------------------------------------------------
        GEN_EXTERN_ERR_DETECT : if (C_REALIGNER_INCLUDED = 1) generate
        
        
           begin
        
 
              sig_tlast_error_undrrun <= '0';  -- not used here
 
              sig_tlast_error_ovrrun  <= '0';  -- not used here
 
              sig_end_stbs_match_err  <= '0';  -- not used here
 
 
           
              sig_tlast_error <= realign2wdc_eop_error and  -- External error detection asserted
                                 not(sig_halt_reg);         -- Suppress TLAST error when in soft shutdown
               
             
             
              -- Special case for pushing error status when timing is such that no 
              -- addresses have been posted to AXI and a TLAST error has been detected  
              -- by the Realigner module and propagated in from the Stream input side.
              sig_spcl_push_err2wsc <= sig_tlast_error_reg     and
                                       not(sig_tlast_err_stop) and
                                       not(sig_addr_chan_rdy );
             
             
             
             
             
             
               
               
           
           end generate GEN_EXTERN_ERR_DETECT;
 
 
 
 
 
 
 
        
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_TLAST_ERR_REG
        --
        -- Process Description:
        --  Implements a sample and hold flop for the flag indicating
        -- that the input Stream TLAST assertion was not at the expected
        -- data beat relative to the commanded number of databeats
        -- from the associated command from the SCC or PCC.
        -------------------------------------------------------------
        IMP_TLAST_ERR_REG : process (primary_aclk)
           begin
             if (primary_aclk'event and primary_aclk = '1') then
                if (mmap_reset = '1') then
                  sig_tlast_error_reg <= '0';
                elsif (sig_tlast_error = '1') then
                  sig_tlast_error_reg <= '1';
                else
                  null;  -- hold current state
                end if; 
             end if;       
           end process IMP_TLAST_ERR_REG; 
           
           
           
           
           
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_TLAST_ERROR_STOP
        --
        -- Process Description:
        --  Implements the flop to generate a stop flag once the TLAST
        -- error condition has been relayed to the Write Status 
        -- Controller. This stop flag is used to prevent any more 
        -- pushes to the Write Status Controller.
        --
        -------------------------------------------------------------
        IMP_TLAST_ERROR_STOP : process (primary_aclk)
           begin
             if (primary_aclk'event and primary_aclk = '1') then
                if (mmap_reset = '1') then
                  sig_tlast_err_stop <= '0';
                elsif (sig_tlast_error_reg   = '1' and
                       sig_push_to_wsc_cmplt = '1') then
                  sig_tlast_err_stop <= '1';
                else
                  null; -- Hold State
                end if; 
             end if;       
           end process IMP_TLAST_ERROR_STOP; 
           
           
           
      
      
      
      end generate GEN_OMIT_INDET_BTT;
   
   













    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_INDET_BTT
    --
    -- If Generate Description:
    --   Includes any Indeterminate BTT Support logic. Primarily
    -- this is a counter for the input stream bytes received. The
    -- received byte count is relayed to the Write Status Controller
    -- for each parent command completed.
    -- When a packet completion is indicated via the EOP marker
    -- assertion, the status to the Write Status Controller also
    -- indicates the EOP condition.
    -- Note that underrun and overrun detection/error flagging
    -- is disabled in Indeterminate BTT Mode.
    --
    ------------------------------------------------------------
    GEN_INDET_BTT : if (C_ENABLE_INDET_BTT = 1) generate
    
      -- local constants
      Constant BYTE_CNTR_WIDTH          : integer := C_SF_BYTES_RCVD_WIDTH;
      Constant NUM_ZEROS_WIDTH          : integer := 8;
      Constant BYTES_PER_DBEAT          : integer := C_STREAM_DWIDTH/8;
      Constant STRBGEN_ADDR_SLICE_WIDTH : integer := 
                                          funct_get_dbeat_residue_width(BYTES_PER_DBEAT);
      
      Constant STRBGEN_ADDR_0            : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
      
      
      
      -- local signals
      signal lsig_byte_cntr             : unsigned(BYTE_CNTR_WIDTH-1 downto 0) := (others => '0');
      signal lsig_byte_cntr_incr_value  : unsigned(BYTE_CNTR_WIDTH-1 downto 0) := (others => '0');
      signal lsig_ld_byte_cntr          : std_logic := '0';
      signal lsig_incr_byte_cntr        : std_logic := '0';
      signal lsig_clr_byte_cntr         : std_logic := '0';
      signal lsig_end_of_cmd_reg        : std_logic := '0';
      signal lsig_eop_s_h_reg           : std_logic := '0';
      signal lsig_eop_reg               : std_logic := '0';
      signal sig_strbgen_addr           : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
      signal sig_strbgen_bytes          : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH   downto 0) := (others => '0');
      
      
      
      
      begin
   
       
        -- Assign the outputs to the Write Status Controller
        data2wsc_eop         <= lsig_eop_reg and 
                                not(sig_next_calc_error_reg);
         
        data2wsc_bytes_rcvd  <= STD_LOGIC_VECTOR(lsig_byte_cntr);
        
 
 
        -- WRSTRB logic ------------------------------
      
      
      
        --sig_strbgen_bytes <= (others => '1'); -- set to the max value
      
        
        -- set the length to the max number of bytes per databeat
        sig_strbgen_bytes <=  STD_LOGIC_VECTOR(TO_UNSIGNED(BYTES_PER_DBEAT, STRBGEN_ADDR_SLICE_WIDTH+1));
        
        
        
        
        
        
        sig_strbgen_addr  <= STD_LOGIC_VECTOR(RESIZE(UNSIGNED(sig_fifo_next_sadddr_lsb), 
                                                     STRBGEN_ADDR_SLICE_WIDTH)) ;
 
 
 
      
      ------------------------------------------------------------
      -- Instance: I_STRT_STRB_GEN 
      --
      -- Description:
      --  Strobe generator used to generate the starting databeat
      -- strobe value for soft shutdown case where the S2MM has to 
      -- flush out all of the transfers that have been committed
      -- to the AXI Write address channel. Starting Strobes must
      -- match the committed address offest for each transfer. 
      -- 
      ------------------------------------------------------------
      I_STRT_STRB_GEN : entity axi_datamover_v5_1_14.axi_datamover_strb_gen2
      generic map (
                            
        C_OP_MODE            =>  0                         , -- 0 = Offset/Length mode
        C_STRB_WIDTH         =>  BYTES_PER_DBEAT           ,   
        C_OFFSET_WIDTH       =>  STRBGEN_ADDR_SLICE_WIDTH  ,   
        C_NUM_BYTES_WIDTH    =>  STRBGEN_ADDR_SLICE_WIDTH+1           
    
        )
      port map (
        
        start_addr_offset    =>  sig_strbgen_addr         , 
        end_addr_offset      =>  STRBGEN_ADDR_0           , -- not used in op mode 0
        num_valid_bytes      =>  sig_strbgen_bytes        , 
        strb_out             =>  sig_sfhalt_next_strt_strb   
    
        );
                                  
     


   
   

        -- Generate the WSTRB to use during soft shutdown 
        sig_halt_strb  <= sig_strt_strb_reg 
          When (sig_first_dbeat   = '1' or
                sig_single_dbeat  = '1')
          Else  (others => '1');

           
                  
        -- Generate the Write Strobes for the MMap Write Data Channel
        -- for the Indeterminate BTT case. Strobes come from the Stream
        -- input from the Indeterminate BTT module during normal operation.
        -- However, during soft shutdown, those strobes become unpredictable
        -- so generated strobes have to be used.
        data2skid_wstrb <=  sig_halt_strb
          When (sig_halt_reg = '1')
        
          Else s2mm_strm_wstrb;
          
          
          
        -- Generate the Stream Ready for the Stream input side
        sig_s2mm_strm_wready <=  sig_halt_reg             or -- force tready if a halt requested
                                 (sig_mmap2data_ready    and -- MMap is accepting the xfers
                                 sig_addr_chan_rdy       and -- xfers are commited on the address channel and 
                                 sig_dqual_rdy           and -- there are commands in the command fifo        
                                 not(sig_calc_error_reg) and -- No internal error                             
                                 not(sig_stop_wvalid));      -- Gate off stream ready immediately after a wlast for 1 clk
                                                             -- or when the soft shutdown has completed
         
        
        -- MMap Write Data Channel Valid Handshaking
        sig_data2mmap_valid <= (s2mm_strm_wvalid        or -- Normal Stream input valid       
                               sig_halt_reg       )    and -- force valid if halt requested       
                               sig_addr_chan_rdy       and -- xfers are commited on the address channel and       
                               sig_dqual_rdy           and -- there are commands in the command fifo        
                               not(sig_calc_error_reg) and -- No internal error
                               not(sig_stop_wvalid);       -- Gate off wvalid immediately after a wlast for 1 clk
                                                           -- or when the soft shutdown has completed
              
      
         
        -- TLAST Error housekeeping for Indeterminate BTT Mode
        -- There is no Underrun/overrun in Stroe and Forward mode 
         
        sig_tlast_error_ovrrun  <= '0'; -- Not used with Indeterminate BTT
        sig_tlast_error_undrrun <= '0'; -- Not used with Indeterminate BTT
        sig_end_stbs_match_err  <= '0'; -- Not used with Indeterminate BTT
        sig_tlast_error         <= '0'; -- Not used with Indeterminate BTT
        sig_tlast_error_reg     <= '0'; -- Not used with Indeterminate BTT
        sig_tlast_err_stop      <= '0'; -- Not used with Indeterminate BTT
        
        
        
        
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_EOP_REG_FLOP
        --
        -- Process Description:
        --  Register the End of Packet marker.
        --
        -------------------------------------------------------------
        IMP_EOP_REG_FLOP : process (primary_aclk)
          begin
            if (primary_aclk'event and primary_aclk = '1') then
               if (mmap_reset = '1') then
        
                 lsig_end_of_cmd_reg <= '0';
                 lsig_eop_reg        <= '0';
               
               
               Elsif (sig_good_strm_dbeat = '1') Then
               
        
                 lsig_end_of_cmd_reg <= sig_next_cmd_cmplt_reg and
                                        s2mm_strm_wlast;
                 
                 lsig_eop_reg        <= s2mm_strm_eop;
               
               else

                 null; -- hold current state  
                   
               end if; 
            end if;       
          end process IMP_EOP_REG_FLOP; 
        
        
        
        
 
        -----  Byte Counter Logic -----------------------------------------------
        -- The Byte counter reflects the actual byte count received on the 
        -- Stream input for each parent command loaded into the S2MM command
        -- FIFO. Thus it counts input bytes until the command complete qualifier
        -- is set and the TLAST input from the Stream input.
      
      
        lsig_clr_byte_cntr        <= lsig_end_of_cmd_reg and   -- Clear if a new stream packet does not start 
                                     not(sig_good_strm_dbeat); -- immediately after the previous one finished.    
        
     
        lsig_ld_byte_cntr         <= lsig_end_of_cmd_reg and -- Only load if a new stream packet starts       
                                     sig_good_strm_dbeat;    -- immediately after the previous one finished.       
        
        lsig_incr_byte_cntr       <= sig_good_strm_dbeat; 
        
        
        lsig_byte_cntr_incr_value <=  RESIZE(UNSIGNED(s2mm_stbs_asserted), 
                                                       BYTE_CNTR_WIDTH);
     
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_BYTE_CMTR
        --
        -- Process Description:
        -- Keeps a running byte count per burst packet loaded into the 
        -- xfer FIFO. It is based on the strobes set on the incoming
        -- Stream dbeat.
        --
        -------------------------------------------------------------
        IMP_BYTE_CMTR : process (primary_aclk)
           begin
             if (primary_aclk'event and primary_aclk = '1') then
               if (mmap_reset         = '1' or
                   lsig_clr_byte_cntr = '1') then 

                 lsig_byte_cntr <= (others => '0');
                 
               elsif (lsig_ld_byte_cntr = '1') then

                 lsig_byte_cntr <= lsig_byte_cntr_incr_value;
                 
               elsif (lsig_incr_byte_cntr = '1') then

                 lsig_byte_cntr <= lsig_byte_cntr + lsig_byte_cntr_incr_value;
                 
               else
                 null;  -- hold current value
               end if; 
             end if;       
           end process IMP_BYTE_CMTR; 
     
     
        
 
   
      end generate GEN_INDET_BTT;
   
   
    
    
    
    
    
    
              
    
    -- Internal logic ------------------------------
    
    sig_good_mmap_dbeat  <= sig_mmap2data_ready and 
                            sig_data2mmap_valid;
    
    
    sig_last_mmap_dbeat  <= sig_good_mmap_dbeat and 
                            sig_data2mmap_last;
     
     
    sig_get_next_dqual   <= sig_last_mmap_dbeat; 
    
    
    
    
    
    
         
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_LAST_DBEAT
    --
    -- Process Description:
    --   This implements a FLOP that creates a pulse
    -- indicating the LAST signal for an outgoing write data channel
    -- has been sent. Note that it is possible to have back to 
    -- back LAST databeats.
    --
    -------------------------------------------------------------
    REG_LAST_DBEAT : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset = '1') then

             sig_last_mmap_dbeat_reg <= '0';
             
           else
             
             sig_last_mmap_dbeat_reg <= sig_last_mmap_dbeat;
             
           end if; 
         end if;       
       end process REG_LAST_DBEAT; 
  
 
 
 
 
    
    
    -----  Write Status Interface Stuff --------------------------
    
    sig_push_to_wsc_cmplt <= sig_push_to_wsc and sig_wsc_ready;
    
    
    sig_set_push2wsc      <= (sig_good_mmap_dbeat and
                             sig_dbeat_cntr_eq_0) or
                             sig_push_err2wsc     or
                             sig_spcl_push_err2wsc;   -- Special case from CR616212
                             
    
    
    
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_INTERR_PUSH_FLOP
    --
    -- Process Description:
    -- Generate a 1 clock wide pulse when a calc error has propagated
    -- from the Command Calculator. This pulse is used to force a 
    -- push of the error status to the Write Status Controller
    -- without a AXI transfer completion.
    --
    -------------------------------------------------------------
    IMP_INTERR_PUSH_FLOP : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset       = '1' or
                sig_push_err2wsc = '1') then
              sig_push_err2wsc <= '0';
            elsif (sig_ld_new_cmd_reg = '1' and
                   sig_calc_error_reg = '1') then
              sig_push_err2wsc <= '1';
            else
              null; -- hold state
            end if; 
         end if;       
       end process IMP_INTERR_PUSH_FLOP; 
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_PUSH2WSC_FLOP
    --
    -- Process Description:
    -- Implements a Sample and hold register for the outbound status
    -- signals to the Write Status Controller (WSC). This register
    -- has to support back to back transfer completions.
    --
    -------------------------------------------------------------
    IMP_PUSH2WSC_FLOP : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset             = '1'  or
               (sig_push_to_wsc_cmplt = '1'  and
                sig_set_push2wsc      = '0')) then
              
              sig_push_to_wsc        <= '0';
              sig_data2wsc_tag       <=  (others => '0');
              sig_data2wsc_calc_err  <=  '0';
              sig_data2wsc_last_err  <=  '0';
              sig_data2wsc_cmd_cmplt <=  '0';
              
            elsif (sig_set_push2wsc   = '1' and 
                   sig_tlast_err_stop = '0') then
              
              sig_push_to_wsc        <= '1';
              sig_data2wsc_tag       <= sig_tag_reg          ;
              sig_data2wsc_calc_err  <= sig_calc_error_reg   ;
              sig_data2wsc_last_err  <= sig_tlast_error_reg or 
                                        sig_tlast_error      ;
              sig_data2wsc_cmd_cmplt <= sig_cmd_cmplt_reg   or 
                                        sig_tlast_error_reg or
                                        sig_tlast_error      ;
              
            else
              null;  -- hold current state
            end if; 
         end if;       
       end process IMP_PUSH2WSC_FLOP; 
     
  
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_LD_NEW_CMD_REG
    --
    -- Process Description:
    -- Registers the flag indicating a new command has been 
    -- loaded. Needs to be a 1 clk wide pulse.
    --
    -------------------------------------------------------------
    IMP_LD_NEW_CMD_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset         = '1' or
                sig_ld_new_cmd_reg = '1') then
              sig_ld_new_cmd_reg <= '0';
            else
              sig_ld_new_cmd_reg <= sig_ld_new_cmd;
            end if; 
         end if;       
       end process IMP_LD_NEW_CMD_REG; 
    
    
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_NXT_LEN_REG
    --
    -- Process Description:
    -- Registers the load control and length value for a command 
    -- passed to the WDC input command interface. The registered
    -- signals are used for the external Indeterminate BTT support
    -- ports.
    --
    -------------------------------------------------------------
    IMP_NXT_LEN_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset         = '1') then

              sig_s2mm_ld_nxt_len <= '0';
              sig_s2mm_wr_len     <= (others => '0');
              
            else
              sig_s2mm_ld_nxt_len <= mstr2data_cmd_valid and
                                     sig_data2mstr_cmd_ready;
              sig_s2mm_wr_len     <= mstr2data_len;
                                     
                                     
            end if; 
         end if;       
       end process IMP_NXT_LEN_REG; 
    
    
    
    
    
    
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_NO_DATA_CNTL_FIFO
    --
    -- If Generate Description:
     --   Omits the input data control FIFO if the requested FIFO
     -- depth is 1. The Data Qualifier Register serves as a 
     -- 1 deep FIFO by itself.
    --
    ------------------------------------------------------------
    GEN_NO_DATA_CNTL_FIFO : if (C_DATA_CNTL_FIFO_DEPTH = 1) generate
    
       
      begin

        -- Command Calculator Handshake output
        sig_data2mstr_cmd_ready <= sig_fifo_wr_cmd_ready;           
       
        sig_fifo_rd_cmd_valid   <= mstr2data_cmd_valid ;
        
        
        
        -- pre 13.1 sig_fifo_wr_cmd_ready  <= sig_dqual_reg_empty     and                                         
        -- pre 13.1                           sig_aposted_cntr_ready  and                                         
        -- pre 13.1                           not(wsc2mstr_halt_pipe) and -- The Wr Status Controller is not stalling
        -- pre 13.1                           not(sig_calc_error_reg);    -- the command execution pipe and there is  
        -- pre 13.1                                                       -- no calculation error being propagated    
        
        sig_fifo_wr_cmd_ready  <= sig_push_dqual_reg;
        
                                                              
        
        sig_fifo_next_tag         <= mstr2data_tag        ;    
        sig_fifo_next_sadddr_lsb  <= mstr2data_saddr_lsb  ;    
        sig_fifo_next_len         <= mstr2data_len        ;    
        sig_fifo_next_strt_strb   <= mstr2data_strt_strb  ;    
        sig_fifo_next_last_strb   <= mstr2data_last_strb  ;    
        sig_fifo_next_drr         <= mstr2data_drr        ;    
        sig_fifo_next_eof         <= mstr2data_eof        ;    
        sig_fifo_next_sequential  <= mstr2data_sequential ;    
        sig_fifo_next_cmd_cmplt   <= mstr2data_cmd_cmplt  ;    
        sig_fifo_next_calc_error  <= mstr2data_calc_error ;    
                                                             
             
   
      end generate GEN_NO_DATA_CNTL_FIFO;
  
  
 
    
    
    
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_DATA_CNTL_FIFO
     --
     -- If Generate Description:
     --   Includes the input data control FIFO if the requested 
     -- FIFO depth is more than 1.
     --
     ------------------------------------------------------------
     GEN_DATA_CNTL_FIFO : if (C_DATA_CNTL_FIFO_DEPTH > 1) generate

       begin

       
         -- Command Calculator Handshake output
         sig_data2mstr_cmd_ready  <= sig_fifo_wr_cmd_ready;           
         
         sig_fifo_wr_cmd_valid    <= mstr2data_cmd_valid  ;
         
         
         -- pop the fifo when dqual reg is pushed
         sig_fifo_rd_cmd_ready    <= sig_push_dqual_reg;     
                                                               

         
         
                              
         -- Format the input fifo data word
         sig_cmd_fifo_data_in  <=   mstr2data_calc_error &
                                    mstr2data_cmd_cmplt  &
                                    mstr2data_sequential &
                                    mstr2data_eof        &
                                    mstr2data_drr        &
                                    mstr2data_last_strb  &
                                    mstr2data_strt_strb  &
                                    mstr2data_len        &
                                    mstr2data_saddr_lsb  &
                                    mstr2data_tag ;
         
          
         -- Rip the output fifo data word
         sig_fifo_next_tag        <= sig_cmd_fifo_data_out((TAG_STRT_INDEX+TAG_WIDTH)-1 downto 
                                                            TAG_STRT_INDEX);                   
         sig_fifo_next_sadddr_lsb <= sig_cmd_fifo_data_out((SADDR_LSB_STRT_INDEX+SADDR_LSB_WIDTH)-1 downto 
                                                            SADDR_LSB_STRT_INDEX);
         sig_fifo_next_len        <= sig_cmd_fifo_data_out((LEN_STRT_INDEX+LEN_WIDTH)-1 downto 
                                                            LEN_STRT_INDEX);
         sig_fifo_next_strt_strb  <= sig_cmd_fifo_data_out((STRT_STRB_STRT_INDEX+STRB_WIDTH)-1 downto 
                                                            STRT_STRB_STRT_INDEX);
         sig_fifo_next_last_strb  <= sig_cmd_fifo_data_out((LAST_STRB_STRT_INDEX+STRB_WIDTH)-1 downto 
                                                            LAST_STRB_STRT_INDEX);
         sig_fifo_next_drr        <= sig_cmd_fifo_data_out(DRR_STRT_INDEX);
         sig_fifo_next_eof        <= sig_cmd_fifo_data_out(EOF_STRT_INDEX);
         sig_fifo_next_sequential <= sig_cmd_fifo_data_out(SEQUENTIAL_STRT_INDEX);
         sig_fifo_next_cmd_cmplt  <= sig_cmd_fifo_data_out(CMD_CMPLT_STRT_INDEX);
         sig_fifo_next_calc_error <= sig_cmd_fifo_data_out(CALC_ERR_STRT_INDEX);

         
         
         
         ------------------------------------------------------------
         -- Instance: I_DATA_CNTL_FIFO 
         --
         -- Description:
         -- Instance for the Command Qualifier FIFO
         --
         ------------------------------------------------------------
          I_DATA_CNTL_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
          generic map (
        
            C_DWIDTH             =>  DCTL_FIFO_WIDTH        , 
            C_DEPTH              =>  C_DATA_CNTL_FIFO_DEPTH , 
            C_IS_ASYNC           =>  USE_SYNC_FIFO          , 
            C_PRIM_TYPE          =>  FIFO_PRIM_TYPE         , 
            C_FAMILY             =>  C_FAMILY                 
           
            )
          port map (
            
            -- Write Clock and reset
            fifo_wr_reset        =>   mmap_reset            , 
            fifo_wr_clk          =>   primary_aclk          , 
            
            -- Write Side
            fifo_wr_tvalid       =>   sig_fifo_wr_cmd_valid , 
            fifo_wr_tready       =>   sig_fifo_wr_cmd_ready , 
            fifo_wr_tdata        =>   sig_cmd_fifo_data_in  , 
            fifo_wr_full         =>   open                  , 
           
           
            -- Read Clock and reset
            fifo_async_rd_reset  =>   mmap_reset            ,   
            fifo_async_rd_clk    =>   primary_aclk          , 
            
            -- Read Side
            fifo_rd_tvalid       =>   sig_fifo_rd_cmd_valid , 
            fifo_rd_tready       =>   sig_fifo_rd_cmd_ready , 
            fifo_rd_tdata        =>   sig_cmd_fifo_data_out , 
            fifo_rd_empty        =>   sig_cmd_fifo_empty      
           
            );
        

       end generate GEN_DATA_CNTL_FIFO;
         
          
    
   
  
  
  
  
  
    -- Data Qualifier Register ------------------------------------
    
    
    sig_ld_new_cmd           <= sig_push_dqual_reg              ;
    sig_dqual_rdy            <= sig_dqual_reg_full              ;
    sig_strt_strb_reg        <= sig_next_strt_strb_reg          ;
    sig_last_strb_reg        <= sig_next_last_strb_reg          ;
    sig_tag_reg              <= sig_next_tag_reg                ;
    sig_cmd_cmplt_reg        <= sig_next_cmd_cmplt_reg          ;
    sig_calc_error_reg       <= sig_next_calc_error_reg         ;
    
    sig_cmd_is_eof           <= sig_next_eof_reg                ;
    
    
    
    -- new for no bubbles between child requests
    sig_sequential_push      <= sig_good_mmap_dbeat and -- MMap handshake qualified
                                sig_last_dbeat      and -- last data beat of transfer
                                sig_next_sequential_reg;-- next queued command is sequential 
                                                        -- to the current command
    
    
    -- pre 13.1 sig_push_dqual_reg        <= (sig_sequential_push   or
    -- pre 13.1                               sig_dqual_reg_empty)  and 
    -- pre 13.1                              sig_fifo_rd_cmd_valid  and
    -- pre 13.1                              sig_aposted_cntr_ready and 
    -- pre 13.1                              not(wsc2mstr_halt_pipe);  -- The Wr Status Controller is not     
    -- pre 13.1                                                        -- stalling the command execution pipe 

    
    sig_push_dqual_reg       <= (sig_sequential_push    or
                                 sig_dqual_reg_empty)   and 
                                sig_fifo_rd_cmd_valid   and
                                sig_aposted_cntr_ready  and 
                                not(sig_calc_error_reg) and -- 13.1 addition => An error has not been propagated
                                not(wsc2mstr_halt_pipe);    -- The Wr Status Controller is not  
                                                            -- stalling the command execution pipe
                                                        







                                                         
    sig_pop_dqual_reg         <= not(sig_next_calc_error_reg) and 
                                 sig_get_next_dqual and 
                                 sig_dqual_reg_full  ; 
    
  
    -- new for no bubbles between child requests
    sig_clr_dqual_reg        <=  mmap_reset         or
                                 (sig_pop_dqual_reg and
                                 not(sig_push_dqual_reg));
  
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_DQUAL_REG
    --
    -- Process Description:
    --    This process implements a register for the Data 
    -- Control and qualifiers. It operates like a 1 deep Sync FIFO.
    --
    -------------------------------------------------------------
    IMP_DQUAL_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_clr_dqual_reg = '1') then
              
              sig_next_tag_reg         <= (others => '0');
              sig_next_strt_strb_reg   <= (others => '0');
              sig_next_last_strb_reg   <= (others => '0');
              sig_next_eof_reg         <= '0'            ;
              sig_next_sequential_reg  <= '0'            ;
              sig_next_cmd_cmplt_reg   <= '0'            ;
              sig_next_calc_error_reg  <= '0'            ;
                                                        
              sig_dqual_reg_empty      <= '1'            ;
              sig_dqual_reg_full       <= '0'            ;
                                                        
            elsif (sig_push_dqual_reg = '1') then
              
              sig_next_tag_reg        <= sig_fifo_next_tag         ;
              sig_next_strt_strb_reg  <= sig_sfhalt_next_strt_strb ;
              sig_next_last_strb_reg  <= sig_fifo_next_last_strb   ;
              sig_next_eof_reg        <= sig_fifo_next_eof         ;
              sig_next_sequential_reg <= sig_fifo_next_sequential  ;
              sig_next_cmd_cmplt_reg  <= sig_fifo_next_cmd_cmplt   ;
              sig_next_calc_error_reg <= sig_fifo_next_calc_error  ;
              
              sig_dqual_reg_empty     <= '0';
              sig_dqual_reg_full      <= '1';
              
            else
              null;  -- don't change state
            end if; 
         end if;       
       end process IMP_DQUAL_REG; 
     

  
  
  
    
    
    -- Address LS Cntr logic  --------------------------
   
    sig_addr_lsb_reg         <= STD_LOGIC_VECTOR(sig_ls_addr_cntr);
    sig_addr_incr_unsgnd     <= TO_UNSIGNED(ADDR_INCR_VALUE, C_SEL_ADDR_WIDTH);
    sig_incr_ls_addr_cntr    <= sig_good_mmap_dbeat;
    
   
   
   
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: DO_ADDR_LSB_CNTR
    --
    -- Process Description:
    --  Implements the LS Address Counter used for controlling
    -- the Write STRB  DeMux during Burst transfers
    --
    -------------------------------------------------------------
    DO_ADDR_LSB_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset         = '1' or
               (sig_pop_dqual_reg  = '1'and
                sig_push_dqual_reg = '0')) then                 -- Clear the Counter
              
              sig_ls_addr_cntr <= (others => '0');
            
            elsif (sig_push_dqual_reg = '1') then               -- Load the Counter
              
              sig_ls_addr_cntr <= unsigned(sig_fifo_next_sadddr_lsb);
            
            elsif (sig_incr_ls_addr_cntr = '1') then            -- Increment the Counter
              
              sig_ls_addr_cntr <= sig_ls_addr_cntr + sig_addr_incr_unsgnd;
            
            else
              null;  -- Hold Current value
            end if; 
         end if;       
       end process DO_ADDR_LSB_CNTR; 
    
    
    
    
    
    
    
    
    
    
    
    
   -- Address Posted Counter Logic --------------------------------------
   
    sig_addr_chan_rdy         <= not(sig_addr_posted_cntr_eq_0 or 
                                     sig_apc_going2zero)         ; -- Gates data channel xfer handshake
    
    sig_aposted_cntr_ready    <= not(sig_addr_posted_cntr_max)   ; -- Gates new command fetching
    
    sig_no_posted_cmds        <= sig_addr_posted_cntr_eq_0       ; -- Used for flushing cmds that are posted
    
 
 
 
    
    sig_incr_addr_posted_cntr <= sig_addr_posted         ;
    
    sig_decr_addr_posted_cntr <= sig_last_mmap_dbeat_reg ;
    
    sig_addr_posted_cntr_eq_0 <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_ZERO)
      Else '0';
    
    sig_addr_posted_cntr_max <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_MAX)
      Else '0';
    
    
    sig_addr_posted_cntr_eq_1 <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_ONE)
      Else '0';
    
    sig_apc_going2zero  <= sig_addr_posted_cntr_eq_1 and
                           sig_decr_addr_posted_cntr and
                           not(sig_incr_addr_posted_cntr);
    
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ADDR_POSTED_FIFO_CNTR
    --
    -- Process Description:
    --    This process implements a counter for the tracking  
    -- if an Address has been posted on the AXI address channel.
    -- The Data Controller must wait for an address to be posted
    -- before proceeding with the corresponding data transfer on
    -- the Data Channel. The counter is also used to track flushing
    -- operations where all transfers commited on the  AXI Address
    -- Channel have to be completed before a halt can occur.
    -------------------------------------------------------------
    IMP_ADDR_POSTED_FIFO_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_addr_posted_cntr <= ADDR_POSTED_ZERO;
              
            elsif (sig_incr_addr_posted_cntr = '1' and
                   sig_decr_addr_posted_cntr = '0' and
                   sig_addr_posted_cntr_max  = '0') then
              
              sig_addr_posted_cntr <= sig_addr_posted_cntr + ADDR_POSTED_ONE ;
              
            elsif (sig_incr_addr_posted_cntr = '0' and
                   sig_decr_addr_posted_cntr = '1' and
                   sig_addr_posted_cntr_eq_0 = '0') then
              
              sig_addr_posted_cntr <= sig_addr_posted_cntr - ADDR_POSTED_ONE ;
              
            else
              null;  -- don't change state
            end if; 
         end if;       
       end process IMP_ADDR_POSTED_FIFO_CNTR; 
 
 
      
      
      
    ------- First/Middle/Last Dbeat detimination -------------------
     
     sig_new_len_eq_0 <= '1'
       When  (sig_fifo_next_len = LEN_OF_ZERO)
       else '0';
     
     
      
      
     -------------------------------------------------------------
     -- Synchronous Process with Sync Reset
     --
     -- Label: DO_FIRST_MID_LAST
     --
     -- Process Description:
     --  Implements the detection of the First/Mid/Last databeat of
     -- a transfer.
     --
     -------------------------------------------------------------
     DO_FIRST_MID_LAST : process (primary_aclk)
        begin
          if (primary_aclk'event and primary_aclk = '1') then
             if (mmap_reset = '1') then
               
               sig_first_dbeat   <= '0';
               sig_last_dbeat    <= '0';
               sig_single_dbeat  <= '0';
             
             elsif (sig_ld_new_cmd = '1') then
               
               sig_first_dbeat   <= not(sig_new_len_eq_0);
               sig_last_dbeat    <= sig_new_len_eq_0;
               sig_single_dbeat  <= sig_new_len_eq_0;
             
             Elsif (sig_dbeat_cntr_eq_1 = '1' and
                    sig_good_mmap_dbeat = '1') Then
             
               sig_first_dbeat   <= '0';
               sig_last_dbeat    <= '1';
               sig_single_dbeat  <= '0';
             
             Elsif (sig_dbeat_cntr_eq_0 = '0' and
                    sig_dbeat_cntr_eq_1 = '0' and
                    sig_good_mmap_dbeat = '1') Then
             
               sig_first_dbeat   <= '0';
               sig_last_dbeat    <= '0';
               sig_single_dbeat  <= '0';
             
             else
               null; -- hold current state
             end if; 
          end if;       
        end process DO_FIRST_MID_LAST; 
    
    
   
   
   
   -------  Data Controller Halted Indication ------------------------------- 
    
 
    data2all_dcntlr_halted <= sig_no_posted_cmds or
                              sig_calc_error_reg;
 
 
    
    
       
       
       
       
       
       
    
    
   -------  Data Beat counter logic ------------------------------- 
    
    
    
    
    
    sig_dbeat_cntr_int  <= TO_INTEGER(sig_dbeat_cntr);
    
    sig_dbeat_cntr_eq_0 <= '1'
      when (sig_dbeat_cntr_int = 0)
      Else '0';
    
    sig_dbeat_cntr_eq_1 <= '1'
      when (sig_dbeat_cntr_int = 1)
      Else '0';
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: DO_DBEAT_CNTR
    --
    -- Process Description:
    -- Implements the transfer data beat counter used to track 
    -- progress of the transfer.
    --
    -------------------------------------------------------------
    DO_DBEAT_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              sig_dbeat_cntr <= (others => '0');
            elsif (sig_ld_new_cmd = '1') then
              sig_dbeat_cntr <= unsigned(sig_fifo_next_len);
            Elsif (sig_good_mmap_dbeat = '1' and
                   sig_dbeat_cntr_eq_0 = '0') Then
              sig_dbeat_cntr <= sig_dbeat_cntr-1;
            else
              null; -- Hold current state
            end if; 
         end if;       
       end process DO_DBEAT_CNTR; 
  
  
  
  
  
  
  
  
  
  
  
  
   -------  Soft Shutdown Logic ------------------------------- 
    
    
    
    
    
    -- Formulate the soft shutdown complete flag
    sig_data2rst_stop_cmplt  <= (sig_halt_reg_dly3        and   -- Normal Mode shutdown
                                 sig_no_posted_cmds       and 
                                 not(sig_calc_error_reg)) or
                                (sig_halt_reg_dly3  and         -- Shutdown after error trap
                                 sig_calc_error_reg);
    
    
              
    
    -- Generate a gate signal to deassert the WVALID output
    -- for 1 clock cycle after a WLAST is issued. This only 
    -- occurs when in soft shutdown mode. 
    sig_stop_wvalid  <= (sig_last_mmap_dbeat_reg and
                        sig_halt_reg) or
                        sig_data2rst_stop_cmplt;
  
  
    
    
    
    -- Assign the output port skid buf control for the
    -- input Stream skid buffer
    data2skid_halt      <= sig_data2skid_halt;
    
    -- Create a 1 clock wide pulse to tell the input
    -- stream skid buffer to shut down.
    sig_data2skid_halt  <=  sig_halt_reg_dly2 and 
                            not(sig_halt_reg_dly3);
    
    
  
  
     
     
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_REQ_REG
    --
    -- Process Description:
    --   Implements the flop for capturing the Halt request from 
    -- the Reset module.
    --
    -------------------------------------------------------------
    IMP_HALT_REQ_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg      <= '0';
            
            elsif (rst2data_stop_request = '1') then
              
              sig_halt_reg <= '1';
            
            else
              null;  -- Hold current State
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG; 
  
  
   
   
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_REQ_REG_DLY
    --
    -- Process Description:
    --   Implements the flops for delaying the halt request by 3
    -- clocks to allow the Address Controller to halt before the
    -- Data Contoller can safely indicate it has exhausted all
    -- transfers committed to the AXI Address Channel by the Address
    -- Controller.
    --
    -------------------------------------------------------------
    IMP_HALT_REQ_REG_DLY : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg_dly1 <= '0';
              sig_halt_reg_dly2 <= '0';
              sig_halt_reg_dly3 <= '0';
            
            else
              
              sig_halt_reg_dly1 <= sig_halt_reg;
              sig_halt_reg_dly2 <= sig_halt_reg_dly1;
              sig_halt_reg_dly3 <= sig_halt_reg_dly2;
            
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG_DLY; 
  
  
   
    
    
    
    
    
    
 
  end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_wr_status_cntl.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_wr_status_cntl.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Write Status Controller.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library axi_datamover_v5_1_14;
  use axi_datamover_v5_1_14.axi_datamover_fifo;
  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_wr_status_cntl is
    generic (
      
      C_ENABLE_INDET_BTT     : Integer range  0 to   1 :=  0;
        -- Specifies if the Indeterminate BTT Module is enabled
        -- for use (outside of this module)
      
      C_SF_BYTES_RCVD_WIDTH  : Integer range  1 to  23 :=  1;
        -- Sets the width of the data2wsc_bytes_rcvd port used for 
        -- relaying the actual number of bytes received when Idet BTT is 
        -- enabled (C_ENABLE_INDET_BTT = 1)
      
      C_STS_FIFO_DEPTH       : Integer range  1 to  32 :=  8;
        -- Specifies the depth of the internal status queue fifo
      
      C_STS_WIDTH            : Integer range  8 to  32 :=  8;
        -- sets the width of the Status ports
      
      C_TAG_WIDTH            : Integer range  1 to 8   :=  4;
        -- Sets the width of the Tag field in the Status reply
      
      C_FAMILY               : String                  := "virtex7"
        -- Specifies the target FPGA device family
      
      
      );
    port (
      
      -- Clock and Reset inputs ------------------------------------------
                                                                        --
      primary_aclk         : in  std_logic;                             --
         -- Primary synchronization clock for the Master side           --
         -- interface and internal logic. It is also used               --
         -- for the User interface synchronization when                 --
         -- C_STSCMD_IS_ASYNC = 0.                                      --
                                                                        --
      -- Reset input                                                    --
      mmap_reset           : in  std_logic;                             --
         -- Reset used for the internal master logic                    --
      --------------------------------------------------------------------
      

     
      
      -- Soft Shutdown Control interface  --------------------------------
                                                                        --
      rst2wsc_stop_request : in  std_logic;                             --
         -- Active high soft stop request to modules                    --
                                                                        --
      wsc2rst_stop_cmplt   : Out std_logic;                             --
        -- Active high indication that the Write status Controller      --
        -- has completed any pending transfers committed by the         --
        -- Address Controller after a stop has been requested by        --
        -- the Reset module.                                            --
                                                                        --
      addr2wsc_addr_posted : In std_logic ;                             --
         -- Indication from the Address Channel Controller to the       --
         -- write Status Controller that an address has been posted     --
         -- to the AXI Address Channel                                  --
      --------------------------------------------------------------------


     
     
      --  Write Response Channel Interface -------------------------------
                                                                        --
      s2mm_bresp          : In std_logic_vector(1 downto 0);            --
         -- The Write response value                                    --
                                                                        --
      s2mm_bvalid         : In std_logic ;                              --
         -- Indication from the Write Response Channel that a new       --
         -- write status input is valid                                 --
                                                                        --
      s2mm_bready         : out std_logic ;                             --
         -- Indication to the Write Response Channel that the           --
         -- Status module is ready for a new status input               --
      --------------------------------------------------------------------


       
       
      -- Command Calculator Interface -------------------------------------
                                                                         --
      calc2wsc_calc_error    : in std_logic ;                            --
         -- Indication from the Command Calculator that a calculation    --
         -- error has occured.                                           --
      ---------------------------------------------------------------------
     
     
      
        
      -- Address Controller Status ----------------------------------------
                                                                         --
      addr2wsc_calc_error    : In std_logic ;                            --
         -- Indication from the Address Channel Controller that it       --
         -- has encountered a calculation error from the command         --
         -- Calculator                                                   --
                                                                         --
      addr2wsc_fifo_empty    : In std_logic ;                            --
         -- Indication from the Address Controller FIFO that it          --
         -- is empty (no commands pending)                               --
      ---------------------------------------------------------------------

                    
                    
                    
      --  Data Controller Status ---------------------------------------------------------
                                                                                        --
      data2wsc_tag           : In std_logic_vector(C_TAG_WIDTH-1 downto 0);             --
         -- The command tag                                                             --
                                                                                        --
      data2wsc_calc_error    : In std_logic ;                                           --
         -- Indication from the Data Channel Controller FIFO that it                    --
         -- has encountered a Calculation error in the command pipe                     --
                                                                                        --
      data2wsc_last_error    : In std_logic ;                                           --
         -- Indication from the Write Data Channel Controller that a                    --
         -- premature TLAST assertion was encountered on the incoming                   --
         -- Stream Channel                                                              --
                                                                                        --
      data2wsc_cmd_cmplt    : In std_logic ;                                            --
         -- Indication from the Data Channel Controller that the                        --
         -- corresponding status is the final status for a parent                       --
         -- command fetched from the command FIFO                                       --
                                                                                        --
      data2wsc_valid         : In std_logic ;                                           --
         -- Indication from the Data Channel Controller FIFO that it                    --
         -- has a new tag/error status to transfer                                      --
                                                                                        --
      wsc2data_ready         : out std_logic ;                                          --
         -- Indication to the Data Channel Controller FIFO that the                     --
         -- Status module is ready for a new tag/error status input                     --
                                                                                        --
                                                                                        --
      data2wsc_eop           : In  std_logic;                                           --
         -- Input from the Write Data Controller indicating that the                    --
         -- associated command status also corresponds to a End of Packet               --
         -- marker for the input Stream. This is only used when Store and               --
         -- Forward is enabled in the S2MM.                                             --
                                                                                        --
      data2wsc_bytes_rcvd    : In  std_logic_vector(C_SF_BYTES_RCVD_WIDTH-1 downto 0);  --
         -- Input from the Write Data Controller indicating the actual                  --
         -- number of bytes received from the Stream input for the                      --
         -- corresponding command status. This is only used when Store and              --
         -- Forward is enabled in the S2MM.                                             --
      ------------------------------------------------------------------------------------
      
      
      
      -- Command/Status Interface --------------------------------------------------------
                                                                                        --
      wsc2stat_status       : Out std_logic_vector(C_STS_WIDTH-1 downto 0);             --
         -- Read Status value collected during a Read Data transfer                     --
         -- Output to the Command/Status Module                                         --
                                                                                        --
      stat2wsc_status_ready : In  std_logic;                                            --
         -- Input from the Command/Status Module indicating that the                    --
         -- Status Reg/FIFO is Full and cannot accept more staus writes                 --
                                                                                        --
      wsc2stat_status_valid : Out std_logic ;                                           --
         -- Control Signal to Write the Status value to the Status                      --
         -- Reg/FIFO                                                                    --
      ------------------------------------------------------------------------------------

  
                                                                             
                                                                             
      -- Address and Data Controller Pipe halt --------------------------------
                                                                             --
      wsc2mstr_halt_pipe    : Out std_logic                                  --
         -- Indication to Halt the Data and Address Command pipeline due     --
         -- to the Status pipe getting full at some point                    --
      -------------------------------------------------------------------------

  
      );
  
  end entity axi_datamover_wr_status_cntl;
  
  
  architecture implementation of axi_datamover_wr_status_cntl is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

 
 
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_set_cnt_width
    --
    -- Function Description:
    --    Sets a count width based on a fifo depth. A depth of 4 or less
    -- is a special case which requires a minimum count width of 3 bits.
    --
    -------------------------------------------------------------------
    function funct_set_cnt_width (fifo_depth : integer) return integer is
    
      Variable temp_cnt_width : Integer := 4;
    
    begin
    
      
      if (fifo_depth <= 4) then
      
         temp_cnt_width := 3;
      
      elsif (fifo_depth <= 8) then
      
         temp_cnt_width := 4;
      
      elsif (fifo_depth <= 16) then
      
         temp_cnt_width := 5;
      
      elsif (fifo_depth <= 32) then
      
         temp_cnt_width := 6;
      
      else  -- fifo depth <= 64
      
         temp_cnt_width := 7;
      
      end if;
      
      Return (temp_cnt_width);
       
       
    end function funct_set_cnt_width;
    
 
 
 
  
    -- Constant Declarations  --------------------------------------------
    
    Constant OKAY                   : std_logic_vector(1 downto 0) := "00";
    Constant EXOKAY                 : std_logic_vector(1 downto 0) := "01";
    Constant SLVERR                 : std_logic_vector(1 downto 0) := "10";
    Constant DECERR                 : std_logic_vector(1 downto 0) := "11";
    Constant STAT_RSVD              : std_logic_vector(3 downto 0) := "0000";
    Constant TAG_WIDTH              : integer := C_TAG_WIDTH;
    Constant STAT_REG_TAG_WIDTH     : integer := 4;
    Constant SYNC_FIFO_SELECT       : integer := 0;
    Constant SRL_FIFO_TYPE          : integer := 2;
    Constant DCNTL_SFIFO_DEPTH      : integer := C_STS_FIFO_DEPTH;
    Constant DCNTL_STATCNT_WIDTH    : integer := funct_set_cnt_width(C_STS_FIFO_DEPTH);-- bits
    Constant DCNTL_HALT_THRES       : unsigned(DCNTL_STATCNT_WIDTH-1 downto 0) := 
                                      TO_UNSIGNED(DCNTL_SFIFO_DEPTH-2,DCNTL_STATCNT_WIDTH);
    Constant DCNTL_STATCNT_ZERO     : unsigned(DCNTL_STATCNT_WIDTH-1 downto 0) := (others => '0');
    Constant DCNTL_STATCNT_MAX      : unsigned(DCNTL_STATCNT_WIDTH-1 downto 0) := 
                                      TO_UNSIGNED(DCNTL_SFIFO_DEPTH,DCNTL_STATCNT_WIDTH);
    Constant DCNTL_STATCNT_ONE      : unsigned(DCNTL_STATCNT_WIDTH-1 downto 0) := 
                                      TO_UNSIGNED(1, DCNTL_STATCNT_WIDTH);
    Constant WRESP_WIDTH            : integer := 2;
    Constant WRESP_SFIFO_WIDTH      : integer := WRESP_WIDTH;
    Constant WRESP_SFIFO_DEPTH      : integer := DCNTL_SFIFO_DEPTH;
    
    Constant ADDR_POSTED_CNTR_WIDTH : integer := funct_set_cnt_width(C_STS_FIFO_DEPTH);-- bits 
    
    
    Constant ADDR_POSTED_ZERO       : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := (others => '0');
    Constant ADDR_POSTED_ONE        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := TO_UNSIGNED(1, ADDR_POSTED_CNTR_WIDTH);
    Constant ADDR_POSTED_MAX        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := (others => '1');
    
    
    -- Signal Declarations  --------------------------------------------
    
    signal sig_valid_status_rdy      : std_logic := '0';
    signal sig_decerr                : std_logic := '0';
    signal sig_slverr                : std_logic := '0';
    signal sig_coelsc_okay_reg       : std_logic := '0';
    signal sig_coelsc_interr_reg     : std_logic := '0';
    signal sig_coelsc_decerr_reg     : std_logic := '0';
    signal sig_coelsc_slverr_reg     : std_logic := '0';
    signal sig_coelsc_tag_reg        : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_pop_coelsc_reg        : std_logic := '0';
    signal sig_push_coelsc_reg       : std_logic := '0';
    signal sig_coelsc_reg_empty      : std_logic := '0';
    signal sig_coelsc_reg_full       : std_logic := '0';
    signal sig_tag2status            : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_data_tag_reg          : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_data_err_reg          : std_logic := '0';
    signal sig_data_last_err_reg     : std_logic := '0';
    signal sig_data_cmd_cmplt_reg    : std_logic := '0';
    signal sig_bresp_reg             : std_logic_vector(1 downto 0) := (others => '0');
    signal sig_push_status           : std_logic := '0';
    Signal sig_status_push_ok        : std_logic := '0';
    signal sig_status_valid          : std_logic := '0';
    signal sig_wsc2data_ready        : std_logic := '0';
    signal sig_s2mm_bready           : std_logic := '0';
    signal sig_wresp_sfifo_in        : std_logic_vector(WRESP_SFIFO_WIDTH-1 downto 0) := (others => '0');
    signal sig_wresp_sfifo_out       : std_logic_vector(WRESP_SFIFO_WIDTH-1 downto 0) := (others => '0');
    signal sig_wresp_sfifo_wr_valid  : std_logic := '0';
    signal sig_wresp_sfifo_wr_ready  : std_logic := '0';
    signal sig_wresp_sfifo_wr_full   : std_logic := '0';
    signal sig_wresp_sfifo_rd_valid  : std_logic := '0';
    signal sig_wresp_sfifo_rd_ready  : std_logic := '0';
    signal sig_wresp_sfifo_rd_empty  : std_logic := '0';
    signal sig_halt_reg              : std_logic := '0';
    signal sig_halt_reg_dly1         : std_logic := '0';
    signal sig_halt_reg_dly2         : std_logic := '0';
    signal sig_halt_reg_dly3         : std_logic := '0';
    signal sig_addr_posted_cntr      : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_posted_cntr_eq_0 : std_logic := '0';
    signal sig_addr_posted_cntr_eq_1 : std_logic := '0';
    signal sig_addr_posted_cntr_max  : std_logic := '0';
    signal sig_decr_addr_posted_cntr : std_logic := '0';
    signal sig_incr_addr_posted_cntr : std_logic := '0';
    signal sig_no_posted_cmds        : std_logic := '0';
    signal sig_addr_posted           : std_logic := '0';
    signal sig_all_cmds_done         : std_logic := '0';
    signal sig_wsc2stat_status       : std_logic_vector(C_STS_WIDTH-1 downto 0) := (others => '0');
    signal sig_dcntl_sfifo_wr_valid  : std_logic := '0';
    signal sig_dcntl_sfifo_wr_ready  : std_logic := '0';
    signal sig_dcntl_sfifo_wr_full   : std_logic := '0';
    signal sig_dcntl_sfifo_rd_valid  : std_logic := '0';
    signal sig_dcntl_sfifo_rd_ready  : std_logic := '0';
    signal sig_dcntl_sfifo_rd_empty  : std_logic := '0';
    signal sig_wdc_statcnt           : unsigned(DCNTL_STATCNT_WIDTH-1 downto 0) := (others => '0');
    signal sig_incr_statcnt          : std_logic := '0';
    signal sig_decr_statcnt          : std_logic := '0';
    signal sig_statcnt_eq_max        : std_logic := '0';
    signal sig_statcnt_eq_0          : std_logic := '0';
    signal sig_statcnt_gt_eq_thres   : std_logic := '0';
    signal sig_wdc_status_going_full : std_logic := '0';
                                           
                                           
                                           
            
  begin --(architecture implementation)
  
    
    -- Assign the ready output to the AXI Write Response Channel
    s2mm_bready           <= sig_s2mm_bready or
                             sig_halt_reg;     -- force bready if a Halt is requested
    
    -- Assign the ready output to the Data Controller status interface
    wsc2data_ready        <= sig_wsc2data_ready;
    
    -- Assign the status valid output control to the Status FIFO
    wsc2stat_status_valid <= sig_status_valid ;
    
    -- Formulate the status output value to the Status FIFO
    wsc2stat_status       <=  sig_wsc2stat_status;
    
     
    -- Formulate the status write request signal
    sig_status_valid      <= sig_push_status;
    
                                           
     
    -- Indicate the desire to push a coelesced status word 
    -- to the Status FIFO
    sig_push_status       <= sig_coelsc_reg_full;
    
    
    
    -- Detect that a push of a new status word is completing
    sig_status_push_ok    <= sig_status_valid and
                             stat2wsc_status_ready; 
    
    sig_pop_coelsc_reg    <= sig_status_push_ok;
    
    
    -- Signal a halt to the execution pipe if new status
    -- is valid but the Status FIFO is not accepting it or
    -- the WDC Status FIFO is going full
    wsc2mstr_halt_pipe    <= (sig_status_valid and
                             not(stat2wsc_status_ready)) or
                             sig_wdc_status_going_full;
 
    
    -- Monitor the Status capture registers to detect a 
    -- qualified Status set and push to the coelescing register
    -- when available to do so
    sig_push_coelsc_reg   <= sig_valid_status_rdy and 
                             sig_coelsc_reg_empty;
    
    --    pre CR616212  sig_valid_status_rdy  <= (sig_wresp_sfifo_rd_valid  and 
    --    pre CR616212                            sig_dcntl_sfifo_rd_valid) or
    --    pre CR616212                           (sig_data_err_reg and
    --    pre CR616212                            sig_dcntl_sfifo_rd_valid);     
          
    sig_valid_status_rdy  <= (sig_wresp_sfifo_rd_valid  and 
                              sig_dcntl_sfifo_rd_valid) or
                             (sig_data_err_reg and
                              sig_dcntl_sfifo_rd_valid) or  -- or Added for CR616212
                             (sig_data_last_err_reg and     -- Added for CR616212
                              sig_dcntl_sfifo_rd_valid);    -- Added for CR616212 
          
 
 
 
    -- Decode the AXI MMap Read Respose       
    sig_decerr  <= '1'
      When sig_bresp_reg = DECERR
      Else '0'; 
           
    sig_slverr  <= '1'
      When sig_bresp_reg = SLVERR
      Else '0';      
          
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_TAG_LE_STAT
    --
    -- If Generate Description:
    -- Populates the TAG bits into the availble Status bits when
    -- the TAG width is less than or equal to the available number
    -- of bits in the Status word. 
    --
    ------------------------------------------------------------
    GEN_TAG_LE_STAT : if (TAG_WIDTH <= STAT_REG_TAG_WIDTH) generate
    
       -- local signals
         signal lsig_temp_tag_small : std_logic_vector(STAT_REG_TAG_WIDTH-1 downto 0) := (others => '0');
         
         
       begin
    
         sig_tag2status <= lsig_temp_tag_small;
         
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: POPULATE_SMALL_TAG
         --
         -- Process Description:
         --
         --
         -------------------------------------------------------------
         POPULATE_SMALL_TAG : process (sig_coelsc_tag_reg)
            begin
         
              -- Set default value
              lsig_temp_tag_small <= (others => '0');
          
              -- Now overload actual TAG bits
              lsig_temp_tag_small(TAG_WIDTH-1 downto 0) <= sig_coelsc_tag_reg;
          
         
            end process POPULATE_SMALL_TAG; 
         
         
       end generate GEN_TAG_LE_STAT;
     
     
     
     
     
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_TAG_GT_STAT
    --
    -- If Generate Description:
    -- Populates the TAG bits into the availble Status bits when
    -- the TAG width is greater than the available number of 
    -- bits in the Status word. The upper bits of the TAG are 
    -- clipped off (discarded). 
    --
    ------------------------------------------------------------
    GEN_TAG_GT_STAT : if (TAG_WIDTH > STAT_REG_TAG_WIDTH) generate
    
       -- local signals
         signal lsig_temp_tag_big : std_logic_vector(STAT_REG_TAG_WIDTH-1 downto 0) := (others => '0');
         
         
       begin
    
         
         sig_tag2status <= lsig_temp_tag_big;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: POPULATE_BIG_TAG
         --
         -- Process Description:
         --
         --
         -------------------------------------------------------------
         POPULATE_SMALL_TAG : process (sig_coelsc_tag_reg)
            begin
         
              -- Set default value
              lsig_temp_tag_big <= (others => '0');
          
              -- Now overload actual TAG bits
              lsig_temp_tag_big <= sig_coelsc_tag_reg(STAT_REG_TAG_WIDTH-1 downto 0);
          
         
            end process POPULATE_SMALL_TAG; 
         
         
       end generate GEN_TAG_GT_STAT;
     
     
     
    
    
    
    
    
    
    
    
    
     
     
    
    
    
    -------------------------------------------------------------------------
    -- Write Response Channel input FIFO and logic
    
                        
    -- BRESP is the only fifo data
    sig_wresp_sfifo_in       <=  s2mm_bresp;     
    
    
    -- The fifo output is already in the right format
    sig_bresp_reg            <=  sig_wresp_sfifo_out;
    
    
    -- Write Side assignments
    sig_wresp_sfifo_wr_valid <=  s2mm_bvalid;
    
    sig_s2mm_bready          <=  sig_wresp_sfifo_wr_ready;
    
    
    -- read Side ready assignment
    sig_wresp_sfifo_rd_ready <=  sig_push_coelsc_reg;
    
    
    
    ------------------------------------------------------------
    -- Instance: I_WRESP_STATUS_FIFO 
    --
    -- Description:
    -- Instance for the AXI Write Response FIFO
    --
    ------------------------------------------------------------
     I_WRESP_STATUS_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
     generic map (
   
       C_DWIDTH             =>  WRESP_SFIFO_WIDTH         ,  
       C_DEPTH              =>  WRESP_SFIFO_DEPTH         ,  
       C_IS_ASYNC           =>  SYNC_FIFO_SELECT          ,  
       C_PRIM_TYPE          =>  SRL_FIFO_TYPE             ,  
       C_FAMILY             =>  C_FAMILY                     
      
       )
     port map (
       
       -- Write Clock and reset
       fifo_wr_reset        =>   mmap_reset               ,  
       fifo_wr_clk          =>   primary_aclk             ,  
       
       -- Write Side
       fifo_wr_tvalid       =>   sig_wresp_sfifo_wr_valid ,  
       fifo_wr_tready       =>   sig_wresp_sfifo_wr_ready ,  
       fifo_wr_tdata        =>   sig_wresp_sfifo_in       ,  
       fifo_wr_full         =>   sig_wresp_sfifo_wr_full  ,  
      
      
       -- Read Clock and reset (not used in Sync mode)
       fifo_async_rd_reset  =>   mmap_reset               ,    
       fifo_async_rd_clk    =>   primary_aclk             ,  
       
       -- Read Side
       fifo_rd_tvalid       =>   sig_wresp_sfifo_rd_valid ,  
       fifo_rd_tready       =>   sig_wresp_sfifo_rd_ready ,  
       fifo_rd_tdata        =>   sig_wresp_sfifo_out      ,  
       fifo_rd_empty        =>   sig_wresp_sfifo_rd_empty    
      
       );
   

    
    
    
    
    
    --------  Write Data Controller Status FIFO Going Full Logic -------------
    
    
    sig_incr_statcnt   <= sig_dcntl_sfifo_wr_valid and
                          sig_dcntl_sfifo_wr_ready;
    
    sig_decr_statcnt   <= sig_dcntl_sfifo_rd_valid and
                          sig_dcntl_sfifo_rd_ready;
    
          
    sig_statcnt_eq_max <= '1'
      when (sig_wdc_statcnt = DCNTL_STATCNT_MAX)
      Else '0';
          
    sig_statcnt_eq_0   <= '1'
      when (sig_wdc_statcnt = DCNTL_STATCNT_ZERO)
      Else '0';
          
    sig_statcnt_gt_eq_thres <= '1'
      when (sig_wdc_statcnt >= DCNTL_HALT_THRES)
      Else '0';
          
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_WDC_GOING_FULL_FLOP
    --
    -- Process Description:
    --  Implements a flop for the WDC Status FIFO going full flag.
    --
    -------------------------------------------------------------
    IMP_WDC_GOING_FULL_FLOP : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset = '1') then
    
             sig_wdc_status_going_full <= '0';
           
           else
    
             sig_wdc_status_going_full <= sig_statcnt_gt_eq_thres;
    
           end if; 
        end if;       
      end process IMP_WDC_GOING_FULL_FLOP; 
    
          
          
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_DCNTL_FIFO_CNTR
    --
    -- Process Description:
    --   Implements a simple counter keeping track of the number
    -- of entries in the WDC Status FIFO. If the Status FIFO gets
    -- too full, the S2MM Data Pipe has to be halted.
    --
    -------------------------------------------------------------
    IMP_DCNTL_FIFO_CNTR : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset = '1') then
    
             sig_wdc_statcnt <= (others => '0');
    
           elsif (sig_incr_statcnt   = '1' and 
                  sig_decr_statcnt   = '0' and
                  sig_statcnt_eq_max = '0') then
    
             sig_wdc_statcnt <= sig_wdc_statcnt + DCNTL_STATCNT_ONE;
    
           elsif (sig_incr_statcnt = '0' and 
                  sig_decr_statcnt = '1' and
                  sig_statcnt_eq_0 = '0') then
    
             sig_wdc_statcnt <= sig_wdc_statcnt - DCNTL_STATCNT_ONE;
    
           else
    
             null; -- Hold current count value
    
           end if; 
        end if;       
      end process IMP_DCNTL_FIFO_CNTR; 
          
          
          
          
          
          
          
          
          
          
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_OMIT_INDET_BTT
    --
    -- If Generate Description:
    --  Implements the logic needed when Indeterminate BTT is
    -- not enabled in the S2MM function.
    --
    ------------------------------------------------------------
    GEN_OMIT_INDET_BTT : if (C_ENABLE_INDET_BTT = 0) generate
    
       -- Local Constants
       Constant DCNTL_SFIFO_WIDTH           : integer := STAT_REG_TAG_WIDTH+3;
       Constant DCNTL_SFIFO_CMD_CMPLT_INDEX : integer := 0;
       Constant DCNTL_SFIFO_TLAST_ERR_INDEX : integer := 1;
       Constant DCNTL_SFIFO_CALC_ERR_INDEX  : integer := 2;
       Constant DCNTL_SFIFO_TAG_INDEX       : integer := DCNTL_SFIFO_CALC_ERR_INDEX+1;
    
       
       -- local signals
       signal sig_dcntl_sfifo_in        : std_logic_vector(DCNTL_SFIFO_WIDTH-1 downto 0) := (others => '0');
       signal sig_dcntl_sfifo_out       : std_logic_vector(DCNTL_SFIFO_WIDTH-1 downto 0) := (others => '0');
    
    
       begin
    
       
         sig_wsc2stat_status       <=  sig_coelsc_okay_reg    &   
                                       sig_coelsc_slverr_reg  &
                                       sig_coelsc_decerr_reg  & 
                                       sig_coelsc_interr_reg  &
                                       sig_tag2status;
         
         
         
         -----------------------------------------------------------------------------
         -- Data Controller Status FIFO and Logic
         
         
         -- Concatonate Input bits to build Dcntl fifo data word
         sig_dcntl_sfifo_in      <=  data2wsc_tag        &   -- bit 3 to tag Width+2
                                     data2wsc_calc_error &   -- bit 2
                                     data2wsc_last_error &   -- bit 1
                                     data2wsc_cmd_cmplt  ;   -- bit 0
         
                       
         -- Rip the DCntl fifo outputs back to constituant pieces              
         sig_data_tag_reg        <= sig_dcntl_sfifo_out((DCNTL_SFIFO_TAG_INDEX+STAT_REG_TAG_WIDTH)-1 downto
                                                        DCNTL_SFIFO_TAG_INDEX);
                                                        
         sig_data_err_reg        <= sig_dcntl_sfifo_out(DCNTL_SFIFO_CALC_ERR_INDEX) ;
         
         sig_data_last_err_reg   <= sig_dcntl_sfifo_out(DCNTL_SFIFO_TLAST_ERR_INDEX);
         
         sig_data_cmd_cmplt_reg  <= sig_dcntl_sfifo_out(DCNTL_SFIFO_CMD_CMPLT_INDEX);
        
         
         
         -- Data Control Valid/Ready assignments
         sig_dcntl_sfifo_wr_valid <= data2wsc_valid     ;
         
         sig_wsc2data_ready       <= sig_dcntl_sfifo_wr_ready; 
         
         
         
         -- read side ready assignment
         sig_dcntl_sfifo_rd_ready <= sig_push_coelsc_reg;
               
                                                  
         
         ------------------------------------------------------------
         -- Instance: I_DATA_CNTL_STATUS_FIFO 
         --
         -- Description:
         -- Instance for the Command Qualifier FIFO
         --
         ------------------------------------------------------------
          I_DATA_CNTL_STATUS_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
          generic map (
        
            C_DWIDTH             =>  DCNTL_SFIFO_WIDTH         , 
            C_DEPTH              =>  DCNTL_SFIFO_DEPTH         , 
            C_IS_ASYNC           =>  SYNC_FIFO_SELECT          , 
            C_PRIM_TYPE          =>  SRL_FIFO_TYPE             , 
            C_FAMILY             =>  C_FAMILY                    
           
            )
          port map (
            
            -- Write Clock and reset
            fifo_wr_reset        =>   mmap_reset               , 
            fifo_wr_clk          =>   primary_aclk             , 
            
            -- Write Side
            fifo_wr_tvalid       =>   sig_dcntl_sfifo_wr_valid , 
            fifo_wr_tready       =>   sig_dcntl_sfifo_wr_ready , 
            fifo_wr_tdata        =>   sig_dcntl_sfifo_in       , 
            fifo_wr_full         =>   sig_dcntl_sfifo_wr_full  , 
           
           
            -- Read Clock and reset (not used in Sync mode)
            fifo_async_rd_reset  =>   mmap_reset               ,   
            fifo_async_rd_clk    =>   primary_aclk             , 
            
            -- Read Side
            fifo_rd_tvalid       =>   sig_dcntl_sfifo_rd_valid , 
            fifo_rd_tready       =>   sig_dcntl_sfifo_rd_ready , 
            fifo_rd_tdata        =>   sig_dcntl_sfifo_out      , 
            fifo_rd_empty        =>   sig_dcntl_sfifo_rd_empty   
           
            );
        

         
         
         
            
            
       
       
         -------------------------------------------------------------
         -- Synchronous Process with Sync Reset
         --
         -- Label: STATUS_COELESC_REG
         --
         -- Process Description:
         --   Implement error status coelescing register. 
         -- Once a bit is set it will remain set until the overall 
         -- status is written to the Status FIFO. 
         -- Tag bits are just registered at each valid dbeat.
         --
         -------------------------------------------------------------
         STATUS_COELESC_REG : process (primary_aclk)
            begin
              if (primary_aclk'event and primary_aclk = '1') then
                 if (mmap_reset         = '1' or
                     sig_pop_coelsc_reg = '1') then
                   
                   sig_coelsc_tag_reg       <= (others => '0');
                   sig_coelsc_interr_reg    <= '0';
                   sig_coelsc_decerr_reg    <= '0';
                   sig_coelsc_slverr_reg    <= '0';
                   sig_coelsc_okay_reg      <= '1'; -- set back to default of "OKAY"
       
                   sig_coelsc_reg_full      <= '0';
                   sig_coelsc_reg_empty     <= '1';
       
       
                   
                 Elsif (sig_push_coelsc_reg = '1') Then
                 
                   sig_coelsc_tag_reg       <= sig_data_tag_reg;                             
                   sig_coelsc_interr_reg    <= sig_data_err_reg      or 
                                               sig_data_last_err_reg or 
                                               sig_coelsc_interr_reg;
                   sig_coelsc_decerr_reg    <= not(sig_data_err_reg) and 
                                               (sig_decerr           or 
                                                sig_coelsc_decerr_reg);
                   sig_coelsc_slverr_reg    <= not(sig_data_err_reg) and 
                                               (sig_slverr           or 
                                                sig_coelsc_slverr_reg);
                   sig_coelsc_okay_reg      <= not(sig_decerr            or 
                                                   sig_coelsc_decerr_reg or
                                                   sig_slverr            or 
                                                   sig_coelsc_slverr_reg or
                                                   sig_data_err_reg      or
                                                   sig_data_last_err_reg or    
                                                   sig_coelsc_interr_reg      
                                                   );
       
                   sig_coelsc_reg_full      <= sig_data_cmd_cmplt_reg;
                   sig_coelsc_reg_empty     <= not(sig_data_cmd_cmplt_reg);
                                                 
                   
                 else
                   
                   null;  -- hold current state
                   
                 end if; 
              end if;       
            end process STATUS_COELESC_REG; 
       
       
       
       
       
       
       end generate GEN_OMIT_INDET_BTT;
    
    
    
    
    
    
    
    
    
    
    
   

    
    
    
    
    
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ENABLE_INDET_BTT
    --
    -- If Generate Description:
    --  Implements the logic needed when Indeterminate BTT is
    -- enabled in the S2MM function. Primary difference is the
    -- addition to the reported status of the End of Packet 
    -- marker (EOP) and the received byte count for the parent 
    -- command.
    --
    ------------------------------------------------------------
    GEN_ENABLE_INDET_BTT : if (C_ENABLE_INDET_BTT = 1) generate
    
       -- Local Constants
       Constant SF_DCNTL_SFIFO_WIDTH         : integer := TAG_WIDTH +
                                                          C_SF_BYTES_RCVD_WIDTH + 3;
       Constant SF_SFIFO_LS_TAG_INDEX        : integer := 0;
       Constant SF_SFIFO_MS_TAG_INDEX        : integer := SF_SFIFO_LS_TAG_INDEX + (TAG_WIDTH-1);
       Constant SF_SFIFO_CALC_ERR_INDEX      : integer := SF_SFIFO_MS_TAG_INDEX+1;
       Constant SF_SFIFO_CMD_CMPLT_INDEX     : integer := SF_SFIFO_CALC_ERR_INDEX+1;
       Constant SF_SFIFO_LS_BYTES_RCVD_INDEX : integer := SF_SFIFO_CMD_CMPLT_INDEX+1;
       Constant SF_SFIFO_MS_BYTES_RCVD_INDEX : integer := SF_SFIFO_LS_BYTES_RCVD_INDEX+
                                                          (C_SF_BYTES_RCVD_WIDTH-1);
       Constant SF_SFIFO_EOP_INDEX           : integer := SF_SFIFO_MS_BYTES_RCVD_INDEX+1;
       Constant BYTES_RCVD_FIELD_WIDTH       : integer := 23;
       
       
       -- local signals
       signal sig_dcntl_sfifo_in        : std_logic_vector(SF_DCNTL_SFIFO_WIDTH-1 downto 0) := (others => '0');
       signal sig_dcntl_sfifo_out       : std_logic_vector(SF_DCNTL_SFIFO_WIDTH-1 downto 0) := (others => '0');
       signal sig_data_bytes_rcvd       : std_logic_vector(C_SF_BYTES_RCVD_WIDTH-1 downto 0) := (others => '0');
       signal sig_data_eop              : std_logic := '0';
       signal sig_coelsc_bytes_rcvd     : std_logic_vector(C_SF_BYTES_RCVD_WIDTH-1 downto 0) := (others => '0');
       signal sig_coelsc_eop            : std_logic := '0';
       signal sig_coelsc_bytes_rcvd_pad : std_logic_vector(BYTES_RCVD_FIELD_WIDTH-1 downto 0) := (others => '0');
 
    
       begin
    
       
                                                                
         sig_wsc2stat_status     <=  sig_coelsc_eop            &
                                     sig_coelsc_bytes_rcvd_pad &
                                     sig_coelsc_okay_reg       &   
                                     sig_coelsc_slverr_reg     &
                                     sig_coelsc_decerr_reg     & 
                                     sig_coelsc_interr_reg     &
                                     sig_tag2status;
         
            
         
         
         -----------------------------------------------------------------------------
         -- Data Controller Status FIFO and Logic


     
         
         -- Concatonate Input bits to build Dcntl fifo input data word
         sig_dcntl_sfifo_in      <=  data2wsc_eop        & -- ms bit
                                     data2wsc_bytes_rcvd & -- bit 7 to C_SF_BYTES_RCVD_WIDTH+7  
                                     data2wsc_cmd_cmplt  & -- bit 6 
                                     data2wsc_calc_error & -- bit 4  
                                     data2wsc_tag;         -- bits 0 to 3
                       
                       
         -- Rip the DCntl fifo outputs back to constituant pieces              
         
         sig_data_eop            <= sig_dcntl_sfifo_out(SF_SFIFO_EOP_INDEX);
         
         sig_data_bytes_rcvd     <= sig_dcntl_sfifo_out(SF_SFIFO_MS_BYTES_RCVD_INDEX downto 
                                                        SF_SFIFO_LS_BYTES_RCVD_INDEX);
         
         sig_data_cmd_cmplt_reg  <= sig_dcntl_sfifo_out(SF_SFIFO_CMD_CMPLT_INDEX);
          
         sig_data_err_reg        <= sig_dcntl_sfifo_out(SF_SFIFO_CALC_ERR_INDEX);
         
         sig_data_tag_reg        <= sig_dcntl_sfifo_out(SF_SFIFO_MS_TAG_INDEX downto 
                                                        SF_SFIFO_LS_TAG_INDEX) ;
         
         
         
         
         -- Data Control Valid/Ready assignments
         sig_dcntl_sfifo_wr_valid <= data2wsc_valid ;
         
         sig_wsc2data_ready       <= sig_dcntl_sfifo_wr_ready; 
         
         
         
         -- read side ready assignment
         sig_dcntl_sfifo_rd_ready <= sig_push_coelsc_reg;
               
                                                  
         
         ------------------------------------------------------------
         -- Instance: I_SF_DATA_CNTL_STATUS_FIFO 
         --
         -- Description:
         -- Instance for the Command Qualifier FIFO when Store and
         -- Forward is included.
         --
         ------------------------------------------------------------
          I_SF_DATA_CNTL_STATUS_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
          generic map (
        
            C_DWIDTH             =>  SF_DCNTL_SFIFO_WIDTH      ,  
            C_DEPTH              =>  DCNTL_SFIFO_DEPTH         ,  
            C_IS_ASYNC           =>  SYNC_FIFO_SELECT          ,  
            C_PRIM_TYPE          =>  SRL_FIFO_TYPE             ,  
            C_FAMILY             =>  C_FAMILY                     
           
            )
          port map (
            
            -- Write Clock and reset
            fifo_wr_reset        =>   mmap_reset               ,  
            fifo_wr_clk          =>   primary_aclk             ,  
            
            -- Write Side
            fifo_wr_tvalid       =>   sig_dcntl_sfifo_wr_valid ,  
            fifo_wr_tready       =>   sig_dcntl_sfifo_wr_ready ,  
            fifo_wr_tdata        =>   sig_dcntl_sfifo_in       ,  
            fifo_wr_full         =>   sig_dcntl_sfifo_wr_full  ,  
           
            -- Read Clock and reset (not used in Sync mode)
            fifo_async_rd_reset  =>   mmap_reset               ,     
            fifo_async_rd_clk    =>   primary_aclk             ,   
            
            -- Read Side
            fifo_rd_tvalid       =>   sig_dcntl_sfifo_rd_valid ,  
            fifo_rd_tready       =>   sig_dcntl_sfifo_rd_ready ,  
            fifo_rd_tdata        =>   sig_dcntl_sfifo_out      ,  
            fifo_rd_empty        =>   sig_dcntl_sfifo_rd_empty    
           
            );
        

         
         
         
            
            
       
       
         -------------------------------------------------------------
         -- Synchronous Process with Sync Reset
         --
         -- Label: SF_STATUS_COELESC_REG
         --
         -- Process Description:
         --   Implement error status coelescing register. 
         -- Once a bit is set it will remain set until the overall 
         -- status is written to the Status FIFO. 
         -- Tag bits are just registered at each valid dbeat.
         --
         -------------------------------------------------------------
         SF_STATUS_COELESC_REG : process (primary_aclk)
            begin
              if (primary_aclk'event and primary_aclk = '1') then
                 if (mmap_reset         = '1' or
                     sig_pop_coelsc_reg = '1') then
                   
                   sig_coelsc_tag_reg       <= (others => '0');
                   sig_coelsc_interr_reg    <= '0';
                   sig_coelsc_decerr_reg    <= '0';
                   sig_coelsc_slverr_reg    <= '0';
                   sig_coelsc_okay_reg      <= '1'; -- set back to default of "OKAY"
                   sig_coelsc_bytes_rcvd    <= (others => '0');
                   sig_coelsc_eop           <= '0';
      
       
                   sig_coelsc_reg_full      <= '0';
                   sig_coelsc_reg_empty     <= '1';
       
       
                   
                 Elsif (sig_push_coelsc_reg = '1') Then
                 
                   sig_coelsc_tag_reg       <= sig_data_tag_reg;                             
                   sig_coelsc_interr_reg    <= sig_data_err_reg      or 
                                               sig_coelsc_interr_reg;
                   sig_coelsc_decerr_reg    <= not(sig_data_err_reg) and 
                                               (sig_decerr           or 
                                                sig_coelsc_decerr_reg);
                   sig_coelsc_slverr_reg    <= not(sig_data_err_reg) and 
                                               (sig_slverr           or 
                                                sig_coelsc_slverr_reg);
                   sig_coelsc_okay_reg      <= not(sig_decerr            or 
                                                   sig_coelsc_decerr_reg or
                                                   sig_slverr            or 
                                                   sig_coelsc_slverr_reg or
                                                   sig_data_err_reg      or
                                                   sig_coelsc_interr_reg      
                                                   );
                   sig_coelsc_bytes_rcvd    <= sig_data_bytes_rcvd;
                   sig_coelsc_eop           <= sig_data_eop;
      
       
                   sig_coelsc_reg_full      <= sig_data_cmd_cmplt_reg;
                   sig_coelsc_reg_empty     <= not(sig_data_cmd_cmplt_reg);
                                                 
                   
                 else
                   
                   null;  -- hold current state
                   
                 end if; 
              end if;       
            end process SF_STATUS_COELESC_REG; 
       
       
       
       
       
       
         
         ------------------------------------------------------------
         -- If Generate
         --
         -- Label: SF_GEN_PAD_BYTES_RCVD
         --
         -- If Generate Description:
         --  Pad the bytes received value with zeros to fill in the 
         -- status field width.
         --
         --
         ------------------------------------------------------------
         SF_GEN_PAD_BYTES_RCVD : if (C_SF_BYTES_RCVD_WIDTH < BYTES_RCVD_FIELD_WIDTH) generate
         
           begin
            
             sig_coelsc_bytes_rcvd_pad(BYTES_RCVD_FIELD_WIDTH-1 downto 
                                       C_SF_BYTES_RCVD_WIDTH) <= (others => '0');
            
            
             sig_coelsc_bytes_rcvd_pad(C_SF_BYTES_RCVD_WIDTH-1 downto 0) <=  sig_coelsc_bytes_rcvd;
            
           
           end generate SF_GEN_PAD_BYTES_RCVD;
       
       
       
         
         
         ------------------------------------------------------------
         -- If Generate
         --
         -- Label: SF_GEN_NO_PAD_BYTES_RCVD
         --
         -- If Generate Description:
         --  No padding required for the bytes received value.
         --
         --
         ------------------------------------------------------------
         SF_GEN_NO_PAD_BYTES_RCVD : if (C_SF_BYTES_RCVD_WIDTH = BYTES_RCVD_FIELD_WIDTH) generate
         
           begin
            
               sig_coelsc_bytes_rcvd_pad <= sig_coelsc_bytes_rcvd; -- no pad required
            
           
           end generate SF_GEN_NO_PAD_BYTES_RCVD;
       
       
          
          
       
       end generate GEN_ENABLE_INDET_BTT;
    
    
    
    
    
    
    
    
    
    
    
   -------  Soft Shutdown Logic ------------------------------- 
    
    
  
  
   -- Address Posted Counter Logic ---------------------t-----------------
   -- Supports soft shutdown by tracking when all commited Write 
   -- transfers to the AXI Bus have had corresponding Write Status 
   -- Reponses Received.   

    
    sig_addr_posted           <= addr2wsc_addr_posted ;
    
    sig_incr_addr_posted_cntr <= sig_addr_posted      ;
    
    sig_decr_addr_posted_cntr <= sig_s2mm_bready  and
                                 s2mm_bvalid          ;
        
    sig_addr_posted_cntr_eq_0 <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_ZERO)
      Else '0';

    sig_addr_posted_cntr_eq_1 <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_ONE)
      Else '0';

    
    sig_addr_posted_cntr_max <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_MAX)
      Else '0';
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ADDR_POSTED_FIFO_CNTR
    --
    -- Process Description:
    --    This process implements a counter for the tracking  
    -- if an Address has been posted on the AXI address channel.
    -- The counter is used to track flushing operations where all
    -- transfers committed on the AXI Address Channel have to
    --  be completed before a halt can occur.
    -------------------------------------------------------------
    IMP_ADDR_POSTED_FIFO_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_addr_posted_cntr <= ADDR_POSTED_ZERO;
              
            elsif (sig_incr_addr_posted_cntr = '1' and
                   sig_decr_addr_posted_cntr = '0' and
                   sig_addr_posted_cntr_max  = '0') then
              
              sig_addr_posted_cntr <= sig_addr_posted_cntr + ADDR_POSTED_ONE ;
              
            elsif (sig_incr_addr_posted_cntr = '0' and
                   sig_decr_addr_posted_cntr = '1' and
                   sig_addr_posted_cntr_eq_0 = '0') then
              
              sig_addr_posted_cntr <= sig_addr_posted_cntr - ADDR_POSTED_ONE ;
              
            else
              null;  -- don't change state
            end if; 
         end if;       
       end process IMP_ADDR_POSTED_FIFO_CNTR; 
 
 
    
    
    wsc2rst_stop_cmplt <= sig_all_cmds_done;
    
    
    
    
    sig_no_posted_cmds <= (sig_addr_posted_cntr_eq_0 and
                            not(addr2wsc_calc_error)) or
                           (sig_addr_posted_cntr_eq_1 and
                            addr2wsc_calc_error);
    
  
                         
    sig_all_cmds_done <=  sig_no_posted_cmds and
                          sig_halt_reg_dly3;   
                               
                               
                               
    
    
  
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_REQ_REG
    --
    -- Process Description:
    --   Implements the flop for capturing the Halt request from 
    -- the Reset module.
    --
    -------------------------------------------------------------
    IMP_HALT_REQ_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg      <= '0';
            
            elsif (rst2wsc_stop_request = '1') then
              
              sig_halt_reg <= '1';
            
            else
              null;  -- Hold current State
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG; 
  
  
   
   
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_REQ_REG_DLY
    --
    -- Process Description:
    --   Implements the flops for delaying the halt request by 3
    -- clocks to allow the Address Controller to halt before the
    -- Data Contoller can safely indicate it has exhausted all
    -- transfers committed to the AXI Address Channel by the Address
    -- Controller.
    --
    -------------------------------------------------------------
    IMP_HALT_REQ_REG_DLY : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg_dly1 <= '0';
              sig_halt_reg_dly2 <= '0';
              sig_halt_reg_dly3 <= '0';
            
            else
              
              sig_halt_reg_dly1 <= sig_halt_reg;
              sig_halt_reg_dly2 <= sig_halt_reg_dly1;
              sig_halt_reg_dly3 <= sig_halt_reg_dly2;
            
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG_DLY; 
  
  
   
      
          
          
  
  end implementation;


-------------------------------------------------------------------------------
-- axi_datamover_skid2mm_buf.vhd
-------------------------------------------------------------------------------
--
-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_datamover_skid2mm_buf.vhd
--
-- Description:     
--  Implements the AXi Skid Buffer in the Option 2 (Registerd outputs) mode.                
--
--  This Module also provides Write Data Bus Mirroring and WSTRB
--  Demuxing to match a narrow Stream to a wider MMap Write 
--  Channel. By doing this in the skid buffer, the resource 
--  utilization of the skid buffer can be minimized by only
--  having to buffer/mux the Stream data width, not the MMap
--  Data width.   
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;


library axi_datamover_v5_1_14; 
use axi_datamover_v5_1_14.axi_datamover_wr_demux;

-------------------------------------------------------------------------------

entity axi_datamover_skid2mm_buf is
  generic (
    
    C_MDATA_WIDTH         : INTEGER range 32 to 1024 := 32 ;
       --  Width of the MMap Write Data bus (in bits)
    C_SDATA_WIDTH         : INTEGER range 8 to 1024 := 32 ;
       --  Width of the Stream Data bus (in bits)
    C_ADDR_LSB_WIDTH      : INTEGER range 1 to 8 := 5
       --  Width of the LS address bus needed to Demux the WSTRB
       
    );
  port (
     
     -- Clock and Reset Inputs -------------------------------------------
                                                                        --
     ACLK         : In  std_logic ;                                     --
     ARST         : In  std_logic ;                                     --
     ---------------------------------------------------------------------
     
      
     -- Slave Side (Wr Data Controller Input Side) -----------------------
                                                                        --
     S_ADDR_LSB   : in  std_logic_vector(C_ADDR_LSB_WIDTH-1 downto 0);  --
     S_VALID      : In  std_logic ;                                     --
     S_READY      : Out std_logic ;                                     --
     S_DATA       : In  std_logic_vector(C_SDATA_WIDTH-1 downto 0);     --
     S_STRB       : In  std_logic_vector((C_SDATA_WIDTH/8)-1 downto 0); --
     S_LAST       : In  std_logic ;                                     --
     ---------------------------------------------------------------------
     

     -- Master Side (MMap Write Data Output Side) ------------------------
     M_VALID      : Out std_logic ;                                     --
     M_READY      : In  std_logic ;                                     --
     M_DATA       : Out std_logic_vector(C_MDATA_WIDTH-1 downto 0);     --
     M_STRB       : Out std_logic_vector((C_MDATA_WIDTH/8)-1 downto 0); --
     M_LAST       : Out std_logic                                       --
     ---------------------------------------------------------------------
     
    );

end entity axi_datamover_skid2mm_buf;


architecture implementation of axi_datamover_skid2mm_buf is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


  Constant IN_DATA_WIDTH       : integer := C_SDATA_WIDTH;
  Constant MM2STRM_WIDTH_RATIO : integer := C_MDATA_WIDTH/C_SDATA_WIDTH;
  
  
-- Signals decalrations -------------------------

  Signal sig_reset_reg         : std_logic := '0';
  signal sig_spcl_s_ready_set  : std_logic := '0';
  signal sig_data_skid_reg     : std_logic_vector(IN_DATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_skid_reg     : std_logic_vector((C_MDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_skid_reg     : std_logic := '0';
  signal sig_skid_reg_en       : std_logic := '0';
  signal sig_data_skid_mux_out : std_logic_vector(IN_DATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_skid_mux_out : std_logic_vector((C_MDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_skid_mux_out : std_logic := '0';
  signal sig_skid_mux_sel      : std_logic := '0';
  signal sig_data_reg_out      : std_logic_vector(IN_DATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_reg_out      : std_logic_vector((C_MDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_reg_out      : std_logic := '0';
  signal sig_data_reg_out_en   : std_logic := '0';
  signal sig_m_valid_out       : std_logic := '0';
  signal sig_m_valid_dup       : std_logic := '0';
  signal sig_m_valid_comb      : std_logic := '0';
  signal sig_s_ready_out       : std_logic := '0';
  signal sig_s_ready_dup       : std_logic := '0';
  signal sig_s_ready_comb      : std_logic := '0';
  signal sig_mirror_data_out   : std_logic_vector(C_MDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_wstrb_demux_out   : std_logic_vector((C_MDATA_WIDTH/8)-1 downto 0) := (others => '0');
                                         
                                         
                                         
 
-- Register duplication attribute assignments to control fanout
-- on handshake output signals  
  
  Attribute KEEP : string; -- declaration
  Attribute EQUIVALENT_REGISTER_REMOVAL : string; -- declaration
  
  Attribute KEEP of sig_m_valid_out : signal is "TRUE"; -- definition
  Attribute KEEP of sig_m_valid_dup : signal is "TRUE"; -- definition
  Attribute KEEP of sig_s_ready_out : signal is "TRUE"; -- definition
  Attribute KEEP of sig_s_ready_dup : signal is "TRUE"; -- definition
  
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_m_valid_out : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_m_valid_dup : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_s_ready_out : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_s_ready_dup : signal is "no"; 
  
  
  

begin --(architecture implementation)

   M_VALID <= sig_m_valid_out;         
   S_READY <= sig_s_ready_out; 
    
   M_STRB  <= sig_strb_reg_out;
   M_LAST  <= sig_last_reg_out;                
   M_DATA  <= sig_mirror_data_out;
            
   -- Assign the special S_READY FLOP set signal
   sig_spcl_s_ready_set <= sig_reset_reg;
  
            
   -- Generate the ouput register load enable control
   sig_data_reg_out_en <= M_READY or not(sig_m_valid_dup);

   -- Generate the skid inpit register load enable control
   sig_skid_reg_en     <= sig_s_ready_dup;
  
   -- Generate the skid mux select control
   sig_skid_mux_sel    <= not(sig_s_ready_dup);
   
   
   -- Skid Mux  
   sig_data_skid_mux_out <=  sig_data_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  S_DATA;
  
   sig_strb_skid_mux_out <=  sig_strb_skid_reg
     When (sig_skid_mux_sel = '1')
     --Else  S_STRB;
     Else  sig_wstrb_demux_out;
  
   sig_last_skid_mux_out <=  sig_last_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  S_LAST;
  
   
   -- m_valid combinational logic        
   sig_m_valid_comb <= S_VALID or
                      (sig_m_valid_dup and
                      (not(sig_s_ready_dup) or
                       not(M_READY)));
   
   
   
   -- s_ready combinational logic        
   sig_s_ready_comb <= M_READY or
                      (sig_s_ready_dup and
                      (not(sig_m_valid_dup) or
                       not(S_VALID)));
   
   
   
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: REG_THE_RST
   --
   -- Process Description:
   -- Register input reset
   --
   -------------------------------------------------------------
   REG_THE_RST : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           
            sig_reset_reg <= ARST;
           
        end if;       
      end process REG_THE_RST; 
   
   
   
   
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: S_READY_FLOP
   --
   -- Process Description:
   -- Registers S_READY handshake signals per Skid Buffer 
   -- Option 2 scheme
   --
   -------------------------------------------------------------
   S_READY_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then

             sig_s_ready_out  <= '0';
             sig_s_ready_dup  <= '0';
            
           Elsif (sig_spcl_s_ready_set = '1') Then
           
             sig_s_ready_out  <= '1';
             sig_s_ready_dup  <= '1';
           
           else

             sig_s_ready_out  <= sig_s_ready_comb;
             sig_s_ready_dup  <= sig_s_ready_comb;
            
           end if; 
        end if;       
      end process S_READY_FLOP; 
   
   
   
   
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: M_VALID_FLOP
   --
   -- Process Description:
   -- Registers M_VALID handshake signals per Skid Buffer 
   -- Option 2 scheme
   --
   -------------------------------------------------------------
   M_VALID_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST                 = '1' or
               sig_spcl_s_ready_set = '1') then -- Fix from AXI DMA

             sig_m_valid_out  <= '0';
             sig_m_valid_dup  <= '0';
            
           else

             sig_m_valid_out  <= sig_m_valid_comb;
             sig_m_valid_dup  <= sig_m_valid_comb;
            
           end if; 
        end if;       
      end process M_VALID_FLOP; 
   
   
   
   
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: SKID_DATA_REG
   --
   -- Process Description:
   -- This process implements the Skid register for the 
   -- Skid Buffer Data signals.
   --
   -------------------------------------------------------------
   SKID_DATA_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           
           if  (sig_skid_reg_en = '1') then
             
             sig_data_skid_reg <= S_DATA;
             
           else
             null;  -- hold current state
           end if;
            
        end if;       
      end process SKID_DATA_REG; 
            
            
   
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: SKID_CNTL_REG
   --
   -- Process Description:
   -- This process implements the Output registers for the 
   -- Skid Buffer Control signals
   --
   -------------------------------------------------------------
   SKID_CNTL_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_strb_skid_reg <= (others => '0');
             sig_last_skid_reg <= '0';
             
           elsif (sig_skid_reg_en = '1') then
             
             sig_strb_skid_reg <= sig_wstrb_demux_out;
             sig_last_skid_reg <= S_LAST;
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process SKID_CNTL_REG; 
            
            
   
            
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: OUTPUT_DATA_REG
   --
   -- Process Description:
   -- This process implements the Output register for the 
   -- Data signals.
   --
   -------------------------------------------------------------
   OUTPUT_DATA_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           
           if (sig_data_reg_out_en = '1') then
             
             sig_data_reg_out <= sig_data_skid_mux_out;
             
           else
             null;  -- hold current state
           end if;
            
        end if;       
      end process OUTPUT_DATA_REG; 
            
            
 
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: OUTPUT_CNTL_REG
   --
   -- Process Description:
   -- This process implements the Output registers for the 
   -- control signals.
   --
   -------------------------------------------------------------
   OUTPUT_CNTL_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_strb_reg_out <= (others => '0');
             sig_last_reg_out <= '0';
             
           elsif (sig_data_reg_out_en = '1') then
             
             sig_strb_reg_out <= sig_strb_skid_mux_out;
             sig_last_reg_out <= sig_last_skid_mux_out;
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process OUTPUT_CNTL_REG; 
            
            
 
 
 
 
 
 
   -------------------------------------------------------------
   -- Combinational Process
   --
   -- Label: DO_WR_DATA_MIRROR
   --
   -- Process Description:
   -- Implement the Write Data Mirror structure
   -- 
   -- Note that it is required that the Stream Width be less than
   -- or equal to the MMap WData width.
   --
   -------------------------------------------------------------
   DO_WR_DATA_MIRROR : process (sig_data_reg_out)
      begin
   
        for slice_index in 0 to MM2STRM_WIDTH_RATIO-1 loop
        
          sig_mirror_data_out(((C_SDATA_WIDTH*slice_index)+C_SDATA_WIDTH)-1 
                              downto C_SDATA_WIDTH*slice_index)
                              
                              <= sig_data_reg_out;
        
        end loop;
 
   
      end process DO_WR_DATA_MIRROR; 
   
 
 
 
 
        
    ------------------------------------------------------------
    -- Instance: I_WSTRB_DEMUX 
    --
    -- Description:
    -- Instance for the Write Strobe DeMux.    
    --
    ------------------------------------------------------------
     I_WSTRB_DEMUX : entity axi_datamover_v5_1_14.axi_datamover_wr_demux
     generic map (
      
       C_SEL_ADDR_WIDTH     =>  C_ADDR_LSB_WIDTH   ,  
       C_MMAP_DWIDTH        =>  C_MDATA_WIDTH      ,  
       C_STREAM_DWIDTH      =>  C_SDATA_WIDTH         
      
       )
     port map (
   
       wstrb_in             =>  S_STRB              , 
       demux_wstrb_out      =>  sig_wstrb_demux_out ,        
       debeat_saddr_lsb     =>  S_ADDR_LSB            
   
       );
   
 
 

end implementation;


-------------------------------------------------------------------------------
-- axi_datamover_skid_buf.vhd
-------------------------------------------------------------------------------
--
-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_datamover_skid_buf.vhd
--
-- Description:     
--  Implements the AXi Skid Buffer in the Option 2 (Registerd outputs) mode.                
--                  
--                  
--                  
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;



-------------------------------------------------------------------------------

entity axi_datamover_skid_buf is
  generic (
    C_WDATA_WIDTH : INTEGER := 32  
       --  Width of the Stream Data bus (in bits) 
               
    );
  port (
    
    -- Clock and Reset Inputs ---------------------------------------------
    aclk         : In  std_logic ;                                       --
    arst         : In  std_logic ;                                       --
    -----------------------------------------------------------------------
    
    
    -- Shutdown control (assert for 1 clk pulse) --------------------------
                                                                         --
    skid_stop    : In std_logic  ;                                       --
    -----------------------------------------------------------------------
    
    
    -- Slave Side (Stream Data Input) -------------------------------------
    s_valid      : In  std_logic ;                                       --
    s_ready      : Out std_logic ;                                       --
    s_data       : In  std_logic_vector(C_WDATA_WIDTH-1 downto 0);       --
    s_strb       : In  std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0);   --
    s_last       : In  std_logic ;                                       --
    -----------------------------------------------------------------------
    

    -- Master Side (Stream Data Output ------------------------------------
    m_valid      : Out std_logic ;                                       --
    m_ready      : In  std_logic ;                                       --
    m_data       : Out std_logic_vector(C_WDATA_WIDTH-1 downto 0);       --
    m_strb       : Out std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0);   --
    m_last       : Out std_logic                                         --
    -----------------------------------------------------------------------
    );

end entity axi_datamover_skid_buf;


architecture implementation of axi_datamover_skid_buf is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";



-- Signals decalrations -------------------------

  Signal sig_reset_reg         : std_logic := '0';
  signal sig_spcl_s_ready_set  : std_logic := '0';
  signal sig_data_skid_reg     : std_logic_vector(C_WDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_skid_reg     : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_skid_reg     : std_logic := '0';
  signal sig_skid_reg_en       : std_logic := '0';
  signal sig_data_skid_mux_out : std_logic_vector(C_WDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_skid_mux_out : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_skid_mux_out : std_logic := '0';
  signal sig_skid_mux_sel      : std_logic := '0';
  signal sig_data_reg_out      : std_logic_vector(C_WDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_reg_out      : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_reg_out      : std_logic := '0';
  signal sig_data_reg_out_en   : std_logic := '0';
  signal sig_m_valid_out       : std_logic := '0';
  signal sig_m_valid_dup       : std_logic := '0';
  signal sig_m_valid_comb      : std_logic := '0';
  signal sig_s_ready_out       : std_logic := '0';
  signal sig_s_ready_dup       : std_logic := '0';
  signal sig_s_ready_comb      : std_logic := '0';
  signal sig_stop_request      : std_logic := '0';
  signal sig_stopped           : std_logic := '0';
  signal sig_sready_stop       : std_logic := '0';
  signal sig_sready_early_stop : std_logic := '0';
  signal sig_sready_stop_set   : std_logic := '0';
  signal sig_sready_stop_reg   : std_logic := '0';
  signal sig_mvalid_stop_reg   : std_logic := '0';
  signal sig_mvalid_stop       : std_logic := '0';
  signal sig_mvalid_early_stop : std_logic := '0';
  signal sig_mvalid_stop_set   : std_logic := '0';
  signal sig_slast_with_stop   : std_logic := '0';
  signal sig_sstrb_stop_mask   : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_sstrb_with_stop   : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  
  

 
-- Register duplication attribute assignments to control fanout
-- on handshake output signals  
  
  Attribute KEEP : string; -- declaration
  Attribute EQUIVALENT_REGISTER_REMOVAL : string; -- declaration
  
  Attribute KEEP of sig_m_valid_out : signal is "TRUE"; -- definition
  Attribute KEEP of sig_m_valid_dup : signal is "TRUE"; -- definition
  Attribute KEEP of sig_s_ready_out : signal is "TRUE"; -- definition
  Attribute KEEP of sig_s_ready_dup : signal is "TRUE"; -- definition
  
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_m_valid_out : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_m_valid_dup : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_s_ready_out : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_s_ready_dup : signal is "no"; 
  
  
  

begin --(architecture implementation)

   m_valid <= sig_m_valid_out;         
   s_ready <= sig_s_ready_out; 
    
   m_strb  <= sig_strb_reg_out;
   m_last  <= sig_last_reg_out;                
   m_data  <= sig_data_reg_out;    
            
  
  
  
   -- Special shutdown logic version od Slast.
   -- A halt request forces a tlast through the skig buffer
   sig_slast_with_stop <= s_last or sig_stop_request;
   sig_sstrb_with_stop <= s_strb or sig_sstrb_stop_mask;
  
  
   -- Assign the special s_ready FLOP set signal
   sig_spcl_s_ready_set <= sig_reset_reg;
  
            
   -- Generate the ouput register load enable control
   sig_data_reg_out_en <= m_ready or not(sig_m_valid_dup);

   -- Generate the skid input register load enable control
   sig_skid_reg_en     <= sig_s_ready_dup;
  
   -- Generate the skid mux select control
   sig_skid_mux_sel    <= not(sig_s_ready_dup);
   
   
   -- Skid Mux  
   sig_data_skid_mux_out <=  sig_data_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  s_data;
  
   sig_strb_skid_mux_out <=  sig_strb_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  sig_sstrb_with_stop;
  
   sig_last_skid_mux_out <=  sig_last_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  sig_slast_with_stop;
  
   
   -- m_valid combinational logic        
   sig_m_valid_comb <= s_valid or
                      (sig_m_valid_dup and
                      (not(sig_s_ready_dup) or
                       not(m_ready)));
   
   
   
   -- s_ready combinational logic        
   sig_s_ready_comb <= m_ready or
                      (sig_s_ready_dup and
                      (not(sig_m_valid_dup) or
                       not(s_valid)));
   
   
   
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: REG_THE_RST
   --
   -- Process Description:
   -- Register input reset
   --
   -------------------------------------------------------------
   REG_THE_RST : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           
            sig_reset_reg <= ARST;
           
        end if;       
      end process REG_THE_RST; 
   
   
   
   
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: S_READY_FLOP
   --
   -- Process Description:
   -- Registers s_ready handshake signals per Skid Buffer 
   -- Option 2 scheme
   --
   -------------------------------------------------------------
   S_READY_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST                  = '1' or
               sig_sready_stop       = '1' or
               sig_sready_early_stop = '1') then  -- Special stop condition

             sig_s_ready_out  <= '0';
             sig_s_ready_dup  <= '0';
            
           Elsif (sig_spcl_s_ready_set = '1') Then
           
             sig_s_ready_out  <= '1';
             sig_s_ready_dup  <= '1';
           
           else

             sig_s_ready_out  <= sig_s_ready_comb;
             sig_s_ready_dup  <= sig_s_ready_comb;
            
           end if; 
        end if;       
      end process S_READY_FLOP; 
   
   
   
   
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: M_VALID_FLOP
   --
   -- Process Description:
   -- Registers m_valid handshake signals per Skid Buffer 
   -- Option 2 scheme
   --
   -------------------------------------------------------------
   M_VALID_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST                  = '1' or
               sig_spcl_s_ready_set  = '1' or    -- Fix from AXI DMA
               sig_mvalid_stop       = '1' or
               sig_mvalid_stop_set   = '1') then -- Special stop condition

             sig_m_valid_out  <= '0';
             sig_m_valid_dup  <= '0';
            
           else

             sig_m_valid_out  <= sig_m_valid_comb;
             sig_m_valid_dup  <= sig_m_valid_comb;
            
           end if; 
        end if;       
      end process M_VALID_FLOP; 
   
   
   
   
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: SKID_REG
   --
   -- Process Description:
   -- This process implements the output registers for the 
   -- Skid Buffer Data signals
   --
   -------------------------------------------------------------
   SKID_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_data_skid_reg <= (others => '0');
             sig_strb_skid_reg <= (others => '0');
             sig_last_skid_reg <= '0';
             
           elsif (sig_skid_reg_en = '1') then
             
             sig_data_skid_reg <= s_data;
             sig_strb_skid_reg <= sig_sstrb_with_stop;
             sig_last_skid_reg <= sig_slast_with_stop;
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process SKID_REG; 
            
            
   
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: OUTPUT_REG
   --
   -- Process Description:
   -- This process implements the output registers for the 
   -- Skid Buffer Data signals
   --
   -------------------------------------------------------------
   OUTPUT_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST                = '1' or
               sig_mvalid_stop_reg = '1') then
             
             sig_data_reg_out <= (others => '0');
             sig_strb_reg_out <= (others => '0');
             sig_last_reg_out <= '0';
             
           elsif (sig_data_reg_out_en = '1') then
             
             sig_data_reg_out <= sig_data_skid_mux_out;
             sig_strb_reg_out <= sig_strb_skid_mux_out;
             sig_last_reg_out <= sig_last_skid_mux_out;
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process OUTPUT_REG; 
            
            
  
  
  
  
  
  
  
  
  
  
  
  
   -------- Special Stop Logic --------------------------------------
   
   
   sig_sready_stop        <= sig_sready_stop_reg; 
  
  
   sig_sready_early_stop  <= skid_stop; -- deassert S_READY immediately
   
  
   sig_sready_stop_set    <= sig_sready_early_stop;
   
                                     
   sig_mvalid_stop        <=  sig_mvalid_stop_reg;
                                     
                                     
   sig_mvalid_early_stop  <= sig_m_valid_dup and
                            m_ready and
                            skid_stop;
  
    
   sig_mvalid_stop_set    <=  sig_mvalid_early_stop or
                              (sig_stop_request and 
                               not(sig_m_valid_dup)) or
                              (sig_m_valid_dup and
                               m_ready         and
                               sig_stop_request);
   
   
   
                             
    
    
                                     
                                     
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_STOP_REQ_FLOP
   --
   -- Process Description:
   -- This process implements the Stop request flop. It is a 
   -- sample and hold register that can only be cleared by reset.
   --
   -------------------------------------------------------------
   IMP_STOP_REQ_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_stop_request    <= '0';
             sig_sstrb_stop_mask <= (others => '0');
             
           elsif (skid_stop = '1') then
             
             sig_stop_request    <= '1';
             sig_sstrb_stop_mask <= (others => '1');
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process IMP_STOP_REQ_FLOP; 
            
            
  
  
  
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_CLR_SREADY_FLOP
   --
   -- Process Description:
   -- This process implements the flag to clear the s_ready 
   -- flop at a stop condition.
   --
   -------------------------------------------------------------
   IMP_CLR_SREADY_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_sready_stop_reg <= '0';
             
           elsif (sig_sready_stop_set  = '1') then
             
             sig_sready_stop_reg <= '1';
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process IMP_CLR_SREADY_FLOP; 
            
            
  
  
  
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_CLR_MVALID_FLOP
   --
   -- Process Description:
   -- This process implements the flag to clear the m_valid 
   -- flop at a stop condition.
   --
   -------------------------------------------------------------
   IMP_CLR_MVALID_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_mvalid_stop_reg <= '0';
             
           elsif (sig_mvalid_stop_set  = '1') then
             
             sig_mvalid_stop_reg <= '1';
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process IMP_CLR_MVALID_FLOP; 
            
            


end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_rd_sf.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_rd_sf.vhd
  --
  -- Description:     
  --    This file implements the AXI DataMover Read (MM2S) Store and Forward module. 
  --  The design utilizes the AXI DataMover's new address pipelining
  --  control function. The design is such that predictive address  
  --  pipelining can be supported on the AXI Read Bus without over-commiting 
  --  the internal Data FIFO and potentially throttling the Read Data Channel 
  --  if the Data FIFO goes full. 
  -- 
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library lib_pkg_v1_0_2;
  use lib_pkg_v1_0_2.lib_pkg.all;
  use lib_pkg_v1_0_2.lib_pkg.clog2;

  library axi_datamover_v5_1_14;
  use axi_datamover_v5_1_14.axi_datamover_sfifo_autord;
  use axi_datamover_v5_1_14.axi_datamover_fifo;

  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_rd_sf is
    generic (
      
      C_SF_FIFO_DEPTH        : Integer range 128 to 8192 := 512;
        -- Sets the desired depth of the internal Data FIFO.
      
      C_MAX_BURST_LEN        : Integer range  2 to  256 :=  16;
        -- Indicates the max burst length being used by the external
        -- AXI4 Master for each AXI4 transfer request.
        
      C_DRE_IS_USED          : Integer range   0 to    1 :=   0;
        -- Indicates if the external Master is utilizing a DRE on
        -- the stream input to this module.
         
      C_DRE_CNTL_FIFO_DEPTH  : Integer range  1 to  32 :=  1;
        -- Specifies the depth of the internal dre control queue fifo
      
      C_DRE_ALIGN_WIDTH      : Integer range  1 to   6 :=  2;
        -- Sets the width of the DRE alignment control ports
      
      C_MMAP_DWIDTH          : Integer range   32 to  1024 := 64;
        -- Sets the AXI4 Memory Mapped Bus Data Width 
      
      C_STREAM_DWIDTH        : Integer range   8 to  1024 :=  32;
        -- Sets the Stream Data Width for the Input and Output
        -- Data streams.
        
      C_STRT_SF_OFFSET_WIDTH : Integer range   1 to 7 :=  2;
        -- Sets the bit width of the starting address offset port
        -- This should be set to log2(C_MMAP_DWIDTH/C_STREAM_DWIDTH)
    C_ENABLE_MM2S_TKEEP             : integer range 0 to 1 := 1; 
        
      C_TAG_WIDTH            : Integer range  1 to   8 :=  4;
        -- Indicates the width of the Tag field of the input DRE command
        
      C_FAMILY               : String  := "virtex7"
        -- Indicates the target FPGA Family.
      
      );
    port (
      
      -- Clock and Reset inputs --------------------------------------------
                                                                          --
      aclk                    : in  std_logic;                            --
         -- Primary synchronization clock for the Master side             --
         -- interface and internal logic. It is also used                 --
         -- for the User interface synchronization when                   --
         -- C_STSCMD_IS_ASYNC = 0.                                        --
                                                                          --
      -- Reset input                                                      --
      reset                   : in  std_logic;                            --
         -- Reset used for the internal syncronization logic              --
      ----------------------------------------------------------------------
    
    
      -- DataMover Read Side Address Pipelining Control Interface ----------
                                                                          --
      ok_to_post_rd_addr      : Out  Std_logic;                           --
        -- Indicates that the transfer token pool has at least            --
        -- one token available to borrow                                  --
                                                                          --
      rd_addr_posted          : In std_logic;                             --
        -- Indication that a read address has been posted to AXI4         --
                                                                          --
      rd_xfer_cmplt           : In std_logic;                             --
        -- Indicates that the Datamover has completed a Read Data         --
        -- transfer on the AXI4                                           --
      ----------------------------------------------------------------------
      
       
        
      -- Read Side Stream In from DataMover MM2S Read Data Controller ----------------------
                                                                                          --
      sf2sin_tready           : Out  Std_logic;                                           --
        -- DRE  Stream READY input                                                        --
                                                                                          --
      sin2sf_tvalid           : In  std_logic;                                            --
        -- DRE Stream VALID Output                                                        --
                                                                                          --
      sin2sf_tdata            : In  std_logic_vector(C_MMAP_DWIDTH-1 downto 0);           --
        -- DRE  Stream DATA input                                                         --
                                                                                          --
      sin2sf_tkeep            : In std_logic_vector((C_MMAP_DWIDTH/8)-1 downto 0);        --   
        -- DRE  Stream STRB input                                                         --
                                                                                          --
      sin2sf_tlast            : In std_logic;                                             --
        -- DRE  Xfer LAST input                                                           --
      --------------------------------------------------------------------------------------

      
      -- RDC Store and Forward Supplimental Controls ---------------------
      -- These are time aligned and qualified with the RDC Stream Input --
                                                                        --
      data2sf_cmd_cmplt       : In std_logic;                           --
      data2sf_dre_flush       : In std_logic;                           --
      --------------------------------------------------------------------
     
     
               
                
      -- DRE Control Interface from the Command Calculator  -----------------------------
                                                                                       --
      dre2mstr_cmd_ready      : Out std_logic ;                                        --
        -- Indication from the DRE that the command is being                           --
        -- accepted from the Command Calculator                                        --
                                                                                       --
      mstr2dre_cmd_valid      : In std_logic;                                          --
        -- The next command valid indication to the DRE                                --
        -- from the Command Calculator                                                 --
                                                                                       --
      mstr2dre_tag            : In std_logic_vector(C_TAG_WIDTH-1 downto 0);           --
        -- The next command tag                                                        --
                                                                                       --
      mstr2dre_dre_src_align  : In std_logic_vector(C_DRE_ALIGN_WIDTH-1 downto 0);     --
        -- The source (input) alignment for the DRE                                    --
                                                                                       --
      mstr2dre_dre_dest_align : In std_logic_vector(C_DRE_ALIGN_WIDTH-1 downto 0);     --
        -- The destinstion (output) alignment for the DRE                              --
                                                                                       --
      -- mstr2dre_btt            : In std_logic_vector(C_BTT_USED-1 downto 0);            --
      --   -- The bytes to transfer value for the input command                           --
                                                                                       --
      mstr2dre_drr            : In std_logic;                                          --
        -- The starting tranfer of a sequence of transfers                             --
                                                                                       --
      mstr2dre_eof            : In std_logic;                                          --
        -- The endiing tranfer of a sequence of transfers                              --
                                                                                       --
      -- mstr2dre_cmd_cmplt      : In std_logic;                                          --
      --   -- The last tranfer command of a sequence of transfers                         --
      --   -- spawned from a single parent command                                        --
                                                                                       --
      mstr2dre_calc_error     : In std_logic;                                          --
        -- Indication if the next command in the calculation pipe                      --
        -- has a calculation error                                                     --
                                                                                       --
      mstr2dre_strt_offset    : In std_logic_vector(C_STRT_SF_OFFSET_WIDTH-1 downto 0);--
        -- Outputs the starting offset of a transfer. This is used with Store          --
        -- and Forward Packer/Unpacker logic                                           --
      -----------------------------------------------------------------------------------



     -- MM2S DRE Control  -------------------------------------------------------------
                                                                                     --
      sf2dre_new_align      : Out std_logic;                                         --
        -- Active high signal indicating new DRE aligment required                   --
                                                                                     --
      sf2dre_use_autodest   : Out std_logic;                                         --
        -- Active high signal indicating to the DRE to use an auto-                  --
        -- calculated desination alignment based on the last transfer                --
                                                                                     --
      sf2dre_src_align      : Out std_logic_vector(C_DRE_ALIGN_WIDTH-1 downto 0);    --
        -- Bit field indicating the byte lane of the first valid data byte           --
        -- being sent to the DRE                                                     --
                                                                                     --
      sf2dre_dest_align     : Out std_logic_vector(C_DRE_ALIGN_WIDTH-1 downto 0);    --
        -- Bit field indicating the desired byte lane of the first valid data byte   --
        -- to be output by the DRE                                                   --
                                                                                     --
      sf2dre_flush          : Out std_logic;                                         --
        -- Active high signal indicating to the DRE to flush the current             --
        -- contents to the output register in preparation of a new alignment         --
        -- that will be comming on the next transfer input                           --
      ---------------------------------------------------------------------------------
               
                
                
                
      -- Stream Out  -----------------------------------------------------------------------
                                                                                          --
      sout2sf_tready          : In  std_logic;                                            --
        -- Write READY input from the Stream Master                                       --
                                                                                          --
      sf2sout_tvalid          : Out  std_logic;                                           --
        -- Write VALID output to the Stream Master                                        --
                                                                                          --
      sf2sout_tdata           : Out  std_logic_vector(C_STREAM_DWIDTH-1 downto 0);        --
        -- Write DATA output to the Stream Master                                         --
                                                                                          --
      sf2sout_tkeep           : Out  std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);    --
        -- Write DATA output to the Stream Master                                         --
                                                                                          --
      sf2sout_tlast           : Out  std_logic                                            --
        -- Write LAST output to the Stream Master                                         --
      --------------------------------------------------------------------------------------
 
      );
  
  end entity axi_datamover_rd_sf;
  
  
  architecture implementation of axi_datamover_rd_sf is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    
    -- Functions ---------------------------------------------------------------------------
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_fifo_cnt_width
    --
    -- Function Description:
    --   simple function to set the width of the data fifo read 
    -- and write count outputs. 
    -------------------------------------------------------------------
    function funct_get_fifo_cnt_width (fifo_depth : integer) 
             return integer is
    
      Variable temp_width : integer := 8;
    
    begin
    
      if (fifo_depth = 1) then
      
         temp_width := 1;
      
      elsif (fifo_depth  = 2) then
      
         temp_width := 2;
      
      elsif (fifo_depth  <= 4) then
      
         temp_width := 3;
      
      elsif (fifo_depth  <= 8) then
      
         temp_width := 4;
      
      elsif (fifo_depth  <= 16) then
      
         temp_width := 5;
      
      elsif (fifo_depth  <= 32) then
      
         temp_width := 6;
      
      elsif (fifo_depth  <= 64) then
      
         temp_width := 7;
      
      elsif (fifo_depth  <= 128) then
      
         temp_width := 8;
      
      elsif (fifo_depth  <= 256) then
      
         temp_width := 9;
      
      elsif (fifo_depth  <= 512) then
      
         temp_width := 10;
      
      elsif (fifo_depth  <= 1024) then
      
         temp_width := 11;
      
      elsif (fifo_depth  <= 2048) then
      
         temp_width := 12;
      
      elsif (fifo_depth  <= 4096) then
      
         temp_width := 13;
      
      else -- assume 8192 depth
      
         temp_width := 14;
      
      end if;
      
      
      Return (temp_width);
     
    
    end function funct_get_fifo_cnt_width;
    
    
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_wrcnt_lsrip
    --
    -- Function Description:
    --   Calculates the ls index of the upper slice of the data fifo
    -- write count needed to repesent one max burst worth of data
    -- present in the fifo.
    --
    -------------------------------------------------------------------
    function funct_get_wrcnt_lsrip (max_burst_dbeats : integer) return integer is
    
      Variable temp_ls_index : Integer := 0;
    
    begin
      
      if (max_burst_dbeats <= 2) then
      
        temp_ls_index := 1;
      
      elsif (max_burst_dbeats <= 4) then
      
        temp_ls_index := 2;
      
      elsif (max_burst_dbeats <= 8) then
      
        temp_ls_index := 3;
      
      elsif (max_burst_dbeats <= 16) then
      
        temp_ls_index := 4;
      
      elsif (max_burst_dbeats <= 32) then
      
        temp_ls_index := 5;
      
      elsif (max_burst_dbeats <= 64) then
      
        temp_ls_index := 6;
      
      elsif (max_burst_dbeats <= 128) then
      
        temp_ls_index := 7;
      
      else
      
        temp_ls_index := 8;
      
      end if;
      
      Return (temp_ls_index);
      
      
    end function funct_get_wrcnt_lsrip;
    
     
     
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_stall_thresh
    --
    -- Function Description:
    -- Calculates the Stall threshold for the input side of the Data
    -- FIFO. If DRE is being used by the DataMover, then the threshold
    -- must be reduced to account for the potential of an extra write
    -- databeat per request (DRE alignment dependent).
    --
    -------------------------------------------------------------------
    function funct_get_stall_thresh (dre_is_used         : integer;
                                     max_xfer_length     : integer;
                                     data_fifo_depth     : integer;
                                     pipeline_delay_clks : integer;
                                     fifo_settling_clks  : integer) return integer is
    
      Constant DRE_PIPE_DELAY             : integer := 2; -- clks
      
      Variable var_num_max_xfers_allowed  : Integer := 0;
      Variable var_dre_dbeat_overhead     : Integer := 0;
      Variable var_delay_fudge_factor     : Integer := 0;    
      Variable var_thresh_headroom        : Integer := 0;
      Variable var_stall_thresh           : Integer := 0;
      
    begin
    
      var_num_max_xfers_allowed := data_fifo_depth/max_xfer_length;
      
      var_dre_dbeat_overhead    := var_num_max_xfers_allowed * dre_is_used;
      
      
      var_delay_fudge_factor    := (dre_is_used * DRE_PIPE_DELAY) + 
                                   pipeline_delay_clks            + 
                                   fifo_settling_clks;
      
      var_thresh_headroom       := max_xfer_length        + 
                                   var_dre_dbeat_overhead + 
                                   var_delay_fudge_factor;
      
      -- Scale the result to be in max transfer length increments
      var_stall_thresh          := (data_fifo_depth - var_thresh_headroom)/max_xfer_length;
      
      Return (var_stall_thresh);
      
      
    end function funct_get_stall_thresh;
    
    
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_size_drecntl_fifo
    --
    -- Function Description:
    --  Assures that the DRE control fifo depth is at least 4 deep else it
    -- is equal to the number of max burst transfers that can fit in the
    -- Store and Forward Data FIFO.
    --
    -------------------------------------------------------------------
    function funct_size_drecntl_fifo (sf_fifo_depth    : integer;
                                      max_burst_length : integer) return integer is
    
      
      Constant NEEDED_FIFO_DEPTH : integer := sf_fifo_depth/max_burst_length;
      
      
      Variable temp_fifo_depth : Integer := 4;
    
    begin
    
      If (NEEDED_FIFO_DEPTH < 4) Then
    
        temp_fifo_depth := 4;
      
      Else 

        temp_fifo_depth := NEEDED_FIFO_DEPTH;
          
      End if;
      
      
      Return (temp_fifo_depth);
      
      
    end function funct_size_drecntl_fifo;
    

     
     

    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_cntr_width
    --
    -- Function Description:
    --  Detirmine the width needed for the address offset counter used
    -- for the data fifo mux selects.
    --
    -------------------------------------------------------------------
    function funct_get_cntr_width (num_count_states : integer) return integer is
    
     Variable lvar_temp_width : Integer := 1;
    
    begin
      
      if (num_count_states <= 2) then
      
         lvar_temp_width := 1;
      
      elsif (num_count_states <= 4) then
      
         lvar_temp_width := 2;
      
      elsif (num_count_states <= 8) then
      
         lvar_temp_width := 3;
      
      elsif (num_count_states <= 16) then
      
         lvar_temp_width := 4;
      
      elsif (num_count_states <= 32) then
      
         lvar_temp_width := 5;
      
      elsif (num_count_states <= 64) then
      
         lvar_temp_width := 6;
      
      Else   -- 128 cnt states
         
         lvar_temp_width := 7;
      
      end if;
      
      
      
      Return (lvar_temp_width);
      
      
      
    end function funct_get_cntr_width;
    
    
    
    
    -- Constants ---------------------------------------------------------------------------
    
    Constant LOGIC_LOW                 : std_logic := '0';
    Constant LOGIC_HIGH                : std_logic := '1';
    
    Constant BLK_MEM_FIFO              : integer := 1;
    Constant SRL_FIFO                  : integer := 0;
    Constant NOT_NEEDED                : integer := 0;
    
    
    Constant MMAP_TKEEP_WIDTH          : integer := C_MMAP_DWIDTH/8; -- bits
    Constant TLAST_WIDTH               : integer := 1;               -- bits
    Constant CMPLT_WIDTH               : integer := 1;               -- bits
    Constant DRE_FLUSH_WIDTH           : integer := 1;               -- bits
    
    
    Constant DATA_FIFO_DEPTH           : integer := C_SF_FIFO_DEPTH;
    Constant DATA_FIFO_CNT_WIDTH       : integer := funct_get_fifo_cnt_width(DATA_FIFO_DEPTH);
    Constant DF_WRCNT_RIP_LS_INDEX     : integer := funct_get_wrcnt_lsrip(C_MAX_BURST_LEN);
    Constant DATA_FIFO_WIDTH           : integer := C_MMAP_DWIDTH    +
                                                    MMAP_TKEEP_WIDTH*C_ENABLE_MM2S_TKEEP +
                                                    TLAST_WIDTH      +
                                                    CMPLT_WIDTH      +
                                                    DRE_FLUSH_WIDTH;
    
    Constant DATA_OUT_LSB_INDEX        : integer := 0;
    Constant DATA_OUT_MSB_INDEX        : integer := C_MMAP_DWIDTH-1;
    
    Constant TKEEP_OUT_LSB_INDEX       : integer := DATA_OUT_MSB_INDEX+1;
    Constant TKEEP_OUT_MSB_INDEX       : integer := (TKEEP_OUT_LSB_INDEX+MMAP_TKEEP_WIDTH*C_ENABLE_MM2S_TKEEP)-1*C_ENABLE_MM2S_TKEEP;
    
    Constant TLAST_OUT_INDEX           : integer := TKEEP_OUT_MSB_INDEX+1*C_ENABLE_MM2S_TKEEP;
    Constant CMPLT_OUT_INDEX           : integer := TLAST_OUT_INDEX+1;
    Constant DRE_FLUSH_OUT_INDEX       : integer := CMPLT_OUT_INDEX+1;
    
    
    Constant TOKEN_POOL_SIZE           : integer := C_SF_FIFO_DEPTH / C_MAX_BURST_LEN;
    
    Constant TOKEN_CNTR_WIDTH          : integer := clog2(TOKEN_POOL_SIZE)+1;
    
    Constant TOKEN_CNT_ZERO            : Unsigned(TOKEN_CNTR_WIDTH-1 downto 0) := 
                                         TO_UNSIGNED(0, TOKEN_CNTR_WIDTH);
    
    Constant TOKEN_CNT_ONE             : Unsigned(TOKEN_CNTR_WIDTH-1 downto 0) := 
                                         TO_UNSIGNED(1, TOKEN_CNTR_WIDTH);
    
    Constant TOKEN_CNT_MAX             : Unsigned(TOKEN_CNTR_WIDTH-1 downto 0) :=  
                                         TO_UNSIGNED(TOKEN_POOL_SIZE, TOKEN_CNTR_WIDTH);
    
    Constant THRESH_COMPARE_WIDTH      : integer := TOKEN_CNTR_WIDTH+2;
    
              
    Constant RD_PATH_PIPE_DEPTH        : integer := 2; -- clocks excluding DRE
    
    Constant WRCNT_SETTLING_TIME       : integer := 2; -- data fifo push or pop settling clocks
    
    
    Constant DRE_COMPENSATION          : integer := 0; -- DRE does not contribute since it is on
                                                       -- the output side of the Store and Forward
    
    Constant RD_ADDR_POST_STALL_THRESH : integer := 
                                         funct_get_stall_thresh(DRE_COMPENSATION   ,
                                                                C_MAX_BURST_LEN    ,
                                                                C_SF_FIFO_DEPTH    ,
                                                                RD_PATH_PIPE_DEPTH ,
                                                                WRCNT_SETTLING_TIME);
    
    Constant RD_ADDR_POST_STALL_THRESH_US : Unsigned(THRESH_COMPARE_WIDTH-1 downto 0) := 
                                            TO_UNSIGNED(RD_ADDR_POST_STALL_THRESH , 
                                                        THRESH_COMPARE_WIDTH);
    
    
    Constant UNCOM_WRCNT_1             : Unsigned(DATA_FIFO_CNT_WIDTH-1 downto 0) := 
                                         TO_UNSIGNED(1, DATA_FIFO_CNT_WIDTH);
    
    Constant UNCOM_WRCNT_0             : Unsigned(DATA_FIFO_CNT_WIDTH-1 downto 0) := 
                                         TO_UNSIGNED(0, DATA_FIFO_CNT_WIDTH);
    

    Constant USE_SYNC_FIFO             : integer := 0;
    Constant SRL_FIFO_PRIM             : integer := 2;
    
    Constant TAG_WIDTH                 : integer := C_TAG_WIDTH;
    Constant SRC_ALIGN_WIDTH           : integer := C_DRE_ALIGN_WIDTH;
    Constant DEST_ALIGN_WIDTH          : integer := C_DRE_ALIGN_WIDTH;
    Constant DRR_WIDTH                 : integer := 1;
    Constant EOF_WIDTH                 : integer := 1;
    Constant CALC_ERR_WIDTH            : integer := 1;
    Constant SF_OFFSET_WIDTH           : integer := C_STRT_SF_OFFSET_WIDTH;
    
    

    
    
    
    -- Signals ---------------------------------------------------------------------------
    
    signal sig_good_sin_strm_dbeat    : std_logic := '0';
    signal sig_strm_sin_ready         : std_logic := '0';
    
    signal sig_good_sout_strm_dbeat   : std_logic := '0';
    signal sig_sout2sf_tready         : std_logic := '0';
    signal sig_sf2sout_tvalid         : std_logic := '0';
    signal sig_sf2sout_tdata          : std_logic_vector(C_STREAM_DWIDTH-1 downto 0) := (others => '0');
    signal sig_sf2sout_tkeep          : std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0) := (others => '0');
    signal sig_sf2sout_tlast          : std_logic := '0';
    
    signal sig_sf2dre_flush           : std_logic := '0';
    
    signal sig_push_data_fifo         : std_logic := '0';
    signal sig_pop_data_fifo          : std_logic := '0';
    signal sig_data_fifo_full         : std_logic := '0';
    signal sig_data_fifo_data_in      : std_logic_vector(DATA_FIFO_WIDTH-1 downto 0) := (others => '0');
    signal sig_data_fifo_dvalid       : std_logic := '0';
    signal sig_data_fifo_data_out     : std_logic_vector(DATA_FIFO_WIDTH-1 downto 0) := (others => '0');
    signal sig_data_fifo_wr_cnt       : std_logic_vector(DATA_FIFO_CNT_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_wr_cnt_unsgnd     : unsigned(DATA_FIFO_CNT_WIDTH-1 downto 0) := (others => '0');
    
    signal sig_wrcnt_mblen_slice      : unsigned(DATA_FIFO_CNT_WIDTH-1 downto 
                                                 DF_WRCNT_RIP_LS_INDEX) := (others => '0');
    
    signal sig_ok_to_post_rd_addr     : std_logic := '0';
    signal sig_rd_addr_posted         : std_logic := '0';
    signal sig_rd_xfer_cmplt          : std_logic := '0';
    signal sig_taking_last_token      : std_logic := '0';
    signal sig_stall_rd_addr_posts    : std_logic := '0';
    
    signal sig_incr_token_cntr        : std_logic := '0';
    signal sig_decr_token_cntr        : std_logic := '0';
    signal sig_token_eq_max           : std_logic := '0';
    signal sig_token_eq_zero          : std_logic := '0';
    signal sig_token_eq_one           : std_logic := '0';
    signal sig_token_cntr             : Unsigned(TOKEN_CNTR_WIDTH-1 downto 0) := (others => '0');
    
    signal sig_tokens_commited        : Unsigned(TOKEN_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_commit_plus_actual     : unsigned(THRESH_COMPARE_WIDTH-1 downto 0) := (others => '0');
    
    signal sig_cntl_fifo_has_data     : std_logic := '0';
    signal sig_get_cntl_fifo_data     : std_logic := '0';
                    
    signal sig_curr_tag_reg           : std_logic_vector(TAG_WIDTH-1 downto 0)        := (others => '0');
    signal sig_curr_src_align_reg     : std_logic_vector(SRC_ALIGN_WIDTH-1 downto 0)  := (others => '0');
    signal sig_curr_dest_align_reg    : std_logic_vector(DEST_ALIGN_WIDTH-1 downto 0) := (others => '0');
    signal sig_curr_drr_reg           : std_logic := '0';
    signal sig_curr_eof_reg           : std_logic := '0';
    signal sig_curr_calc_error_reg    : std_logic := '0';
    signal sig_curr_strt_offset_reg   : std_logic_vector(SF_OFFSET_WIDTH-1 downto 0) := (others => '0');
    
    signal sig_ld_dre_cntl_reg        : std_logic := '0';
    
    signal sig_dfifo_data_out         : std_logic_vector(C_MMAP_DWIDTH-1 downto 0)    := (others => '0');
    signal sig_dfifo_tkeep_out        : std_logic_vector(MMAP_TKEEP_WIDTH-1 downto 0) := (others => '0');
    signal sig_dfifo_tlast_out        : std_logic := '0';
    signal sig_dfifo_cmd_cmplt_out    : std_logic := '0';
    signal sig_dfifo_dre_flush_out    : std_logic := '0';
    
  
  begin --(architecture implementation)
  
   
   
    -- Read Side (MM2S) Control Flags port connections
    ok_to_post_rd_addr       <= sig_ok_to_post_rd_addr ;
    sig_rd_addr_posted       <= rd_addr_posted         ;
    sig_rd_xfer_cmplt        <= rd_xfer_cmplt          ;
    
    
    
    --  Output Stream Port connections
    sig_sout2sf_tready       <= sout2sf_tready        ;
    sf2sout_tvalid           <= sig_sf2sout_tvalid    ;
    sf2sout_tdata            <= sig_sf2sout_tdata     ; 
    --sf2sout_tkeep            <= sig_sf2sout_tkeep     ;
    sf2sout_tlast            <= sig_sf2sout_tlast and
                                sig_sf2sout_tvalid    ;
    
   



GEN_MM2S_TKEEP_ENABLE4 : if C_ENABLE_MM2S_TKEEP = 1 generate
begin

    sf2sout_tkeep            <= sig_sf2sout_tkeep     ;
 
end generate GEN_MM2S_TKEEP_ENABLE4;

GEN_MM2S_TKEEP_DISABLE4 : if C_ENABLE_MM2S_TKEEP = 0 generate
begin

  sf2sout_tkeep        <= (others => '1');

end generate GEN_MM2S_TKEEP_DISABLE4;




 
    -- Input Stream port connections 
    sf2sin_tready            <= sig_strm_sin_ready;
    
    sig_strm_sin_ready       <= not(sig_data_fifo_full); -- Throttle if Read Side Data fifo goes full.
                                                         -- This should never happen if read address 
                                                         -- posting control is working properly.
    
    
    -- Stream transfer qualifiers
    
    sig_good_sin_strm_dbeat  <= sin2sf_tvalid and
                                sig_strm_sin_ready;
                               
 
    sig_good_sout_strm_dbeat <= sig_sf2sout_tvalid and  
                                sig_sout2sf_tready;     
  
  
  
  
  
  
  
 ---------------------------------------------------------------- 
 -- Unpacking Logic    ------------------------------------------
 ---------------------------------------------------------------- 
 
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: OMIT_UNPACKING
    --
    -- If Generate Description:
    --    Omits any unpacking logic in the Store and Forward module.
    -- The Stream and MMap data widths are the same. The Data FIFO
    -- output can be connected directly to the stream outputs.
    --
    ------------------------------------------------------------
    OMIT_UNPACKING : if (C_MMAP_DWIDTH = C_STREAM_DWIDTH) generate
    
      signal lsig_cmd_loaded          : std_logic := '0';
      signal lsig_ld_cmd              : std_logic := '0';
      signal lsig_cmd_cmplt_dbeat     : std_logic := '0';
      signal lsig_cmd_cmplt           : std_logic := '0';
      
      begin
   
            
            
        -- Data FIFO Output to the stream attachments
        sig_sf2sout_tvalid      <=  sig_data_fifo_dvalid and 
                                    lsig_cmd_loaded     ;
        
        sig_sf2sout_tdata       <=  sig_dfifo_data_out  ;
        
        sig_sf2sout_tkeep       <=  sig_dfifo_tkeep_out ;
        
        sig_sf2sout_tlast       <=  sig_dfifo_tlast_out ;
        
        sig_sf2dre_flush        <=  sig_dfifo_dre_flush_out   ;
        
         
         
        -- Control for reading the Data FIFO
        sig_pop_data_fifo       <=  lsig_cmd_loaded    and
                                    sig_sout2sf_tready and 
                                    sig_data_fifo_dvalid;
        
        -- Control for reading the Command/Offset FIFO
        sig_get_cntl_fifo_data  <= lsig_ld_cmd ;
      
        
        -- Control for loading the DRE Control Reg
        sig_ld_dre_cntl_reg     <= lsig_ld_cmd ;
        
       
        lsig_cmd_cmplt_dbeat    <= sig_dfifo_cmd_cmplt_out and
                                   lsig_cmd_loaded         and
                                   sig_data_fifo_dvalid    and
                                   sig_sout2sf_tready  ;

     
        -- Generate the control that loads the DRE
        lsig_ld_cmd             <= (sig_cntl_fifo_has_data and  -- startup or gap case
                                    not(lsig_cmd_loaded))  or
                                   (sig_cntl_fifo_has_data and  -- back to back commands
                                    lsig_cmd_cmplt_dbeat);
                                      
         
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_CMD_LOADED
        --
        -- Process Description:
        --  Implements the flop indicating a command from the cmd fifo
        -- has been loaded into the DRE Output Register.
        --
        -------------------------------------------------------------
        IMP_CMD_LOADED : process (aclk)
          begin
            if (aclk'event and aclk = '1') then
              if (reset = '1') then
       
                lsig_cmd_loaded <= '0';
       
              Elsif (lsig_ld_cmd = '1' ) Then
              
                lsig_cmd_loaded <= '1';
              
              elsif (sig_cntl_fifo_has_data = '0' and   -- No more commands queued and
                     lsig_cmd_cmplt_dbeat   = '1') then
       
                lsig_cmd_loaded <= '0';
       
              else
       
                null;  -- Hold Current State
       
              end if; 
            end if;       
          end process IMP_CMD_LOADED; 
        
        
        
       
       
       
      
      end generate OMIT_UNPACKING;
    
     
    
    
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: INCLUDE_UNPACKING
    --
    -- If Generate Description:
    --    Includes unpacking logic in the Store and Forward module.
    -- The MMap Data bus is wider than the Stream width.
    --
    ------------------------------------------------------------
    INCLUDE_UNPACKING : if (C_MMAP_DWIDTH > C_STREAM_DWIDTH) generate
    
      Constant MMAP2STRM_WIDTH_RATO  : integer := C_MMAP_DWIDTH/C_STREAM_DWIDTH;
    
      Constant DATA_SLICE_WIDTH      : integer := C_STREAM_DWIDTH;
      
      Constant TKEEP_SLICE_WIDTH     : integer := C_STREAM_DWIDTH/8;
      
      Constant FLAG_SLICE_WIDTH      : integer := TLAST_WIDTH;
      
      Constant OFFSET_CNTR_WIDTH     : integer := funct_get_cntr_width(MMAP2STRM_WIDTH_RATO);
      
      Constant OFFSET_CNT_ONE        : unsigned(OFFSET_CNTR_WIDTH-1 downto 0) := 
                                       TO_UNSIGNED(1, OFFSET_CNTR_WIDTH);
      
      Constant OFFSET_CNT_MAX        : unsigned(OFFSET_CNTR_WIDTH-1 downto 0) := 
                                       TO_UNSIGNED(MMAP2STRM_WIDTH_RATO-1, OFFSET_CNTR_WIDTH);
      
      
      
      -- Types -----------------------------------------------------------------------------
      type lsig_data_slice_type is array(MMAP2STRM_WIDTH_RATO-1 downto 0) of
                    std_logic_vector(DATA_SLICE_WIDTH-1 downto 0);

      type lsig_tkeep_slice_type is array(MMAP2STRM_WIDTH_RATO downto 0) of
                    std_logic_vector(TKEEP_SLICE_WIDTH-1 downto 0);

      type lsig_flag_slice_type is array(MMAP2STRM_WIDTH_RATO-1 downto 0) of
                    std_logic_vector(FLAG_SLICE_WIDTH-1 downto 0);


       
      -- local signals
      
      signal lsig_0ffset_cntr         : unsigned(OFFSET_CNTR_WIDTH-1 downto 0) := (others => '0');
      signal lsig_ld_offset           : std_logic := '0';
      signal lsig_incr_offset         : std_logic := '0';
      signal lsig_offset_cntr_eq_max  : std_logic := '0';
      signal lsig_fifo_data_out_wide  : lsig_data_slice_type;
      signal lsig_fifo_tkeep_out_wide : lsig_tkeep_slice_type;
      signal lsig_mux_sel             : integer range 0 to MMAP2STRM_WIDTH_RATO-1;
      signal lsig_data_mux_out        : std_logic_vector(DATA_SLICE_WIDTH-1 downto 0) ;
      signal lsig_tkeep_mux_out       : std_logic_vector(TKEEP_SLICE_WIDTH-1 downto 0);
      signal lsig_tlast_out           : std_logic := '0';
      signal lsig_dre_flush_out       : std_logic := '0';
      signal lsig_this_fifo_wrd_done  : std_logic := '0';
      signal lsig_cmd_loaded          : std_logic := '0';
      signal lsig_cmd_cmplt_dbeat     : std_logic := '0';
      signal lsig_cmd_cmplt           : std_logic := '0';
      signal lsig_next_slice_tkeep_0  : std_logic := '0';
       
        
      begin
    
       
       sig_sf2sout_tvalid      <= sig_data_fifo_dvalid and 
                                  lsig_cmd_loaded      ;
       
       sig_sf2sout_tdata       <= lsig_data_mux_out    ;
       
       sig_sf2sout_tkeep       <= lsig_tkeep_mux_out(TKEEP_SLICE_WIDTH-1 downto 0);
      
       sig_sf2sout_tlast       <= lsig_tlast_out       ;
      
       sig_sf2dre_flush        <= lsig_dre_flush_out   ;
       

       
       
       -- Control for reading the Data FIFO
       sig_pop_data_fifo       <= lsig_this_fifo_wrd_done and
                                  lsig_cmd_loaded         and
                                  sig_sout2sf_tready      and 
                                  sig_data_fifo_dvalid;
     
       -- Control for reading the Command/Offset FIFO
       sig_get_cntl_fifo_data  <= lsig_ld_offset;
       
       
       -- Control for loading the DRE Control Reg
       sig_ld_dre_cntl_reg     <= lsig_ld_offset ;
       
       
       lsig_next_slice_tkeep_0 <= lsig_fifo_tkeep_out_wide(lsig_mux_sel+1)(0);
       
       
       -- Detirmine if a Command Complete condition exists
       lsig_cmd_cmplt  <= '1'
         when (sig_dfifo_cmd_cmplt_out = '1' and
               lsig_next_slice_tkeep_0 = '0')
         Else '0';
       
       
       
       -- Detirmine if a TLAST condition exists
       -- From the RDC via the Data FIFO
       lsig_tlast_out <= '1'
         when (sig_dfifo_tlast_out     = '1' and
               lsig_next_slice_tkeep_0 = '0')
         Else '0';
        
       
       -- Detimine if a DRE Flush condition exists
       -- From the RDC via the Data FIFO
       lsig_dre_flush_out <= '1'
         when (sig_dfifo_dre_flush_out = '1' and
               lsig_next_slice_tkeep_0 = '0')
         Else '0';
      
      
      
       
       
       lsig_cmd_cmplt_dbeat <= lsig_cmd_cmplt          and
                               lsig_cmd_loaded         and
                               sig_data_fifo_dvalid    and
                               sig_sout2sf_tready  ;

     
       
       -- Check to see if the FIFO output word is finished. This occurs
       -- when the offset counter is at max value or the tlast from the
       -- fifo is set and the LS TKEED of the next MS Slice is zero.
       lsig_this_fifo_wrd_done  <= '1'
         When (lsig_offset_cntr_eq_max = '1' or
              (lsig_cmd_cmplt_dbeat    = '1' and
               lsig_next_slice_tkeep_0 = '0'))
         Else '0';
      
       
        
       -- Generate the control that loads the starting address
       -- offset for the next input packet
       lsig_ld_offset          <= (sig_cntl_fifo_has_data and  -- startup or gap case
                                   not(lsig_cmd_loaded))  or
                                  (sig_cntl_fifo_has_data and  -- back to back commands
                                   lsig_cmd_cmplt_dbeat);
                                  
       -- Generate the control for incrementing the offset counter
       lsig_incr_offset        <= sig_good_sout_strm_dbeat;
       
       
       -- Check to see if the offset counter has reached its max
       -- value
       lsig_offset_cntr_eq_max <=  '1'
         when  (lsig_0ffset_cntr = OFFSET_CNT_MAX)
         Else '0';
       
       
       
        
        
       -------------------------------------------------------------
       -- Synchronous Process with Sync Reset
       --
       -- Label: IMP_CMD_LOADED
       --
       -- Process Description:
       --  Implements the flop indicating a command from the cmd fifo
       -- has been loaded into the unpacker control logic.
       --
       -------------------------------------------------------------
       IMP_CMD_LOADED : process (aclk)
         begin
           if (aclk'event and aclk = '1') then
             if (reset = '1') then
      
               lsig_cmd_loaded <= '0';
      
             Elsif (lsig_ld_offset = '1' ) Then
             
               lsig_cmd_loaded <= '1';
             
             elsif (sig_cntl_fifo_has_data = '0' and   -- No more commands queued
                    lsig_cmd_cmplt_dbeat   = '1') then
      
               lsig_cmd_loaded <= '0';
      
             else
      
               null;  -- Hold Current State
      
             end if; 
           end if;       
         end process IMP_CMD_LOADED; 
       
       
       
       
       
       
       -------------------------------------------------------------
       -- Synchronous Process with Sync Reset
       --
       -- Label: IMP_OFFSET_CNTR
       --
       -- Process Description:
       --  Implements the address offset counter that is used to 
       -- generate the data and tkeep mux selects.
       -- Note that the counter has to be loaded with the starting
       -- offset plus one to sync up with the data input.
       -------------------------------------------------------------
       IMP_OFFSET_CNTR : process (aclk)
         begin
           if (aclk'event and aclk = '1') then
              if (reset = '1') then
       
                lsig_0ffset_cntr <= (others => '0');
       
              Elsif (lsig_ld_offset = '1') Then
              
                lsig_0ffset_cntr <= UNSIGNED(sig_curr_strt_offset_reg);
              
              elsif (lsig_incr_offset = '1') then
       
                lsig_0ffset_cntr <= lsig_0ffset_cntr + OFFSET_CNT_ONE;
       
              else
       
                null;  -- Hold Current State
       
              end if; 
           end if;       
         end process IMP_OFFSET_CNTR; 
       
       
       
       
     
       ------------------------------------------------------------
       -- For Generate
       --
       -- Label: DO_DATA_CONVERTER
       --
       -- For Generate Description:
       --   This ForGen converts the FIFO output data and tkeep from a single
       -- std logic vector type to a vector of slices. 
       --
       ------------------------------------------------------------
       DO_DATA_CONVERTER : for slice_index in 1 to MMAP2STRM_WIDTH_RATO generate
       
       begin
        
         lsig_fifo_data_out_wide(slice_index-1) <=
                            sig_dfifo_data_out((slice_index*DATA_SLICE_WIDTH)-1 downto 
                            (slice_index-1)*DATA_SLICE_WIDTH);
        
         
         lsig_fifo_tkeep_out_wide(slice_index-1) <=
                            sig_dfifo_tkeep_out((slice_index*TKEEP_SLICE_WIDTH)-1 downto 
                            (slice_index-1)*TKEEP_SLICE_WIDTH);
         
        
       end generate DO_DATA_CONVERTER;
     
     
       -- Assign the extra tkeep slice to all zeros to allow for detection
       -- of the data word done when the ls tkeep bit of the next tkeep 
       -- slice is zero and the offset count is pointing to the last slice
       -- position.
       lsig_fifo_tkeep_out_wide(MMAP2STRM_WIDTH_RATO) <= (others => '0');

       
       
       -- Mux the appropriate data and tkeep slice to the stream output
       lsig_mux_sel       <=  TO_INTEGER(lsig_0ffset_cntr);
     
       lsig_data_mux_out  <=  lsig_fifo_data_out_wide(lsig_mux_sel) ; 
     
       lsig_tkeep_mux_out(TKEEP_SLICE_WIDTH-1 downto 0) <=  lsig_fifo_tkeep_out_wide(lsig_mux_sel);
       
      
       
       
      end generate INCLUDE_UNPACKING;
     
   
   
   
   
   
   
   
     
    
    
  
  
      ------------------------------------------------------------
      -- If Generate
      --
      -- Label: OMIT_DRE_CNTL
      --
      -- If Generate Description:
      --   This IfGen is used to omit the DRE control logic and 
      -- minimize the Control FIFO when MM2S DRE is not included
      -- in the MM2S. 
      --
      ------------------------------------------------------------
      OMIT_DRE_CNTL : if (C_DRE_IS_USED = 0) generate
      
        
        -- Constant Declarations ------------------------------------------------------------------
        
        Constant USE_SYNC_FIFO           : integer := 0;
        Constant SRL_FIFO_PRIM           : integer := 2;
        
        Constant TAG_WIDTH               : integer := C_TAG_WIDTH;
        Constant DRR_WIDTH               : integer := 1;
        Constant EOF_WIDTH               : integer := 1;
        Constant CALC_ERR_WIDTH          : integer := 1;
        Constant SF_OFFSET_WIDTH         : integer := C_STRT_SF_OFFSET_WIDTH;
        
        Constant SF_OFFSET_FIFO_DEPTH    : integer := funct_size_drecntl_fifo(C_DRE_CNTL_FIFO_DEPTH,
                                                                              C_MAX_BURST_LEN);
        
        
        Constant SF_OFFSET_FIFO_WIDTH    : Integer := TAG_WIDTH        +  -- Tag field
                                                      DRR_WIDTH        +  -- DRE Re-alignment Request Flag Field
                                                      EOF_WIDTH        +  -- EOF flag field
                                                      CALC_ERR_WIDTH   +  -- Calc error flag
                                                      SF_OFFSET_WIDTH;    -- Store and Forward Offset

        Constant TAG_STRT_INDEX          : integer := 0;
        Constant DRR_STRT_INDEX          : integer := TAG_STRT_INDEX + TAG_WIDTH;
        Constant EOF_STRT_INDEX          : integer := DRR_STRT_INDEX + DRR_WIDTH;
        Constant CALC_ERR_STRT_INDEX     : integer := EOF_STRT_INDEX + EOF_WIDTH;
        Constant SF_OFFSET_STRT_INDEX    : integer := CALC_ERR_STRT_INDEX+CALC_ERR_WIDTH;
        
       
        
        -- Signal Declarations --------------------------------------------------------------------
        
        signal sig_offset_fifo_data_in   : std_logic_vector(SF_OFFSET_FIFO_WIDTH-1 downto 0) := (others => '0');
        signal sig_offset_fifo_data_out  : std_logic_vector(SF_OFFSET_FIFO_WIDTH-1 downto 0) := (others => '0');
        signal sig_offset_fifo_wr_valid  : std_logic := '0';
        signal sig_offset_fifo_wr_ready  : std_logic := '0';
        signal sig_offset_fifo_rd_valid  : std_logic := '0';
        signal sig_offset_fifo_rd_ready  : std_logic := '0';
             
        
        
        begin
      
          
          -- PCC DRE Command interface handshake 
          dre2mstr_cmd_ready        <= sig_offset_fifo_wr_ready ;
          sig_offset_fifo_wr_valid  <= mstr2dre_cmd_valid       ;
          
          
          -- No DRE so no controls
          sf2dre_new_align          <= '0';
          sf2dre_use_autodest       <= '0';
          sf2dre_src_align          <= (others => '0');
          sf2dre_dest_align         <= (others => '0');
          sf2dre_flush              <= '0';
      
          -- No DRE so no alignment values
          sig_curr_src_align_reg    <= (others => '0');
          sig_curr_dest_align_reg   <= (others => '0');
          
          
          
          -- Format the input data word for the Offset FIFO Queue
          sig_offset_fifo_data_in   <= mstr2dre_strt_offset  &  -- MS field
                                       mstr2dre_calc_error   &   
                                       mstr2dre_eof          &   
                                       mstr2dre_drr          &   
                                       mstr2dre_tag;             -- LS Field
          
          
          
          sig_cntl_fifo_has_data    <= sig_offset_fifo_rd_valid ;
          sig_offset_fifo_rd_ready  <= sig_get_cntl_fifo_data   ;
          
          
          -- Rip the output fifo data word
          sig_curr_tag_reg          <= sig_offset_fifo_data_out((TAG_STRT_INDEX+TAG_WIDTH)-1 downto TAG_STRT_INDEX);
          sig_curr_drr_reg          <= sig_offset_fifo_data_out(DRR_STRT_INDEX);
          sig_curr_eof_reg          <= sig_offset_fifo_data_out(EOF_STRT_INDEX);
          sig_curr_calc_error_reg   <= sig_offset_fifo_data_out(CALC_ERR_STRT_INDEX);
          sig_curr_strt_offset_reg  <= sig_offset_fifo_data_out((SF_OFFSET_STRT_INDEX+SF_OFFSET_WIDTH)-1 downto 
                                                                SF_OFFSET_STRT_INDEX);


          
          
          
          
               
          ------------------------------------------------------------
          -- Instance: I_DRE_CNTL_FIFO
          --
          -- Description:
          -- Instance for the Offset Control FIFO. This is still needed
          -- by the unpacker logic to get the starting offset at the
          -- begining of an input packet coming out of the Store and 
          -- Forward data FIFO.
          --
          ------------------------------------------------------------
          I_DRE_CNTL_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
          generic map (

            C_DWIDTH             =>  SF_OFFSET_FIFO_WIDTH   , 
            C_DEPTH              =>  SF_OFFSET_FIFO_DEPTH   , 
            C_IS_ASYNC           =>  USE_SYNC_FIFO          , 
            C_PRIM_TYPE          =>  SRL_FIFO_PRIM          , 
            C_FAMILY             =>  C_FAMILY                 

            )
          port map (

            -- Write Clock and reset
            fifo_wr_reset        =>  reset                    , 
            fifo_wr_clk          =>  aclk                     , 

            -- Write Side
            fifo_wr_tvalid       =>  sig_offset_fifo_wr_valid , 
            fifo_wr_tready       =>  sig_offset_fifo_wr_ready , 
            fifo_wr_tdata        =>  sig_offset_fifo_data_in  , 
            fifo_wr_full         =>  open                     , 


            -- Read Clock and reset
            fifo_async_rd_reset  =>  aclk                     , 
            fifo_async_rd_clk    =>  reset                    , 

            -- Read Side
            fifo_rd_tvalid       =>  sig_offset_fifo_rd_valid , 
            fifo_rd_tready       =>  sig_offset_fifo_rd_ready , 
            fifo_rd_tdata        =>  sig_offset_fifo_data_out , 
            fifo_rd_empty        =>  open                    

            );





      
        end generate OMIT_DRE_CNTL;
      
      
      
      
      
     
     
     
      
      
      
      ------------------------------------------------------------
      -- If Generate
      --
      -- Label: INCLUDE_DRE_CNTL
      --
      -- If Generate Description:
      --   This IfGen is used to include the DRE control logic and 
      -- Control FIFO when MM2S DRE is included in the MM2S.
      --
      --
      ------------------------------------------------------------
      INCLUDE_DRE_CNTL : if (C_DRE_IS_USED = 1) generate
      
        
        -- Constant Declarations
        
        Constant DRECNTL_FIFO_DEPTH    : integer := funct_size_drecntl_fifo(C_DRE_CNTL_FIFO_DEPTH,
                                                                            C_MAX_BURST_LEN);
        
        
        Constant DRECNTL_FIFO_WIDTH    : Integer := TAG_WIDTH        +  -- Tag field
                                                    SRC_ALIGN_WIDTH  +  -- Source align field width
                                                    DEST_ALIGN_WIDTH +  -- Dest align field width
                                                    DRR_WIDTH        +  -- DRE Re-alignment Request Flag Field
                                                    EOF_WIDTH        +  -- EOF flag field
                                                    CALC_ERR_WIDTH   +  -- Calc error flag
                                                    SF_OFFSET_WIDTH;    -- Store and Forward Offset

        Constant TAG_STRT_INDEX        : integer := 0;
        Constant SRC_ALIGN_STRT_INDEX  : integer := TAG_STRT_INDEX + TAG_WIDTH;
        Constant DEST_ALIGN_STRT_INDEX : integer := SRC_ALIGN_STRT_INDEX + SRC_ALIGN_WIDTH;
        Constant DRR_STRT_INDEX        : integer := DEST_ALIGN_STRT_INDEX + DEST_ALIGN_WIDTH;
        Constant EOF_STRT_INDEX        : integer := DRR_STRT_INDEX + DRR_WIDTH;
        Constant CALC_ERR_STRT_INDEX   : integer := EOF_STRT_INDEX + EOF_WIDTH;
        Constant SF_OFFSET_STRT_INDEX  : integer := CALC_ERR_STRT_INDEX+CALC_ERR_WIDTH;
        
        
        
        
        
        signal sig_cmd_fifo_data_in        : std_logic_vector(DRECNTL_FIFO_WIDTH-1 downto 0) := (others => '0');
        signal sig_cmd_fifo_data_out       : std_logic_vector(DRECNTL_FIFO_WIDTH-1 downto 0) := (others => '0');
        signal sig_fifo_wr_cmd_valid       : std_logic := '0';
        signal sig_fifo_wr_cmd_ready       : std_logic := '0';
        signal sig_fifo_rd_cmd_valid       : std_logic := '0';
        signal sig_fifo_rd_cmd_ready       : std_logic := '0';
             
        signal sig_dre_align_ready         : std_logic := '0';
        signal sig_dre_align_valid_reg     : std_logic := '0';
        signal sig_dre_use_autodest_reg    : std_logic := '0';
        signal sig_dre_src_align_reg       : std_logic_vector(SRC_ALIGN_WIDTH-1 downto 0)  := (others => '0');
        signal sig_dre_dest_align_reg      : std_logic_vector(DEST_ALIGN_WIDTH-1 downto 0) := (others => '0');
        signal sig_dre_flush_reg           : std_logic := '0';
             
        begin
 
 
          
          -- Assign the DRE Control Outputs
          sf2dre_new_align      <= sig_dre_align_valid_reg;
          sf2dre_use_autodest   <= sig_dre_use_autodest_reg;
          sf2dre_src_align      <= sig_dre_src_align_reg;
          sf2dre_dest_align     <= sig_dre_dest_align_reg;
          sf2dre_flush          <= sig_sf2dre_flush;      -- from RDC via data FIFO
      
 
          
          -- PCC DRE Command interface handshake 
          dre2mstr_cmd_ready    <= sig_fifo_wr_cmd_ready;
          sig_fifo_wr_cmd_valid <= mstr2dre_cmd_valid   ;
          
          -- Format the input data word for the DRE Control FIFO Queue
          sig_cmd_fifo_data_in  <=  mstr2dre_strt_offset    &
                                    mstr2dre_calc_error     &
                                    mstr2dre_eof            &   
                                    mstr2dre_drr            &   
                                    mstr2dre_dre_dest_align &   
                                    mstr2dre_dre_src_align  &   
                                    mstr2dre_tag;   
          
          
          
          
          
          -- Formulate the DRE Control FIFO Read signaling
          sig_cntl_fifo_has_data   <= sig_fifo_rd_cmd_valid  ;
          sig_fifo_rd_cmd_ready    <= sig_get_cntl_fifo_data ;
          
          
          
          
                                       
                              
          -- Rip the output fifo data word
          sig_curr_tag_reg         <= sig_cmd_fifo_data_out((TAG_STRT_INDEX+TAG_WIDTH)-1 downto TAG_STRT_INDEX);
          sig_curr_src_align_reg   <= sig_cmd_fifo_data_out((SRC_ALIGN_STRT_INDEX+SRC_ALIGN_WIDTH)-1 downto 
                                                            SRC_ALIGN_STRT_INDEX);
          sig_curr_dest_align_reg  <= sig_cmd_fifo_data_out((DEST_ALIGN_STRT_INDEX+DEST_ALIGN_WIDTH)-1 downto 
                                                            DEST_ALIGN_STRT_INDEX);
          sig_curr_drr_reg         <= sig_cmd_fifo_data_out(DRR_STRT_INDEX);
          sig_curr_eof_reg         <= sig_cmd_fifo_data_out(EOF_STRT_INDEX);
          sig_curr_calc_error_reg  <= sig_cmd_fifo_data_out(CALC_ERR_STRT_INDEX);
          sig_curr_strt_offset_reg <= sig_cmd_fifo_data_out((SF_OFFSET_STRT_INDEX+SF_OFFSET_WIDTH)-1 downto 
                                                            SF_OFFSET_STRT_INDEX);


               
                              
                              
                                       
          ------------------------------------------------------------
          -- Instance: I_DRE_CNTL_FIFO
          --
          -- Description:
          -- Instance for the DRE Control FIFO
          --
          ------------------------------------------------------------
          I_DRE_CNTL_FIFO : entity axi_datamover_v5_1_14.axi_datamover_fifo
          generic map (

            C_DWIDTH             =>  DRECNTL_FIFO_WIDTH     , 
            C_DEPTH              =>  DRECNTL_FIFO_DEPTH     , 
            C_IS_ASYNC           =>  USE_SYNC_FIFO          , 
            C_PRIM_TYPE          =>  SRL_FIFO_PRIM          , 
            C_FAMILY             =>  C_FAMILY                 

            )
          port map (

            -- Write Clock and reset
            fifo_wr_reset        =>   reset                 , 
            fifo_wr_clk          =>   aclk                  , 

            -- Write Side
            fifo_wr_tvalid       =>   sig_fifo_wr_cmd_valid , 
            fifo_wr_tready       =>   sig_fifo_wr_cmd_ready , 
            fifo_wr_tdata        =>   sig_cmd_fifo_data_in  , 
            fifo_wr_full         =>   open                  , 


            -- Read Clock and reset
            fifo_async_rd_reset  =>   aclk                  , 
            fifo_async_rd_clk    =>   reset                 , 

            -- Read Side
            fifo_rd_tvalid       =>   sig_fifo_rd_cmd_valid , 
            fifo_rd_tready       =>   sig_fifo_rd_cmd_ready , 
            fifo_rd_tdata        =>   sig_cmd_fifo_data_out , 
            fifo_rd_empty        =>   open                    

            );



          
  
          
           
           
          
          -------------------------------------------------------------------------
          -- DRE Control Register
          -------------------------------------------------------------------------
          
           
          
          -- The DRE will auto-flush on a received TLAST so a commanded Flush 
          -- is not needed.                                                             
          sig_dre_flush_reg         <= '0';
  
  

          -------------------------------------------------------------
          -- Synchronous Process with Sync Reset
          --
          -- Label: IMP_CNTL_REG
          --
          -- Process Description:
          --  Implements the DRE alignment Output Register.
          --
          -------------------------------------------------------------
          IMP_CNTL_REG : process (aclk)
            begin
              if (aclk'event and aclk = '1') then
                if (reset = '1') then
         
                  sig_dre_use_autodest_reg  <= '0'             ;
                  sig_dre_src_align_reg     <= (others => '0') ;
                  sig_dre_dest_align_reg    <= (others => '0') ;
                
                Elsif (sig_ld_dre_cntl_reg = '1' ) Then
                
                  sig_dre_use_autodest_reg  <= not(sig_curr_drr_reg)   ;
                  sig_dre_src_align_reg     <= sig_curr_src_align_reg  ;
                  sig_dre_dest_align_reg    <= sig_curr_dest_align_reg ;
                
                
                Elsif (sig_good_sout_strm_dbeat = '1') Then
                
                  sig_dre_use_autodest_reg  <= '0'             ;
                  sig_dre_src_align_reg     <= (others => '0') ;
                  sig_dre_dest_align_reg    <= (others => '0') ;
                
                else
         
                  null;  -- Hold Current State
         
                end if; 
              end if;       
            end process IMP_CNTL_REG; 
          
        
        
       
            
            
          -------------------------------------------------------------
          -- Synchronous Process with Sync Reset
          --
          -- Label: IMP_DRE_CNTL_VALID_REG
          --
          -- Process Description:
          --  Implements the DRE Alignment valid Register.
          --
          -------------------------------------------------------------
          IMP_DRE_CNTL_VALID_REG : process (aclk)
            begin
              if (aclk'event and aclk = '1') then
                if (reset = '1') then
         
                  sig_dre_align_valid_reg   <= '0' ; 
                
                Elsif (sig_ld_dre_cntl_reg = '1' ) Then
                
                  sig_dre_align_valid_reg   <= '1'  ; 
                
                
                Elsif (sig_good_sout_strm_dbeat = '1') Then
                
                  sig_dre_align_valid_reg   <= '0' ; 
                
                else
         
                  null;  -- Hold Current State
         
                end if; 
              end if;       
            end process IMP_DRE_CNTL_VALID_REG; 
          
        
        
 
      
        end generate INCLUDE_DRE_CNTL;
  
 
  
  
  
  
  
  
  
  
  
  
  
 
 ---------------------------------------------------------------- 
 -- Token Counter Logic  
 -- Predicting fifo space availability at some point in the  
 -- future is based on managing a virtual pool of transfer tokens.
 -- A token represents 1 max length burst worth of space in the
 -- Data FIFO. 
 ---------------------------------------------------------------- 
    
    
    -- calculate how many tokens are commited to pending transfers
    sig_tokens_commited <= TOKEN_CNT_MAX - sig_token_cntr;
    
    
    
    -- Decrement the token counter when a token is
    -- borrowed
    sig_decr_token_cntr <= '1'
      when (sig_rd_addr_posted = '1' and 
            sig_token_eq_zero  = '0')
      else '0';
    
    
    -- Increment the token counter when a  
    -- token is returned.
    sig_incr_token_cntr <= '1'
      when (sig_rd_xfer_cmplt = '1' and 
            sig_token_eq_max  = '0')
      else '0';
  
    
    
    -- Detect when the xfer token count is at max value
    sig_token_eq_max <= '1' 
     when (sig_token_cntr = TOKEN_CNT_MAX)
     Else '0';
  
    -- Detect when the xfer token count is at one
    sig_token_eq_one <= '1' 
     when (sig_token_cntr = TOKEN_CNT_ONE)
     Else '0';
  
    -- Detect when the xfer token count is at zero
    sig_token_eq_zero <= '1' 
     when (sig_token_cntr = TOKEN_CNT_ZERO)
     Else '0';
  
    -- Look ahead to see if the xfer token pool is going empty
    sig_taking_last_token <= '1'
      When (sig_token_eq_one   = '1' and
            sig_rd_addr_posted = '1')
      Else '0';
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_TOKEN_CNTR
    --
    -- Process Description:
    -- Implements the Token counter
    --
    -------------------------------------------------------------
    IMP_TOKEN_CNTR : process (aclk)
      begin
        if (aclk'event and aclk = '1') then
          if (reset  = '1' ) then 

            sig_token_cntr <= TOKEN_CNT_MAX;
            
          elsif (sig_incr_token_cntr = '1' and
                 sig_decr_token_cntr = '0') then

            sig_token_cntr <= sig_token_cntr + TOKEN_CNT_ONE;
            
          elsif (sig_incr_token_cntr = '0' and
                 sig_decr_token_cntr = '1') then

            sig_token_cntr <= sig_token_cntr - TOKEN_CNT_ONE;
            
          else
            null;  -- hold current value
          end if; 
        end if;       
      end process IMP_TOKEN_CNTR; 

 
     
     
 
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_TOKEN_AVAIL_FLAG
    --
    -- Process Description:
    --   Implements the flag indicating that the AXI Read Master
    -- can post a read address request on the AXI4 bus.
    --
    -- Read address posting can occur if:
    --
    --  - The write side LEN fifo is not empty.                   
    --  - The commited plus actual Data FIFO space is less than 
    --    the stall threshold (a max length read burst can fit 
    --    in the data FIFO without overflow).   
    --  - The max allowed commited read count has not been reached.      
    --
    -- The flag is cleared after each address has been posted to
    -- ensure a second unauthorized post does not occur.
    -------------------------------------------------------------
    IMP_TOKEN_AVAIL_FLAG : process (aclk)
      begin
        if (aclk'event and aclk = '1') then
           if (reset              = '1' or
               sig_rd_addr_posted = '1') then
    
             sig_ok_to_post_rd_addr <= '0';
    
           else
    
             sig_ok_to_post_rd_addr <= not(sig_stall_rd_addr_posts) and -- the commited Data FIFO space is approaching full 
                                       not(sig_token_eq_zero)       and -- max allowed pending reads has not been reached
                                       not(sig_taking_last_token);      -- the max allowed pending reads is about to be reached
    
           end if; 
        end if;       
      end process IMP_TOKEN_AVAIL_FLAG; 
 
 
 
  
  
    
    
 
 
  
  
    
    
 ---------------------------------------------------------------- 
 -- Data FIFO Logic ------------------------------------------
 ---------------------------------------------------------------- 
 
   
GEN_MM2S_TKEEP_ENABLE3 : if C_ENABLE_MM2S_TKEEP = 1 generate
begin

    
    -- FIFO Output ripping to components
    sig_dfifo_data_out       <=  sig_data_fifo_data_out(DATA_OUT_MSB_INDEX downto
                                                        DATA_OUT_LSB_INDEX);
     
    sig_dfifo_tkeep_out      <=  sig_data_fifo_data_out(TKEEP_OUT_MSB_INDEX downto
                                                        TKEEP_OUT_LSB_INDEX);
     
    sig_dfifo_tlast_out      <=  sig_data_fifo_data_out(TLAST_OUT_INDEX) ;
    
    sig_dfifo_cmd_cmplt_out  <=  sig_data_fifo_data_out(CMPLT_OUT_INDEX) ;
  
    sig_dfifo_dre_flush_out  <=  sig_data_fifo_data_out(DRE_FLUSH_OUT_INDEX) ;
  

end generate GEN_MM2S_TKEEP_ENABLE3;

GEN_MM2S_TKEEP_DISABLE3 : if C_ENABLE_MM2S_TKEEP = 0 generate
begin
    
    -- FIFO Output ripping to components
    sig_dfifo_data_out       <=  sig_data_fifo_data_out(DATA_OUT_MSB_INDEX downto
                                                        DATA_OUT_LSB_INDEX);
     
    sig_dfifo_tkeep_out      <=  (others => '1');
     
    sig_dfifo_tlast_out      <=  sig_data_fifo_data_out(TLAST_OUT_INDEX) ;
    
    sig_dfifo_cmd_cmplt_out  <=  sig_data_fifo_data_out(CMPLT_OUT_INDEX) ;
  
    sig_dfifo_dre_flush_out  <=  sig_data_fifo_data_out(DRE_FLUSH_OUT_INDEX) ;
  


end generate GEN_MM2S_TKEEP_DISABLE3;


 
  
    
    -- Stall Threshold calculations
    sig_fifo_wr_cnt_unsgnd   <= UNSIGNED(sig_data_fifo_wr_cnt);
 
    sig_wrcnt_mblen_slice    <= sig_fifo_wr_cnt_unsgnd(DATA_FIFO_CNT_WIDTH-1 downto 
                                                       DF_WRCNT_RIP_LS_INDEX);
    
    sig_commit_plus_actual   <= RESIZE(sig_tokens_commited, THRESH_COMPARE_WIDTH) +
                                RESIZE(sig_wrcnt_mblen_slice, THRESH_COMPARE_WIDTH);
    
    
    -- Compare the commited read space plus the actual used space against the
    -- stall threshold. Assert the read address posting stall flag if the
    -- threshold is met or exceeded.
    sig_stall_rd_addr_posts  <= '1'
      when (sig_commit_plus_actual > RD_ADDR_POST_STALL_THRESH_US)
      Else '0';
    
    
    
    
    -- FIFO Rd/WR Controls
    sig_push_data_fifo <= sig_good_sin_strm_dbeat;
    
    -- sig_pop_data_fifo  <= sig_sout2sf_tready and 
    --                       sig_data_fifo_dvalid;
    
    
      
GEN_MM2S_TKEEP_ENABLE2 : if C_ENABLE_MM2S_TKEEP = 1 generate
begin

 
    -- Concatonate the Stream inputs into the single FIFO data in value 
    sig_data_fifo_data_in <= data2sf_dre_flush &  -- ms Field
                             data2sf_cmd_cmplt &  
                             sin2sf_tlast      &
                             sin2sf_tkeep      & 
                             sin2sf_tdata;        -- ls field


end generate GEN_MM2S_TKEEP_ENABLE2;

GEN_MM2S_TKEEP_DISABLE2 : if C_ENABLE_MM2S_TKEEP = 0 generate
begin
 
    -- Concatonate the Stream inputs into the single FIFO data in value 
    sig_data_fifo_data_in <= data2sf_dre_flush &  -- ms Field
                             data2sf_cmd_cmplt &  
                             sin2sf_tlast      &
                             --sin2sf_tkeep      & 
                             sin2sf_tdata;        -- ls field



end generate GEN_MM2S_TKEEP_DISABLE2;

   
   
                                                    
    ------------------------------------------------------------
    -- Instance: I_DATA_FIFO 
    --
    -- Description:
    --  Implements the Store and Forward data FIFO (synchronous)   
    --
    ------------------------------------------------------------
    I_DATA_FIFO : entity axi_datamover_v5_1_14.axi_datamover_sfifo_autord
    generic map (

      C_DWIDTH                =>  DATA_FIFO_WIDTH       ,  
      C_DEPTH                 =>  DATA_FIFO_DEPTH       ,  
      C_DATA_CNT_WIDTH        =>  DATA_FIFO_CNT_WIDTH   ,  
      C_NEED_ALMOST_EMPTY     =>  NOT_NEEDED            ,  
      C_NEED_ALMOST_FULL      =>  NOT_NEEDED            ,  
      C_USE_BLKMEM            =>  BLK_MEM_FIFO          ,  
      C_FAMILY                =>  C_FAMILY                 

      )
    port map (

     -- Inputs 
      SFIFO_Sinit             =>  reset                  , 
      SFIFO_Clk               =>  aclk                   , 
      SFIFO_Wr_en             =>  sig_push_data_fifo     , 
      SFIFO_Din               =>  sig_data_fifo_data_in  , 
      SFIFO_Rd_en             =>  sig_pop_data_fifo      , 
      SFIFO_Clr_Rd_Data_Valid =>  LOGIC_LOW              , 
      
     -- Outputs
      SFIFO_DValid            =>  sig_data_fifo_dvalid   , 
      SFIFO_Dout              =>  sig_data_fifo_data_out , 
      SFIFO_Full              =>  sig_data_fifo_full     , 
      SFIFO_Empty             =>  open                   , 
      SFIFO_Almost_full       =>  open                   , 
      SFIFO_Almost_empty      =>  open                   , 
      SFIFO_Rd_count          =>  open                   ,  
      SFIFO_Rd_count_minus1   =>  open                   ,  
      SFIFO_Wr_count          =>  sig_data_fifo_wr_cnt   ,  
      SFIFO_Rd_ack            =>  open                     

    );



 
 
 
 
 
 
  
  
  end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_wr_sf.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_wr_sf.vhd
  --
  -- Description:     
  --    This file implements the AXI DataMover Write (S2MM) Store and Forward module. 
  --  The design utilizes the AXI DataMover's new address pipelining
  --  control function.  This module buffers write data and provides status and 
  --  control features such that the DataMover Write Master is only allowed 
  --  to post AXI WRite Requests if the associated write data needed to complete
  --  the Write Data transfer is present in the Data FIFO. In addition, the Write 
  --  side logic is such that Write transfer requests can be pipelined to the 
  --  AXI4 bus based on the Data FIFO contents but ahead of the actual Write Data
  --  transfers.
  -- 
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library lib_pkg_v1_0_2;
  library lib_srl_fifo_v1_0_2;
  use lib_pkg_v1_0_2.lib_pkg.all;
  use lib_pkg_v1_0_2.lib_pkg.clog2;
  use lib_srl_fifo_v1_0_2.srl_fifo_f;
  
 

  library axi_datamover_v5_1_14;
  use axi_datamover_v5_1_14.axi_datamover_sfifo_autord;

  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_wr_sf is
    generic (
      
      C_WR_ADDR_PIPE_DEPTH   : Integer range 1 to 30 := 4;
        -- This parameter indicates the depth of the DataMover
        -- write address pipelining queues for the Main data transport
        -- channels. The effective address pipelining on the AXI4 
        -- Write Address Channel will be the value assigned plus 2. 
      
      C_SF_FIFO_DEPTH        : Integer range 128 to 8192 := 512;
        -- Sets the desired depth of the internal Data FIFO.
      
    --   C_MAX_BURST_LEN        : Integer range  16 to  256 :=  16;
    --     -- Indicates the max burst length being used by the external
    --     -- AXI4 Master for each AXI4 transfer request.
        
    --   C_DRE_IS_USED          : Integer range   0 to    1 :=   0;
    --     -- Indicates if the external Master is utilizing a DRE on
    --     -- the stream input to this module.
         
      C_MMAP_DWIDTH          : Integer range   32 to  1024 := 64;
        -- Sets the AXI4 Memory Mapped Bus Data Width 
      
      C_STREAM_DWIDTH        : Integer range   8 to  1024 :=  16;
        -- Sets the Stream Data Width for the Input and Output
        -- Data streams.
  
      C_STRT_OFFSET_WIDTH    : Integer range   1 to 7 :=  2;
        -- Sets the bit width of the starting address offset port
        -- This should be set to log2(C_MMAP_DWIDTH/C_STREAM_DWIDTH)
        
      C_FAMILY               : String  := "virtex7"
        -- Indicates the target FPGA Family.
      
      );
    port (
      
      -- Clock and Reset inputs -----------------------------------------------       
                                                                             --       
      aclk                    : in  std_logic;                               --       
         -- Primary synchronization clock for the Master side                --       
         -- interface and internal logic. It is also used                    --       
         -- for the User interface synchronization when                      --       
         -- C_STSCMD_IS_ASYNC = 0.                                           --       
                                                                             --       
      -- Reset input                                                         --       
      reset                   : in  std_logic;                               --       
         -- Reset used for the internal syncronization logic                 --       
      -------------------------------------------------------------------------
      
      
    
    
      -- Slave Stream Input  ------------------------------------------------------------ 
                                                                                       -- 
      sf2sin_tready           : Out Std_logic;                                         -- 
        -- DRE  Stream READY input                                                     -- 
                                                                                       -- 
      sin2sf_tvalid           : In  std_logic;                                         -- 
        -- DRE Stream VALID Output                                                     -- 
                                                                                       -- 
      sin2sf_tdata            : In  std_logic_vector(C_STREAM_DWIDTH-1 downto 0);      --    
        -- DRE  Stream DATA input                                                      -- 
                                                                                       -- 
      sin2sf_tkeep            : In  std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);  --       
        -- DRE  Stream STRB input                                                      -- 
                                                                                       -- 
      sin2sf_tlast            : In  std_logic;                                         -- 
        -- DRE  Xfer LAST input                                                        --
                                                                                       -- 
      sin2sf_error            : In  std_logic;                                         -- 
        -- Stream Underrun/Overrun error input                                         --
      ----------------------------------------------------------------------------------- 
  
 
  
      -- Starting Address Offset Input  ------------------------------------------------- 
                                                                                       --
      sin2sf_strt_addr_offset : In std_logic_vector(C_STRT_OFFSET_WIDTH-1 downto 0);   -- 
        -- Used by Packing logic to set the initial data slice position for the        --
        -- packing operation. Packing is only needed if the MMap and Stream Data       --
        -- widths do not match.                                                        -- 
      ----------------------------------------------------------------------------------- 
               
                
      -- DataMover Write Side Address Pipelining Control Interface ---------------------- 
                                                                                       -- 
      ok_to_post_wr_addr      : Out  Std_logic;                                        -- 
        -- Indicates that the internal FIFO has enough data                            -- 
        -- physically present to supply one more max length                            -- 
        -- burst transfer or a completion burst                                        -- 
        -- (tlast asserted)                                                            -- 
                                                                                       -- 
      wr_addr_posted          : In std_logic;                                          -- 
        -- Indication that a write address has been posted to AXI4                     -- 
                                                                                       -- 
                                                                                       -- 
      wr_xfer_cmplt           : In  Std_logic;                                         -- 
        -- Indicates that the Datamover has completed a Write Data                     -- 
        -- transfer on the AXI4                                                        -- 
                                                                                       -- 
                                                                                       -- 
      wr_ld_nxt_len           : in  std_logic;                                         -- 
        -- Active high pulse indicating a new transfer LEN qualifier                   -- 
        -- has been queued to the DataMover Write Data Controller                      -- 
                                                                                       -- 
      wr_len                  : in  std_logic_vector(7 downto 0);                      -- 
        -- The actual LEN qualifier value that has been queued to the                  -- 
        -- DataMover Write Data Controller                                             --
      ----------------------------------------------------------------------------------- 


         
      
      -- Write Side Stream Out to DataMover S2MM ---------------------------------------- 
                                                                                       -- 
      sout2sf_tready          : In  std_logic;                                         -- 
        -- Write READY input from the Stream Master                                    -- 
                                                                                       -- 
      sf2sout_tvalid          : Out  std_logic;                                        -- 
        -- Write VALID output to the Stream Master                                     -- 
                                                                                       -- 
      sf2sout_tdata           : Out  std_logic_vector(C_MMAP_DWIDTH-1 downto 0);       -- 
        -- Write DATA output to the Stream Master                                      -- 
                                                                                       -- 
      sf2sout_tkeep           : Out  std_logic_vector((C_MMAP_DWIDTH/8)-1 downto 0);   -- 
        -- Write DATA output to the Stream Master                                      -- 
                                                                                       -- 
      sf2sout_tlast           : Out  std_logic;                                        -- 
        -- Write LAST output to the Stream Master                                      --
                                                                                       -- 
      sf2sout_error           : Out  std_logic                                         -- 
        -- Stream Underrun/Overrun error input                                         --
      ----------------------------------------------------------------------------------- 
     
 
      );
  
  end entity axi_datamover_wr_sf;
  
  
  architecture implementation of axi_datamover_wr_sf is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    
    -- Functions ---------------------------------------------------------------------------
 
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_pwr2_depth
    --
    -- Function Description:
    --  Rounds up to the next power of 2 depth value in an input
    --  range of 1 to 8192
    --
    -------------------------------------------------------------------
    function funct_get_pwr2_depth (min_depth : integer) return integer is
     
      Variable var_temp_depth : Integer := 16;
     
    begin
       
      
      if (min_depth = 1) then
      
         var_temp_depth := 1;
      
      elsif (min_depth  = 2) then
      
         var_temp_depth := 2;
      
      elsif (min_depth  <= 4) then
      
         var_temp_depth := 4;
      
      elsif (min_depth  <= 8) then
      
         var_temp_depth := 8;
      
      elsif (min_depth  <= 16) then
      
         var_temp_depth := 16;
      
      elsif (min_depth  <= 32) then
      
         var_temp_depth := 32;
      
      elsif (min_depth  <= 64) then
      
         var_temp_depth := 64;
      
      elsif (min_depth  <= 128) then
      
         var_temp_depth := 128;
      
      elsif (min_depth  <= 256) then
      
         var_temp_depth := 256;
      
      elsif (min_depth  <= 512) then
      
         var_temp_depth := 512;
      
      elsif (min_depth  <= 1024) then
      
         var_temp_depth := 1024;
      
      elsif (min_depth  <= 2048) then
      
         var_temp_depth := 2048;
      
      elsif (min_depth  <= 4096) then
      
         var_temp_depth := 4096;
      
      else -- assume 8192 depth
      
         var_temp_depth := 8192;
      
      end if;
      
       
       
      Return (var_temp_depth);
       
       
    end function funct_get_pwr2_depth;
    
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_fifo_cnt_width
    --
    -- Function Description:
    --   simple function to set the width of the data fifo read 
    -- and write count outputs. 
    -------------------------------------------------------------------
    function funct_get_fifo_cnt_width (fifo_depth : integer) 
             return integer is
    
      Variable temp_width : integer := 8;
    
    begin
    
      if (fifo_depth = 1) then
      
         temp_width := 1;
      
      elsif (fifo_depth  = 2) then
      
         temp_width := 2;
      
      elsif (fifo_depth  <= 4) then
      
         temp_width := 3;
      
      elsif (fifo_depth  <= 8) then
      
         temp_width := 4;
      
      elsif (fifo_depth  <= 16) then
      
         temp_width := 5;
      
      elsif (fifo_depth  <= 32) then
      
         temp_width := 6;
      
      elsif (fifo_depth  <= 64) then
      
         temp_width := 7;
      
      elsif (fifo_depth  <= 128) then
      
         temp_width := 8;
      
      elsif (fifo_depth  <= 256) then
      
         temp_width := 9;
      
      elsif (fifo_depth  <= 512) then
      
         temp_width := 10;
      
      elsif (fifo_depth  <= 1024) then
      
         temp_width := 11;
      
      elsif (fifo_depth  <= 2048) then
      
         temp_width := 12;
      
      elsif (fifo_depth  <= 4096) then
      
         temp_width := 13;
      
      else -- assume 8192 depth
      
         temp_width := 14;
      
      end if;
      
      
      Return (temp_width);
     
    
    end function funct_get_fifo_cnt_width;
    
   
   
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_cntr_width
    --
    -- Function Description:
    --  This function calculates the needed counter bit width from the 
    -- number of count sates needed (input).
    --
    -------------------------------------------------------------------
    function funct_get_cntr_width (num_cnt_values : integer) return integer is
    
      Variable temp_cnt_width : Integer := 0;
    
    begin
    
      
      if (num_cnt_values <= 2) then
      
        temp_cnt_width := 1;
      
      elsif (num_cnt_values <= 4) then
      
        temp_cnt_width := 2;
      
      elsif (num_cnt_values <= 8) then
      
        temp_cnt_width := 3;
      
      elsif (num_cnt_values <= 16) then
      
        temp_cnt_width := 4;
      
      elsif (num_cnt_values <= 32) then
      
        temp_cnt_width := 5;
      
      elsif (num_cnt_values <= 64) then
      
        temp_cnt_width := 6;
      
      elsif (num_cnt_values <= 128) then
      
        temp_cnt_width := 7;
      
      else
      
        temp_cnt_width := 8;
      
      end if;
      
      
      
      
      Return (temp_cnt_width);
      
      
    end function funct_get_cntr_width;
    

    
    
    
    -- Constants ---------------------------------------------------------------------------
    
    Constant LOGIC_LOW                 : std_logic := '0';
    Constant LOGIC_HIGH                : std_logic := '1';
    
    Constant BLK_MEM_FIFO              : integer := 1;
    Constant SRL_FIFO                  : integer := 0;
    Constant NOT_NEEDED                : integer := 0;
    
    
    Constant WSTB_WIDTH                : integer := C_MMAP_DWIDTH/8; -- bits
    Constant TLAST_WIDTH               : integer := 1;               -- bits
    Constant EOP_ERR_WIDTH             : integer := 1;               -- bits
    
    
    
    Constant DATA_FIFO_DEPTH           : integer := C_SF_FIFO_DEPTH;
    Constant DATA_FIFO_CNT_WIDTH       : integer := funct_get_fifo_cnt_width(DATA_FIFO_DEPTH);
    -- Constant DF_WRCNT_RIP_LS_INDEX     : integer := funct_get_wrcnt_lsrip(C_MAX_BURST_LEN);
    
     
    Constant DATA_FIFO_WIDTH           : integer := C_MMAP_DWIDTH +
                                                    --WSTB_WIDTH     +
                                                    TLAST_WIDTH   +
                                                    EOP_ERR_WIDTH;
    
    Constant DATA_OUT_MSB_INDEX        : integer := C_MMAP_DWIDTH-1;
    Constant DATA_OUT_LSB_INDEX        : integer := 0;
    
   --  Constant TSTRB_OUT_LSB_INDEX       : integer := DATA_OUT_MSB_INDEX+1;
   --  Constant TSTRB_OUT_MSB_INDEX       : integer := (TSTRB_OUT_LSB_INDEX+WSTB_WIDTH)-1;
    
   -- Constant TLAST_OUT_INDEX           : integer := TSTRB_OUT_MSB_INDEX+1;
    Constant TLAST_OUT_INDEX           : integer := DATA_OUT_MSB_INDEX+1;
    
    Constant EOP_ERR_OUT_INDEX         : integer := TLAST_OUT_INDEX+1;
    
    
    Constant WR_LEN_FIFO_DWIDTH        : integer := 8;
    Constant WR_LEN_FIFO_DEPTH         : integer := funct_get_pwr2_depth(C_WR_ADDR_PIPE_DEPTH + 2);
    
    Constant LEN_CNTR_WIDTH            : integer := 8;
    Constant LEN_CNT_ZERO              : Unsigned(LEN_CNTR_WIDTH-1 downto 0) := 
                                         TO_UNSIGNED(0, LEN_CNTR_WIDTH);
    Constant LEN_CNT_ONE               : Unsigned(LEN_CNTR_WIDTH-1 downto 0) := 
                                         TO_UNSIGNED(1, LEN_CNTR_WIDTH);
    
    Constant WR_XFER_CNTR_WIDTH        : integer := 8;
    Constant WR_XFER_CNT_ZERO          : Unsigned(WR_XFER_CNTR_WIDTH-1 downto 0) := 
                                         TO_UNSIGNED(0, WR_XFER_CNTR_WIDTH);
    Constant WR_XFER_CNT_ONE           : Unsigned(WR_XFER_CNTR_WIDTH-1 downto 0) := 
                                         TO_UNSIGNED(1, WR_XFER_CNTR_WIDTH);
    
    Constant UNCOM_WRCNT_1             : Unsigned(DATA_FIFO_CNT_WIDTH-1 downto 0) := 
                                         TO_UNSIGNED(1, DATA_FIFO_CNT_WIDTH);
    
    Constant UNCOM_WRCNT_0             : Unsigned(DATA_FIFO_CNT_WIDTH-1 downto 0) := 
                                         TO_UNSIGNED(0, DATA_FIFO_CNT_WIDTH);
    
    
    
    
    
    
    
    
    -- Signals ---------------------------------------------------------------------------
    
    
    signal sig_good_sin_strm_dbeat    : std_logic := '0';
    signal sig_strm_sin_ready         : std_logic := '0';
    
    signal sig_sout2sf_tready         : std_logic := '0';
    signal sig_sf2sout_tvalid         : std_logic := '0';
    signal sig_sf2sout_tdata          : std_logic_vector(C_MMAP_DWIDTH-1 downto 0) := (others => '0');
    signal sig_sf2sout_tkeep          : std_logic_vector(WSTB_WIDTH-1 downto 0) := (others => '0');
    signal sig_sf2sout_tlast          : std_logic := '0';
    
    signal sig_push_data_fifo         : std_logic := '0';
    signal sig_pop_data_fifo          : std_logic := '0';
    signal sig_data_fifo_full         : std_logic := '0';
    signal sig_data_fifo_data_in      : std_logic_vector(DATA_FIFO_WIDTH-1 downto 0) := (others => '0');
    signal sig_data_fifo_dvalid       : std_logic := '0';
    signal sig_data_fifo_data_out     : std_logic_vector(DATA_FIFO_WIDTH-1 downto 0) := (others => '0');
    
    signal sig_ok_to_post_wr_addr     : std_logic := '0';
    signal sig_wr_addr_posted         : std_logic := '0';
    signal sig_wr_xfer_cmplt          : std_logic := '0';
    
    signal sig_wr_ld_nxt_len          : std_logic := '0';
    signal sig_push_len_fifo          : std_logic := '0';
    signal sig_pop_len_fifo           : std_logic := '0';
    signal sig_len_fifo_full          : std_logic := '0';
    signal sig_len_fifo_empty         : std_logic := '0';
    signal sig_len_fifo_data_in       : std_logic_vector(WR_LEN_FIFO_DWIDTH-1 downto 0) := (others => '0');
    signal sig_len_fifo_data_out      : std_logic_vector(WR_LEN_FIFO_DWIDTH-1 downto 0) := (others => '0');
    signal sig_len_fifo_len_out_un    : unsigned(WR_LEN_FIFO_DWIDTH-1 downto 0) := (others => '0');
  
    signal sig_uncom_wrcnt            : unsigned(DATA_FIFO_CNT_WIDTH-1 downto 0) := (others => '0');
    signal sig_sub_len_uncom_wrcnt    : std_logic := '0';
    signal sig_incr_uncom_wrcnt       : std_logic := '0';
    signal sig_resized_fifo_len       : unsigned(DATA_FIFO_CNT_WIDTH-1 downto 0) := (others => '0');
    signal sig_num_wr_dbeats_needed   : unsigned(DATA_FIFO_CNT_WIDTH-1 downto 0) := (others => '0');
    signal sig_enough_dbeats_rcvd     : std_logic := '0';
                    
    signal sig_sf2sout_eop_err_out    : std_logic := '0';
    
    signal sig_good_fifo_write        : std_logic := '0';
    
  
  begin --(architecture implementation)
  
   
   
    -- Write Side (S2MM) Control Flags port connections
    ok_to_post_wr_addr       <= sig_ok_to_post_wr_addr ;
    sig_wr_addr_posted       <= wr_addr_posted         ;
    sig_wr_xfer_cmplt        <= wr_xfer_cmplt          ;
  
    sig_wr_ld_nxt_len        <= wr_ld_nxt_len          ;
    sig_len_fifo_data_in     <= wr_len                 ;
  
    
    
    --  Output Stream Port connections
    sig_sout2sf_tready       <= sout2sf_tready          ;
    sf2sout_tvalid           <= sig_sf2sout_tvalid      ;
    sf2sout_tdata            <= sig_sf2sout_tdata       ; 
    sf2sout_tkeep            <= sig_sf2sout_tkeep       ;
    sf2sout_tlast            <= sig_sf2sout_tlast and
                                sig_sf2sout_tvalid      ;
    sf2sout_error            <= sig_sf2sout_eop_err_out ;
                               
                               
    
    -- Input Stream port connections 
    sf2sin_tready            <= sig_strm_sin_ready;
    
                                                          
    sig_good_sin_strm_dbeat  <= sin2sf_tvalid and
                                sig_strm_sin_ready;
                               
 
  
    
    
 ---------------------------------------------------------------- 
 -- Packing Logic      ------------------------------------------
 ---------------------------------------------------------------- 
 
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: OMIT_PACKING
    --
    -- If Generate Description:
    --    Omits any packing logic in the Store and Forward module.
    -- The Stream and MMap data widths are the same.
    --
    ------------------------------------------------------------
    OMIT_PACKING : if (C_MMAP_DWIDTH = C_STREAM_DWIDTH) generate
    
       begin
    
        
         sig_good_fifo_write   <= sig_good_sin_strm_dbeat;
         
         sig_strm_sin_ready    <= not(sig_data_fifo_full); 
         
         
         sig_push_data_fifo    <= sig_good_sin_strm_dbeat;
       
          
         -- Concatonate the Stream inputs into the single FIFO data in value 
         sig_data_fifo_data_in <= sin2sf_error &
                                  sin2sf_tlast &
                                --  sin2sf_tkeep & 
                                  sin2sf_tdata;
       
       end generate OMIT_PACKING;
     
     
    
    
  
  
  
  
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: INCLUDE_PACKING
    --
    -- If Generate Description:
    --    Includes packing logic in the Store and Forward module.
    -- The MMap Data bus is wider than the Stream width.
    --
    ------------------------------------------------------------
    INCLUDE_PACKING : if (C_MMAP_DWIDTH > C_STREAM_DWIDTH) generate
    
      Constant MMAP2STRM_WIDTH_RATO  : integer := C_MMAP_DWIDTH/C_STREAM_DWIDTH;
    
      Constant DATA_SLICE_WIDTH      : integer := C_STREAM_DWIDTH;
      
      Constant FLAG_SLICE_WIDTH      : integer := TLAST_WIDTH     + 
                                                  EOP_ERR_WIDTH;
      
      
      
      
      
      
      Constant OFFSET_CNTR_WIDTH     : integer := funct_get_cntr_width(MMAP2STRM_WIDTH_RATO);
      
      Constant OFFSET_CNT_ONE        : unsigned(OFFSET_CNTR_WIDTH-1 downto 0) := 
                                       TO_UNSIGNED(1, OFFSET_CNTR_WIDTH);
      
      Constant OFFSET_CNT_MAX        : unsigned(OFFSET_CNTR_WIDTH-1 downto 0) := 
                                       TO_UNSIGNED(MMAP2STRM_WIDTH_RATO-1, OFFSET_CNTR_WIDTH);
      
      
      
      
      -- Types -----------------------------------------------------------------------------
      type lsig_data_slice_type is array(MMAP2STRM_WIDTH_RATO-1 downto 0) of
                    std_logic_vector(DATA_SLICE_WIDTH-1 downto 0);

      type lsig_flag_slice_type is array(MMAP2STRM_WIDTH_RATO-1 downto 0) of
                    std_logic_vector(FLAG_SLICE_WIDTH-1 downto 0);


       
      -- local signals
      
      signal lsig_data_slice_reg      : lsig_data_slice_type;
      signal lsig_flag_slice_reg      : lsig_flag_slice_type;
      
       
      signal lsig_reg_segment         : std_logic_vector(DATA_SLICE_WIDTH-1 downto 0)   := (others => '0');
      signal lsig_segment_ld          : std_logic_vector(MMAP2STRM_WIDTH_RATO-1 downto 0) := (others => '0');
      signal lsig_segment_clr         : std_logic_vector(MMAP2STRM_WIDTH_RATO-1 downto 0) := (others => '0');
      
      signal lsig_0ffset_to_to_use    : unsigned(OFFSET_CNTR_WIDTH-1 downto 0) := (others => '0');
      
      signal lsig_0ffset_cntr         : unsigned(OFFSET_CNTR_WIDTH-1 downto 0) := (others => '0');
      signal lsig_ld_offset           : std_logic := '0';
      signal lsig_incr_offset         : std_logic := '0';
      signal lsig_offset_cntr_eq_max  : std_logic := '0';
      
      signal lsig_combined_data       : std_logic_vector(C_MMAP_DWIDTH-1 downto 0) := (others => '0');
      
      
      signal lsig_tlast_or            : std_logic := '0';
      signal lsig_eop_err_or          : std_logic := '0';
      
      signal lsig_partial_tlast_or    : std_logic_vector(MMAP2STRM_WIDTH_RATO downto 0) := (others => '0');
      signal lsig_partial_eop_err_or  : std_logic_vector(MMAP2STRM_WIDTH_RATO downto 0) := (others => '0');
      
      signal lsig_packer_full         : std_logic := '0';
      signal lsig_packer_empty        : std_logic := '0';
      signal lsig_set_packer_full     : std_logic := '0';
      signal lsig_good_push2fifo      : std_logic := '0';
      signal lsig_first_dbeat         : std_logic := '0';
      
        
        
        
      begin
    
       
       -- Assign the flag indicating that a fifo write is going
       -- to occur at the next rising clock edge.
       sig_good_fifo_write     <=  lsig_good_push2fifo;
       
       
       -- Generate the stream ready
       sig_strm_sin_ready       <= not(lsig_packer_full) or
                                   lsig_good_push2fifo ; 
       
       
       -- Format the FIFO input data 
       sig_data_fifo_data_in   <= lsig_eop_err_or    &   -- MS Bit
                                  lsig_tlast_or      &
                                  lsig_combined_data  ;  -- LS Bits
        
       
       -- Generate a write to the Data FIFO input
       sig_push_data_fifo      <= lsig_packer_full;
       
       
       -- Generate a flag indicating a write to the DataFIFO 
       -- is going to complete 
       lsig_good_push2fifo    <=  lsig_packer_full and
                                  not(sig_data_fifo_full);
       
       -- Generate the control that loads the starting address
       -- offset for the next input packet
       lsig_ld_offset          <= lsig_first_dbeat and
                                  sig_good_sin_strm_dbeat;
                                  
       -- Generate the control for incrementing the offset counter
       lsig_incr_offset        <= sig_good_sin_strm_dbeat;
       
       
       -- Generate a flag indicating the packer input register
       -- array is full or has loaded the last data beat of
       -- the input paket
       lsig_set_packer_full    <=  sig_good_sin_strm_dbeat  and
                                   (sin2sf_tlast            or 
                                    lsig_offset_cntr_eq_max);

       -- Check to see if the offset counter has reached its max
       -- value
       lsig_offset_cntr_eq_max <=  '1'
         --when  (lsig_0ffset_cntr = OFFSET_CNT_MAX)
         when  (lsig_0ffset_to_to_use = OFFSET_CNT_MAX)
         Else '0';
       
       
       -- Mux between the input start offset and the offset counter
       -- output to use for the packer slice load control.  
       lsig_0ffset_to_to_use <= UNSIGNED(sin2sf_strt_addr_offset) 
         when (lsig_first_dbeat = '1')
         Else lsig_0ffset_cntr;
       
        
        
       -------------------------------------------------------------
       -- Synchronous Process with Sync Reset
       --
       -- Label: IMP_OFFSET_LD_MARKER
       --
       -- Process Description:
       --  Implements the flop indicating the first databeat of
       -- an input data packet.
       --
       -------------------------------------------------------------
       IMP_OFFSET_LD_MARKER : process (aclk)
         begin
           if (aclk'event and aclk = '1') then
              if (reset = '1') then
       
                lsig_first_dbeat <= '1';
       
              elsif (sig_good_sin_strm_dbeat = '1' and
                     sin2sf_tlast            = '0') then
       
                lsig_first_dbeat <= '0';
       
              Elsif (sig_good_sin_strm_dbeat = '1' and
                     sin2sf_tlast            = '1') Then
              
                lsig_first_dbeat <= '1';
              
              else
       
                null;  -- Hold Current State
       
              end if; 
           end if;       
         end process IMP_OFFSET_LD_MARKER; 
       
       
       
       
       
       
       -------------------------------------------------------------
       -- Synchronous Process with Sync Reset
       --
       -- Label: IMP_OFFSET_CNTR
       --
       -- Process Description:
       --  Implements the address offset counter that is used to 
       -- steer the data loads into the packer register slices.
       -- Note that the counter has to be loaded with the starting
       -- offset plus one to sync up with the data input.
       -------------------------------------------------------------
       IMP_OFFSET_CNTR : process (aclk)
         begin
           if (aclk'event and aclk = '1') then
              if (reset = '1') then
       
                lsig_0ffset_cntr <= (others => '0');
       
              Elsif (lsig_ld_offset = '1') Then
              
               lsig_0ffset_cntr <= UNSIGNED(sin2sf_strt_addr_offset) + OFFSET_CNT_ONE;
              
              elsif (lsig_incr_offset = '1') then
       
                lsig_0ffset_cntr <= lsig_0ffset_cntr + OFFSET_CNT_ONE;
       
              else
       
                null;  -- Hold Current State
       
              end if; 
           end if;       
         end process IMP_OFFSET_CNTR; 
       
       
       
       
       
       -------------------------------------------------------------
       -- Synchronous Process with Sync Reset
       --
       -- Label: IMP_PACK_REG_FULL
       --
       -- Process Description:
       --   Implements the Packer Register full/empty flags
       --
       -------------------------------------------------------------
       IMP_PACK_REG_FULL : process (aclk)
         begin
           if (aclk'event and aclk = '1') then
              if (reset = '1') then
       
                lsig_packer_full  <= '0';
                lsig_packer_empty <= '1';
       
              Elsif (lsig_set_packer_full = '1' and
                     lsig_packer_full     = '0') Then
              
                lsig_packer_full  <= '1';
                lsig_packer_empty <= '0';
       
              elsif (lsig_set_packer_full = '0' and
                     lsig_good_push2fifo  = '1') then
              
                lsig_packer_full  <= '0';
                lsig_packer_empty <= '1';
       
              else
       
                null;  -- Hold Current State
       
              end if; 
           end if;       
         end process IMP_PACK_REG_FULL; 
       
       
       
       
       
       
       ------------------------------------------------------------
       -- For Generate
       --
       -- Label: DO_REG_SLICES
       --
       -- For Generate Description:
       --
       --  Implements the Packng Register Slices
       --
       --
       ------------------------------------------------------------
       DO_REG_SLICES : for slice_index in 0 to MMAP2STRM_WIDTH_RATO-1 generate


       
       begin
       
        
         -- generate the register load enable for each slice segment based
         -- on the address offset count value
         lsig_segment_ld(slice_index) <= '1'
           when (sig_good_sin_strm_dbeat = '1' and
                TO_INTEGER(lsig_0ffset_to_to_use) = slice_index)
           Else '0';
         
         
        
         -------------------------------------------------------------
         -- Synchronous Process with Sync Reset
         --
         -- Label: IMP_DATA_SLICE
         --
         -- Process Description:
         --   Implement a data register slice for the packer. 
         --
         -------------------------------------------------------------
         IMP_DATA_SLICE : process (aclk)
           begin
             if (aclk'event and aclk = '1') then
               if (reset = '1') then
        
                 lsig_data_slice_reg(slice_index) <= (others => '0');
        
               elsif (lsig_segment_ld(slice_index) = '1') then
        
                 lsig_data_slice_reg(slice_index) <= sin2sf_tdata;
        
               -- optional clear of slice reg 
               elsif (lsig_segment_ld(slice_index) = '0' and
                      lsig_good_push2fifo          = '1') then

                 lsig_data_slice_reg(slice_index) <= (others => '0');
        
               else
        
                 null;  -- Hold Current State
        
               end if; 
             end if;       
           end process IMP_DATA_SLICE; 
        
        
         
         -------------------------------------------------------------
         -- Synchronous Process with Sync Reset
         --
         -- Label: IMP_FLAG_SLICE
         --
         -- Process Description:
         --   Implement a flag register slice for the packer.
         --
         -------------------------------------------------------------
         IMP_FLAG_SLICE : process (aclk)
           begin
             if (aclk'event and aclk = '1') then
               if (reset = '1') then
        
                 lsig_flag_slice_reg(slice_index) <= (others => '0');
        
               elsif (lsig_segment_ld(slice_index) = '1') then
        
                 lsig_flag_slice_reg(slice_index) <= sin2sf_tlast & -- bit 1
                                                     sin2sf_error;  -- bit 0
               
               elsif (lsig_segment_ld(slice_index) = '0' and
                      lsig_good_push2fifo          = '1') then

                 lsig_flag_slice_reg(slice_index) <= (others => '0');
        
               else
        
                 null;  -- Hold Current State
        
               end if; 
             end if;       
           end process IMP_FLAG_SLICE; 
        
        
        
         
       end generate DO_REG_SLICES;
       
       
        
        
        
        
                                                                                
       -- Do the OR functions of the Flags -------------------------------------
       lsig_tlast_or   <= lsig_partial_tlast_or(MMAP2STRM_WIDTH_RATO-1) ;
       lsig_eop_err_or <= lsig_partial_eop_err_or(MMAP2STRM_WIDTH_RATO-1);
       
       lsig_partial_tlast_or(0)   <= lsig_flag_slice_reg(0)(1);
       lsig_partial_eop_err_or(0) <= lsig_flag_slice_reg(0)(0);
       

       
       ------------------------------------------------------------
       -- For Generate
       --
       -- Label: DO_FLAG_OR
       --
       -- For Generate Description:
       --  Implement the OR of the TLAST and EOP Error flags.
       --
       --
       --
       ------------------------------------------------------------
       DO_FLAG_OR : for slice_index in 1 to MMAP2STRM_WIDTH_RATO-1 generate
       
       begin
     
          lsig_partial_tlast_or(slice_index)   <= lsig_partial_tlast_or(slice_index-1) or
                                                  --lsig_partial_tlast_or(slice_index);
                                                  lsig_flag_slice_reg(slice_index)(1);
                                                  
                                                  
                                                  
                                                  
          lsig_partial_eop_err_or(slice_index) <= lsig_partial_eop_err_or(slice_index-1) or
                                                  --lsig_partial_eop_err_or(slice_index); 
                                                  lsig_flag_slice_reg(slice_index)(0);
       
       end generate DO_FLAG_OR;

     
        
        
     
       ------------------------------------------------------------
       -- For Generate
       --
       -- Label: DO_DATA_COMBINER
       --
       -- For Generate Description:
       --   Combines the Data Slice register outputs into a single
       -- vector for input to the Data FIFO.
       --
       --
       ------------------------------------------------------------
       DO_DATA_COMBINER : for slice_index in 1 to MMAP2STRM_WIDTH_RATO generate
       
       begin
        
         lsig_combined_data((slice_index*DATA_SLICE_WIDTH)-1 downto 
                            (slice_index-1)*DATA_SLICE_WIDTH) <=
                            lsig_data_slice_reg(slice_index-1);
        
        
       end generate DO_DATA_COMBINER;
     
     
     
     
     
       
       
      end generate INCLUDE_PACKING;
     
     
    
    
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
  
     
 ---------------------------------------------------------------- 
 -- Data FIFO Logic    ------------------------------------------
 ---------------------------------------------------------------- 
 
     
     -- FIFO Input attachments
     
     -- sig_push_data_fifo <= sig_good_sin_strm_dbeat;
     
     
     -- -- Concatonate the Stream inputs into the single FIFO data in value 
     -- sig_data_fifo_data_in <= sin2sf_error &
     --                          sin2sf_tlast &
     --                          sin2sf_tkeep & 
     --                          sin2sf_tdata;
 
    
    
     -- FIFO Output to output stream attachments
     sig_sf2sout_tvalid      <=  sig_data_fifo_dvalid ;
     
     sig_sf2sout_tdata       <=  sig_data_fifo_data_out(DATA_OUT_MSB_INDEX downto
                                                        DATA_OUT_LSB_INDEX);
     
     -- sig_sf2sout_tkeep       <=  sig_data_fifo_data_out(TSTRB_OUT_MSB_INDEX downto
     --                                                    TSTRB_OUT_LSB_INDEX);
     
     -- When this Store and Forward is enabled, the Write Data Controller ignores the 
     -- TKEEP input so this is not sent through the FIFO.
     sig_sf2sout_tkeep       <=  (others => '1');
     
     
     
     sig_sf2sout_tlast       <=  sig_data_fifo_data_out(TLAST_OUT_INDEX) ;
     
     sig_sf2sout_eop_err_out <=  sig_data_fifo_data_out(EOP_ERR_OUT_INDEX) ;
     
     
     -- FIFO Rd/WR Controls
     
     
     sig_pop_data_fifo  <= sig_sout2sf_tready and 
                           sig_data_fifo_dvalid;
     
                                                     
    ------------------------------------------------------------
    -- Instance: I_DATA_FIFO 
    --
    -- Description:
    --  Implements the Store and Forward data FIFO (synchronous)   
    --
    ------------------------------------------------------------
    I_DATA_FIFO : entity axi_datamover_v5_1_14.axi_datamover_sfifo_autord
    generic map (

      C_DWIDTH                =>  DATA_FIFO_WIDTH       ,  
      C_DEPTH                 =>  DATA_FIFO_DEPTH       ,  
      C_DATA_CNT_WIDTH        =>  DATA_FIFO_CNT_WIDTH   ,  
      C_NEED_ALMOST_EMPTY     =>  NOT_NEEDED            ,  
      C_NEED_ALMOST_FULL      =>  NOT_NEEDED            ,  
      C_USE_BLKMEM            =>  BLK_MEM_FIFO          ,  
      C_FAMILY                =>  C_FAMILY                 

      )
    port map (

     -- Inputs 
      SFIFO_Sinit             =>  reset                  , 
      SFIFO_Clk               =>  aclk                   , 
      SFIFO_Wr_en             =>  sig_push_data_fifo     , 
      SFIFO_Din               =>  sig_data_fifo_data_in  , 
      SFIFO_Rd_en             =>  sig_pop_data_fifo      , 
      SFIFO_Clr_Rd_Data_Valid =>  LOGIC_LOW              , 
      
     -- Outputs
      SFIFO_DValid            =>  sig_data_fifo_dvalid   , 
      SFIFO_Dout              =>  sig_data_fifo_data_out , 
      SFIFO_Full              =>  sig_data_fifo_full     , 
      SFIFO_Empty             =>  open                   , 
      SFIFO_Almost_full       =>  open                   , 
      SFIFO_Almost_empty      =>  open                   , 
      SFIFO_Rd_count          =>  open                   ,  
      SFIFO_Rd_count_minus1   =>  open                   ,  
      SFIFO_Wr_count          =>  open                   ,  
      SFIFO_Rd_ack            =>  open                     

    );



 
 
 
 
 
 
 
 
 
 
 
-------------------------------------------------------------------- 
-- Write Side Control Logic  
--------------------------------------------------------------------

    -- Convert the LEN fifo data output to unsigned
    sig_len_fifo_len_out_un <= unsigned(sig_len_fifo_data_out);
   
    -- Resize the unsigned LEN output to the Data FIFO writecount width
    sig_resized_fifo_len    <= RESIZE(sig_len_fifo_len_out_un , DATA_FIFO_CNT_WIDTH);
   
    
    -- The actual number of databeats needed for the queued write transfer
    -- is the current LEN fifo output plus 1.
    sig_num_wr_dbeats_needed <= sig_resized_fifo_len + UNCOM_WRCNT_1;
   
   
    -- Compare the uncommited receved data beat count to that needed
    -- for the next queued write request.
    sig_enough_dbeats_rcvd <= '1'
      When (sig_num_wr_dbeats_needed <= sig_uncom_wrcnt)
      else '0';
    
    
    
    
    -- Increment the uncommited databeat counter on a good input
    -- stream databeat (Read Side of SF)
   -- sig_incr_uncom_wrcnt    <=  sig_good_sin_strm_dbeat;
    sig_incr_uncom_wrcnt    <=  sig_good_fifo_write;
   

    -- Subtract the current number of databeats needed from the
    -- uncommited databeat counter when the associated transfer
    -- address/qualifiers have been posted to the AXI Write 
    -- Address Channel
    sig_sub_len_uncom_wrcnt <= sig_wr_addr_posted;
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_UNCOM_DBEAT_CNTR
    --
    -- Process Description:
    -- Implements the counter that keeps track of the received read
    -- data beat count that has not been commited to a transfer on  
    -- the write side with a Write Address posting.
    --
    -------------------------------------------------------------
    IMP_UNCOM_DBEAT_CNTR : process (aclk)
      begin
        if (aclk'event and aclk = '1') then
          if (reset            = '1') then 

            sig_uncom_wrcnt <= UNCOM_WRCNT_0;
            
          elsif (sig_incr_uncom_wrcnt    = '1' and
                 sig_sub_len_uncom_wrcnt = '1') then

            sig_uncom_wrcnt <= sig_uncom_wrcnt - sig_resized_fifo_len;
            
          elsif (sig_incr_uncom_wrcnt    = '1' and
                 sig_sub_len_uncom_wrcnt = '0') then

            sig_uncom_wrcnt <= sig_uncom_wrcnt + UNCOM_WRCNT_1;
            
          elsif (sig_incr_uncom_wrcnt    = '0' and
                 sig_sub_len_uncom_wrcnt = '1') then

            sig_uncom_wrcnt <= sig_uncom_wrcnt - sig_num_wr_dbeats_needed;
            
          else
            null;  -- hold current value
          end if; 
        end if;       
      end process IMP_UNCOM_DBEAT_CNTR; 
    
    
    





  
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_WR_ADDR_POST_FLAG
   --
   -- Process Description:
   --   Implements the flag indicating that the pending write
   -- transfer's data beat count has been received on the input
   -- side of the Data FIFO. This means the Write side can post
   -- the associated write address to the AXI4 bus and the 
   -- associated write data transfer can complete without CDMA
   -- throttling the Write Data Channel.     
   --
   -- The flag is cleared immediately after an address is posted
   -- to prohibit a second unauthorized posting while the control
   -- logic stabilizes to the next LEN FIFO value
   --.
   -------------------------------------------------------------
   IMP_WR_ADDR_POST_FLAG : process (aclk)
     begin
       if (aclk'event and aclk = '1') then
          if (reset              = '1' or
              sig_wr_addr_posted = '1') then
   
            sig_ok_to_post_wr_addr <= '0';
   
          else
   
            sig_ok_to_post_wr_addr <= not(sig_len_fifo_empty) and
                                      sig_enough_dbeats_rcvd; 
   
          end if; 
       end if;       
     end process IMP_WR_ADDR_POST_FLAG; 


 
   
   
   
   
   
   -------------------------------------------------------------
   -- LEN FIFO logic 
   -- The LEN FIFO stores the xfer lengths needed for each queued 
   -- write transfer in the DataMover S2MM Write Data Controller.  
    
   sig_push_len_fifo    <= sig_wr_ld_nxt_len and
                           not(sig_len_fifo_full);


   sig_pop_len_fifo     <= wr_addr_posted and
                           not(sig_len_fifo_empty);
  
 
 


   ------------------------------------------------------------
   -- Instance: I_WR_LEN_FIFO 
   --
   -- Description:
   -- Implement the LEN FIFO using SRL FIFO elements    
   --
   ------------------------------------------------------------
   I_WR_LEN_FIFO : entity lib_srl_fifo_v1_0_2.srl_fifo_f
   generic map (

     C_DWIDTH      =>  WR_LEN_FIFO_DWIDTH   ,  
     C_DEPTH       =>  WR_LEN_FIFO_DEPTH    ,  
     C_FAMILY      =>  C_FAMILY      

     )
   port map (

     Clk           =>  aclk                  ,  
     Reset         =>  reset                 ,  
     FIFO_Write    =>  sig_push_len_fifo     ,  
     Data_In       =>  sig_len_fifo_data_in  ,  
     FIFO_Read     =>  sig_pop_len_fifo      ,  
     Data_Out      =>  sig_len_fifo_data_out ,  
     FIFO_Empty    =>  sig_len_fifo_empty    ,  
     FIFO_Full     =>  sig_len_fifo_full     ,  
     Addr          =>  open                
 
     );

    

   
     
 
 
 
  
  
  end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_stbs_set.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_stbs_set.vhd
  --
  -- Description:     
  --    This file implements a module to count the number of strobe bits that 
  --    are asserted active high on the input strobe bus. This module does not
  --    support sparse strobe assertions.              
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  


  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_stbs_set is
    generic (
      
      C_STROBE_WIDTH    : Integer range 1 to 128 := 8
        -- Specifies the width (in bits) of the input strobe bus.
      
      );
    port (
      
      -- Input Strobe bus ----------------------------------------------------
                                                                            --
      tstrb_in          : in  std_logic_vector(C_STROBE_WIDTH-1 downto 0);  --
      ------------------------------------------------------------------------
      
      
      -- Asserted Strobes count output ---------------------------------------
                                                                            --
      num_stbs_asserted : Out std_logic_vector(7 downto 0)                  --
        -- Indicates the number of asserted tstrb_in bits                   --
      ------------------------------------------------------------------------
     
      );
  
  end entity axi_datamover_stbs_set;
  
  
  architecture implementation of axi_datamover_stbs_set is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    -- Function
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_8bit_stbs_set
    --
    -- Function Description:
    --  Implements an 8-bit lookup table for calculating the number
    -- of asserted bits within an 8-bit strobe vector.
    --
    -- Note that this function assumes that asserted strobes are 
    -- contiguous with each other (no sparse strobe assertions). 
    --
    -------------------------------------------------------------------
    function funct_8bit_stbs_set (strb_8 : std_logic_vector(7 downto 0)) return unsigned is
    
      Constant ASSERTED_VALUE_WIDTH : integer := 4;-- 4 bits needed
      
      
      Variable lvar_num_set : Integer range 0 to 8 := 0;
    
    begin
    
      case strb_8 is
        
        -------  1 bit --------------------------
        when "00000001" | "00000010" | "00000100" | "00001000" | 
             "00010000" | "00100000" | "01000000" | "10000000" =>
        
          lvar_num_set := 1;
        
        
        -------  2 bit --------------------------
        when "00000011" | "00000110" | "00001100" | "00011000" | 
             "00110000" | "01100000" | "11000000"  =>
        
          lvar_num_set := 2;
        
        
        -------  3 bit --------------------------
        when "00000111" | "00001110" | "00011100" | "00111000" | 
             "01110000" | "11100000"   =>
        
          lvar_num_set := 3;
        
        
        -------  4 bit --------------------------
        when "00001111" | "00011110" | "00111100" | "01111000" | 
             "11110000"    =>
        
          lvar_num_set := 4;
        
        
        -------  5 bit --------------------------
        when "00011111" | "00111110" | "01111100" | "11111000"  =>
        
          lvar_num_set := 5;
        
        
        -------  6 bit --------------------------
        when "00111111" | "01111110" | "11111100"  =>
        
          lvar_num_set := 6;
        
        
        -------  7 bit --------------------------
        when "01111111" | "11111110"   =>
        
          lvar_num_set := 7;
        
        
        -------  8 bit --------------------------
        when "11111111"    =>
        
          lvar_num_set := 8;
        
        
        ------- all zeros or sparse strobes ------
        When others =>  
        
          lvar_num_set := 0;
        
      end case;
      
      
      Return (TO_UNSIGNED(lvar_num_set, ASSERTED_VALUE_WIDTH));
       
       
      
    end function funct_8bit_stbs_set;
    
    
    
    
    
    
    -- Constants
    
    Constant LOGIC_LOW              : std_logic := '0';
    Constant LOGIC_HIGH             : std_logic := '1';
    Constant BITS_FOR_STBS_ASSERTED : integer := 8; -- increments of 8 bits
    Constant NUM_ZEROS_WIDTH        : integer := BITS_FOR_STBS_ASSERTED;
    
    
    -- Signals
    
    signal sig_strb_input           : std_logic_vector(C_STROBE_WIDTH-1 downto 0) := (others => '0');
    signal sig_stbs_asserted        : std_logic_vector(BITS_FOR_STBS_ASSERTED-1 downto 0) := (others => '0');


    
    
  begin --(architecture implementation)
  
   
   num_stbs_asserted     <= sig_stbs_asserted;
   
   sig_strb_input        <= tstrb_in         ;
    
    
    
    
 
 
    -------------------------------------------------------------------------
    ----------------  Asserted TSTRB calculation logic  --------------------- 
    -------------------------------------------------------------------------
    
    
 
   
     
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_1_STRB
     --
     -- If Generate Description:
     --   1-bit strobe bus width case
     --
     --
     ------------------------------------------------------------
     GEN_1_STRB : if (C_STROBE_WIDTH = 1) generate
     
     
        begin
     
          -------------------------------------------------------------
          -- Combinational Process
          --
          -- Label: IMP_1BIT_STRB
          --
          -- Process Description:
          --
          --
          -------------------------------------------------------------
          IMP_1BIT_STRB : process (sig_strb_input)
             begin
          
               
               -- Concatonate the strobe to the ls bit of
               -- the asserted value
               sig_stbs_asserted <= "0000000" &
                                    sig_strb_input(0);
          
             end process IMP_1BIT_STRB; 
        
        end generate GEN_1_STRB;
   
   
   
   
   
   
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_2_STRB
     --
     -- If Generate Description:
     --   2-bit strobe bus width case
     --
     --
     ------------------------------------------------------------
     GEN_2_STRB : if (C_STROBE_WIDTH = 2) generate
     
     
        signal lsig_num_set     : integer range 0 to 2 := 0;
        signal lsig_strb_vect   : std_logic_vector(1 downto 0) := (others => '0');
        
        begin
     
          
          lsig_strb_vect <=  sig_strb_input;
          
          
          -------------------------------------------------------------
          -- Combinational Process
          --
          -- Label: IMP_2BIT_STRB
          --
          -- Process Description:
          --  Calculates the number of strobes set fo the 2-bit 
          -- strobe case
          --
          -------------------------------------------------------------
          IMP_2BIT_STRB : process (lsig_strb_vect)
             begin
              
               case lsig_strb_vect is
                 when "01" | "10" =>
                   lsig_num_set <= 1;
                 when "11" =>
                   lsig_num_set <= 2;
                 when others =>
                   lsig_num_set <= 0;
               end case;
               
             end process IMP_2BIT_STRB; 
             
          
          sig_stbs_asserted <= STD_LOGIC_VECTOR(TO_UNSIGNED(lsig_num_set,
                                                            BITS_FOR_STBS_ASSERTED));
     
        
        end generate GEN_2_STRB;
   
   
   
   
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_4_STRB
     --
     -- If Generate Description:
     --   4-bit strobe bus width case
     --
     --
     ------------------------------------------------------------
     GEN_4_STRB : if (C_STROBE_WIDTH = 4) generate
     
     
       signal lsig_strb_vect   : std_logic_vector(7 downto 0) := (others => '0');
        
       begin
     
          
         lsig_strb_vect <=  "0000" & sig_strb_input; -- make and 8-bit vector 
                                                     -- for the function call
          
          
         sig_stbs_asserted <= STD_LOGIC_VECTOR(RESIZE(funct_8bit_stbs_set(lsig_strb_vect),
                                                      BITS_FOR_STBS_ASSERTED));
     
     
       end generate GEN_4_STRB;
   
   
  
  
   
   
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_8_STRB
     --
     -- If Generate Description:
     --   8-bit strobe bus width case
     --
     --
     ------------------------------------------------------------
     GEN_8_STRB : if (C_STROBE_WIDTH = 8) generate
     
     
       signal lsig_strb_vect   : std_logic_vector(7 downto 0) := (others => '0');
        
       begin
     
          
         lsig_strb_vect <=  sig_strb_input; -- make and 8-bit vector 
                                            -- for the function call
          
          
         sig_stbs_asserted <= STD_LOGIC_VECTOR(RESIZE(funct_8bit_stbs_set(lsig_strb_vect),
                                                           BITS_FOR_STBS_ASSERTED));
     
     
       end generate GEN_8_STRB;
   
   
   
   
   
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_16_STRB
     --
     -- If Generate Description:
     --   16-bit strobe bus width case
     --
     --
     ------------------------------------------------------------
     GEN_16_STRB : if (C_STROBE_WIDTH = 16) generate
     
       Constant RESULT_BIT_WIDTH : integer := 8;
       
       signal lsig_strb_vect1    : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect2    : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_num_in_stbs1  : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs2  : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_total     : unsigned(RESULT_BIT_WIDTH-1 downto 0) := (others => '0');
        
       begin
     
          
         lsig_strb_vect1   <=  sig_strb_input(7 downto 0); -- make and 8-bit vector 
                                                           -- for the function call
          
         lsig_strb_vect2   <=  sig_strb_input(15 downto 8); -- make and 8-bit vector 
                                                            -- for the function call
          
          
         lsig_num_in_stbs1 <=  funct_8bit_stbs_set(lsig_strb_vect1) ;
          
         lsig_num_in_stbs2 <=  funct_8bit_stbs_set(lsig_strb_vect2) ;
          
          
         lsig_num_total    <= RESIZE(lsig_num_in_stbs1 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs2 , RESULT_BIT_WIDTH);
          
          
          
         sig_stbs_asserted <= STD_LOGIC_VECTOR(lsig_num_total);
     
     
       end generate GEN_16_STRB;
   
   
   
   
   
   
   
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_32_STRB
     --
     -- If Generate Description:
     --   32-bit strobe bus width case
     --
     --
     ------------------------------------------------------------
     GEN_32_STRB : if (C_STROBE_WIDTH = 32) generate
     
       Constant RESULT_BIT_WIDTH : integer := 8;
       
       signal lsig_strb_vect1   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect2   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect3   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect4   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_num_in_stbs1 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs2 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs3 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs4 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_total    : unsigned(RESULT_BIT_WIDTH-1 downto 0) := (others => '0');
        
       begin
     
          
         lsig_strb_vect1   <=  sig_strb_input(7 downto 0);   -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect2   <=  sig_strb_input(15 downto 8);  -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect3   <=  sig_strb_input(23 downto 16); -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect4   <=  sig_strb_input(31 downto 24); -- make and 8-bit vector 
                                                       -- for the function call
          
          
         lsig_num_in_stbs1 <=  funct_8bit_stbs_set(lsig_strb_vect1) ;
          
         lsig_num_in_stbs2 <=  funct_8bit_stbs_set(lsig_strb_vect2) ;
         
         lsig_num_in_stbs3 <=  funct_8bit_stbs_set(lsig_strb_vect3) ;
         
         lsig_num_in_stbs4 <=  funct_8bit_stbs_set(lsig_strb_vect4) ;
          
          
         lsig_num_total    <= RESIZE(lsig_num_in_stbs1 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs2 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs3 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs4 , RESULT_BIT_WIDTH);
          
          
          
         sig_stbs_asserted <= STD_LOGIC_VECTOR(lsig_num_total);
     
     
       end generate GEN_32_STRB;
   
   
   
 
 
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_64_STRB
     --
     -- If Generate Description:
     --   64-bit strobe bus width case
     --
     --
     ------------------------------------------------------------
     GEN_64_STRB : if (C_STROBE_WIDTH = 64) generate
     
       Constant RESULT_BIT_WIDTH : integer := 8;
       
       signal lsig_strb_vect1   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect2   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect3   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect4   : std_logic_vector(7 downto 0) := (others => '0');
       
       signal lsig_strb_vect5   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect6   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect7   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect8   : std_logic_vector(7 downto 0) := (others => '0');
       
       
       
       signal lsig_num_in_stbs1 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs2 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs3 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs4 : unsigned(3 downto 0) := (others => '0');
       
       signal lsig_num_in_stbs5 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs6 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs7 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs8 : unsigned(3 downto 0) := (others => '0');
       
       signal lsig_num_total    : unsigned(RESULT_BIT_WIDTH-1 downto 0) := (others => '0');
        
       begin
     
          
         lsig_strb_vect1   <=  sig_strb_input(7 downto 0);   -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect2   <=  sig_strb_input(15 downto 8);  -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect3   <=  sig_strb_input(23 downto 16); -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect4   <=  sig_strb_input(31 downto 24); -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect5   <=  sig_strb_input(39 downto 32);   -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect6   <=  sig_strb_input(47 downto 40);  -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect7   <=  sig_strb_input(55 downto 48); -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect8   <=  sig_strb_input(63 downto 56); -- make and 8-bit vector 
                                                       -- for the function call
          
          
          
         lsig_num_in_stbs1 <=  funct_8bit_stbs_set(lsig_strb_vect1) ;
          
         lsig_num_in_stbs2 <=  funct_8bit_stbs_set(lsig_strb_vect2) ;
         
         lsig_num_in_stbs3 <=  funct_8bit_stbs_set(lsig_strb_vect3) ;
         
         lsig_num_in_stbs4 <=  funct_8bit_stbs_set(lsig_strb_vect4) ;
   
   
         lsig_num_in_stbs5 <=  funct_8bit_stbs_set(lsig_strb_vect5) ;
   
         lsig_num_in_stbs6 <=  funct_8bit_stbs_set(lsig_strb_vect6) ;
   
         lsig_num_in_stbs7 <=  funct_8bit_stbs_set(lsig_strb_vect7) ;
   
         lsig_num_in_stbs8 <=  funct_8bit_stbs_set(lsig_strb_vect8) ;
   
          
          
         lsig_num_total    <= RESIZE(lsig_num_in_stbs1 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs2 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs3 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs4 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs5 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs6 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs7 , RESULT_BIT_WIDTH) +
                              RESIZE(lsig_num_in_stbs8 , RESULT_BIT_WIDTH);
          
          
          
         sig_stbs_asserted <= STD_LOGIC_VECTOR(lsig_num_total);
     
     
       end generate GEN_64_STRB;
   
   
   
 
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_128_STRB
     --
     -- If Generate Description:
     --   128-bit strobe bus width case
     --
     --
     ------------------------------------------------------------
     GEN_128_STRB : if (C_STROBE_WIDTH = 128) generate
     
       Constant RESULT_BIT_WIDTH : integer := 8;
       
       signal lsig_strb_vect1    : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect2    : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect3    : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect4    : std_logic_vector(7 downto 0) := (others => '0');
       
       signal lsig_strb_vect5    : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect6    : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect7    : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect8    : std_logic_vector(7 downto 0) := (others => '0');
       
       signal lsig_strb_vect9    : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect10   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect11   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect12   : std_logic_vector(7 downto 0) := (others => '0');
       
       signal lsig_strb_vect13   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect14   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect15   : std_logic_vector(7 downto 0) := (others => '0');
       signal lsig_strb_vect16   : std_logic_vector(7 downto 0) := (others => '0');
       
       
       
       signal lsig_num_in_stbs1  : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs2  : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs3  : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs4  : unsigned(3 downto 0) := (others => '0');
       
       signal lsig_num_in_stbs5  : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs6  : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs7  : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs8  : unsigned(3 downto 0) := (others => '0');
       
       signal lsig_num_in_stbs9  : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs10 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs11 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs12 : unsigned(3 downto 0) := (others => '0');
       
       signal lsig_num_in_stbs13 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs14 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs15 : unsigned(3 downto 0) := (others => '0');
       signal lsig_num_in_stbs16 : unsigned(3 downto 0) := (others => '0');
       
       signal lsig_num_total     : unsigned(RESULT_BIT_WIDTH-1 downto 0) := (others => '0');
        
       begin
     
          
         lsig_strb_vect1   <=  sig_strb_input(7 downto 0);   -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect2   <=  sig_strb_input(15 downto 8);  -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect3   <=  sig_strb_input(23 downto 16); -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect4   <=  sig_strb_input(31 downto 24); -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect5   <=  sig_strb_input(39 downto 32);   -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect6   <=  sig_strb_input(47 downto 40);  -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect7   <=  sig_strb_input(55 downto 48); -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect8   <=  sig_strb_input(63 downto 56); -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect9   <=  sig_strb_input(71 downto 64);   -- make and 8-bit vector 
                                                       -- for the function call
          
         lsig_strb_vect10  <=  sig_strb_input(79 downto 72);  -- make and 8-bit vector 
                                                      -- for the function call
          
         lsig_strb_vect11  <=  sig_strb_input(87 downto 80); -- make and 8-bit vector 
                                                      -- for the function call
          
         lsig_strb_vect12  <=  sig_strb_input(95 downto 88); -- make and 8-bit vector 
                                                      -- for the function call
          
         lsig_strb_vect13  <=  sig_strb_input(103 downto 96);   -- make and 8-bit vector 
                                                      -- for the function call
          
         lsig_strb_vect14  <=  sig_strb_input(111 downto 104);  -- make and 8-bit vector 
                                                      -- for the function call
          
         lsig_strb_vect15  <=  sig_strb_input(119 downto 112); -- make and 8-bit vector 
                                                      -- for the function call
          
         lsig_strb_vect16  <=  sig_strb_input(127 downto 120); -- make and 8-bit vector 
                                                       -- for the function call
          
          
          
         lsig_num_in_stbs1  <=  funct_8bit_stbs_set(lsig_strb_vect1) ;
          
         lsig_num_in_stbs2  <=  funct_8bit_stbs_set(lsig_strb_vect2) ;
         
         lsig_num_in_stbs3  <=  funct_8bit_stbs_set(lsig_strb_vect3) ;
         
         lsig_num_in_stbs4  <=  funct_8bit_stbs_set(lsig_strb_vect4) ;
   
   
         lsig_num_in_stbs5  <=  funct_8bit_stbs_set(lsig_strb_vect5) ;
   
         lsig_num_in_stbs6  <=  funct_8bit_stbs_set(lsig_strb_vect6) ;
   
         lsig_num_in_stbs7  <=  funct_8bit_stbs_set(lsig_strb_vect7) ;
   
         lsig_num_in_stbs8  <=  funct_8bit_stbs_set(lsig_strb_vect8) ;
         
   
         lsig_num_in_stbs9  <=  funct_8bit_stbs_set(lsig_strb_vect9) ;
          
         lsig_num_in_stbs10 <=  funct_8bit_stbs_set(lsig_strb_vect10) ;
         
         lsig_num_in_stbs11 <=  funct_8bit_stbs_set(lsig_strb_vect11) ;
         
         lsig_num_in_stbs12 <=  funct_8bit_stbs_set(lsig_strb_vect12) ;
   
   
         lsig_num_in_stbs13 <=  funct_8bit_stbs_set(lsig_strb_vect13) ;
   
         lsig_num_in_stbs14 <=  funct_8bit_stbs_set(lsig_strb_vect14) ;
   
         lsig_num_in_stbs15 <=  funct_8bit_stbs_set(lsig_strb_vect15) ;
   
         lsig_num_in_stbs16 <=  funct_8bit_stbs_set(lsig_strb_vect16) ;
   
          
          
         lsig_num_total     <= RESIZE(lsig_num_in_stbs1  , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs2  , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs3  , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs4  , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs5  , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs6  , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs7  , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs8  , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs9  , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs10 , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs11 , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs12 , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs13 , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs14 , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs15 , RESULT_BIT_WIDTH) +
                               RESIZE(lsig_num_in_stbs16 , RESULT_BIT_WIDTH);
          
          
          
         sig_stbs_asserted  <= STD_LOGIC_VECTOR(lsig_num_total);
     
     
       end generate GEN_128_STRB;
   
  
  
  end implementation;


  -------------------------------------------------------------------------------
  -- axi_datamover_stbs_set_nodre.vhd
  -------------------------------------------------------------------------------
  --
  -- *************************************************************************
  --                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_datamover_stbs_set_nodre.vhd
  --
  -- Description:     
  --    This file implements a module to count the number of strobe bits that 
  --    are asserted active high on the input strobe bus. This module does not
  --    support sparse strobe assertions.              
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  


  
  -------------------------------------------------------------------------------
  
  entity axi_datamover_stbs_set_nodre is
    generic (
      
      C_STROBE_WIDTH    : Integer range 1 to 128 := 8
        -- Specifies the width (in bits) of the input strobe bus.
      
      );
    port (
      
      -- Input Strobe bus ----------------------------------------------------
                                                                            --
      tstrb_in          : in  std_logic_vector(C_STROBE_WIDTH-1 downto 0);  --
      ------------------------------------------------------------------------
      
      
      -- Asserted Strobes count output ---------------------------------------
                                                                            --
      num_stbs_asserted : Out std_logic_vector(7 downto 0)                  --
        -- Indicates the number of asserted tstrb_in bits                   --
      ------------------------------------------------------------------------
     
      );
  
  end entity axi_datamover_stbs_set_nodre;
  
  
  architecture implementation of axi_datamover_stbs_set_nodre is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    -- Function
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_8bit_stbs_set
    --
    -- Function Description:
    --  Implements an 8-bit lookup table for calculating the number
    -- of asserted bits within an 8-bit strobe vector.
    --
    -- Note that this function assumes that asserted strobes are 
    -- contiguous with each other (no sparse strobe assertions). 
    --
    -------------------------------------------------------------------
    function funct_8bit_stbs_set (strb_8 : std_logic_vector(7 downto 0)) return unsigned is
    
      Constant ASSERTED_VALUE_WIDTH : integer := 4;-- 4 bits needed
      
      
      Variable lvar_num_set : Integer range 0 to 8 := 0;
    
    begin
    
      case strb_8 is
        
        -------  1 bit --------------------------
        when "00000001" =>
        
          lvar_num_set := 1;
        
        
        -------  2 bit --------------------------
        when "00000011" =>
        
          lvar_num_set := 2;
        
        
        -------  3 bit --------------------------
        when "00000111" =>
        
          lvar_num_set := 3;
        
        
        -------  4 bit --------------------------
        when "00001111" =>
             
        
          lvar_num_set := 4;
        
        
        -------  5 bit --------------------------
        when "00011111" =>
        
          lvar_num_set := 5;
        
        
        -------  6 bit --------------------------
        when "00111111" =>
        
          lvar_num_set := 6;
        
        
        -------  7 bit --------------------------
        when "01111111" =>
        
          lvar_num_set := 7;
        
        
        -------  8 bit --------------------------
        when "11111111" =>
        
          lvar_num_set := 8;
        
        
        ------- all zeros or sparse strobes ------
        When others =>  
        
          lvar_num_set := 0;
        
      end case;
      
      
      Return (TO_UNSIGNED(lvar_num_set, ASSERTED_VALUE_WIDTH));
       
       
      
    end function funct_8bit_stbs_set;


    function funct_256bit_stbs_set (strb_3 : std_logic_vector(2 downto 0)) return unsigned is
    
      Constant ASSERTED_VALUE_WIDTH : integer := 5;-- 4 bits needed
      
      
      Variable lvar_num_set : Integer range 0 to 24 := 0;
    
    begin
    
      case strb_3 is
        
--        when "0000000" =>
        
--          lvar_num_set := 0;
        -------  1 bit --------------------------
        when "001" =>
        
          lvar_num_set := 8;
        
        -------  2 bit --------------------------
        when "011" =>
        
          lvar_num_set := 16;
        
        -------  3 bit --------------------------
        when "111" =>
        
          lvar_num_set := 24;

        ------- all zeros or sparse strobes ------
        When others =>  
        
          lvar_num_set := 0;
        
      end case;
      
      
      Return (TO_UNSIGNED(lvar_num_set, ASSERTED_VALUE_WIDTH));
       
       
      
    end function funct_256bit_stbs_set;




    function funct_512bit_stbs_set (strb_3 : std_logic_vector(6 downto 0)) return unsigned is
    
      Constant ASSERTED_VALUE_WIDTH : integer := 6;-- 4 bits needed
      
      
      Variable lvar_num_set : Integer range 0 to 56 := 0;
    
    begin
    
      case strb_3 is
        
--        when "0000000" =>
        
--          lvar_num_set := 0;
        -------  1 bit --------------------------
        when "0000001" =>
        
          lvar_num_set := 8;
        
        -------  2 bit --------------------------
        when "0000011" =>
        
          lvar_num_set := 16;
        
        -------  3 bit --------------------------
        when "0000111" =>
        
          lvar_num_set := 24;

        when "0001111" =>
        
          lvar_num_set := 32;

        when "0011111" =>
        
          lvar_num_set := 40;

        when "0111111" =>
        
          lvar_num_set := 48;

        when "1111111" =>
        
          lvar_num_set := 56;

        ------- all zeros or sparse strobes ------
        When others =>  
        
          lvar_num_set := 0;
        
      end case;
      
      
      Return (TO_UNSIGNED(lvar_num_set, ASSERTED_VALUE_WIDTH));
       
       
      
    end function funct_512bit_stbs_set;


    function funct_1024bit_stbs_set (strb_3 : std_logic_vector(14 downto 0)) return unsigned is
    
      Constant ASSERTED_VALUE_WIDTH : integer := 7;-- 4 bits needed
      
      
      Variable lvar_num_set : Integer range 0 to 120 := 0;
    
    begin
    
      case strb_3 is
        
        -------  1 bit --------------------------
        when "000000000000001" =>
        
          lvar_num_set := 8;
        
        -------  2 bit --------------------------
        when "000000000000011" =>
        
          lvar_num_set := 16;
        
        -------  3 bit --------------------------
        when "000000000000111" =>
        
          lvar_num_set := 24;

        when "000000000001111" =>
        
          lvar_num_set := 32;

        when "000000000011111" =>
        
          lvar_num_set := 40;

        when "000000000111111" =>
        
          lvar_num_set := 48;

        when "000000001111111" =>
        
          lvar_num_set := 56;

        when "000000011111111" =>
        
          lvar_num_set := 64;

        when "000000111111111" =>
        
          lvar_num_set := 72;

        when "000001111111111" =>
        
          lvar_num_set := 80;

        when "000011111111111" =>
        
          lvar_num_set := 88;

        when "000111111111111" =>
        
          lvar_num_set := 96;

        when "001111111111111" =>
        
          lvar_num_set := 104;

        when "011111111111111" =>
        
          lvar_num_set := 112;

        when "111111111111111" =>
        
          lvar_num_set := 120;

        ------- all zeros or sparse strobes ------
        When others =>  
        
          lvar_num_set := 0;
        
      end case;
      
      
      Return (TO_UNSIGNED(lvar_num_set, ASSERTED_VALUE_WIDTH));
       
       
      
    end function funct_1024bit_stbs_set;


--    function funct_8bit_stbs_set (strb_8 : std_logic_vector(7 downto 0)) return unsigned is
--    
--      Constant ASSERTED_VALUE_WIDTH : integer := 4;-- 4 bits needed
--      
--      
--      Variable lvar_num_set : Integer range 0 to 8 := 0;
--    
--    begin
--    
--      case strb_8 is
--        
----        -------  1 bit --------------------------
--        when "00000001" | "00000010" | "00000100" | "00001000" | 
--             "00010000" | "00100000" | "01000000" | "10000000" =>
--        
--          lvar_num_set := 1;
--        
--        
--        -------  2 bit --------------------------
--        when "00000011" | "00000110" | "00001100" | "00011000" | 
--             "00110000" | "01100000" | "11000000"  =>
--        
--          lvar_num_set := 2;
--        
--        
--        -------  3 bit --------------------------
--        when "00000111" | "00001110" | "00011100" | "00111000" | 
--             "01110000" | "11100000"   =>
--        
--          lvar_num_set := 3;
--        
--        
--        --